<?php
session_start();

// Check authentication
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

header('Content-Type: application/json');

$dataFile = 'data.json';

function readData() {
    global $dataFile;
    if (!file_exists($dataFile)) {
        return ['categories' => [], 'tasks' => []];
    }
    $content = file_get_contents($dataFile);
    return json_decode($content, true) ?: ['categories' => [], 'tasks' => []];
}

function writeData($data) {
    global $dataFile;
    return file_put_contents($dataFile, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function generateId($prefix = 'item') {
    return $prefix . '_' . uniqid();
}

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

try {
    switch ($method) {
        case 'GET':
            if ($action === 'data') {
                echo json_encode(readData());
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid action']);
            }
            break;

        case 'POST':
            $input = json_decode(file_get_contents('php://input'), true);
            $data = readData();

            switch ($action) {
                case 'add_category':
                    $category = [
                        'id' => generateId('cat'),
                        'name' => $input['name'],
                        'icon' => $input['icon'] ?? '📁',
                        'color' => $input['color'] ?? '#3b82f6'
                    ];
                    $data['categories'][] = $category;
                    writeData($data);
                    echo json_encode($category);
                    break;

                case 'add_task':
                    $task = [
                        'id' => generateId('task'),
                        'title' => $input['title'],
                        'description' => $input['description'] ?? '',
                        'category' => $input['category'],
                        'priority' => $input['priority'] ?? 'medium',
                        'completed' => false,
                        'created_at' => date('c'),
                        'due_date' => $input['due_date'] ?? null,
                        'subtasks' => $input['subtasks'] ?? []
                    ];
                    $data['tasks'][] = $task;
                    writeData($data);
                    echo json_encode($task);
                    break;

                case 'update_task':
                    $taskId = $input['id'];
                    foreach ($data['tasks'] as &$task) {
                        if ($task['id'] === $taskId) {
                            $task['title'] = $input['title'];
                            $task['description'] = $input['description'] ?? '';
                            $task['category'] = $input['category'];
                            $task['priority'] = $input['priority'] ?? 'medium';
                            $task['due_date'] = $input['due_date'] ?? null;
                            $task['subtasks'] = $input['subtasks'] ?? [];
                            break;
                        }
                    }
                    writeData($data);
                    echo json_encode(['success' => true]);
                    break;

                case 'toggle_task':
                    $taskId = $input['id'];
                    foreach ($data['tasks'] as &$task) {
                        if ($task['id'] === $taskId) {
                            $task['completed'] = !$task['completed'];
                            break;
                        }
                    }
                    writeData($data);
                    echo json_encode(['success' => true]);
                    break;

                case 'toggle_subtask':
                    $taskId = $input['task_id'];
                    $subtaskId = $input['subtask_id'];
                    foreach ($data['tasks'] as &$task) {
                        if ($task['id'] === $taskId) {
                            foreach ($task['subtasks'] as &$subtask) {
                                if ($subtask['id'] === $subtaskId) {
                                    $subtask['completed'] = !$subtask['completed'];
                                    break 2;
                                }
                            }
                        }
                    }
                    writeData($data);
                    echo json_encode(['success' => true]);
                    break;

                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid action']);
            }
            break;

        case 'DELETE':
            $data = readData();
            
            switch ($action) {
                case 'delete_task':
                    $taskId = $_GET['id'];
                    $data['tasks'] = array_filter($data['tasks'], function($task) use ($taskId) {
                        return $task['id'] !== $taskId;
                    });
                    $data['tasks'] = array_values($data['tasks']);
                    writeData($data);
                    echo json_encode(['success' => true]);
                    break;

                case 'delete_category':
                    $categoryId = $_GET['id'];
                    $data['categories'] = array_filter($data['categories'], function($category) use ($categoryId) {
                        return $category['id'] !== $categoryId;
                    });
                    $data['categories'] = array_values($data['categories']);
                    writeData($data);
                    echo json_encode(['success' => true]);
                    break;

                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid action']);
            }
            break;

        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}
?>

