import React, { useState, useEffect } from 'react';

const WhatsAppLinkGenerator = () => {
  const [phoneNumber, setPhoneNumber] = useState('');
  const [message, setMessage] = useState('');
  const [whatsappLink, setWhatsappLink] = useState('');
  const [copied, setCopied] = useState(false);

  // Format nomor telepon - menghapus karakter yang tidak diinginkan dan mengubah format
  const formatPhoneNumber = (number) => {
    // Menghapus semua karakter non-digit
    let cleaned = number.replace(/[^\d]/g, '');
    
    // Mengganti 08 dengan 628 jika dimulai dengan 08
    if (cleaned.startsWith('08')) {
      cleaned = '62' + cleaned.substring(2);
    }
    
    // Mengganti 8 dengan 628 jika dimulai dengan 8 (kasus lain)
    else if (cleaned.startsWith('8')) {
      cleaned = '62' + cleaned;
    }
    
    // Pastikan nomor dimulai dengan 62 untuk Indonesia
    else if (!cleaned.startsWith('62') && cleaned.length > 0) {
      cleaned = '62' + cleaned;
    }
    
    return cleaned;
  };

  // Generate link WhatsApp
  const generateLink = () => {
    const formattedNumber = formatPhoneNumber(phoneNumber);
    if (formattedNumber.length < 10) {
      return '';
    }
    
    // Encode pesan untuk URL
    const encodedMessage = encodeURIComponent(message);
    
    // Buat link WhatsApp
    return `https://wa.me/${formattedNumber}${encodedMessage ? `?text=${encodedMessage}` : ''}`;
  };

  // Update link saat input berubah
  useEffect(() => {
    setWhatsappLink(generateLink());
  }, [phoneNumber, message]);

  // Fungsi untuk menyalin link ke clipboard
  const copyToClipboard = () => {
    if (whatsappLink) {
      navigator.clipboard.writeText(whatsappLink)
        .then(() => {
          setCopied(true);
          setTimeout(() => setCopied(false), 2000);
        })
        .catch(err => {
          console.error('Gagal menyalin: ', err);
        });
    }
  };

  // Fungsi untuk langsung membuka link
  const openWhatsApp = () => {
    if (whatsappLink) {
      window.open(whatsappLink, '_blank');
    }
  };

  return (
    <div className="max-w-md mx-auto p-6 bg-white rounded-lg shadow-lg">
      <h1 className="text-2xl font-bold text-green-600 mb-6 text-center">Pembuat Link WhatsApp</h1>
      
      <div className="mb-4">
        <label className="block text-gray-700 mb-2" htmlFor="phone">
          Nomor WhatsApp
        </label>
        <input
          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
          id="phone"
          type="text"
          placeholder="Contoh: 081234567890, 628123456789"
          value={phoneNumber}
          onChange={(e) => setPhoneNumber(e.target.value)}
        />
        {phoneNumber && (
          <p className="mt-1 text-sm text-gray-600">
            Format yang akan digunakan: {formatPhoneNumber(phoneNumber)}
          </p>
        )}
      </div>
      
      <div className="mb-4">
        <label className="block text-gray-700 mb-2" htmlFor="message">
          Pesan (Opsional)
        </label>
        <textarea
          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-green-500"
          id="message"
          rows="3"
          placeholder="Ketik pesan Anda di sini..."
          value={message}
          onChange={(e) => setMessage(e.target.value)}
        ></textarea>
      </div>
      
      {whatsappLink && (
        <div className="mb-4 p-3 bg-gray-100 rounded-md break-all">
          <p className="text-sm font-medium text-gray-700 mb-1">Link WhatsApp:</p>
          <a 
            href={whatsappLink} 
            target="_blank" 
            className="text-green-600 hover:underline"
          >
            {whatsappLink}
          </a>
        </div>
      )}
      
      <div className="flex space-x-2">
        <button
          className="flex-1 bg-green-600 text-white py-2 px-4 rounded-md hover:bg-green-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          onClick={copyToClipboard}
          disabled={!whatsappLink}
        >
          {copied ? 'Tersalin!' : 'Salin Link'}
        </button>
        
        <button
          className="flex-1 bg-green-600 text-white py-2 px-4 rounded-md hover:bg-green-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
          onClick={openWhatsApp}
          disabled={!whatsappLink}
        >
          Buka WhatsApp
        </button>
      </div>
      
      <p className="mt-6 text-xs text-gray-500 text-center">
        Masukkan nomor telepon dengan format apapun, sistem akan otomatis mengubahnya menjadi format yang benar untuk Indonesia.
      </p>
    </div>
  );
};

export default WhatsAppLinkGenerator;
