<?php
/**
 * File instalasi database Tugasin
 * 
 * File ini digunakan untuk:
 * 1. Membuat database baru jika belum ada
 * 2. Membuat tabel-tabel yang diperlukan
 * 3. Menambahkan data awal seperti tema dan user admin
 * 4. Menambahkan contoh data untuk demonstrasi
 * 
 * @author Claude
 * @version 2.0
 */

// Nonaktifkan pelaporan error untuk produksi
// error_reporting(0);
// ini_set('display_errors', 0);

// Aktifkan pelaporan error untuk debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Mulai sesi
session_start();

// Header HTML
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalasi Tugasin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            background-color: #f8f9fa;
            color: #333;
            padding: 20px;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        h1 {
            color: #4CAF50;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }
        
        h2 {
            color: #2196F3;
            margin: 20px 0 10px;
        }
        
        pre {
            background-color: #f5f5f5;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            font-family: Consolas, monospace;
            font-size: 14px;
            line-height: 1.4;
            margin-bottom: 20px;
        }
        
        .success {
            color: #4CAF50;
        }
        
        .error {
            color: #F44336;
        }
        
        .warning {
            color: #FF9800;
        }
        
        .info {
            color: #2196F3;
        }
        
        .button {
            display: inline-block;
            background-color: #4CAF50;
            color: white;
            padding: 10px 15px;
            text-decoration: none;
            border-radius: 4px;
            transition: background-color 0.3s;
            border: none;
            cursor: pointer;
            font-size: 16px;
            margin-top: 10px;
        }
        
        .button:hover {
            background-color: #45a049;
        }
        
        .button-secondary {
            background-color: #2196F3;
        }
        
        .button-secondary:hover {
            background-color: #0b7dda;
        }
        
        .button-danger {
            background-color: #F44336;
        }
        
        .button-danger:hover {
            background-color: #d32f2f;
        }
        
        .step {
            background-color: white;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 15px;
            border-left: 4px solid #4CAF50;
        }
        
        .step h3 {
            margin-top: 0;
            color: #4CAF50;
        }
        
        .step.failed {
            border-left-color: #F44336;
        }
        
        .step.failed h3 {
            color: #F44336;
        }
        
        .step.warning {
            border-left-color: #FF9800;
        }
        
        .step.warning h3 {
            color: #FF9800;
        }
        
        .log {
            max-height: 200px;
            overflow-y: auto;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        input[type="text"],
        input[type="password"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }
        
        .progress-container {
            margin: 20px 0;
        }
        
        .progress-bar {
            height: 20px;
            background-color: #e0e0e0;
            border-radius: 10px;
            position: relative;
            overflow: hidden;
        }
        
        .progress {
            height: 100%;
            background-color: #4CAF50;
            border-radius: 10px;
            transition: width 0.3s ease;
            width: 0%;
        }
        
        .progress-text {
            text-align: center;
            margin-top: 5px;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1><i class="fas fa-cogs"></i> Instalasi Database Tugasin</h1>

<?php
// Cek jika formulir konfigurasi database telah disubmit
$db_config_submitted = isset($_POST['db_host']) && isset($_POST['db_user']);
$create_admin_submitted = isset($_POST['admin_username']) && isset($_POST['admin_password']);
$demo_data_submitted = isset($_POST['create_demo_data']);

// Langkah instalasi
$step = isset($_GET['step']) ? (int)$_GET['step'] : (
    $db_config_submitted ? 2 : (
        $create_admin_submitted ? 3 : (
            $demo_data_submitted ? 4 : 1
        )
    )
);

// Variabel status
$success = true;
$log = [];
$error_message = '';

// Fungsi untuk menambahkan log
function addLog($message, $type = 'info') {
    global $log;
    $log[] = ['message' => $message, 'type' => $type];
}

// Fungsi untuk menampilkan log
function showLog() {
    global $log;
    echo '<div class="log">';
    foreach ($log as $entry) {
        $class = $entry['type'] === 'success' ? 'success' : ($entry['type'] === 'error' ? 'error' : ($entry['type'] === 'warning' ? 'warning' : 'info'));
        echo '<div class="' . $class . '">';
        if ($entry['type'] === 'success') {
            echo '<i class="fas fa-check-circle"></i> ';
        } else if ($entry['type'] === 'error') {
            echo '<i class="fas fa-times-circle"></i> ';
        } else if ($entry['type'] === 'warning') {
            echo '<i class="fas fa-exclamation-triangle"></i> ';
        } else {
            echo '<i class="fas fa-info-circle"></i> ';
        }
        echo $entry['message'] . '</div>';
    }
    echo '</div>';
}

// Langkah 1: Tampilkan formulir konfigurasi database
if ($step === 1) {
    // Cek versi PHP
    $php_version = phpversion();
    $php_version_ok = version_compare($php_version, '7.0.0', '>=');
    
    // Cek ekstensi MySQL
    $mysql_extension_ok = extension_loaded('mysqli');
    
    // Cek jika folder dapat ditulis
    $folder_writable = is_writable('./');
    
    // Sambungkan config.php
    $config_exists = file_exists('config.php');
    
    // Tampilkan hasil cek
    echo '<div class="step' . ($php_version_ok ? '' : ' failed') . '">';
    echo '<h3>Cek Versi PHP</h3>';
    echo '<p>Versi PHP: ' . $php_version . '</p>';
    echo '<p class="' . ($php_version_ok ? 'success' : 'error') . '">' . ($php_version_ok ? 'OK' : 'PHP 7.0.0 atau lebih tinggi diperlukan') . '</p>';
    echo '</div>';
    
    echo '<div class="step' . ($mysql_extension_ok ? '' : ' failed') . '">';
    echo '<h3>Cek Ekstensi MySQL</h3>';
    echo '<p class="' . ($mysql_extension_ok ? 'success' : 'error') . '">' . ($mysql_extension_ok ? 'Ekstensi MySQLi tersedia' : 'Ekstensi MySQLi tidak tersedia') . '</p>';
    echo '</div>';
    
    echo '<div class="step' . ($folder_writable ? '' : ' failed') . '">';
    echo '<h3>Cek Izin Folder</h3>';
    echo '<p class="' . ($folder_writable ? 'success' : 'error') . '">' . ($folder_writable ? 'Folder dapat ditulis' : 'Folder tidak dapat ditulis') . '</p>';
    echo '</div>';
    
    echo '<div class="step' . ($config_exists ? ' warning' : '') . '">';
    echo '<h3>Cek Konfigurasi</h3>';
    echo '<p class="' . ($config_exists ? 'warning' : 'info') . '">' . ($config_exists ? 'File config.php sudah ada dan akan ditimpa' : 'File config.php akan dibuat') . '</p>';
    echo '</div>';
    
    // Jika semua cek berhasil, tampilkan formulir konfigurasi database
    if ($php_version_ok && $mysql_extension_ok && $folder_writable) {
        echo '<h2>Konfigurasi Database</h2>';
        echo '<form method="post" action="install.php">';
        echo '<div class="form-group">';
        echo '<label for="db_host">Host Database:</label>';
        echo '<input type="text" id="db_host" name="db_host" value="localhost" required>';
        echo '</div>';
        
        echo '<div class="form-group">';
        echo '<label for="db_user">Username Database:</label>';
        echo '<input type="text" id="db_user" name="db_user" value="root" required>';
        echo '</div>';
        
        echo '<div class="form-group">';
        echo '<label for="db_pass">Password Database:</label>';
        echo '<input type="password" id="db_pass" name="db_pass">';
        echo '</div>';
        
        echo '<div class="form-group">';
        echo '<label for="db_name">Nama Database:</label>';
        echo '<input type="text" id="db_name" name="db_name" value="tugasin_db" required>';
        echo '</div>';
        
        echo '<button type="submit" class="button">Lanjutkan <i class="fas fa-arrow-right"></i></button>';
        echo '</form>';
    } else {
        echo '<p class="error">Beberapa persyaratan tidak terpenuhi. Harap perbaiki masalah di atas sebelum melanjutkan.</p>';
    }
}
// Langkah 2: Buat database dan tabel
else if ($step === 2 && $db_config_submitted) {
    // Ambil parameter database dari POST
    $db_host = $_POST['db_host'];
    $db_user = $_POST['db_user'];
    $db_pass = $_POST['db_pass'];
    $db_name = $_POST['db_name'];
    
    // Simpan konfigurasi untuk langkah selanjutnya
    $_SESSION['db_host'] = $db_host;
    $_SESSION['db_user'] = $db_user;
    $_SESSION['db_pass'] = $db_pass;
    $_SESSION['db_name'] = $db_name;
    
    // Coba koneksi ke database
    try {
        $db = new mysqli($db_host, $db_user, $db_pass);
        
        if ($db->connect_error) {
            throw new Exception("Koneksi database gagal: " . $db->connect_error);
        }
        
        addLog("Koneksi ke database berhasil", "success");
        
        // Buat database jika belum ada
        addLog("Membuat database $db_name...");
        $db->query("CREATE DATABASE IF NOT EXISTS `$db_name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        
        if ($db->error) {
            throw new Exception("Gagal membuat database: " . $db->error);
        }
        
        addLog("Database $db_name berhasil dibuat", "success");
        $db->select_db($db_name);
        
        // Definisi struktur tabel
        $tables = [
            // Tabel users - menyimpan data pengguna
            'users' => "CREATE TABLE IF NOT EXISTS `users` (
                `username` VARCHAR(50) PRIMARY KEY,
                `password` VARCHAR(255) NOT NULL,
                `nama` VARCHAR(100) NOT NULL,
                `nim` VARCHAR(50) NOT NULL,
                `role` ENUM('user', 'super_admin') DEFAULT 'user',
                `expires` DATE DEFAULT NULL,
                `theme` VARCHAR(50) DEFAULT 'default'
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel mapel - menyimpan daftar mata pelajaran untuk setiap pengguna
            'mapel' => "CREATE TABLE IF NOT EXISTS `mapel` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nama_mapel` VARCHAR(100) NOT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE,
                UNIQUE KEY `unique_mapel` (`username`, `nama_mapel`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel diskusi - menyimpan data diskusi
            'diskusi' => "CREATE TABLE IF NOT EXISTS `diskusi` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nomor` VARCHAR(100) NOT NULL,
                `tanggal_dibuat` DATE NOT NULL,
                `deadline` DATE DEFAULT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel diskusi_selesai - menyimpan mata pelajaran yang sudah selesai diskusi
            'diskusi_selesai' => "CREATE TABLE IF NOT EXISTS `diskusi_selesai` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `diskusi_id` INT NOT NULL,
                `mapel_id` INT NOT NULL,
                FOREIGN KEY (`diskusi_id`) REFERENCES `diskusi`(`id`) ON DELETE CASCADE,
                FOREIGN KEY (`mapel_id`) REFERENCES `mapel`(`id`) ON DELETE CASCADE,
                UNIQUE KEY `unique_diskusi_mapel` (`diskusi_id`, `mapel_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel diskusi_archive - menyimpan diskusi yang sudah selesai sepenuhnya
            'diskusi_archive' => "CREATE TABLE IF NOT EXISTS `diskusi_archive` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nomor` VARCHAR(100) NOT NULL,
                `tanggal_dibuat` DATE NOT NULL,
                `deadline` DATE DEFAULT NULL,
                `tanggal_selesai` DATE NOT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel tugas - menyimpan data tugas
            'tugas' => "CREATE TABLE IF NOT EXISTS `tugas` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nomor` VARCHAR(100) NOT NULL,
                `tanggal_dibuat` DATE NOT NULL,
                `deadline` DATE DEFAULT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel tugas_selesai - menyimpan mata pelajaran yang sudah selesai tugas
            'tugas_selesai' => "CREATE TABLE IF NOT EXISTS `tugas_selesai` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `tugas_id` INT NOT NULL,
                `mapel_id` INT NOT NULL,
                FOREIGN KEY (`tugas_id`) REFERENCES `tugas`(`id`) ON DELETE CASCADE,
                FOREIGN KEY (`mapel_id`) REFERENCES `mapel`(`id`) ON DELETE CASCADE,
                UNIQUE KEY `unique_tugas_mapel` (`tugas_id`, `mapel_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel tugas_archive - menyimpan tugas yang sudah selesai sepenuhnya
            'tugas_archive' => "CREATE TABLE IF NOT EXISTS `tugas_archive` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nomor` VARCHAR(100) NOT NULL,
                `tanggal_dibuat` DATE NOT NULL,
                `deadline` DATE DEFAULT NULL,
                `tanggal_selesai` DATE NOT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel baca - menyimpan data baca
            'baca' => "CREATE TABLE IF NOT EXISTS `baca` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `mapel_id` INT NOT NULL,
                `jumlah_bab` INT NOT NULL DEFAULT 1,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE,
                FOREIGN KEY (`mapel_id`) REFERENCES `mapel`(`id`) ON DELETE CASCADE,
                UNIQUE KEY `unique_baca_mapel` (`username`, `mapel_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel baca_selesai - menyimpan bab yang sudah selesai dibaca
            'baca_selesai' => "CREATE TABLE IF NOT EXISTS `baca_selesai` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `baca_id` INT NOT NULL,
                `bab_nomor` INT NOT NULL,
                FOREIGN KEY (`baca_id`) REFERENCES `baca`(`id`) ON DELETE CASCADE,
                UNIQUE KEY `unique_baca_bab` (`baca_id`, `bab_nomor`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel baca_archive - menyimpan data baca yang sudah selesai sepenuhnya
            'baca_archive' => "CREATE TABLE IF NOT EXISTS `baca_archive` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `username` VARCHAR(50) NOT NULL,
                `nama_mapel` VARCHAR(100) NOT NULL,
                `jumlah_bab` INT NOT NULL,
                `tanggal_selesai` DATE NOT NULL,
                FOREIGN KEY (`username`) REFERENCES `users`(`username`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci",
            
            // Tabel themes - menyimpan definisi tema
            'themes' => "CREATE TABLE IF NOT EXISTS `themes` (
                `id` VARCHAR(50) PRIMARY KEY,
                `nama` VARCHAR(50) NOT NULL,
                `primary_color` VARCHAR(20) NOT NULL,
                `secondary_color` VARCHAR(20) NOT NULL,
                `danger_color` VARCHAR(20) NOT NULL,
                `warning_color` VARCHAR(20) NOT NULL,
                `info_color` VARCHAR(20) NOT NULL,
                `dark_color` VARCHAR(20) NOT NULL,
                `light_color` VARCHAR(20) NOT NULL,
                `success_color` VARCHAR(20) NOT NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci"
        ];
        
        // Buat tabel
        foreach ($tables as $table_name => $query) {
            addLog("Membuat tabel $table_name...");
            $db->query($query);
            
            if ($db->error) {
                addLog("Error: " . $db->error, "error");
                $success = false;
            } else {
                addLog("Tabel $table_name berhasil dibuat", "success");
            }
        }
        
        // Tambahkan tema default
        addLog("Menambahkan tema default...");
        $themes = [
            ['default', 'Default', '#4CAF50', '#45a049', '#f44336', '#ff9800', '#2196F3', '#333', '#f4f4f4', '#4CAF50'],
            ['blue', 'Biru', '#2196F3', '#0b7dda', '#f44336', '#ff9800', '#00bcd4', '#333', '#f4f4f4', '#4CAF50'],
            ['dark', 'Gelap', '#333', '#444', '#f44336', '#ff9800', '#2196F3', '#222', '#444', '#4CAF50'],
            ['purple', 'Ungu', '#9c27b0', '#7B1FA2', '#f44336', '#ff9800', '#2196F3', '#333', '#f4f4f4', '#4CAF50'],
            ['yellow', 'Kuning', '#FFC107', '#FFA000', '#f44336', '#FF5722', '#03A9F4', '#333', '#f4f4f4', '#4CAF50'],
            ['red', 'Merah', '#F44336', '#D32F2F', '#E91E63', '#ff9800', '#2196F3', '#333', '#f4f4f4', '#4CAF50'],
            ['teal', 'Teal', '#009688', '#00796B', '#f44336', '#ff9800', '#2196F3', '#333', '#f4f4f4', '#4CAF50']
        ];
        
        $stmt = $db->prepare("INSERT IGNORE INTO themes (id, nama, primary_color, secondary_color, danger_color, warning_color, info_color, dark_color, light_color, success_color) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        foreach ($themes as $theme) {
            $stmt->bind_param("ssssssssss", $theme[0], $theme[1], $theme[2], $theme[3], $theme[4], $theme[5], $theme[6], $theme[7], $theme[8], $theme[9]);
            $stmt->execute();
        }
        
        addLog("Tema default berhasil ditambahkan", "success");
        
        // Buat file konfigurasi
        addLog("Membuat file konfigurasi...");
        $config_content = "<?php
/**
 * File konfigurasi database
 * 
 * Berisi parameter koneksi database dan konstanta sistem
 */

// Parameter koneksi database
define('DB_HOST', '$db_host');
define('DB_USER', '$db_user');
define('DB_PASS', '$db_pass');
define('DB_NAME', '$db_name');

// Konstanta aplikasi
define('APP_NAME', 'Tugasin');
define('APP_VERSION', '2.0');

// Konstanta path
define('BASE_URL', '/tugasin/');
define('ASSETS_URL', BASE_URL . 'assets/');

// Pengaturan session
ini_set('session.cookie_httponly', 1);
session_start();

// Zona waktu Indonesia
date_default_timezone_set('Asia/Jakarta');

// Koneksi database
try {
    \$db = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if (\$db->connect_error) {
        throw new Exception(\"Koneksi database gagal: \" . \$db->connect_error);
    }
    
    \$db->set_charset(\"utf8mb4\");
} catch (Exception \$e) {
    die(\"Error: \" . \$e->getMessage());
}";
        
        if (file_put_contents('config.php', $config_content)) {
            addLog("File config.php berhasil dibuat", "success");
        } else {
            addLog("Gagal membuat file config.php", "error");
            $success = false;
        }
        
        // Tampilkan formulir admin
        if ($success) {
            echo '<div class="progress-container">';
            echo '<div class="progress-bar"><div class="progress" style="width: 40%;"></div></div>';
            echo '<div class="progress-text">40% Selesai</div>';
            echo '</div>';
            
            echo '<h2>Buat Akun Admin</h2>';
            echo '<form method="post" action="install.php">';
            echo '<div class="form-group">';
            echo '<label for="admin_username">Username Admin:</label>';
            echo '<input type="text" id="admin_username" name="admin_username" value="rizaldi" required>';
            echo '</div>';
            
            echo '<div class="form-group">';
            echo '<label for="admin_password">Password Admin:</label>';
            echo '<input type="password" id="admin_password" name="admin_password" value="admin123" required>';
            echo '</div>';
            
            echo '<div class="form-group">';
            echo '<label for="admin_nama">Nama Admin:</label>';
            echo '<input type="text" id="admin_nama" name="admin_nama" value="Super Admin" required>';
            echo '</div>';
            
            echo '<div class="form-group">';
            echo '<label for="admin_nim">NIM Admin:</label>';
            echo '<input type="text" id="admin_nim" name="admin_nim" value="ADMIN" required>';
            echo '</div>';
            
            echo '<button type="submit" class="button">Lanjutkan <i class="fas fa-arrow-right"></i></button>';
            echo '</form>';
        } else {
            echo '<p class="error">Terjadi kesalahan dalam pembuatan database dan tabel. Harap periksa log di atas untuk detailnya.</p>';
            echo '<a href="install.php" class="button">Coba Lagi</a>';
        }
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        addLog("Error: " . $error_message, "error");
        $success = false;
        
        echo '<p class="error">' . $error_message . '</p>';
        echo '<a href="install.php" class="button">Coba Lagi</a>';
    }
    
    showLog();
}
// Langkah 3: Buat akun admin dan pindah ke langkah demo data
else if ($step === 3 && $create_admin_submitted) {
    // Ambil parameter admin dari POST
    $admin_username = $_POST['admin_username'];
    $admin_password = $_POST['admin_password'];
    $admin_nama = $_POST['admin_nama'];
    $admin_nim = $_POST['admin_nim'];
    
    // Mendapatkan koneksi database
    $db_host = $_SESSION['db_host'];
    $db_user = $_SESSION['db_user'];
    $db_pass = $_SESSION['db_pass'];
    $db_name = $_SESSION['db_name'];
    
    try {
        $db = new mysqli($db_host, $db_user, $db_pass, $db_name);
        
        if ($db->connect_error) {
            throw new Exception("Koneksi database gagal: " . $db->connect_error);
        }
        
        addLog("Koneksi ke database berhasil", "success");
        
        // Tambahkan user admin
        addLog("Menambahkan user admin...");
        $admin_role = 'super_admin';
        
        // Hash password
        $hashed_password = password_hash($admin_password, PASSWORD_DEFAULT);
        
        // Cek apakah user sudah ada
        $stmt = $db->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->bind_param("s", $admin_username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            // Update user yang sudah ada
            $stmt = $db->prepare("UPDATE users SET password = ?, nama = ?, nim = ?, role = ? WHERE username = ?");
            $stmt->bind_param("sssss", $hashed_password, $admin_nama, $admin_nim, $admin_role, $admin_username);
            $stmt->execute();
            
            if ($db->error) {
                throw new Exception("Gagal mengupdate user admin: " . $db->error);
            }
            
            addLog("User admin berhasil diupdate", "success");
        } else {
            // Tambah user baru
            $stmt = $db->prepare("INSERT INTO users (username, password, nama, nim, role) VALUES (?, ?, ?, ?, ?)");
            $stmt->bind_param("sssss", $admin_username, $hashed_password, $admin_nama, $admin_nim, $admin_role);
            $stmt->execute();
            
            if ($db->error) {
                throw new Exception("Gagal menambahkan user admin: " . $db->error);
            }
            
            addLog("User admin berhasil ditambahkan", "success");
        }
        
        // Detail admin
        addLog("Username: $admin_username");
        addLog("Password: " . str_repeat('*', strlen($admin_password)) . " (tersimpan dengan aman)");
        
        // Tambahkan mata pelajaran default untuk admin
        addLog("Menambahkan mata pelajaran default untuk admin...");
        $default_mapel = [
            'Bank dan Lembaga Keuangan Non Bank',
            'Komunikasi Bisnis',
            'Statistika Ekonomi',
            'Pengantar Akuntansi',
            'Pengantar Ekonomi Mikro',
            'Perekonomian Indonesia'
        ];
        
        // Hapus mapel yang sudah ada untuk admin ini
        $stmt = $db->prepare("DELETE FROM mapel WHERE username = ?");
        $stmt->bind_param("s", $admin_username);
        $stmt->execute();
        
        // Tambahkan mapel baru
        $stmt = $db->prepare("INSERT INTO mapel (username, nama_mapel) VALUES (?, ?)");
        foreach ($default_mapel as $mapel) {
            $stmt->bind_param("ss", $admin_username, $mapel);
            $stmt->execute();
        }
        
        if ($db->error) {
            addLog("Peringatan: Beberapa mata pelajaran mungkin tidak berhasil ditambahkan", "warning");
        } else {
            addLog("Mata pelajaran default berhasil ditambahkan", "success");
        }
        
        // Tampilkan pilihan untuk membuat data demo
        echo '<div class="progress-container">';
        echo '<div class="progress-bar"><div class="progress" style="width: 80%;"></div></div>';
        echo '<div class="progress-text">80% Selesai</div>';
        echo '</div>';
        
        echo '<h2>Data Demo</h2>';
        echo '<p>Apakah Anda ingin membuat data demo untuk user admin? Data demo mencakup contoh diskusi, tugas, dan data baca.</p>';
        
        echo '<form method="post" action="install.php">';
        echo '<div class="form-group">';
        echo '<label>';
        echo '<input type="checkbox" name="create_demo_data" value="1" checked> Ya, buat data demo';
        echo '</label>';
        echo '</div>';
        
        echo '<button type="submit" class="button">Lanjutkan <i class="fas fa-arrow-right"></i></button>';
        echo '</form>';
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        addLog("Error: " . $error_message, "error");
        $success = false;
        
        echo '<p class="error">' . $error_message . '</p>';
        echo '<a href="install.php?step=2" class="button">Kembali ke Langkah Sebelumnya</a>';
    }
    
    showLog();
}
// Langkah 4: Tambahkan data demo jika dipilih
else if ($step === 4 && $demo_data_submitted) {
    // Mendapatkan koneksi database
    $db_host = $_SESSION['db_host'];
    $db_user = $_SESSION['db_user'];
    $db_pass = $_SESSION['db_pass'];
    $db_name = $_SESSION['db_name'];
    
    // Apakah perlu membuat data demo
    $create_demo = isset($_POST['create_demo_data']) && $_POST['create_demo_data'] == 1;
    
    try {
        $db = new mysqli($db_host, $db_user, $db_pass, $db_name);
        
        if ($db->connect_error) {
            throw new Exception("Koneksi database gagal: " . $db->connect_error);
        }
        
        addLog("Koneksi ke database berhasil", "success");
        
        // Dapatkan username admin dari database (yang pertama dengan role super_admin)
        $result = $db->query("SELECT username FROM users WHERE role = 'super_admin' LIMIT 1");
        $admin_username = $result->fetch_assoc()['username'];
        
        if ($create_demo) {
            addLog("Membuat data demo untuk user $admin_username...");
            
            // Tambahkan contoh diskusi
            addLog("Menambahkan contoh diskusi...");
            $diskusi_data = [
                ["Diskusi 1", date("Y-m-d"), date("Y-m-d", strtotime("+1 week"))],
                ["Diskusi Kelompok 2", date("Y-m-d", strtotime("-3 days")), date("Y-m-d", strtotime("+2 weeks"))],
                ["Diskusi Final", date("Y-m-d", strtotime("-1 week")), date("Y-m-d", strtotime("+3 days"))]
            ];
            
            $stmt = $db->prepare("INSERT INTO diskusi (username, nomor, tanggal_dibuat, deadline) VALUES (?, ?, ?, ?)");
            foreach ($diskusi_data as $diskusi) {
                $stmt->bind_param("ssss", $admin_username, $diskusi[0], $diskusi[1], $diskusi[2]);
                $stmt->execute();
                
                if ($db->error) {
                    addLog("Error menambahkan diskusi: " . $db->error, "error");
                    break;
                }
            }
            
            if (!$db->error) {
                addLog("Contoh diskusi berhasil ditambahkan", "success");
            }
            
            // Tambahkan contoh tugas
            addLog("Menambahkan contoh tugas...");
            $tugas_data = [
                ["Tugas 1", date("Y-m-d"), date("Y-m-d", strtotime("+5 days"))],
                ["Tugas Praktikum", date("Y-m-d", strtotime("-2 days")), date("Y-m-d", strtotime("+10 days"))],
                ["Tugas Individu", date("Y-m-d", strtotime("-1 week")), date("Y-m-d", strtotime("-1 day"))]
            ];
            
            $stmt = $db->prepare("INSERT INTO tugas (username, nomor, tanggal_dibuat, deadline) VALUES (?, ?, ?, ?)");
            foreach ($tugas_data as $tugas) {
                $stmt->bind_param("ssss", $admin_username, $tugas[0], $tugas[1], $tugas[2]);
                $stmt->execute();
                
                if ($db->error) {
                    addLog("Error menambahkan tugas: " . $db->error, "error");
                    break;
                }
            }
            
            if (!$db->error) {
                addLog("Contoh tugas berhasil ditambahkan", "success");
            }
            
            // Tambahkan contoh baca
            addLog("Menambahkan contoh data baca...");
            
            // Dapatkan ID mapel
            $result = $db->query("SELECT id, nama_mapel FROM mapel WHERE username = '$admin_username' LIMIT 3");
            $mapel_ids = [];
            while ($row = $result->fetch_assoc()) {
                $mapel_ids[$row['id']] = $row['nama_mapel'];
            }
            
            if (count($mapel_ids) > 0) {
                $stmt = $db->prepare("INSERT INTO baca (username, mapel_id, jumlah_bab) VALUES (?, ?, ?)");
                
                $i = 0;
                foreach ($mapel_ids as $id => $nama) {
                    $jumlah_bab = 10 + $i;
                    $stmt->bind_param("sii", $admin_username, $id, $jumlah_bab);
                    $stmt->execute();
                    
                    if ($db->error) {
                        addLog("Error menambahkan data baca: " . $db->error, "error");
                        break;
                    }
                    
                    // Tambahkan beberapa bab selesai
                    $baca_id = $db->insert_id;
                    $stmt_bab = $db->prepare("INSERT INTO baca_selesai (baca_id, bab_nomor) VALUES (?, ?)");
                    
                    for ($j = 1; $j <= 3 + $i; $j++) {
                        $stmt_bab->bind_param("ii", $baca_id, $j);
                        $stmt_bab->execute();
                    }
                    
                    $i++;
                }
                
                if (!$db->error) {
                    addLog("Contoh data baca berhasil ditambahkan", "success");
                }
            } else {
                addLog("Tidak ada mata pelajaran yang tersedia untuk data baca", "warning");
            }
            
            // Tambahkan contoh data arsip
            addLog("Menambahkan contoh data arsip...");
            
            // Diskusi arsip
            $stmt = $db->prepare("INSERT INTO diskusi_archive (username, nomor, tanggal_dibuat, deadline, tanggal_selesai) VALUES (?, ?, ?, ?, ?)");
            $diskusi_archive = ["Diskusi Arsip", date("Y-m-d", strtotime("-2 weeks")), date("Y-m-d", strtotime("-1 week")), date("Y-m-d", strtotime("-3 days"))];
            $stmt->bind_param("sssss", $admin_username, $diskusi_archive[0], $diskusi_archive[1], $diskusi_archive[2], $diskusi_archive[3]);
            $stmt->execute();
            
            // Tugas arsip
            $stmt = $db->prepare("INSERT INTO tugas_archive (username, nomor, tanggal_dibuat, deadline, tanggal_selesai) VALUES (?, ?, ?, ?, ?)");
            $tugas_archive = ["Tugas Arsip", date("Y-m-d", strtotime("-3 weeks")), date("Y-m-d", strtotime("-2 weeks")), date("Y-m-d", strtotime("-10 days"))];
            $stmt->bind_param("sssss", $admin_username, $tugas_archive[0], $tugas_archive[1], $tugas_archive[2], $tugas_archive[3]);
            $stmt->execute();
            
            // Baca arsip
            if (count($mapel_ids) > 0) {
                $mapel_name = reset($mapel_ids);
                $stmt = $db->prepare("INSERT INTO baca_archive (username, nama_mapel, jumlah_bab, tanggal_selesai) VALUES (?, ?, ?, ?)");
                $bab_count = 8;
                $selesai_date = date("Y-m-d", strtotime("-1 week"));
                $stmt->bind_param("ssis", $admin_username, $mapel_name, $bab_count, $selesai_date);
                $stmt->execute();
            }
            
            addLog("Contoh data arsip berhasil ditambahkan", "success");
        } else {
            addLog("Pembuatan data demo dilewati", "info");
        }
        
        // Instalasi selesai
        echo '<div class="progress-container">';
        echo '<div class="progress-bar"><div class="progress" style="width: 100%;"></div></div>';
        echo '<div class="progress-text">100% Selesai</div>';
        echo '</div>';
        
        echo '<h2 class="success"><i class="fas fa-check-circle"></i> Instalasi Selesai!</h2>';
        echo '<p>Aplikasi Tugasin berhasil diinstal dan siap digunakan.</p>';
        
        // Tampilkan detail akses
        echo '<div class="step">';
        echo '<h3>Informasi Akses</h3>';
        echo '<p><strong>URL Aplikasi:</strong> <a href="index.php">index.php</a></p>';
        echo '<p><strong>Username Admin:</strong> ' . htmlspecialchars($admin_username) . '</p>';
        echo '<p><strong>Password Admin:</strong> (password yang Anda masukkan sebelumnya)</p>';
        echo '</div>';
        
        // Langkah-langkah selanjutnya
        echo '<div class="step">';
        echo '<h3>Langkah Selanjutnya</h3>';
        echo '<ol>';
        echo '<li>Login ke aplikasi dengan menggunakan akun admin.</li>';
        echo '<li>Tambahkan pengguna baru melalui menu Pengaturan (jika Anda adalah admin).</li>';
        echo '<li>Ubah password admin Anda untuk keamanan yang lebih baik.</li>';
        echo '<li>Mulai gunakan aplikasi Tugasin!</li>';
        echo '</ol>';
        echo '</div>';
        
        // Tindakan keamanan
        echo '<div class="step warning">';
        echo '<h3><i class="fas fa-exclamation-triangle"></i> Catatan Keamanan</h3>';
        echo '<p>Untuk alasan keamanan, sebaiknya hapus atau ubah izin file <code>install.php</code> setelah instalasi selesai.</p>';
        echo '<p>File instalasi ini dapat digunakan oleh pihak yang tidak bertanggung jawab untuk menimpa database Anda.</p>';
        echo '</div>';
        
        echo '<a href="index.php" class="button button-secondary">Mulai Menggunakan Aplikasi</a>';
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
        addLog("Error: " . $error_message, "error");
        $success = false;
        
        echo '<p class="error">' . $error_message . '</p>';
        echo '<a href="install.php?step=3" class="button">Kembali ke Langkah Sebelumnya</a>';
    }
    
    showLog();
}
?>

    </div>
    
    <script>
        // Sembunyikan progress bar pada halaman pertama
        document.addEventListener('DOMContentLoaded', function() {
            const urlParams = new URLSearchParams(window.location.search);
            const step = urlParams.get('step') || <?php echo $step; ?>;
            
            if (step == 1) {
                const progressContainers = document.querySelectorAll('.progress-container');
                progressContainers.forEach(function(container) {
                    container.style.display = 'none';
                });
            }
        });
    </script>
</body>
</html>