<?php
/**
 * File fungsi umum
 * 
 * Berisi fungsi-fungsi umum untuk aplikasi
 */

// Pastikan file config.php sudah di-include sebelum file ini

/**
 * Redirect ke halaman lain
 * 
 * @param string $url URL tujuan redirect
 * @return void
 */
function redirect($url) {
    header("Location: $url");
    exit();
}

/**
 * Mendapatkan URL saat ini
 * 
 * @return string URL saat ini
 */
function getCurrentUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $uri = $_SERVER['REQUEST_URI'];
    return "$protocol://$host$uri";
}

/**
 * Mendapatkan URL dasar
 * 
 * @return string URL dasar
 */
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $baseDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
    return "$protocol://$host$baseDir";
}

/**
 * Mengecek apakah user sudah login
 * 
 * @return bool Sudah login atau belum
 */
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * Mengecek apakah user adalah super admin
 * 
 * @return bool Super admin atau bukan
 */
function isSuperAdmin() {
    return isLoggedIn() && isset($_SESSION['role']) && $_SESSION['role'] === 'super_admin';
}

/**
 * Menampilkan pesan error
 * 
 * @param string $message Pesan error
 * @return string HTML pesan error
 */
function showError($message) {
    return '<div class="alert alert-danger" role="alert">' . $message . '</div>';
}

/**
 * Menampilkan pesan sukses
 * 
 * @param string $message Pesan sukses
 * @return string HTML pesan sukses
 */
function showSuccess($message) {
    return '<div class="alert alert-success" role="alert">' . $message . '</div>';
}

/**
 * Menampilkan pesan informasi
 * 
 * @param string $message Pesan informasi
 * @return string HTML pesan informasi
 */
function showInfo($message) {
    return '<div class="alert alert-info" role="alert">' . $message . '</div>';
}

/**
 * Mendapatkan status deadline
 * 
 * @param string $deadline Tanggal deadline
 * @return array Status deadline (badge_class, badge_text)
 */
function getDeadlineStatus($deadline) {
    if (empty($deadline)) {
        return [
            'badge_class' => 'badge-info',
            'badge_text' => 'Tidak ada deadline'
        ];
    }
    
    $days_left = hitungSisaHari($deadline);
    
    if ($days_left < 0) {
        return [
            'badge_class' => 'badge-danger',
            'badge_text' => 'Terlambat ' . abs($days_left) . ' hari'
        ];
    } else if ($days_left == 0) {
        return [
            'badge_class' => 'badge-warning',
            'badge_text' => 'Hari ini'
        ];
    } else if ($days_left <= 3) {
        return [
            'badge_class' => 'badge-warning',
            'badge_text' => 'Tersisa ' . $days_left . ' hari'
        ];
    } else {
        return [
            'badge_class' => 'badge-info',
            'badge_text' => 'Tersisa ' . $days_left . ' hari'
        ];
    }
}

/**
 * Membuat ID unik untuk CSS atau JS
 * 
 * @param string $prefix Awalan ID
 * @return string ID unik
 */
function generateUniqueId($prefix = '') {
    return $prefix . bin2hex(random_bytes(4));
}

/**
 * Mempertahankan input setelah form submit gagal
 * 
 * @param string $key Nama input
 * @param string $default Nilai default jika tidak ada
 * @return string Nilai input
 */
function old($key, $default = '') {
    return isset($_REQUEST[$key]) ? htmlspecialchars($_REQUEST[$key]) : $default;
}

/**
 * Mempertahankan pilihan select setelah form submit gagal
 * 
 * @param string $key Nama input
 * @param string $value Nilai yang dicek
 * @return string Atribut selected jika cocok
 */
function selected($key, $value) {
    return isset($_REQUEST[$key]) && $_REQUEST[$key] == $value ? 'selected' : '';
}

/**
 * Mempertahankan pilihan checkbox setelah form submit gagal
 * 
 * @param string $key Nama input
 * @param string $value Nilai yang dicek
 * @return string Atribut checked jika cocok
 */
function checked($key, $value) {
    return isset($_REQUEST[$key]) && $_REQUEST[$key] == $value ? 'checked' : '';
}

/**
 * Sanitasi input
 * 
 * @param string $input Input yang akan disanitasi
 * @return string Input yang sudah disanitasi
 */
function sanitize($input) {
    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

/**
 * Memotong teks menjadi lebih pendek
 * 
 * @param string $text Teks yang akan dipotong
 * @param int $length Panjang maksimal
 * @param string $append Teks tambahan di akhir
 * @return string Teks yang sudah dipotong
 */
function truncateText($text, $length = 100, $append = '...') {
    if (strlen($text) > $length) {
        $text = substr($text, 0, $length) . $append;
    }
    
    return $text;
}

/**
 * Mendapatkan tab aktif saat ini
 * 
 * @param string $default Tab default jika tidak ada
 * @return string Tab aktif
 */
function getCurrentTab($default = 'diskusi') {
    return isset($_GET['tab']) ? $_GET['tab'] : $default;
}

/**
 * Cek apakah tab tertentu sedang aktif
 * 
 * @param string $tab Nama tab yang dicek
 * @return string Kelas 'active' jika tab aktif
 */
function isActiveTab($tab) {
    return getCurrentTab() === $tab ? 'active' : '';
}

/**
 * Mendapatkan tema pengguna
 * 
 * @param mysqli $db Koneksi database
 * @param string $username Username pengguna
 * @return array Data tema
 */
function getUserTheme($db, $username) {
    // Dapatkan ID tema dari pengguna
    $stmt = $db->prepare("SELECT theme FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $themeId = $user['theme'];
        
        // Dapatkan data tema
        $theme = getThemeById($db, $themeId);
        
        if ($theme) {
            return $theme;
        }
    }
    
    // Tema default jika tidak ditemukan
    return [
        'id' => 'default',
        'nama' => 'Default',
        'primary_color' => '#4CAF50',
        'secondary_color' => '#45a049',
        'danger_color' => '#f44336',
        'warning_color' => '#ff9800',
        'info_color' => '#2196F3',
        'dark_color' => '#333',
        'light_color' => '#f4f4f4',
        'success_color' => '#4CAF50'
    ];
}

/**
 * Menghasilkan HTML untuk notifikasi (toast)
 * 
 * @param string $message Pesan notifikasi
 * @param string $type Tipe notifikasi (success, error, info, warning)
 * @return string HTML notifikasi
 */
function notifyToast($message, $type = 'success') {
    $id = generateUniqueId('toast-');
    
    switch ($type) {
        case 'error':
            $bg_color = 'var(--danger-color)';
            $icon = 'fa-exclamation-circle';
            break;
        case 'warning':
            $bg_color = 'var(--warning-color)';
            $icon = 'fa-exclamation-triangle';
            break;
        case 'info':
            $bg_color = 'var(--info-color)';
            $icon = 'fa-info-circle';
            break;
        case 'success':
        default:
            $bg_color = 'var(--success-color)';
            $icon = 'fa-check-circle';
            break;
    }
    
    return <<<HTML
    <div id="$id" class="toast" style="background-color: $bg_color;">
        <div class="toast-icon">
            <i class="fas $icon"></i>
        </div>
        <div class="toast-body">$message</div>
        <button class="toast-close" onclick="closeToast('$id')">
            <i class="fas fa-times"></i>
        </button>
    </div>
    <script>
        showToast('$id');
    </script>
    HTML;
}

/**
 * Filter array berdasarkan kunci dan nilai
 * 
 * @param array $array Array yang akan difilter
 * @param string $key Kunci yang dicari
 * @param mixed $value Nilai yang dicari
 * @return array Array hasil filter
 */
function array_filter_by_key_value($array, $key, $value) {
    return array_filter($array, function($item) use ($key, $value) {
        return isset($item[$key]) && $item[$key] == $value;
    });
}

/**
 * Menampilkan loading spinner
 * 
 * @param string $id ID spinner
 * @param string $size Ukuran spinner (sm, md, lg)
 * @param string $color Warna spinner (mengambil dari tema: primary, secondary, info, warning, danger, success)
 * @return string HTML spinner
 */
function showSpinner($id = 'spinner', $size = 'md', $color = 'primary') {
    $sizeClass = '';
    
    switch ($size) {
        case 'sm':
            $sizeClass = 'spinner-sm';
            break;
        case 'lg':
            $sizeClass = 'spinner-lg';
            break;
        case 'md':
        default:
            $sizeClass = '';
            break;
    }
    
    $colorClass = "spinner-$color";
    
    return <<<HTML
    <div id="$id" class="spinner $sizeClass $colorClass">
        <div class="bounce1"></div>
        <div class="bounce2"></div>
        <div class="bounce3"></div>
    </div>
    HTML;
}

/**
 * Mendapatkan kelas ikon untuk status tugas/diskusi
 * 
 * @param int $selesai_count Jumlah yang sudah selesai
 * @param int $total_count Total item
 * @return string Nama kelas ikon
 */
function getStatusIconClass($selesai_count, $total_count) {
    if ($total_count == 0) {
        return 'fa-question-circle text-info';
    }
    
    $percentage = ($selesai_count / $total_count) * 100;
    
    if ($percentage >= 100) {
        return 'fa-check-circle text-success';
    } else if ($percentage >= 75) {
        return 'fa-thumbs-up text-primary';
    } else if ($percentage >= 50) {
        return 'fa-spinner text-primary';
    } else if ($percentage >= 25) {
        return 'fa-hourglass-half text-warning';
    } else {
        return 'fa-hourglass-start text-warning';
    }
}

/**
 * Mendapatkan warna untuk tema tertentu
 * 
 * @param string $colorName Nama warna (primary, secondary, dll)
 * @param array $theme Data tema
 * @return string Kode warna
 */
function getThemeColor($colorName, $theme) {
    $colorKey = $colorName . '_color';
    
    if (isset($theme[$colorKey])) {
        return $theme[$colorKey];
    }
    
    // Default warna
    $defaults = [
        'primary_color' => '#4CAF50',
        'secondary_color' => '#45a049',
        'danger_color' => '#f44336',
        'warning_color' => '#ff9800',
        'info_color' => '#2196F3',
        'dark_color' => '#333',
        'light_color' => '#f4f4f4',
        'success_color' => '#4CAF50'
    ];
    
    return isset($defaults[$colorKey]) ? $defaults[$colorKey] : '#000000';
}

/**
 * Mendapatkan warna kontras dari warna dasar (untuk teks)
 * 
 * @param string $color Kode warna dalam format hex (#RRGGBB)
 * @return string Warna kontras (#FFFFFF atau #000000)
 */
function getContrastColor($color) {
    // Hapus karakter # jika ada
    $color = ltrim($color, '#');
    
    // Konversi ke RGB
    $r = hexdec(substr($color, 0, 2));
    $g = hexdec(substr($color, 2, 2));
    $b = hexdec(substr($color, 4, 2));
    
    // Rumus luminansi (perkiraan kecerahan)
    $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
    
    // Gunakan teks putih untuk warna gelap, dan teks hitam untuk warna terang
    return $luminance > 0.5 ? '#000000' : '#FFFFFF';
}

/**
 * Menampilkan warna tema kustom dalam CSS
 * 
 * @param array $theme Data tema
 * @return string Kode CSS untuk tema
 */
function generateThemeColors($theme) {
    $primaryColor = getThemeColor('primary', $theme);
    $secondaryColor = getThemeColor('secondary', $theme);
    $dangerColor = getThemeColor('danger', $theme);
    $warningColor = getThemeColor('warning', $theme);
    $infoColor = getThemeColor('info', $theme);
    $darkColor = getThemeColor('dark', $theme);
    $lightColor = getThemeColor('light', $theme);
    $successColor = getThemeColor('success', $theme);
    
    $primaryText = getContrastColor($primaryColor);
    $secondaryText = getContrastColor($secondaryColor);
    $dangerText = getContrastColor($dangerColor);
    $warningText = getContrastColor($warningColor);
    $infoText = getContrastColor($infoColor);
    $darkText = getContrastColor($darkColor);
    $lightText = getContrastColor($lightColor);
    $successText = getContrastColor($successColor);
    
    return <<<CSS
    :root {
        --primary-color: $primaryColor;
        --secondary-color: $secondaryColor;
        --danger-color: $dangerColor;
        --warning-color: $warningColor;
        --info-color: $infoColor;
        --dark-color: $darkColor;
        --light-color: $lightColor;
        --success-color: $successColor;
        
        --primary-text: $primaryText;
        --secondary-text: $secondaryText;
        --danger-text: $dangerText;
        --warning-text: $warningText;
        --info-text: $infoText;
        --dark-text: $darkText;
        --light-text: $lightText;
        --success-text: $successText;
        
        --border-radius: 8px;
        --shadow-sm: 0 1px 3px rgba(0,0,0,0.08);
        --shadow-md: 0 2px 5px rgba(0,0,0,0.1);
        --shadow-lg: 0 3px 10px rgba(0,0,0,0.2);
    }
    CSS;
}

/**
 * Mengenkripsi password
 * 
 * @param string $password Password yang akan dienkripsi
 * @return string Password terenkripsi
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Memverifikasi password
 * 
 * @param string $password Password yang diinput
 * @param string $hashedPassword Password terenkripsi dari database
 * @return bool Password cocok atau tidak
 */
function verifyPassword($password, $hashedPassword) {
    // Jika masih menggunakan plain text (kompatibilitas dengan versi lama)
    if ($password === $hashedPassword) {
        return true;
    }
    
    // Verifikasi dengan password_verify
    return password_verify($password, $hashedPassword);
}