// AJAX functions for navigation without page refresh
function loadContent(url, updateHistory = true) {
    // Show loading indicator
    document.getElementById('loadingIndicator').style.display = 'flex';
    
    // Extract the URL without query parameters for history state
    const baseUrl = url.split('?')[0];
    const queryParams = url.includes('?') ? url.split('?')[1] : '';
    
    // Use fetch to get the content
    fetch(url)
        .then(response => response.text())
        .then(html => {
            // Create a temporary element to parse the HTML
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            
            // Extract the main content
            const newContent = doc.querySelector('.main-content');
            if (newContent) {
                // Update the main content
                document.querySelector('.main-content').innerHTML = newContent.innerHTML;
                
                // Update the sidebar active state
                updateSidebarActiveState(url);
                
                // Update browser history if needed
                if (updateHistory) {
                    window.history.pushState({url: url}, '', url);
                }
                
                // Reinitialize any scripts that need to run after content load
                initializeScripts();
            }
            
            // Hide loading indicator
            document.getElementById('loadingIndicator').style.display = 'none';
        })
        .catch(error => {
            console.error('Error loading content:', error);
            // Hide loading indicator
            document.getElementById('loadingIndicator').style.display = 'none';
            // Fallback to traditional navigation on error
            window.location.href = url;
        });
}

// Update sidebar active state based on current URL
function updateSidebarActiveState(url) {
    // Extract category and status from URL
    const urlParams = new URLSearchParams(url.includes('?') ? url.split('?')[1] : '');
    const category = urlParams.get('category') || 'work';
    const status = urlParams.get('status') || 'current';
    
    // Remove active class from all sidebar links
    document.querySelectorAll('.sidebar-menu a').forEach(link => {
        link.classList.remove('active');
    });
    
    // Add active class to current category
    document.querySelector(`.sidebar-menu a[href*="category=${category}"]`).classList.add('active');
    
    // Add active class to current status
    document.querySelector(`.sidebar-menu a[href*="status=${status}"]`).classList.add('active');
    
    // Update mobile navigation active state
    document.querySelectorAll('.mobile-category-nav a').forEach(link => {
        link.classList.remove('active');
    });
    document.querySelector(`.mobile-category-nav a[href*="category=${category}"]`)?.classList.add('active');
}

// Initialize scripts after content load
function initializeScripts() {
    // Reinitialize date pickers
    if (typeof flatpickr !== 'undefined') {
        const dateInputs = document.querySelectorAll('#start_date, #deadline');
        if (dateInputs.length > 0) {
            dateInputs.forEach(input => {
                flatpickr(input, {
                    dateFormat: "Y-m-d",
                    defaultDate: "today",
                    locale: {
                        weekdays: {
                            shorthand: ["Min", "Sen", "Sel", "Rab", "Kam", "Jum", "Sab"],
                            longhand: ["Minggu", "Senin", "Selasa", "Rabu", "Kamis", "Jumat", "Sabtu"]
                        },
                        months: {
                            shorthand: ["Jan", "Feb", "Mar", "Apr", "Mei", "Jun", "Jul", "Agt", "Sep", "Okt", "Nov", "Des"],
                            longhand: ["Januari", "Februari", "Maret", "April", "Mei", "Juni", "Juli", "Agustus", "September", "Oktober", "November", "Desember"]
                        }
                    }
                });
            });
        }
    }
    
    // Reinitialize modal functionality
    const modal = document.getElementById("taskModal");
    const closeModal = document.getElementById("closeModal");
    const cancelTask = document.getElementById("cancelTask");
    
    if (modal && closeModal && cancelTask) {
        closeModal.addEventListener("click", function() {
            modal.style.display = "none";
        });
        
        cancelTask.addEventListener("click", function() {
            modal.style.display = "none";
        });
        
        window.addEventListener("click", function(event) {
            if (event.target == modal) {
                modal.style.display = "none";
            }
        });
    }
    
    // Reinitialize edit buttons
    const editButtons = document.querySelectorAll('.edit-task-btn');
    editButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const taskId = this.getAttribute('data-id');
            editTask(taskId);
        });
    });
    
    // Reinitialize add task buttons
    const addTaskBtn = document.getElementById("addTaskBtn");
    const floatingAddTaskBtn = document.getElementById("floatingAddTaskBtn");
    const emptyAddTaskBtn = document.getElementById("emptyAddTaskBtn");
    
    if (addTaskBtn) {
        addTaskBtn.addEventListener("click", openAddTaskModal);
    }
    
    if (floatingAddTaskBtn) {
        floatingAddTaskBtn.addEventListener("click", openAddTaskModal);
    }
    
    if (emptyAddTaskBtn) {
        emptyAddTaskBtn.addEventListener("click", openAddTaskModal);
    }
    
    // Reinitialize category select change event
    const categorySelect = document.getElementById("category");
    const subjectsGroup = document.getElementById("subjectsGroup");
    
    if (categorySelect && subjectsGroup) {
        categorySelect.addEventListener("change", function() {
            if (this.value === "college") {
                subjectsGroup.style.display = "block";
            } else {
                subjectsGroup.style.display = "none";
            }
        });
    }
    
    // Reinitialize calendar cell click events
    document.querySelectorAll('.calendar td[data-date]').forEach(cell => {
        cell.addEventListener('click', function() {
            const dateStr = this.getAttribute('data-date');
            resetForm();
            if (window.startDatePicker && window.deadlinePicker) {
                window.startDatePicker.setDate(dateStr);
                window.deadlinePicker.setDate(dateStr);
            }
            
            // Default to current category
            const currentCategory = document.querySelector('.sidebar-menu a.active[href*="category="]')?.getAttribute('href')?.match(/category=([^&]*)/)?.[1] || 'work';
            if (categorySelect) {
                categorySelect.value = currentCategory;
                
                // Show subjects if category is college
                if (currentCategory === "college" && subjectsGroup) {
                    subjectsGroup.style.display = "block";
                }
            }
            
            const modalTitle = document.getElementById("modalTitle")?.querySelector("span");
            const saveTaskBtn = document.getElementById("saveTaskBtn");
            
            if (modalTitle) modalTitle.textContent = "Tambah Tugas Baru";
            if (saveTaskBtn) saveTaskBtn.name = "add_task";
            if (modal) modal.style.display = "flex";
        });
    });
    
    // Add AJAX navigation to all links
    addAjaxNavigation();
}

// Add AJAX navigation to all internal links
function addAjaxNavigation() {
    // Get all links in the sidebar and main content
    const links = document.querySelectorAll('.sidebar-menu a, .mobile-category-nav a, .task-actions a:not(.edit-task-btn), .subject-pill');
    
    links.forEach(link => {
        // Skip links that should perform default action
        if (link.classList.contains('edit-task-btn') || 
            link.getAttribute('href') === '#' || 
            link.getAttribute('href')?.startsWith('http') ||
            link.getAttribute('href')?.includes('logout') ||
            link.getAttribute('onclick')) {
            return;
        }
        
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const url = this.getAttribute('href');
            loadContent(url);
        });
    });
}

// Handle browser back/forward buttons
window.addEventListener('popstate', function(event) {
    if (event.state && event.state.url) {
        loadContent(event.state.url, false);
    } else {
        loadContent(window.location.href, false);
    }
});

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Store initial state
    window.history.replaceState({url: window.location.href}, '', window.location.href);
    
    // Add loading indicator to the page
    const loadingIndicator = document.createElement('div');
    loadingIndicator.id = 'loadingIndicator';
    loadingIndicator.innerHTML = '<div class="spinner"></div>';
    document.body.appendChild(loadingIndicator);
    
    // Add styles for loading indicator
    const style = document.createElement('style');
    style.textContent = `
        #loadingIndicator {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(255, 255, 255, 0.7);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        .spinner {
            width: 40px;
            height: 40px;
            border: 4px solid rgba(67, 97, 238, 0.3);
            border-radius: 50%;
            border-top: 4px solid var(--primary-color);
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    `;
    document.head.appendChild(style);
    
    // Initialize scripts
    initializeScripts();
});
