// sw.js - Service Worker for Sleep Tracker PWA
// Progressive Web App functionality

const CACHE_NAME = 'sleep-tracker-v2.0';
const urlsToCache = [
    '/',
    '/index.php',
    '/mobile.php',
    '/analytics.php',
    '/export.php',
    '/api.php',
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css',
    'https://cdnjs.cloudflare.com/ajax/libs/Chart.js/3.9.1/chart.min.js'
];

// Install event - cache resources
self.addEventListener('install', (event) => {
    console.log('Sleep Tracker SW: Installing...');
    
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                console.log('Sleep Tracker SW: Caching app shell');
                return cache.addAll(urlsToCache);
            })
            .then(() => {
                console.log('Sleep Tracker SW: Installation complete');
                return self.skipWaiting();
            })
            .catch((error) => {
                console.error('Sleep Tracker SW: Installation failed', error);
            })
    );
});

// Activate event - clean up old caches
self.addEventListener('activate', (event) => {
    console.log('Sleep Tracker SW: Activating...');
    
    event.waitUntil(
        caches.keys().then((cacheNames) => {
            return Promise.all(
                cacheNames.map((cacheName) => {
                    if (cacheName !== CACHE_NAME) {
                        console.log('Sleep Tracker SW: Deleting old cache', cacheName);
                        return caches.delete(cacheName);
                    }
                })
            );
        }).then(() => {
            console.log('Sleep Tracker SW: Activation complete');
            return self.clients.claim();
        })
    );
});

// Fetch event - serve from cache, fallback to network
self.addEventListener('fetch', (event) => {
    // Skip non-GET requests
    if (event.request.method !== 'GET') {
        return;
    }
    
    // Skip API requests for real-time data
    if (event.request.url.includes('/api.php') && 
        (event.request.url.includes('action=') || event.request.url.includes('POST'))) {
        return;
    }
    
    event.respondWith(
        caches.match(event.request)
            .then((response) => {
                // Return cached version or fetch from network
                if (response) {
                    console.log('Sleep Tracker SW: Serving from cache', event.request.url);
                    return response;
                }
                
                console.log('Sleep Tracker SW: Fetching from network', event.request.url);
                return fetch(event.request).then((response) => {
                    // Don't cache if not a valid response
                    if (!response || response.status !== 200 || response.type !== 'basic') {
                        return response;
                    }
                    
                    // Cache the response for future use
                    const responseToCache = response.clone();
                    caches.open(CACHE_NAME)
                        .then((cache) => {
                            cache.put(event.request, responseToCache);
                        });
                    
                    return response;
                });
            })
            .catch(() => {
                // Return offline page if available
                if (event.request.destination === 'document') {
                    return caches.match('/mobile.php');
                }
            })
    );
});

// Background sync for offline data submission
self.addEventListener('sync', (event) => {
    console.log('Sleep Tracker SW: Background sync triggered', event.tag);
    
    if (event.tag === 'background-sync-sleep-data') {
        event.waitUntil(syncSleepData());
    }
    
    if (event.tag === 'background-sync-activity-data') {
        event.waitUntil(syncActivityData());
    }
});

// Push notification handling
self.addEventListener('push', (event) => {
    console.log('Sleep Tracker SW: Push notification received');
    
    const options = {
        body: 'Time to check your sleep schedule!',
        icon: '/icon-192x192.png',
        badge: '/badge-72x72.png',
        vibrate: [100, 50, 100],
        data: {
            dateOfArrival: Date.now(),
            primaryKey: 1
        },
        actions: [
            {
                action: 'open-app',
                title: 'Open Sleep Tracker',
                icon: '/icon-192x192.png'
            },
            {
                action: 'dismiss',
                title: 'Dismiss',
                icon: '/close-icon.png'
            }
        ]
    };
    
    if (event.data) {
        const notificationData = event.data.json();
        options.body = notificationData.body || options.body;
        options.data = notificationData.data || options.data;
    }
    
    event.waitUntil(
        self.registration.showNotification('Sleep Tracker', options)
    );
});

// Notification click handling
self.addEventListener('notificationclick', (event) => {
    console.log('Sleep Tracker SW: Notification clicked', event.action);
    
    event.notification.close();
    
    if (event.action === 'open-app') {
        event.waitUntil(
            clients.openWindow('/')
        );
    } else if (event.action === 'dismiss') {
        // Just close the notification
        return;
    } else {
        // Default action - open app
        event.waitUntil(
            clients.openWindow('/')
        );
    }
});

// Message handling from main thread
self.addEventListener('message', (event) => {
    console.log('Sleep Tracker SW: Message received', event.data);
    
    if (event.data && event.data.type === 'SKIP_WAITING') {
        self.skipWaiting();
    }
    
    if (event.data && event.data.type === 'CACHE_SLEEP_DATA') {
        cacheSleepData(event.data.data);
    }
    
    if (event.data && event.data.type === 'GET_CACHED_DATA') {
        getCachedSleepData().then(data => {
            event.ports[0].postMessage({ type: 'CACHED_DATA', data: data });
        });
    }
});

// Periodic background sync (for supported browsers)
self.addEventListener('periodicsync', (event) => {
    console.log('Sleep Tracker SW: Periodic sync triggered', event.tag);
    
    if (event.tag === 'sleep-reminder') {
        event.waitUntil(sendSleepReminder());
    }
    
    if (event.tag === 'data-backup') {
        event.waitUntil(performDataBackup());
    }
});

// Helper functions
async function syncSleepData() {
    try {
        const offlineData = await getOfflineSleepData();
        
        for (const data of offlineData) {
            await fetch('/sleep_tracker.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(data)
            });
        }
        
        // Clear offline data after successful sync
        await clearOfflineSleepData();
        
        console.log('Sleep Tracker SW: Sleep data synced successfully');
        
        // Show success notification
        self.registration.showNotification('Sleep Tracker', {
            body: 'Offline sleep data has been synced!',
            icon: '/icon-192x192.png',
            tag: 'sync-success'
        });
        
    } catch (error) {
        console.error('Sleep Tracker SW: Failed to sync sleep data', error);
        throw error;
    }
}

async function syncActivityData() {
    try {
        const offlineData = await getOfflineActivityData();
        
        for (const data of offlineData) {
            await fetch('/sleep_tracker.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(data)
            });
        }
        
        await clearOfflineActivityData();
        console.log('Sleep Tracker SW: Activity data synced successfully');
        
    } catch (error) {
        console.error('Sleep Tracker SW: Failed to sync activity data', error);
        throw error;
    }
}

async function sendSleepReminder() {
    try {
        // Get user's typical bedtime from cache or API
        const userData = await getCachedUserData();
        const currentTime = new Date();
        const currentHour = currentTime.getHours();
        
        // Check if it's close to typical bedtime
        if (userData && userData.typicalBedtime) {
            const bedtimeHour = parseInt(userData.typicalBedtime.split(':')[0]);
            const reminderHour = bedtimeHour - 1; // 1 hour before bedtime
            
            if (currentHour === reminderHour) {
                await self.registration.showNotification('Sleep Tracker Reminder', {
                    body: 'Hey Rizaldi! Time to start winding down for sleep 🌙',
                    icon: '/icon-192x192.png',
                    badge: '/badge-72x72.png',
                    vibrate: [200, 100, 200],
                    tag: 'sleep-reminder',
                    actions: [
                        {
                            action: 'start-sleep',
                            title: 'Start Sleep Session',
                            icon: '/sleep-icon.png'
                        },
                        {
                            action: 'snooze',
                            title: 'Remind in 30 mins',
                            icon: '/snooze-icon.png'
                        }
                    ]
                });
            }
        }
        
    } catch (error) {
        console.error('Sleep Tracker SW: Failed to send sleep reminder', error);
    }
}

async function performDataBackup() {
    try {
        // Get all cached sleep data
        const sleepData = await getCachedSleepData();
        
        if (sleepData && sleepData.length > 0) {
            // Send backup request to server
            await fetch('/api.php/backup', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-API-Key': 'rizaldi_mobile_app'
                },
                body: JSON.stringify({
                    data: sleepData,
                    timestamp: Date.now(),
                    source: 'service-worker-backup'
                })
            });
            
            console.log('Sleep Tracker SW: Data backup completed');
        }
        
    } catch (error) {
        console.error('Sleep Tracker SW: Failed to perform data backup', error);
    }
}

// IndexedDB operations for offline storage
async function cacheSleepData(data) {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            const transaction = db.transaction(['sleepData'], 'readwrite');
            const store = transaction.objectStore('sleepData');
            
            store.put({
                id: Date.now(),
                data: data,
                timestamp: new Date(),
                synced: false
            });
            
            transaction.oncomplete = () => resolve();
            transaction.onerror = () => reject(transaction.error);
        };
        
        request.onupgradeneeded = (event) => {
            const db = event.target.result;
            
            if (!db.objectStoreNames.contains('sleepData')) {
                const store = db.createObjectStore('sleepData', { keyPath: 'id' });
                store.createIndex('timestamp', 'timestamp', { unique: false });
                store.createIndex('synced', 'synced', { unique: false });
            }
            
            if (!db.objectStoreNames.contains('activityData')) {
                const store = db.createObjectStore('activityData', { keyPath: 'id' });
                store.createIndex('timestamp', 'timestamp', { unique: false });
                store.createIndex('synced', 'synced', { unique: false });
            }
            
            if (!db.objectStoreNames.contains('userData')) {
                db.createObjectStore('userData', { keyPath: 'key' });
            }
        };
    });
}

async function getCachedSleepData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            
            if (!db.objectStoreNames.contains('sleepData')) {
                resolve([]);
                return;
            }
            
            const transaction = db.transaction(['sleepData'], 'readonly');
            const store = transaction.objectStore('sleepData');
            const getAllRequest = store.getAll();
            
            getAllRequest.onsuccess = () => resolve(getAllRequest.result);
            getAllRequest.onerror = () => reject(getAllRequest.error);
        };
    });
}

async function getOfflineSleepData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            
            if (!db.objectStoreNames.contains('sleepData')) {
                resolve([]);
                return;
            }
            
            const transaction = db.transaction(['sleepData'], 'readonly');
            const store = transaction.objectStore('sleepData');
            const index = store.index('synced');
            const getRequest = index.getAll(false);
            
            getRequest.onsuccess = () => {
                const results = getRequest.result.map(item => item.data);
                resolve(results);
            };
            getRequest.onerror = () => reject(getRequest.error);
        };
    });
}

async function clearOfflineSleepData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            const transaction = db.transaction(['sleepData'], 'readwrite');
            const store = transaction.objectStore('sleepData');
            const index = store.index('synced');
            const getRequest = index.getAll(false);
            
            getRequest.onsuccess = () => {
                const items = getRequest.result;
                
                items.forEach(item => {
                    item.synced = true;
                    store.put(item);
                });
                
                transaction.oncomplete = () => resolve();
                transaction.onerror = () => reject(transaction.error);
            };
        };
    });
}

async function getOfflineActivityData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            
            if (!db.objectStoreNames.contains('activityData')) {
                resolve([]);
                return;
            }
            
            const transaction = db.transaction(['activityData'], 'readonly');
            const store = transaction.objectStore('activityData');
            const index = store.index('synced');
            const getRequest = index.getAll(false);
            
            getRequest.onsuccess = () => {
                const results = getRequest.result.map(item => item.data);
                resolve(results);
            };
            getRequest.onerror = () => reject(getRequest.error);
        };
    });
}

async function clearOfflineActivityData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            const transaction = db.transaction(['activityData'], 'readwrite');
            const store = transaction.objectStore('activityData');
            const index = store.index('synced');
            const getRequest = index.getAll(false);
            
            getRequest.onsuccess = () => {
                const items = getRequest.result;
                
                items.forEach(item => {
                    item.synced = true;
                    store.put(item);
                });
                
                transaction.oncomplete = () => resolve();
                transaction.onerror = () => reject(transaction.error);
            };
        };
    });
}

async function getCachedUserData() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('SleepTrackerDB', 1);
        
        request.onerror = () => reject(request.error);
        request.onsuccess = () => {
            const db = request.result;
            
            if (!db.objectStoreNames.contains('userData')) {
                resolve(null);
                return;
            }
            
            const transaction = db.transaction(['userData'], 'readonly');
            const store = transaction.objectStore('userData');
            const getRequest = store.get('preferences');
            
            getRequest.onsuccess = () => {
                resolve(getRequest.result ? getRequest.result.data : null);
            };
            getRequest.onerror = () => reject(getRequest.error);
        };
    });
}

// Version update notification
self.addEventListener('message', (event) => {
    if (event.data && event.data.type === 'CHECK_UPDATE') {
        // Check for app updates
        caches.has(CACHE_NAME).then(hasCache => {
            if (hasCache) {
                event.ports[0].postMessage({
                    type: 'UPDATE_AVAILABLE',
                    version: '2.0'
                });
            }
        });
    }
});

console.log('Sleep Tracker Service Worker v2.0 loaded successfully! 🌙');