<?php
// setup.php - Sleep Tracker Setup & Installation

$setupComplete = false;
$messages = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Create necessary directories
        $directories = ['data', 'backup'];
        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                if (mkdir($dir, 0755, true)) {
                    $messages[] = "✅ Direktori '$dir' berhasil dibuat";
                } else {
                    throw new Exception("❌ Gagal membuat direktori '$dir'");
                }
            } else {
                $messages[] = "ℹ️ Direktori '$dir' sudah ada";
            }
        }
        
        // Create sample data if requested
        if (isset($_POST['create_sample'])) {
            $currentMonth = date('Y-m');
            $sampleDataFile = "data/{$currentMonth}.json";
            
            if (!file_exists($sampleDataFile)) {
                $sampleData = createSampleData();
                if (file_put_contents($sampleDataFile, json_encode($sampleData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                    $messages[] = "✅ Data contoh berhasil dibuat untuk bulan " . date('F Y');
                } else {
                    throw new Exception("❌ Gagal membuat data contoh");
                }
            } else {
                $messages[] = "ℹ️ Data bulan ini sudah ada, tidak membuat data contoh";
            }
        }
        
        // Create .htaccess for security
        $htaccessContent = "# Sleep Tracker Security\n";
        $htaccessContent .= "# Protect data and backup directories\n";
        $htaccessContent .= "<Files ~ \"\\.(json|log)$\">\n";
        $htaccessContent .= "    Order allow,deny\n";
        $htaccessContent .= "    Deny from all\n";
        $htaccessContent .= "</Files>\n\n";
        $htaccessContent .= "# Protect setup file after installation\n";
        $htaccessContent .= "<Files \"setup.php\">\n";
        $htaccessContent .= "    Order allow,deny\n";
        $htaccessContent .= "    Deny from all\n";
        $htaccessContent .= "</Files>\n";
        
        if (file_put_contents('.htaccess', $htaccessContent)) {
            $messages[] = "✅ File keamanan .htaccess berhasil dibuat";
        }
        
        // Check PHP version and extensions
        $phpVersion = phpversion();
        if (version_compare($phpVersion, '7.0', '>=')) {
            $messages[] = "✅ PHP version: $phpVersion (Compatible)";
        } else {
            $messages[] = "⚠️ PHP version: $phpVersion (Recommend PHP 7.0+)";
        }
        
        // Check required functions
        $requiredFunctions = ['json_encode', 'json_decode', 'file_get_contents', 'file_put_contents', 'mkdir', 'glob'];
        foreach ($requiredFunctions as $func) {
            if (function_exists($func)) {
                $messages[] = "✅ Function '$func' tersedia";
            } else {
                throw new Exception("❌ Function '$func' tidak tersedia");
            }
        }
        
        // Check write permissions
        if (is_writable('.')) {
            $messages[] = "✅ Direktori aplikasi dapat ditulis";
        } else {
            throw new Exception("❌ Direktori aplikasi tidak dapat ditulis. Ubah permission ke 755.");
        }
        
        $setupComplete = true;
        $messages[] = "🎉 Setup berhasil! Aplikasi Sleep Tracker siap digunakan.";
        
    } catch (Exception $e) {
        $messages[] = $e->getMessage();
    }
}

function createSampleData() {
    $sampleData = [];
    
    // Create sample data for last 7 days
    for ($i = 7; $i >= 1; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        $dayOfWeek = date('w', strtotime($date));
        $isWeekend = ($dayOfWeek == 0 || $dayOfWeek == 6);
        
        $entry = [
            'date' => $date,
            'sleepSessions' => [],
            'workActivity' => $isWeekend ? 'ringan' : ['ringan', 'sedang', 'berat'][rand(0, 2)],
            'location' => ['rumah', 'keluar', 'jauh'][rand(0, 2)],
            'notes' => 'Data contoh untuk demonstrasi aplikasi',
            'createdAt' => date('Y-m-d H:i:s', strtotime($date)),
            'updatedAt' => date('Y-m-d H:i:s', strtotime($date))
        ];
        
        // Add night sleep session
        $nightBedTime = $isWeekend ? rand(23, 24) : rand(21, 23);
        $nightBedMinute = rand(0, 59);
        $nightDuration = $isWeekend ? rand(8, 10) + rand(0, 9)/10 : rand(6, 8) + rand(0, 9)/10;
        
        $bedTime = sprintf('%02d:%02d', $nightBedTime % 24, $nightBedMinute);
        $wakeTimeMinutes = ($nightBedTime * 60 + $nightBedMinute + ($nightDuration * 60)) % (24 * 60);
        $wakeTime = sprintf('%02d:%02d', floor($wakeTimeMinutes / 60), $wakeTimeMinutes % 60);
        
        $entry['sleepSessions'][] = [
            'type' => 'malam',
            'bedTime' => $bedTime,
            'wakeTime' => $wakeTime,
            'duration' => round($nightDuration, 1),
            'cycles' => round($nightDuration / 1.5),
            'quality' => $nightDuration >= 7 ? 'Excellent' : ($nightDuration >= 6 ? 'Good' : 'Fair'),
            'createdAt' => date('Y-m-d H:i:s', strtotime($date))
        ];
        
        // Sometimes add afternoon nap
        if (rand(0, 3) == 0) { // 25% chance
            $napBedTime = rand(13, 15);
            $napDuration = rand(1, 3) + rand(0, 9)/10;
            
            $napBedTimeStr = sprintf('%02d:%02d', $napBedTime, rand(0, 59));
            $napWakeTimeMinutes = ($napBedTime * 60 + ($napDuration * 60)) % (24 * 60);
            $napWakeTime = sprintf('%02d:%02d', floor($napWakeTimeMinutes / 60), $napWakeTimeMinutes % 60);
            
            $entry['sleepSessions'][] = [
                'type' => 'siang',
                'bedTime' => $napBedTimeStr,
                'wakeTime' => $napWakeTime,
                'duration' => round($napDuration, 1),
                'cycles' => round($napDuration / 1.5),
                'quality' => $napDuration <= 2 ? 'Excellent' : 'Good',
                'createdAt' => date('Y-m-d H:i:s', strtotime($date))
            ];
        }
        
        $sampleData[] = $entry;
    }
    
    return $sampleData;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup - Sleep Tracker Rizaldi</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 800px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 40px;
        }

        .header {
            text-align: center;
            margin-bottom: 40px;
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .setup-form {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .form-group {
            margin-bottom: 25px;
        }

        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 600;
            color: #555;
        }

        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 10px;
            border: 2px solid #e9ecef;
        }

        .checkbox-group input[type="checkbox"] {
            width: 20px;
            height: 20px;
        }

        .btn {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            width: 100%;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .btn:disabled {
            background: #6c757d;
            cursor: not-allowed;
            transform: none;
        }

        .messages {
            background: white;
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
        }

        .message {
            padding: 10px 15px;
            margin: 8px 0;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 14px;
        }

        .message.success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }

        .message.info {
            background: #d1ecf1;
            border: 1px solid #bee5eb;
            color: #0c5460;
        }

        .message.warning {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
        }

        .message.error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }

        .success-panel {
            background: linear-gradient(45deg, #4CAF50, #45a049);
            color: white;
            border-radius: 15px;
            padding: 30px;
            text-align: center;
            margin-top: 20px;
        }

        .success-panel h2 {
            margin-bottom: 15px;
            font-size: 1.8em;
        }

        .success-panel .btn {
            background: rgba(255,255,255,0.2);
            margin-top: 20px;
            width: auto;
            padding: 12px 25px;
        }

        .features {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }

        .feature {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            text-align: center;
        }

        .feature i {
            font-size: 2em;
            color: #667eea;
            margin-bottom: 10px;
        }

        .system-check {
            background: #e8f4fd;
            border: 1px solid #b8daff;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
        }

        .system-check h3 {
            color: #004085;
            margin-bottom: 15px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-cog"></i> Setup Sleep Tracker</h1>
            <p>Instalasi & Konfigurasi Aplikasi Pengelola Jam Tidur Rizaldi</p>
        </div>

        <?php if (!$setupComplete): ?>
            <div class="system-check">
                <h3><i class="fas fa-server"></i> Pengecekan Sistem</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px;">
                    <div>
                        <strong>PHP Version:</strong><br>
                        <span style="color: <?php echo version_compare(phpversion(), '7.0', '>=') ? '#28a745' : '#dc3545'; ?>">
                            <?php echo phpversion(); ?>
                        </span>
                    </div>
                    <div>
                        <strong>JSON Support:</strong><br>
                        <span style="color: <?php echo function_exists('json_encode') ? '#28a745' : '#dc3545'; ?>">
                            <?php echo function_exists('json_encode') ? 'Available' : 'Not Available'; ?>
                        </span>
                    </div>
                    <div>
                        <strong>File Permissions:</strong><br>
                        <span style="color: <?php echo is_writable('.') ? '#28a745' : '#dc3545'; ?>">
                            <?php echo is_writable('.') ? 'Writable' : 'Not Writable'; ?>
                        </span>
                    </div>
                </div>
            </div>

            <div class="setup-form">
                <h2 style="color: #667eea; margin-bottom: 20px;">
                    <i class="fas fa-rocket"></i> Mulai Setup
                </h2>
                
                <form method="POST">
                    <div class="form-group">
                        <label><i class="fas fa-database"></i> Opsi Setup Data:</label>
                        <div class="checkbox-group">
                            <input type="checkbox" id="create_sample" name="create_sample" checked>
                            <label for="create_sample" style="margin: 0;">
                                Buat data contoh untuk demonstrasi (7 hari terakhir)
                            </label>
                        </div>
                        <small style="color: #666; margin-top: 10px; display: block;">
                            Data contoh membantu untuk melihat cara kerja prediksi AI dan fitur-fitur aplikasi
                        </small>
                    </div>

                    <button type="submit" class="btn">
                        <i class="fas fa-play"></i> Mulai Setup Aplikasi
                    </button>
                </form>
            </div>

            <div class="features">
                <div class="feature">
                    <i class="fas fa-brain"></i>
                    <h3>AI Prediction</h3>
                    <p>Prediksi cerdas berdasarkan pola tidur personal</p>
                </div>
                <div class="feature">
                    <i class="fas fa-chart-line"></i>
                    <h3>Multi Sessions</h3>
                    <p>Tracking multiple sleep sessions per hari</p>
                </div>
                <div class="feature">
                    <i class="fas fa-shield-alt"></i>
                    <h3>Secure Storage</h3>
                    <p>Data disimpan aman dalam format JSON</p>
                </div>
                <div class="feature">
                    <i class="fas fa-mobile-alt"></i>
                    <h3>Responsive</h3>
                    <p>Dapat diakses dari berbagai device</p>
                </div>
            </div>
        <?php endif; ?>

        <?php if (!empty($messages)): ?>
            <div class="messages">
                <h3><i class="fas fa-terminal"></i> Log Setup:</h3>
                <?php foreach ($messages as $message): ?>
                    <div class="message <?php 
                        if (strpos($message, '✅') !== false) echo 'success';
                        elseif (strpos($message, 'ℹ️') !== false) echo 'info';
                        elseif (strpos($message, '⚠️') !== false) echo 'warning';
                        elseif (strpos($message, '❌') !== false) echo 'error';
                        else echo 'info';
                    ?>">
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if ($setupComplete): ?>
            <div class="success-panel">
                <h2><i class="fas fa-check-circle"></i> Setup Berhasil!</h2>
                <p>Aplikasi Sleep Tracker telah berhasil dikonfigurasi dan siap digunakan.</p>
                <p><strong>Selamat datang, Rizaldi!</strong></p>
                
                <a href="index.php" class="btn">
                    <i class="fas fa-arrow-right"></i> Mulai Gunakan Aplikasi
                </a>
                
                <div style="margin-top: 25px; font-size: 0.9em; opacity: 0.9;">
                    <p><strong>Tips untuk memulai:</strong></p>
                    <p>• Input data tidur harian untuk mendapatkan prediksi yang akurat</p>
                    <p>• Sistem AI akan belajar dari pola tidur Anda</p>
                    <p>• Prediksi akan semakin akurat setelah 1-2 minggu penggunaan</p>
                </div>
            </div>
        <?php endif; ?>

        <div style="text-align: center; margin-top: 30px; color: #666; font-size: 0.9em;">
            <p><i class="fas fa-code"></i> Sleep Tracker v2.0 - Developed for Rizaldi</p>
            <p>Made with ❤️ using PHP & Smart AI Algorithms</p>
        </div>
    </div>
</body>
</html>