<?php
// mobile.php - Mobile Optimized Sleep Tracker

session_start();
require_once 'config.php';

// Mobile-specific functions
function getMobileData() {
    $currentDate = date('Y-m-d');
    $currentMonth = date('Y-m');
    
    // Load current month data
    $dataFile = "data/{$currentMonth}.json";
    $monthData = [];
    if (file_exists($dataFile)) {
        $monthData = json_decode(file_get_contents($dataFile), true) ?: [];
    }
    
    // Get today's data
    $todayData = null;
    foreach ($monthData as $entry) {
        if ($entry['date'] === $currentDate) {
            $todayData = $entry;
            break;
        }
    }
    
    if (!$todayData) {
        $todayData = [
            'date' => $currentDate,
            'sleepSessions' => [],
            'workActivity' => '',
            'location' => '',
            'notes' => ''
        ];
    }
    
    // Calculate quick stats
    $totalSessions = 0;
    $totalDuration = 0;
    $recentDays = 0;
    
    $cutoffDate = date('Y-m-d', strtotime('-7 days'));
    
    foreach ($monthData as $entry) {
        if ($entry['date'] >= $cutoffDate && !empty($entry['sleepSessions'])) {
            $recentDays++;
            foreach ($entry['sleepSessions'] as $session) {
                $totalSessions++;
                $totalDuration += $session['duration'];
            }
        }
    }
    
    $avgDuration = $totalSessions > 0 ? round($totalDuration / $totalSessions, 1) : 0;
    
    return [
        'today' => $todayData,
        'stats' => [
            'recent_days' => $recentDays,
            'total_sessions' => $totalSessions,
            'avg_duration' => $avgDuration
        ]
    ];
}

// Handle quick actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    try {
        switch ($_POST['action']) {
            case 'quick_sleep':
                $result = addQuickSleep($_POST);
                echo json_encode(['success' => true, 'data' => $result]);
                break;
                
            case 'end_sleep':
                $result = endSleep($_POST);
                echo json_encode(['success' => true, 'data' => $result]);
                break;
                
            case 'quick_activity':
                $result = updateQuickActivity($_POST);
                echo json_encode(['success' => true, 'data' => $result]);
                break;
                
            default:
                echo json_encode(['success' => false, 'error' => 'Unknown action']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

function addQuickSleep($data) {
    $currentTime = date('H:i');
    $currentDate = date('Y-m-d');
    $type = $data['type'] ?? 'malam';
    
    // Store sleep start in session for later completion
    $_SESSION['sleep_start'] = [
        'type' => $type,
        'bedTime' => $currentTime,
        'date' => $currentDate,
        'timestamp' => time()
    ];
    
    return [
        'message' => 'Sleep session started',
        'bedTime' => $currentTime,
        'type' => $type
    ];
}

function endSleep($data) {
    if (!isset($_SESSION['sleep_start'])) {
        throw new Exception('No active sleep session found');
    }
    
    $sleepStart = $_SESSION['sleep_start'];
    $currentTime = date('H:i');
    $currentDate = date('Y-m-d');
    
    // Calculate duration
    $duration = SleepTrackerConfig::calculateSleepDuration($sleepStart['bedTime'], $currentTime);
    
    if ($duration < 0.5) {
        throw new Exception('Sleep duration too short');
    }
    
    // Save to data file
    $month = substr($currentDate, 0, 7);
    $dataFile = "data/{$month}.json";
    $monthData = file_exists($dataFile) ? json_decode(file_get_contents($dataFile), true) : [];
    if (!$monthData) $monthData = [];
    
    // Find or create daily entry
    $dayEntry = null;
    foreach ($monthData as &$entry) {
        if ($entry['date'] === $currentDate) {
            $dayEntry = &$entry;
            break;
        }
    }
    
    if (!$dayEntry) {
        $newEntry = [
            'date' => $currentDate,
            'sleepSessions' => [],
            'workActivity' => '',
            'location' => '',
            'notes' => '',
            'createdAt' => date('Y-m-d H:i:s'),
            'updatedAt' => date('Y-m-d H:i:s')
        ];
        $monthData[] = $newEntry;
        $dayEntry = &$monthData[count($monthData) - 1];
    }
    
    // Add sleep session
    $cycles = round($duration / 1.5);
    $quality = SleepTrackerConfig::getSleepQuality($cycles, $sleepStart['type']);
    
    $session = [
        'type' => $sleepStart['type'],
        'bedTime' => $sleepStart['bedTime'],
        'wakeTime' => $currentTime,
        'duration' => $duration,
        'cycles' => $cycles,
        'quality' => $quality,
        'createdAt' => date('Y-m-d H:i:s')
    ];
    
    $dayEntry['sleepSessions'][] = $session;
    $dayEntry['updatedAt'] = date('Y-m-d H:i:s');
    
    // Save data
    file_put_contents($dataFile, json_encode($monthData, JSON_PRETTY_PRINT));
    
    // Clear session
    unset($_SESSION['sleep_start']);
    
    return [
        'message' => 'Sleep session completed',
        'session' => $session
    ];
}

function updateQuickActivity($data) {
    $currentDate = date('Y-m-d');
    $month = substr($currentDate, 0, 7);
    $dataFile = "data/{$month}.json";
    
    $monthData = file_exists($dataFile) ? json_decode(file_get_contents($dataFile), true) : [];
    if (!$monthData) $monthData = [];
    
    // Find or create daily entry
    $dayEntry = null;
    foreach ($monthData as &$entry) {
        if ($entry['date'] === $currentDate) {
            $dayEntry = &$entry;
            break;
        }
    }
    
    if (!$dayEntry) {
        $newEntry = [
            'date' => $currentDate,
            'sleepSessions' => [],
            'workActivity' => $data['activity'] ?? '',
            'location' => $data['location'] ?? '',
            'notes' => '',
            'createdAt' => date('Y-m-d H:i:s'),
            'updatedAt' => date('Y-m-d H:i:s')
        ];
        $monthData[] = $newEntry;
    } else {
        $dayEntry['workActivity'] = $data['activity'] ?? $dayEntry['workActivity'];
        $dayEntry['location'] = $data['location'] ?? $dayEntry['location'];
        $dayEntry['updatedAt'] = date('Y-m-d H:i:s');
    }
    
    file_put_contents($dataFile, json_encode($monthData, JSON_PRETTY_PRINT));
    
    return ['message' => 'Activity updated'];
}

$mobileData = getMobileData();
$activeSleep = isset($_SESSION['sleep_start']) ? $_SESSION['sleep_start'] : null;
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="mobile-web-app-capable" content="yes">
    <title>Sleep Tracker Mobile - Rizaldi</title>
    <link rel="apple-touch-icon" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🌙</text></svg>">
    <link rel="icon" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><text y='.9em' font-size='90'>🌙</text></svg>">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 0;
            margin: 0;
            overflow-x: hidden;
            user-select: none;
        }

        .mobile-container {
            max-width: 100%;
            margin: 0 auto;
            background: white;
            min-height: 100vh;
            position: relative;
        }

        .mobile-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px 15px 15px;
            text-align: center;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .mobile-header h1 {
            font-size: 1.4em;
            margin-bottom: 5px;
        }

        .mobile-header .subtitle {
            font-size: 0.9em;
            opacity: 0.9;
        }

        .quick-stats {
            display: flex;
            justify-content: space-around;
            padding: 15px;
            background: #f8f9fa;
            border-bottom: 1px solid #e9ecef;
        }

        .stat-item {
            text-align: center;
            flex: 1;
        }

        .stat-number {
            font-size: 1.3em;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 2px;
        }

        .stat-label {
            font-size: 0.75em;
            color: #666;
            text-transform: uppercase;
        }

        .active-sleep-banner {
            background: linear-gradient(45deg, #28a745, #20c997);
            color: white;
            padding: 15px;
            text-align: center;
            animation: pulse 2s infinite;
        }

        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.8; }
            100% { opacity: 1; }
        }

        .mobile-content {
            padding: 20px 15px;
        }

        .action-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border: 1px solid #f0f0f0;
        }

        .action-card h3 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 1.1em;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .quick-actions {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-bottom: 15px;
        }

        .quick-btn {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 12px;
            padding: 15px 10px;
            font-size: 0.9em;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 5px;
            min-height: 70px;
        }

        .quick-btn:active {
            transform: scale(0.95);
        }

        .quick-btn.sleep {
            background: linear-gradient(45deg, #667eea, #764ba2);
        }

        .quick-btn.nap {
            background: linear-gradient(45deg, #17a2b8, #007bff);
        }

        .quick-btn.activity {
            background: linear-gradient(45deg, #28a745, #20c997);
        }

        .quick-btn.end {
            background: linear-gradient(45deg, #dc3545, #fd7e14);
        }

        .quick-btn i {
            font-size: 1.2em;
        }

        .sessions-list {
            max-height: 200px;
            overflow-y: auto;
        }

        .session-item {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 12px;
            margin-bottom: 8px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .session-info {
            flex: 1;
        }

        .session-type {
            font-weight: 600;
            color: #667eea;
            font-size: 0.9em;
        }

        .session-time {
            color: #666;
            font-size: 0.8em;
        }

        .session-duration {
            background: #667eea;
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.7em;
            font-weight: 600;
        }

        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: white;
            border-top: 1px solid #e9ecef;
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
            z-index: 100;
        }

        .nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            text-decoration: none;
            color: #666;
            font-size: 0.7em;
            padding: 5px;
            min-width: 60px;
        }

        .nav-item.active {
            color: #667eea;
        }

        .nav-item i {
            font-size: 1.2em;
            margin-bottom: 2px;
        }

        .floating-btn {
            position: fixed;
            bottom: 80px;
            right: 20px;
            width: 60px;
            height: 60px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 50%;
            font-size: 1.5em;
            cursor: pointer;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
            z-index: 99;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
        }

        .modal-content {
            background: white;
            margin: 20% auto;
            padding: 20px;
            border-radius: 15px;
            width: 90%;
            max-width: 300px;
            text-align: center;
        }

        .modal-content h3 {
            color: #667eea;
            margin-bottom: 15px;
        }

        .modal-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }

        .modal-btn {
            flex: 1;
            padding: 10px;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
        }

        .modal-btn.primary {
            background: #667eea;
            color: white;
        }

        .modal-btn.secondary {
            background: #f8f9fa;
            color: #666;
        }

        .loading {
            display: none;
            text-align: center;
            padding: 20px;
            color: #667eea;
        }

        .notification {
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            background: #28a745;
            color: white;
            padding: 10px 20px;
            border-radius: 20px;
            font-size: 0.9em;
            z-index: 1001;
            display: none;
        }

        .notification.error {
            background: #dc3545;
        }

        /* Responsive adjustments */
        @media (max-width: 360px) {
            .quick-actions {
                grid-template-columns: 1fr;
            }
            
            .quick-stats {
                flex-direction: column;
                gap: 10px;
            }
            
            .stat-item {
                text-align: left;
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
        }

        /* Dark mode support */
        @media (prefers-color-scheme: dark) {
            .mobile-container {
                background: #1a1a1a;
                color: white;
            }
            
            .action-card {
                background: #2d2d2d;
                border-color: #404040;
            }
            
            .session-item {
                background: #404040;
            }
            
            .bottom-nav {
                background: #2d2d2d;
                border-top-color: #404040;
            }
            
            .modal-content {
                background: #2d2d2d;
                color: white;
            }
        }

        /* PWA-like styling */
        .safe-area {
            padding-bottom: env(safe-area-inset-bottom);
        }
    </style>
</head>
<body>
    <div class="mobile-container">
        <!-- Header -->
        <div class="mobile-header">
            <h1>🌙 Sleep Tracker</h1>
            <div class="subtitle">Halo Rizaldi! <?php echo date('d M Y'); ?></div>
        </div>

        <!-- Active Sleep Banner -->
        <?php if ($activeSleep): ?>
            <div class="active-sleep-banner">
                <i class="fas fa-bed"></i>
                Tidur <?php echo ucfirst($activeSleep['type']); ?> aktif sejak <?php echo $activeSleep['bedTime']; ?>
                <div style="font-size: 0.8em; margin-top: 5px;">
                    Durasi: <span id="sleepDuration"><?php echo gmdate("H:i", time() - $activeSleep['timestamp']); ?></span>
                </div>
            </div>
        <?php endif; ?>

        <!-- Quick Stats -->
        <div class="quick-stats">
            <div class="stat-item">
                <div class="stat-number"><?php echo $mobileData['stats']['recent_days']; ?></div>
                <div class="stat-label">Hari Aktif</div>
            </div>
            <div class="stat-item">
                <div class="stat-number"><?php echo $mobileData['stats']['total_sessions']; ?></div>
                <div class="stat-label">Sesi Tidur</div>
            </div>
            <div class="stat-item">
                <div class="stat-number"><?php echo $mobileData['stats']['avg_duration']; ?>h</div>
                <div class="stat-label">Rata-rata</div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="mobile-content safe-area">
            <!-- Quick Actions -->
            <div class="action-card">
                <h3><i class="fas fa-zap"></i> Quick Actions</h3>
                
                <?php if (!$activeSleep): ?>
                    <div class="quick-actions">
                        <button class="quick-btn sleep" onclick="startSleep('malam')">
                            <i class="fas fa-moon"></i>
                            <span>Tidur Malam</span>
                        </button>
                        <button class="quick-btn nap" onclick="startSleep('siang')">
                            <i class="fas fa-sun"></i>
                            <span>Tidur Siang</span>
                        </button>
                    </div>
                <?php else: ?>
                    <div class="quick-actions">
                        <button class="quick-btn end" onclick="endSleep()" style="grid-column: 1 / -1;">
                            <i class="fas fa-stop"></i>
                            <span>Akhiri Tidur</span>
                        </button>
                    </div>
                <?php endif; ?>

                <div class="quick-actions">
                    <button class="quick-btn activity" onclick="quickActivity('ringan')">
                        <i class="fas fa-laptop"></i>
                        <span>Kerja Ringan</span>
                    </button>
                    <button class="quick-btn activity" onclick="quickActivity('berat')">
                        <i class="fas fa-dumbbell"></i>
                        <span>Kerja Berat</span>
                    </button>
                </div>
            </div>

            <!-- Today's Sessions -->
            <div class="action-card">
                <h3><i class="fas fa-bed"></i> Tidur Hari Ini (<?php echo count($mobileData['today']['sleepSessions']); ?>)</h3>
                
                <div class="sessions-list">
                    <?php if (empty($mobileData['today']['sleepSessions'])): ?>
                        <div style="text-align: center; color: #666; padding: 20px;">
                            <i class="fas fa-info-circle"></i><br>
                            Belum ada sesi tidur hari ini
                        </div>
                    <?php else: ?>
                        <?php foreach ($mobileData['today']['sleepSessions'] as $session): ?>
                            <div class="session-item">
                                <div class="session-info">
                                    <div class="session-type">
                                        <i class="fas fa-<?php echo $session['type'] === 'malam' ? 'moon' : 'sun'; ?>"></i>
                                        Tidur <?php echo ucfirst($session['type']); ?>
                                    </div>
                                    <div class="session-time">
                                        <?php echo $session['bedTime']; ?> - <?php echo $session['wakeTime']; ?>
                                    </div>
                                </div>
                                <div class="session-duration">
                                    <?php echo $session['duration']; ?>h
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Bottom Navigation -->
        <div class="bottom-nav">
            <a href="mobile.php" class="nav-item active">
                <i class="fas fa-home"></i>
                <span>Home</span>
            </a>
            <a href="index.php" class="nav-item">
                <i class="fas fa-desktop"></i>
                <span>Desktop</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="export.php" class="nav-item">
                <i class="fas fa-download"></i>
                <span>Export</span>
            </a>
        </div>

        <!-- Floating Action Button -->
        <button class="floating-btn" onclick="showQuickSleepModal()">
            <i class="fas fa-plus"></i>
        </button>
    </div>

    <!-- Modals -->
    <div id="quickSleepModal" class="modal">
        <div class="modal-content">
            <h3>Pilih Jenis Tidur</h3>
            <div style="display: grid; gap: 10px;">
                <button class="modal-btn primary" onclick="startSleep('pagi')">
                    <i class="fas fa-sunrise"></i> Tidur Pagi
                </button>
                <button class="modal-btn primary" onclick="startSleep('siang')">
                    <i class="fas fa-sun"></i> Tidur Siang
                </button>
                <button class="modal-btn primary" onclick="startSleep('sore')">
                    <i class="fas fa-sunset"></i> Tidur Sore
                </button>
                <button class="modal-btn primary" onclick="startSleep('malam')">
                    <i class="fas fa-moon"></i> Tidur Malam
                </button>
            </div>
            <div class="modal-actions">
                <button class="modal-btn secondary" onclick="closeModal()">Batal</button>
            </div>
        </div>
    </div>

    <div id="loading" class="loading">
        <i class="fas fa-spinner fa-spin" style="font-size: 2em;"></i>
        <div>Processing...</div>
    </div>

    <div id="notification" class="notification"></div>

    <script>
        // Sleep duration counter
        <?php if ($activeSleep): ?>
            setInterval(() => {
                const startTime = <?php echo $activeSleep['timestamp']; ?> * 1000;
                const now = new Date().getTime();
                const duration = now - startTime;
                
                const hours = Math.floor(duration / (1000 * 60 * 60));
                const minutes = Math.floor((duration % (1000 * 60 * 60)) / (1000 * 60));
                
                document.getElementById('sleepDuration').textContent = 
                    String(hours).padStart(2, '0') + ':' + String(minutes).padStart(2, '0');
            }, 1000);
        <?php endif; ?>

        function showNotification(message, type = 'success') {
            const notification = document.getElementById('notification');
            notification.textContent = message;
            notification.className = `notification ${type}`;
            notification.style.display = 'block';
            
            setTimeout(() => {
                notification.style.display = 'none';
            }, 3000);
        }

        function showLoading() {
            document.getElementById('loading').style.display = 'block';
        }

        function hideLoading() {
            document.getElementById('loading').style.display = 'none';
        }

        function showQuickSleepModal() {
            document.getElementById('quickSleepModal').style.display = 'block';
        }

        function closeModal() {
            document.querySelectorAll('.modal').forEach(modal => {
                modal.style.display = 'none';
            });
        }

        async function startSleep(type) {
            closeModal();
            showLoading();
            
            try {
                const formData = new FormData();
                formData.append('action', 'quick_sleep');
                formData.append('type', type);
                
                const response = await fetch('mobile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(`Tidur ${type} dimulai!`);
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showNotification(result.error, 'error');
                }
            } catch (error) {
                showNotification('Error: ' + error.message, 'error');
            } finally {
                hideLoading();
            }
        }

        async function endSleep() {
            if (!confirm('Akhiri sesi tidur sekarang?')) return;
            
            showLoading();
            
            try {
                const formData = new FormData();
                formData.append('action', 'end_sleep');
                
                const response = await fetch('mobile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('Sesi tidur selesai!');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showNotification(result.error, 'error');
                }
            } catch (error) {
                showNotification('Error: ' + error.message, 'error');
            } finally {
                hideLoading();
            }
        }

        async function quickActivity(activity) {
            showLoading();
            
            try {
                const formData = new FormData();
                formData.append('action', 'quick_activity');
                formData.append('activity', activity);
                formData.append('location', 'rumah'); // Default location
                
                const response = await fetch('mobile.php', {
                    method: 'POST',
                    body: formData
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification(`Aktivitas ${activity} tercatat!`);
                } else {
                    showNotification(result.error, 'error');
                }
            } catch (error) {
                showNotification('Error: ' + error.message, 'error');
            } finally {
                hideLoading();
            }
        }

        // Prevent zoom on double tap
        let lastTouchEnd = 0;
        document.addEventListener('touchend', function (event) {
            var now = (new Date()).getTime();
            if (now - lastTouchEnd <= 300) {
                event.preventDefault();
            }
            lastTouchEnd = now;
        }, false);

        // Add haptic feedback for iOS
        function vibrate() {
            if (navigator.vibrate) {
                navigator.vibrate(50);
            }
        }

        // Add vibration to buttons
        document.querySelectorAll('.quick-btn, .modal-btn').forEach(btn => {
            btn.addEventListener('touchstart', vibrate);
        });

        // Service Worker registration for PWA
        if ('serviceWorker' in navigator) {
            window.addEventListener('load', function() {
                navigator.serviceWorker.register('/sw.js')
                    .then(function(registration) {
                        console.log('ServiceWorker registration successful');
                    })
                    .catch(function(err) {
                        console.log('ServiceWorker registration failed');
                    });
            });
        }

        // Handle back button
        window.addEventListener('popstate', function(event) {
            closeModal();
        });

        // Auto-refresh data every 5 minutes
        setInterval(() => {
            if (!document.getElementById('loading').style.display) {
                location.reload();
            }
        }, 300000);
    </script>
</body>
</html>