<?php
// maintenance.php - Sleep Tracker Maintenance & Cleanup System

require_once 'config.php';

class SleepTrackerMaintenance {
    
    private $logFile;
    private $results = [];
    
    public function __construct() {
        $this->logFile = 'logs/maintenance_' . date('Y-m-d') . '.log';
        $this->log("Maintenance session started");
    }
    
    /**
     * Run all maintenance tasks
     */
    public function runAll() {
        $this->log("Starting comprehensive maintenance");
        
        $tasks = [
            'cleanupOldBackups' => 'Cleanup Old Backup Files',
            'cleanupTempFiles' => 'Cleanup Temporary Files',
            'cleanupOldLogs' => 'Cleanup Old Log Files',
            'optimizeDataFiles' => 'Optimize Data Files',
            'validateDataIntegrity' => 'Validate Data Integrity',
            'generateBackup' => 'Generate Fresh Backup',
            'updateStatistics' => 'Update Statistics Cache',
            'cleanupRateLimits' => 'Cleanup Rate Limit Files',
            'archiveOldData' => 'Archive Old Data',
            'systemHealthCheck' => 'System Health Check'
        ];
        
        foreach ($tasks as $method => $description) {
            $this->log("Running: $description");
            $startTime = microtime(true);
            
            try {
                $result = $this->$method();
                $duration = round((microtime(true) - $startTime) * 1000, 2);
                
                $this->results[$method] = [
                    'status' => 'success',
                    'description' => $description,
                    'result' => $result,
                    'duration_ms' => $duration
                ];
                
                $this->log("✅ $description completed in {$duration}ms");
                
            } catch (Exception $e) {
                $duration = round((microtime(true) - $startTime) * 1000, 2);
                
                $this->results[$method] = [
                    'status' => 'error',
                    'description' => $description,
                    'error' => $e->getMessage(),
                    'duration_ms' => $duration
                ];
                
                $this->log("❌ $description failed: " . $e->getMessage());
            }
        }
        
        $this->log("Maintenance session completed");
        return $this->results;
    }
    
    /**
     * Cleanup old backup files
     */
    public function cleanupOldBackups() {
        $backupFiles = glob('backup/*.json');
        $cleaned = 0;
        $totalSize = 0;
        
        // Group by month
        $monthlyBackups = [];
        foreach ($backupFiles as $file) {
            $filename = basename($file);
            if (preg_match('/^(\d{4}-\d{2})_/', $filename, $matches)) {
                $month = $matches[1];
                if (!isset($monthlyBackups[$month])) {
                    $monthlyBackups[$month] = [];
                }
                $monthlyBackups[$month][] = $file;
            }
        }
        
        // Keep only recent backups per month
        foreach ($monthlyBackups as $month => $files) {
            // Sort by modification time
            usort($files, function($a, $b) {
                return filemtime($b) - filemtime($a);
            });
            
            // Keep only the latest N backups per month
            $toDelete = array_slice($files, SleepTrackerConfig::MAX_BACKUP_FILES_PER_MONTH);
            
            foreach ($toDelete as $file) {
                $size = filesize($file);
                if (unlink($file)) {
                    $cleaned++;
                    $totalSize += $size;
                }
            }
        }
        
        return [
            'files_deleted' => $cleaned,
            'space_freed_kb' => round($totalSize / 1024, 2)
        ];
    }
    
    /**
     * Cleanup temporary files
     */
    public function cleanupTempFiles() {
        $tempFiles = glob('temp/*');
        $cleaned = 0;
        $totalSize = 0;
        $cutoffTime = time() - 3600; // Files older than 1 hour
        
        foreach ($tempFiles as $file) {
            if (is_file($file) && filemtime($file) < $cutoffTime) {
                $size = filesize($file);
                if (unlink($file)) {
                    $cleaned++;
                    $totalSize += $size;
                }
            }
        }
        
        return [
            'files_deleted' => $cleaned,
            'space_freed_kb' => round($totalSize / 1024, 2)
        ];
    }
    
    /**
     * Cleanup old log files
     */
    public function cleanupOldLogs() {
        $logFiles = glob('logs/*.log');
        $cleaned = 0;
        $totalSize = 0;
        $cutoffTime = time() - (30 * 24 * 3600); // 30 days
        
        foreach ($logFiles as $file) {
            if (filemtime($file) < $cutoffTime) {
                $size = filesize($file);
                if (unlink($file)) {
                    $cleaned++;
                    $totalSize += $size;
                }
            }
        }
        
        return [
            'files_deleted' => $cleaned,
            'space_freed_kb' => round($totalSize / 1024, 2)
        ];
    }
    
    /**
     * Optimize data files
     */
    public function optimizeDataFiles() {
        $dataFiles = glob('data/*.json');
        $optimized = 0;
        $spaceSaved = 0;
        
        foreach ($dataFiles as $file) {
            $originalSize = filesize($file);
            $data = json_decode(file_get_contents($file), true);
            
            if ($data) {
                // Sort by date
                usort($data, function($a, $b) {
                    return strtotime($a['date']) - strtotime($b['date']);
                });
                
                // Remove duplicates and validate
                $uniqueData = [];
                $seenDates = [];
                
                foreach ($data as $entry) {
                    if (!in_array($entry['date'], $seenDates)) {
                        // Validate and clean entry
                        $cleanEntry = $this->cleanDataEntry($entry);
                        if ($cleanEntry) {
                            $uniqueData[] = $cleanEntry;
                            $seenDates[] = $entry['date'];
                        }
                    }
                }
                
                // Save optimized data
                $optimizedContent = json_encode($uniqueData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                file_put_contents($file, $optimizedContent);
                
                $newSize = filesize($file);
                $spaceSaved += ($originalSize - $newSize);
                $optimized++;
            }
        }
        
        return [
            'files_optimized' => $optimized,
            'space_saved_kb' => round($spaceSaved / 1024, 2)
        ];
    }
    
    /**
     * Validate data integrity
     */
    public function validateDataIntegrity() {
        $dataFiles = glob('data/*.json');
        $issues = [];
        $totalRecords = 0;
        $validRecords = 0;
        
        foreach ($dataFiles as $file) {
            $data = json_decode(file_get_contents($file), true);
            
            if (!$data) {
                $issues[] = "Invalid JSON in file: " . basename($file);
                continue;
            }
            
            foreach ($data as $index => $entry) {
                $totalRecords++;
                
                // Check required fields
                if (!isset($entry['date']) || !isset($entry['sleepSessions'])) {
                    $issues[] = "Missing required fields in " . basename($file) . " at index $index";
                    continue;
                }
                
                // Validate date format
                if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $entry['date'])) {
                    $issues[] = "Invalid date format in " . basename($file) . ": " . $entry['date'];
                    continue;
                }
                
                // Validate sleep sessions
                foreach ($entry['sleepSessions'] as $sessionIndex => $session) {
                    if (!isset($session['bedTime']) || !isset($session['wakeTime']) || !isset($session['duration'])) {
                        $issues[] = "Invalid sleep session in " . basename($file) . " at entry $index, session $sessionIndex";
                        continue 2;
                    }
                }
                
                $validRecords++;
            }
        }
        
        return [
            'total_records' => $totalRecords,
            'valid_records' => $validRecords,
            'issues_found' => count($issues),
            'issues' => array_slice($issues, 0, 10) // Show first 10 issues
        ];
    }
    
    /**
     * Generate fresh backup
     */
    public function generateBackup() {
        $timestamp = date('Y-m-d_H-i-s');
        $backupFile = "backup/maintenance_backup_{$timestamp}.json";
        
        $allData = [];
        $dataFiles = glob('data/*.json');
        
        foreach ($dataFiles as $file) {
            $monthData = json_decode(file_get_contents($file), true) ?: [];
            $allData[basename($file, '.json')] = $monthData;
        }
        
        $backupData = [
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => 'maintenance_system',
            'version' => SleepTrackerConfig::APP_VERSION,
            'owner' => SleepTrackerConfig::APP_OWNER,
            'data' => $allData,
            'statistics' => $this->generateStatistics($allData)
        ];
        
        $success = file_put_contents($backupFile, json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        return [
            'backup_file' => basename($backupFile),
            'size_kb' => round(filesize($backupFile) / 1024, 2),
            'success' => $success !== false
        ];
    }
    
    /**
     * Update statistics cache
     */
    public function updateStatistics() {
        $cacheFile = 'temp/statistics_cache.json';
        
        $allData = [];
        $dataFiles = glob('data/*.json');
        
        foreach ($dataFiles as $file) {
            $monthData = json_decode(file_get_contents($file), true) ?: [];
            $allData = array_merge($allData, $monthData);
        }
        
        $statistics = $this->generateStatistics($allData);
        $statistics['cached_at'] = date('Y-m-d H:i:s');
        $statistics['cache_expires'] = date('Y-m-d H:i:s', strtotime('+1 hour'));
        
        $success = file_put_contents($cacheFile, json_encode($statistics, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        
        return [
            'cache_file' => basename($cacheFile),
            'size_kb' => round(filesize($cacheFile) / 1024, 2),
            'records_processed' => count($allData),
            'success' => $success !== false
        ];
    }
    
    /**
     * Cleanup rate limit files
     */
    public function cleanupRateLimits() {
        $rateLimitFile = 'temp/rate_limits.json';
        $cleaned = 0;
        
        if (file_exists($rateLimitFile)) {
            $rateLimits = json_decode(file_get_contents($rateLimitFile), true) ?: [];
            $currentTime = time();
            $originalCount = count($rateLimits);
            
            // Remove entries older than 1 hour
            $rateLimits = array_filter($rateLimits, function($timestamp) use ($currentTime) {
                return ($currentTime - $timestamp) < 3600;
            });
            
            $cleaned = $originalCount - count($rateLimits);
            file_put_contents($rateLimitFile, json_encode($rateLimits));
        }
        
        return [
            'entries_cleaned' => $cleaned
        ];
    }
    
    /**
     * Archive old data
     */
    public function archiveOldData() {
        $cutoffDate = date('Y-m-d', strtotime('-' . SleepTrackerConfig::DATA_RETENTION_MONTHS . ' months'));
        $archived = 0;
        $archivedSize = 0;
        
        $dataFiles = glob('data/*.json');
        
        foreach ($dataFiles as $file) {
            $month = basename($file, '.json');
            
            if ($month < substr($cutoffDate, 0, 7)) {
                // Create archive directory if not exists
                if (!file_exists('archive')) {
                    mkdir('archive', 0755, true);
                }
                
                $archiveFile = "archive/" . basename($file);
                $size = filesize($file);
                
                if (rename($file, $archiveFile)) {
                    $archived++;
                    $archivedSize += $size;
                }
            }
        }
        
        return [
            'files_archived' => $archived,
            'size_archived_kb' => round($archivedSize / 1024, 2)
        ];
    }
    
    /**
     * System health check
     */
    public function systemHealthCheck() {
        $health = SleepTrackerConfig::healthCheck();
        $systemInfo = SleepTrackerConfig::getSystemInfo();
        $stats = SleepTrackerConfig::getAppStats();
        
        $issues = [];
        
        // Check health status
        if ($health['status'] !== 'healthy') {
            foreach ($health['checks'] as $check => $result) {
                if (!$result) {
                    $issues[] = "Health check failed: $check";
                }
            }
        }
        
        // Check disk space
        $freeSpace = disk_free_space('.');
        $totalSpace = disk_total_space('.');
        $usedPercent = (($totalSpace - $freeSpace) / $totalSpace) * 100;
        
        if ($usedPercent > 90) {
            $issues[] = "Low disk space: " . round($usedPercent, 1) . "% used";
        }
        
        // Check memory usage
        $memoryUsage = memory_get_usage(true);
        $memoryLimit = ini_get('memory_limit');
        
        if ($memoryLimit != -1) {
            $memoryLimitBytes = $this->parseMemoryLimit($memoryLimit);
            $memoryPercent = ($memoryUsage / $memoryLimitBytes) * 100;
            
            if ($memoryPercent > 80) {
                $issues[] = "High memory usage: " . round($memoryPercent, 1) . "%";
            }
        }
        
        return [
            'overall_status' => empty($issues) ? 'healthy' : 'issues_detected',
            'issues' => $issues,
            'disk_free_mb' => round($freeSpace / 1024 / 1024, 2),
            'disk_used_percent' => round($usedPercent, 1),
            'memory_usage_mb' => round($memoryUsage / 1024 / 1024, 2),
            'app_stats' => $stats
        ];
    }
    
    /**
     * Clean and validate data entry
     */
    private function cleanDataEntry($entry) {
        // Required fields
        if (!isset($entry['date']) || !isset($entry['sleepSessions'])) {
            return null;
        }
        
        $cleanEntry = [
            'date' => $entry['date'],
            'sleepSessions' => [],
            'workActivity' => $entry['workActivity'] ?? '',
            'location' => $entry['location'] ?? '',
            'notes' => $entry['notes'] ?? '',
            'createdAt' => $entry['createdAt'] ?? date('Y-m-d H:i:s'),
            'updatedAt' => date('Y-m-d H:i:s')
        ];
        
        // Clean sleep sessions
        foreach ($entry['sleepSessions'] as $session) {
            if (isset($session['bedTime']) && isset($session['wakeTime']) && isset($session['type'])) {
                $cleanSession = [
                    'type' => $session['type'],
                    'bedTime' => $session['bedTime'],
                    'wakeTime' => $session['wakeTime'],
                    'duration' => $session['duration'] ?? SleepTrackerConfig::calculateSleepDuration($session['bedTime'], $session['wakeTime']),
                    'cycles' => $session['cycles'] ?? round(($session['duration'] ?? 0) / 1.5),
                    'quality' => $session['quality'] ?? 'Unknown',
                    'createdAt' => $session['createdAt'] ?? date('Y-m-d H:i:s')
                ];
                
                $cleanEntry['sleepSessions'][] = $cleanSession;
            }
        }
        
        return $cleanEntry;
    }
    
    /**
     * Generate comprehensive statistics
     */
    private function generateStatistics($allData) {
        $stats = [
            'total_days' => 0,
            'total_sessions' => 0,
            'total_sleep_hours' => 0,
            'average_daily_hours' => 0,
            'quality_distribution' => [],
            'type_distribution' => [],
            'monthly_summary' => []
        ];
        
        $flatData = [];
        foreach ($allData as $monthData) {
            $flatData = array_merge($flatData, $monthData);
        }
        
        foreach ($flatData as $entry) {
            if (!empty($entry['sleepSessions'])) {
                $stats['total_days']++;
                
                foreach ($entry['sleepSessions'] as $session) {
                    $stats['total_sessions']++;
                    $stats['total_sleep_hours'] += $session['duration'];
                    
                    $quality = $session['quality'];
                    $type = $session['type'];
                    
                    $stats['quality_distribution'][$quality] = ($stats['quality_distribution'][$quality] ?? 0) + 1;
                    $stats['type_distribution'][$type] = ($stats['type_distribution'][$type] ?? 0) + 1;
                }
                
                $month = substr($entry['date'], 0, 7);
                if (!isset($stats['monthly_summary'][$month])) {
                    $stats['monthly_summary'][$month] = ['days' => 0, 'sessions' => 0, 'hours' => 0];
                }
                
                $stats['monthly_summary'][$month]['days']++;
                $stats['monthly_summary'][$month]['sessions'] += count($entry['sleepSessions']);
                
                foreach ($entry['sleepSessions'] as $session) {
                    $stats['monthly_summary'][$month]['hours'] += $session['duration'];
                }
            }
        }
        
        if ($stats['total_days'] > 0) {
            $stats['average_daily_hours'] = round($stats['total_sleep_hours'] / $stats['total_days'], 2);
        }
        
        return $stats;
    }
    
    /**
     * Parse PHP memory limit
     */
    private function parseMemoryLimit($limit) {
        $limit = trim($limit);
        $unit = strtolower($limit[strlen($limit) - 1]);
        $value = (int)$limit;
        
        switch ($unit) {
            case 'g': return $value * 1024 * 1024 * 1024;
            case 'm': return $value * 1024 * 1024;
            case 'k': return $value * 1024;
            default: return $value;
        }
    }
    
    /**
     * Log maintenance activity
     */
    private function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message" . PHP_EOL;
        file_put_contents($this->logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * Get maintenance results
     */
    public function getResults() {
        return $this->results;
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    echo "Sleep Tracker Maintenance System\n";
    echo "=================================\n\n";
    
    $maintenance = new SleepTrackerMaintenance();
    $results = $maintenance->runAll();
    
    echo "Maintenance Results:\n";
    echo "==================\n\n";
    
    foreach ($results as $task => $result) {
        $status = $result['status'] === 'success' ? '✅' : '❌';
        echo "$status {$result['description']}\n";
        
        if ($result['status'] === 'success' && isset($result['result'])) {
            foreach ($result['result'] as $key => $value) {
                echo "   - " . ucwords(str_replace('_', ' ', $key)) . ": $value\n";
            }
        } elseif ($result['status'] === 'error') {
            echo "   - Error: {$result['error']}\n";
        }
        
        echo "   - Duration: {$result['duration_ms']}ms\n\n";
    }
    
    exit(0);
}

// Web interface
if (basename($_SERVER['PHP_SELF']) === 'maintenance.php') {
    // Handle AJAX requests
    if (isset($_GET['action']) && $_GET['action'] === 'run') {
        header('Content-Type: application/json');
        
        $maintenance = new SleepTrackerMaintenance();
        $results = $maintenance->runAll();
        
        echo json_encode([
            'success' => true,
            'results' => $results,
            'summary' => [
                'total_tasks' => count($results),
                'successful_tasks' => count(array_filter($results, function($r) { return $r['status'] === 'success'; })),
                'failed_tasks' => count(array_filter($results, function($r) { return $r['status'] === 'error'; }))
            ]
        ]);
        exit;
    }
    
    // Display web interface
    ?>
    <!DOCTYPE html>
    <html lang="id">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Maintenance - Sleep Tracker</title>
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
        <style>
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                padding: 20px;
                margin: 0;
            }

            .container {
                max-width: 1000px;
                margin: 0 auto;
                background: rgba(255, 255, 255, 0.95);
                border-radius: 20px;
                padding: 30px;
                box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            }

            .header {
                text-align: center;
                margin-bottom: 30px;
            }

            .header h1 {
                color: #667eea;
                margin-bottom: 10px;
            }

            .maintenance-btn {
                background: linear-gradient(45deg, #667eea, #764ba2);
                color: white;
                padding: 15px 30px;
                border: none;
                border-radius: 25px;
                font-size: 16px;
                cursor: pointer;
                margin: 20px 0;
            }

            .results {
                display: none;
                margin-top: 30px;
            }

            .task-result {
                background: white;
                border-radius: 10px;
                padding: 15px;
                margin: 10px 0;
                border-left: 4px solid;
            }

            .task-result.success {
                border-left-color: #28a745;
            }

            .task-result.error {
                border-left-color: #dc3545;
            }

            .loading {
                text-align: center;
                padding: 20px;
                display: none;
            }

            .nav-links {
                text-align: center;
                margin-bottom: 20px;
            }

            .nav-links a {
                display: inline-block;
                padding: 8px 15px;
                margin: 0 5px;
                background: #667eea;
                color: white;
                text-decoration: none;
                border-radius: 20px;
                font-size: 0.9em;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1><i class="fas fa-tools"></i> System Maintenance</h1>
                <p>Perawatan dan Optimasi Sleep Tracker</p>
            </div>

            <div class="nav-links">
                <a href="index.php"><i class="fas fa-home"></i> Dashboard</a>
                <a href="analytics.php"><i class="fas fa-chart-bar"></i> Analytics</a>
                <a href="config.php"><i class="fas fa-cog"></i> Config</a>
            </div>

            <div style="text-align: center;">
                <button id="runMaintenance" class="maintenance-btn">
                    <i class="fas fa-play"></i> Run Full Maintenance
                </button>
                
                <div class="loading" id="loading">
                    <i class="fas fa-spinner fa-spin" style="font-size: 2em; color: #667eea;"></i>
                    <p>Running maintenance tasks...</p>
                </div>
            </div>

            <div class="results" id="results"></div>
        </div>

        <script>
            document.getElementById('runMaintenance').addEventListener('click', function() {
                this.style.display = 'none';
                document.getElementById('loading').style.display = 'block';
                document.getElementById('results').style.display = 'none';
                
                fetch('?action=run')
                    .then(response => response.json())
                    .then(data => {
                        document.getElementById('loading').style.display = 'none';
                        displayResults(data);
                    })
                    .catch(error => {
                        document.getElementById('loading').style.display = 'none';
                        alert('Error running maintenance: ' + error);
                    });
            });

            function displayResults(data) {
                const resultsDiv = document.getElementById('results');
                resultsDiv.innerHTML = '';
                
                if (data.success) {
                    let html = `<h3>Maintenance Completed</h3>`;
                    html += `<p><strong>Summary:</strong> ${data.summary.successful_tasks}/${data.summary.total_tasks} tasks completed successfully</p>`;
                    
                    for (const [task, result] of Object.entries(data.results)) {
                        html += `<div class="task-result ${result.status}">`;
                        html += `<h4>${result.status === 'success' ? '✅' : '❌'} ${result.description}</h4>`;
                        
                        if (result.status === 'success' && result.result) {
                            for (const [key, value] of Object.entries(result.result)) {
                                html += `<p><strong>${key.replace(/_/g, ' ')}:</strong> ${value}</p>`;
                            }
                        } else if (result.status === 'error') {
                            html += `<p><strong>Error:</strong> ${result.error}</p>`;
                        }
                        
                        html += `<p><small>Duration: ${result.duration_ms}ms</small></p>`;
                        html += `</div>`;
                    }
                    
                    resultsDiv.innerHTML = html;
                }
                
                resultsDiv.style.display = 'block';
                document.getElementById('runMaintenance').style.display = 'inline-block';
            }
        </script>
    </body>
    </html>
    <?php
}
?>