<?php
session_start();

// Pastikan direktori data dan backup ada
if (!file_exists('data')) {
    mkdir('data', 0755, true);
}
if (!file_exists('backup')) {
    mkdir('backup', 0755, true);
}

// Get current date for display
$currentDate = date('Y-m-d');
$currentMonth = date('Y-m');
$currentTime = date('H:i');

// Load current month data
$dataFile = "data/{$currentMonth}.json";
$monthData = [];
if (file_exists($dataFile)) {
    $monthData = json_decode(file_get_contents($dataFile), true) ?: [];
}

// Get today's data if exists
$todayData = [];
foreach ($monthData as $entry) {
    if ($entry['date'] === $currentDate) {
        $todayData = $entry;
        break;
    }
}

// Initialize sleep sessions if not exists
if (!isset($todayData['sleepSessions'])) {
    $todayData['sleepSessions'] = [];
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sleep Tracker - Rizaldi</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 30px;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            color: #333;
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .greeting {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            padding: 15px 25px;
            border-radius: 15px;
            text-align: center;
            margin-bottom: 30px;
            font-size: 1.2em;
            font-weight: 600;
        }

        .main-content {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 30px;
            margin-bottom: 30px;
        }

        .card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            border: 1px solid rgba(102, 126, 234, 0.1);
        }

        .card h2 {
            color: #667eea;
            margin-bottom: 20px;
            font-size: 1.4em;
            border-bottom: 2px solid #f0f0f0;
            padding-bottom: 10px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #555;
        }

        .form-row {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        input, select, textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e1e5e9;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
        }

        input:focus, select:focus, textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .btn {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .btn-success {
            background: linear-gradient(45deg, #4CAF50, #45a049);
        }

        .btn-danger {
            background: linear-gradient(45deg, #f44336, #d32f2f);
        }

        .btn-secondary {
            background: linear-gradient(45deg, #6c757d, #545b62);
        }

        .sleep-sessions {
            margin-top: 20px;
        }

        .session-item {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 15px;
            display: flex;
            justify-content: between;
            align-items: center;
        }

        .session-info {
            flex: 1;
        }

        .session-type {
            font-weight: 600;
            color: #667eea;
            margin-bottom: 5px;
        }

        .session-time {
            color: #666;
            font-size: 0.9em;
        }

        .session-duration {
            background: #667eea;
            color: white;
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8em;
            margin-left: 15px;
        }

        .delete-session {
            background: #dc3545;
            color: white;
            border: none;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            cursor: pointer;
            margin-left: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .add-session-form {
            background: #e8f2ff;
            border: 2px dashed #667eea;
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
            display: none;
        }

        .add-session-form.active {
            display: block;
        }

        .prediction-card {
            grid-column: 1 / -1;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .prediction-card h2 {
            color: white;
            border-bottom-color: rgba(255,255,255,0.3);
        }

        .predictions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .prediction-item {
            background: rgba(255,255,255,0.1);
            border-radius: 10px;
            padding: 20px;
            backdrop-filter: blur(10px);
        }

        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }

        .stat-number {
            font-size: 2em;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #666;
            font-size: 0.9em;
        }

        .sleep-cycle {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-top: 30px;
            text-align: center;
        }

        .cycle-visualization {
            display: flex;
            justify-content: space-between;
            margin: 20px 0;
            height: 60px;
            align-items: end;
        }

        .cycle-bar {
            width: 15%;
            border-radius: 5px 5px 0 0;
            position: relative;
            transition: all 0.3s ease;
        }

        .cycle-bar.light { background: #a8d8f0; height: 30px; }
        .cycle-bar.deep { background: #667eea; height: 50px; }
        .cycle-bar.rem { background: #764ba2; height: 40px; }

        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #4CAF50;
            color: white;
            padding: 15px 20px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
            transform: translateX(400px);
            transition: transform 0.3s ease;
            z-index: 1000;
        }

        .notification.show {
            transform: translateX(0);
        }

        .notification.error {
            background: #f44336;
        }

        .data-status {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            border-left: 4px solid #667eea;
        }

        @media (max-width: 768px) {
            .main-content {
                grid-template-columns: 1fr;
            }
            
            .predictions {
                grid-template-columns: 1fr;
            }
            
            .form-row {
                flex-direction: column;
            }

            .session-item {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
        }

        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-moon"></i> Sleep Tracker</h1>
            <p>Pengelola Jam Tidur Pribadi dengan Prediksi Cerdas</p>
        </div>

        <div class="greeting">
            <i class="fas fa-hand-wave"></i>
            Halo Rizaldi! Selamat datang di aplikasi sleep tracker pribadi Anda
            <br><small><?php echo date('l, d F Y'); ?></small>
        </div>

        <div class="data-status">
            <strong><i class="fas fa-database"></i> Status Data:</strong> 
            <span id="dataStatus">
                <?php 
                $totalSessions = 0;
                foreach ($monthData as $entry) {
                    if (isset($entry['sleepSessions'])) {
                        $totalSessions += count($entry['sleepSessions']);
                    }
                }
                echo count($monthData) . " hari tercatat bulan ini • " . $totalSessions . " sesi tidur";
                ?>
            </span>
            <div style="font-size: 0.9em; color: #666; margin-top: 5px;">
                <i class="fas fa-save"></i> Auto-backup: <?php echo date('d/m/Y H:i'); ?>
            </div>
        </div>

        <div class="main-content">
            <!-- Input Form -->
            <div class="card">
                <h2><i class="fas fa-plus-circle"></i> Input Jam Tidur Hari Ini</h2>
                
                <!-- Today's Sessions -->
                <div class="sleep-sessions">
                    <h3 style="color: #667eea; margin-bottom: 15px;">
                        <i class="fas fa-bed"></i> Sesi Tidur Hari Ini (<?php echo count($todayData['sleepSessions']); ?>)
                    </h3>
                    
                    <div id="sessionsContainer">
                        <?php if (empty($todayData['sleepSessions'])): ?>
                            <div class="session-item" style="text-align: center; color: #666;">
                                <i class="fas fa-info-circle"></i> Belum ada sesi tidur hari ini
                            </div>
                        <?php else: ?>
                            <?php foreach ($todayData['sleepSessions'] as $index => $session): ?>
                                <div class="session-item">
                                    <div class="session-info">
                                        <div class="session-type">
                                            <i class="fas fa-<?php echo $session['type'] === 'malam' ? 'moon' : 'sun'; ?>"></i>
                                            Tidur <?php echo ucfirst($session['type']); ?>
                                        </div>
                                        <div class="session-time">
                                            <?php echo $session['bedTime']; ?> - <?php echo $session['wakeTime']; ?>
                                        </div>
                                    </div>
                                    <div class="session-duration">
                                        <?php echo $session['duration']; ?>h
                                    </div>
                                    <button class="delete-session" onclick="deleteSession(<?php echo $index; ?>)">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>

                    <button class="btn" onclick="toggleAddForm()" style="width: 100%; margin-top: 15px;">
                        <i class="fas fa-plus"></i> Tambah Waktu Tidur
                    </button>
                </div>

                <!-- Add Session Form -->
                <div class="add-session-form" id="addSessionForm">
                    <form id="sleepForm" action="sleep_tracker.php" method="POST">
                        <input type="hidden" name="action" value="add_session">
                        <input type="hidden" name="date" value="<?php echo $currentDate; ?>">

                        <div class="form-group">
                            <label><i class="fas fa-clock"></i> Jenis Tidur:</label>
                            <select name="sleepType" required>
                                <option value="">Pilih jenis tidur...</option>
                                <option value="pagi">Tidur Pagi (04:00-10:00)</option>
                                <option value="siang">Tidur Siang (11:00-15:00)</option>
                                <option value="sore">Tidur Sore (16:00-19:00)</option>
                                <option value="malam">Tidur Malam (20:00-03:59)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label><i class="fas fa-bed"></i> Jam Tidur:</label>
                            <div class="form-row">
                                <input type="time" name="bedTime" required value="<?php echo $currentTime; ?>">
                                <span style="min-width: 30px; text-align: center;">sampai</span>
                                <input type="time" name="wakeTime" required>
                            </div>
                        </div>

                        <div class="form-row">
                            <button type="submit" class="btn btn-success" style="flex: 1;">
                                <i class="fas fa-save"></i> Simpan Sesi
                            </button>
                            <button type="button" class="btn btn-secondary" onclick="toggleAddForm()" style="flex: 1;">
                                <i class="fas fa-times"></i> Batal
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Daily Activity Form -->
                <div style="margin-top: 30px; padding-top: 20px; border-top: 2px solid #f0f0f0;">
                    <h3 style="color: #667eea; margin-bottom: 15px;">
                        <i class="fas fa-chart-line"></i> Aktivitas Hari Ini
                    </h3>
                    
                    <form action="sleep_tracker.php" method="POST">
                        <input type="hidden" name="action" value="update_activity">
                        <input type="hidden" name="date" value="<?php echo $currentDate; ?>">

                        <div class="form-group">
                            <label><i class="fas fa-briefcase"></i> Aktivitas Pekerjaan:</label>
                            <select name="workActivity" required>
                                <option value="">Pilih aktivitas...</option>
                                <option value="ringan" <?php echo (isset($todayData['workActivity']) && $todayData['workActivity'] === 'ringan') ? 'selected' : ''; ?>>
                                    Ringan (Kantor/WFH)
                                </option>
                                <option value="sedang" <?php echo (isset($todayData['workActivity']) && $todayData['workActivity'] === 'sedang') ? 'selected' : ''; ?>>
                                    Sedang (Meeting/Presentasi)
                                </option>
                                <option value="berat" <?php echo (isset($todayData['workActivity']) && $todayData['workActivity'] === 'berat') ? 'selected' : ''; ?>>
                                    Berat (Fisik/Deadline)
                                </option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label><i class="fas fa-map-marker-alt"></i> Lokasi Aktivitas:</label>
                            <select name="location" required>
                                <option value="">Pilih lokasi...</option>
                                <option value="rumah" <?php echo (isset($todayData['location']) && $todayData['location'] === 'rumah') ? 'selected' : ''; ?>>
                                    Di Rumah
                                </option>
                                <option value="keluar" <?php echo (isset($todayData['location']) && $todayData['location'] === 'keluar') ? 'selected' : ''; ?>>
                                    Keluar Rumah (Lokal)
                                </option>
                                <option value="jauh" <?php echo (isset($todayData['location']) && $todayData['location'] === 'jauh') ? 'selected' : ''; ?>>
                                    Bepergian Jauh
                                </option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label><i class="fas fa-sticky-note"></i> Catatan (Opsional):</label>
                            <textarea name="notes" rows="3" placeholder="Catatan tambahan tentang hari ini..."><?php echo isset($todayData['notes']) ? htmlspecialchars($todayData['notes']) : ''; ?></textarea>
                        </div>

                        <button type="submit" class="btn" style="width: 100%;">
                            <i class="fas fa-save"></i> Update Aktivitas Harian
                        </button>
                    </form>
                </div>
            </div>

            <!-- Quick Stats -->
            <div class="card">
                <h2><i class="fas fa-chart-bar"></i> Statistik Cepat</h2>
                <div class="stats" id="quickStats">
                    <?php
                    // Calculate quick stats
                    $totalDuration = 0;
                    $totalSessions = 0;
                    $qualityCount = 0;
                    
                    foreach ($monthData as $entry) {
                        if (isset($entry['sleepSessions'])) {
                            foreach ($entry['sleepSessions'] as $session) {
                                $totalDuration += $session['duration'];
                                $totalSessions++;
                                if ($session['duration'] >= 1.5 && $session['duration'] <= 3) {
                                    $qualityCount++;
                                }
                            }
                        }
                    }
                    
                    $avgDuration = $totalSessions > 0 ? round($totalDuration / $totalSessions, 1) : 0;
                    $qualityPercent = $totalSessions > 0 ? round(($qualityCount / $totalSessions) * 100) : 0;
                    ?>
                    <div class="stat-card">
                        <div class="stat-number"><?php echo $avgDuration; ?>h</div>
                        <div class="stat-label">Rata-rata Tidur</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?php echo $totalSessions; ?></div>
                        <div class="stat-label">Total Sesi</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?php echo $qualityPercent; ?>%</div>
                        <div class="stat-label">Kualitas Tidur</div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-number"><?php echo count($monthData); ?></div>
                        <div class="stat-label">Hari Tercatat</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Predictions -->
        <div class="card prediction-card">
            <h2><i class="fas fa-crystal-ball"></i> Prediksi Tidur 3 Hari ke Depan</h2>
            <div class="predictions" id="predictions">
                <?php include 'generate_predictions.php'; ?>
            </div>
        </div>

        <!-- Sleep Cycle Visualization -->
        <div class="sleep-cycle">
            <h2 style="color: #667eea; margin-bottom: 20px;">
                <i class="fas fa-chart-area"></i> Analisis Pola Tidur
            </h2>
            <div id="sleepAnalysis">
                <?php if (!empty($todayData['sleepSessions'])): ?>
                    <div style="text-align: left;">
                        <h3 style="color: #667eea; margin-bottom: 15px;">Pola Tidur Hari Ini:</h3>
                        <?php
                        $totalDailyDuration = 0;
                        foreach ($todayData['sleepSessions'] as $session) {
                            $totalDailyDuration += $session['duration'];
                            $cycles = round($session['duration'] / 1.5);
                            echo "<div style='background: #f8f9fa; padding: 10px; margin: 10px 0; border-radius: 8px;'>";
                            echo "<strong>Tidur " . ucfirst($session['type']) . ":</strong> ";
                            echo $session['duration'] . "h (" . $cycles . " siklus) ";
                            echo "<small style='color: #666;'>• " . $session['bedTime'] . " - " . $session['wakeTime'] . "</small>";
                            echo "</div>";
                        }
                        echo "<div style='background: #667eea; color: white; padding: 15px; border-radius: 10px; margin-top: 15px; text-align: center;'>";
                        echo "<strong>Total Tidur Hari Ini: " . round($totalDailyDuration, 1) . " jam</strong>";
                        echo "</div>";
                        ?>
                    </div>
                <?php else: ?>
                    <div style="text-align: center; color: #666; padding: 40px;">
                        <i class="fas fa-bed" style="font-size: 3em; margin-bottom: 20px; opacity: 0.3;"></i>
                        <p>Belum ada data tidur untuk dianalisis</p>
                        <p><small>Mulai input data tidur untuk melihat analisis pola</small></p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Notification -->
    <div id="notification" class="notification"></div>

    <script>
        function toggleAddForm() {
            const form = document.getElementById('addSessionForm');
            form.classList.toggle('active');
        }

        function deleteSession(sessionIndex) {
            if (confirm('Yakin ingin menghapus sesi tidur ini?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'sleep_tracker.php';
                
                const actionInput = document.createElement('input');
                actionInput.type = 'hidden';
                actionInput.name = 'action';
                actionInput.value = 'delete_session';
                
                const dateInput = document.createElement('input');
                dateInput.type = 'hidden';
                dateInput.name = 'date';
                dateInput.value = '<?php echo $currentDate; ?>';
                
                const indexInput = document.createElement('input');
                indexInput.type = 'hidden';
                indexInput.name = 'sessionIndex';
                indexInput.value = sessionIndex;
                
                form.appendChild(actionInput);
                form.appendChild(dateInput);
                form.appendChild(indexInput);
                
                document.body.appendChild(form);
                form.submit();
            }
        }

        function showNotification(message, type = 'success') {
            const notification = document.getElementById('notification');
            notification.textContent = message;
            notification.className = `notification ${type} show`;
            
            setTimeout(() => {
                notification.classList.remove('show');
            }, 3000);
        }

        // Show notification if there's a message from PHP
        <?php if (isset($_SESSION['message'])): ?>
            showNotification('<?php echo $_SESSION['message']; ?>', '<?php echo $_SESSION['message_type'] ?? 'success'; ?>');
            <?php unset($_SESSION['message'], $_SESSION['message_type']); ?>
        <?php endif; ?>

        // Auto-refresh predictions every 5 minutes
        setInterval(() => {
            fetch('sleep_tracker.php?action=get_predictions')
                .then(response => response.text())
                .then(html => {
                    document.getElementById('predictions').innerHTML = html;
                });
        }, 300000);

        // Form validation
        document.getElementById('sleepForm').addEventListener('submit', function(e) {
            const bedTime = document.querySelector('input[name="bedTime"]').value;
            const wakeTime = document.querySelector('input[name="wakeTime"]').value;
            
            if (!bedTime || !wakeTime) {
                e.preventDefault();
                showNotification('Mohon lengkapi jam tidur dan bangun!', 'error');
                return;
            }

            // Calculate duration for validation
            const bedDate = new Date(`2000-01-01T${bedTime}`);
            let wakeDate = new Date(`2000-01-01T${wakeTime}`);
            
            if (wakeDate <= bedDate) {
                wakeDate.setDate(wakeDate.getDate() + 1);
            }
            
            const duration = (wakeDate - bedDate) / (1000 * 60 * 60);
            
            if (duration > 12) {
                e.preventDefault();
                showNotification('Durasi tidur terlalu lama! Maksimal 12 jam.', 'error');
                return;
            }
            
            if (duration < 0.5) {
                e.preventDefault();
                showNotification('Durasi tidur terlalu pendek! Minimal 30 menit.', 'error');
                return;
            }
        });
    </script>
</body>
</html>