<?php
// generate_predictions.php - AI Sleep Prediction System

function loadAllSleepData() {
    $allData = [];
    $dataFiles = glob('data/*.json');
    
    foreach ($dataFiles as $file) {
        $monthData = json_decode(file_get_contents($file), true) ?: [];
        $allData = array_merge($allData, $monthData);
    }
    
    // Sort by date
    usort($allData, function($a, $b) {
        return strtotime($a['date']) - strtotime($b['date']);
    });
    
    return $allData;
}

function analyzeWeekdayPattern($allData, $targetDayOfWeek) {
    $weekdayData = [];
    
    foreach ($allData as $entry) {
        $dayOfWeek = date('w', strtotime($entry['date']));
        if ($dayOfWeek == $targetDayOfWeek && !empty($entry['sleepSessions'])) {
            $weekdayData[] = $entry;
        }
    }
    
    return array_slice($weekdayData, -10); // Last 10 occurrences
}

function calculateOptimalBedtime($sessions, $workActivity, $location) {
    if (empty($sessions)) {
        return ['22:00', 8.0]; // Default
    }
    
    $nightSessions = array_filter($sessions, function($session) {
        return $session['type'] === 'malam';
    });
    
    if (empty($nightSessions)) {
        return ['22:00', 8.0];
    }
    
    // Calculate weighted average (recent data has more weight)
    $totalWeight = 0;
    $weightedBedMinutes = 0;
    $weightedDuration = 0;
    
    foreach ($nightSessions as $index => $session) {
        $weight = $index + 1; // More recent = higher weight
        $bedMinutes = (int)substr($session['bedTime'], 0, 2) * 60 + (int)substr($session['bedTime'], 3, 2);
        
        // Handle times after midnight
        if ($bedMinutes < 12 * 60) { // Before noon = next day
            $bedMinutes += 24 * 60;
        }
        
        $weightedBedMinutes += $bedMinutes * $weight;
        $weightedDuration += $session['duration'] * $weight;
        $totalWeight += $weight;
    }
    
    $avgBedMinutes = round($weightedBedMinutes / $totalWeight);
    $avgDuration = $weightedDuration / $totalWeight;
    
    // Adjust based on activity and location
    if ($workActivity === 'berat') {
        $avgDuration += 0.5; // Need more sleep after heavy work
        $avgBedMinutes -= 30; // Sleep earlier
    } elseif ($workActivity === 'ringan') {
        $avgBedMinutes += 15; // Can sleep a bit later
    }
    
    if ($location === 'jauh') {
        $avgDuration += 1.0; // Travel fatigue
        $avgBedMinutes -= 45; // Sleep much earlier
    } elseif ($location === 'keluar') {
        $avgDuration += 0.3;
        $avgBedMinutes -= 15;
    }
    
    // Convert back to time format
    $avgBedMinutes = $avgBedMinutes % (24 * 60);
    $hours = floor($avgBedMinutes / 60);
    $minutes = $avgBedMinutes % 60;
    
    $bedTime = sprintf('%02d:%02d', $hours, $minutes);
    
    return [$bedTime, round($avgDuration, 1)];
}

function generateSleepRecommendation($allData, $targetDate) {
    $dayOfWeek = date('w', strtotime($targetDate));
    $isWeekend = ($dayOfWeek == 0 || $dayOfWeek == 6);
    $dayName = date('l', strtotime($targetDate));
    
    // Analyze historical pattern for this day of week
    $weekdayData = analyzeWeekdayPattern($allData, $dayOfWeek);
    
    if (empty($weekdayData)) {
        // No historical data, use smart defaults
        $bedTime = $isWeekend ? '23:30' : '22:30';
        $duration = $isWeekend ? 8.5 : 7.5;
        $confidence = 30;
    } else {
        // Extract all sleep sessions from relevant days
        $allSessions = [];
        $activities = [];
        $locations = [];
        
        foreach ($weekdayData as $entry) {
            $allSessions = array_merge($allSessions, $entry['sleepSessions']);
            if (!empty($entry['workActivity'])) {
                $activities[] = $entry['workActivity'];
            }
            if (!empty($entry['location'])) {
                $locations[] = $entry['location'];
            }
        }
        
        // Predict activity and location based on historical pattern
        $predictedActivity = !empty($activities) ? array_count_values($activities) : ['sedang' => 1];
        $predictedActivity = array_keys($predictedActivity, max($predictedActivity))[0];
        
        $predictedLocation = !empty($locations) ? array_count_values($locations) : ['rumah' => 1];
        $predictedLocation = array_keys($predictedLocation, max($predictedLocation))[0];
        
        list($bedTime, $duration) = calculateOptimalBedtime($allSessions, $predictedActivity, $predictedLocation);
        
        // Weekend adjustments
        if ($isWeekend) {
            $bedTimeMinutes = (int)substr($bedTime, 0, 2) * 60 + (int)substr($bedTime, 3, 2);
            $bedTimeMinutes += 30; // Sleep 30 min later on weekends
            $bedTimeMinutes = $bedTimeMinutes % (24 * 60);
            
            $hours = floor($bedTimeMinutes / 60);
            $minutes = $bedTimeMinutes % 60;
            $bedTime = sprintf('%02d:%02d', $hours, $minutes);
            
            $duration += 0.5; // Sleep 30 min longer on weekends
        }
        
        $confidence = min(95, 40 + (count($weekdayData) * 8)); // More data = higher confidence
    }
    
    // Calculate wake time
    $bedTimeMinutes = (int)substr($bedTime, 0, 2) * 60 + (int)substr($bedTime, 3, 2);
    $wakeTimeMinutes = ($bedTimeMinutes + ($duration * 60)) % (24 * 60);
    
    $wakeHours = floor($wakeTimeMinutes / 60);
    $wakeMinutes = $wakeTimeMinutes % 60;
    $wakeTime = sprintf('%02d:%02d', $wakeHours, $wakeMinutes);
    
    // Calculate sleep cycles (90 minutes each)
    $cycles = round($duration / 1.5);
    
    // Determine sleep quality prediction
    $quality = 'Good';
    if ($cycles >= 5 && $cycles <= 6) {
        $quality = 'Excellent';
    } elseif ($cycles < 4 || $cycles > 7) {
        $quality = 'Fair';
    }
    
    return [
        'date' => $targetDate,
        'dayName' => $dayName,
        'bedTime' => $bedTime,
        'wakeTime' => $wakeTime,
        'duration' => $duration,
        'cycles' => $cycles,
        'quality' => $quality,
        'confidence' => $confidence,
        'isWeekend' => $isWeekend,
        'predictedActivity' => $predictedActivity ?? 'sedang',
        'predictedLocation' => $predictedLocation ?? 'rumah'
    ];
}

function getRecommendationIcon($quality) {
    switch ($quality) {
        case 'Excellent': return '🌟';
        case 'Good': return '👍';
        case 'Fair': return '⚠️';
        default: return '❌';
    }
}

function getActivityIcon($activity) {
    switch ($activity) {
        case 'ringan': return '💻';
        case 'sedang': return '📋';
        case 'berat': return '💪';
        default: return '📋';
    }
}

function getLocationIcon($location) {
    switch ($location) {
        case 'rumah': return '🏠';
        case 'keluar': return '🚶';
        case 'jauh': return '✈️';
        default: return '🏠';
    }
}

// Generate predictions for next 3 days
$allData = loadAllSleepData();
$predictions = [];

for ($i = 1; $i <= 3; $i++) {
    $targetDate = date('Y-m-d', strtotime("+{$i} days"));
    $predictions[] = generateSleepRecommendation($allData, $targetDate);
}

// Display predictions
if (empty($allData) || count($allData) < 3) {
    echo '<div class="prediction-item" style="grid-column: 1 / -1; text-align: center;">';
    echo '<div class="prediction-date">🤖 Sistem Learning</div>';
    echo '<div style="font-size: 1.1em; margin-top: 10px;">';
    echo 'Butuh minimal 3 hari data untuk memberikan prediksi yang akurat.<br>';
    echo 'Current data: ' . count($allData) . '/3 hari';
    echo '<br><small style="opacity: 0.8; margin-top: 10px; display: block;">Mulai input data tidur untuk mendapatkan rekomendasi personal</small>';
    echo '</div>';
    echo '</div>';
} else {
    foreach ($predictions as $prediction) {
        $confidenceColor = $prediction['confidence'] >= 70 ? '#4CAF50' : ($prediction['confidence'] >= 50 ? '#FF9800' : '#f44336');
        $qualityIcon = getRecommendationIcon($prediction['quality']);
        $activityIcon = getActivityIcon($prediction['predictedActivity']);
        $locationIcon = getLocationIcon($prediction['predictedLocation']);
        
        echo '<div class="prediction-item">';
        echo '<div class="prediction-date">';
        echo $prediction['isWeekend'] ? '🎯' : '📅';
        echo ' ' . $prediction['dayName'] . '<br>';
        echo '<small>' . date('d M Y', strtotime($prediction['date'])) . '</small>';
        echo '</div>';
        
        echo '<div class="prediction-time">';
        echo '🛏️ ' . $prediction['bedTime'] . ' - 🌅 ' . $prediction['wakeTime'];
        echo '</div>';
        
        echo '<div style="font-size: 0.9em; margin-top: 8px;">';
        echo 'Durasi: ' . $prediction['duration'] . 'h • ' . $prediction['cycles'] . ' siklus ' . $qualityIcon;
        echo '</div>';
        
        echo '<div style="font-size: 0.85em; margin-top: 8px; opacity: 0.9;">';
        echo $activityIcon . ' ' . ucfirst($prediction['predictedActivity']) . ' • ';
        echo $locationIcon . ' ' . ucfirst($prediction['predictedLocation']);
        echo '</div>';
        
        echo '<div style="font-size: 0.8em; margin-top: 8px;">';
        echo '<span style="background: rgba(255,255,255,0.2); padding: 3px 8px; border-radius: 10px;">';
        echo 'AI Confidence: <span style="color: ' . $confidenceColor . '; font-weight: bold;">' . $prediction['confidence'] . '%</span>';
        echo '</span>';
        echo '</div>';
        
        // Add smart tips based on prediction
        if ($prediction['confidence'] < 50) {
            echo '<div style="font-size: 0.75em; margin-top: 8px; opacity: 0.8;">';
            echo '💡 Tip: Input lebih banyak data untuk prediksi yang lebih akurat';
            echo '</div>';
        } elseif ($prediction['cycles'] < 5) {
            echo '<div style="font-size: 0.75em; margin-top: 8px; opacity: 0.8;">';
            echo '💡 Tip: Pertimbangkan tidur lebih awal untuk siklus optimal';
            echo '</div>';
        } elseif ($prediction['duration'] > 9) {
            echo '<div style="font-size: 0.75em; margin-top: 8px; opacity: 0.8;">';
            echo '💡 Tip: Durasi tidur mungkin terlalu lama, coba kurangi sedikit';
            echo '</div>';
        }
        
        echo '</div>';
    }
    
    // Add AI insights
    echo '<div class="prediction-item" style="grid-column: 1 / -1; background: rgba(255,255,255,0.2); text-align: center;">';
    echo '<div class="prediction-date">🧠 AI Insights</div>';
    echo '<div style="font-size: 0.95em; margin-top: 10px; line-height: 1.4;">';
    
    // Analyze sleep pattern trends
    $recentData = array_slice($allData, -7); // Last 7 days
    $totalRecentDuration = 0;
    $totalRecentSessions = 0;
    
    foreach ($recentData as $entry) {
        foreach ($entry['sleepSessions'] as $session) {
            $totalRecentDuration += $session['duration'];
            $totalRecentSessions++;
        }
    }
    
    if ($totalRecentSessions > 0) {
        $avgRecentDuration = round($totalRecentDuration / $totalRecentSessions, 1);
        
        if ($avgRecentDuration > 8) {
            echo '😴 Anda cenderung tidur lebih lama dari rata-rata minggu ini (' . $avgRecentDuration . 'h). ';
        } elseif ($avgRecentDuration < 6) {
            echo '⚡ Anda kurang tidur minggu ini (' . $avgRecentDuration . 'h). Pertimbangkan tidur lebih awal. ';
        } else {
            echo '✅ Pola tidur Anda cukup konsisten minggu ini (' . $avgRecentDuration . 'h). ';
        }
        
        // Weekend vs weekday analysis
        $weekdayDuration = 0;
        $weekendDuration = 0;
        $weekdayCount = 0;
        $weekendCount = 0;
        
        foreach ($recentData as $entry) {
            $dayOfWeek = date('w', strtotime($entry['date']));
            $isWeekend = ($dayOfWeek == 0 || $dayOfWeek == 6);
            
            foreach ($entry['sleepSessions'] as $session) {
                if ($isWeekend) {
                    $weekendDuration += $session['duration'];
                    $weekendCount++;
                } else {
                    $weekdayDuration += $session['duration'];
                    $weekdayCount++;
                }
            }
        }
        
        if ($weekdayCount > 0 && $weekendCount > 0) {
            $avgWeekday = round($weekdayDuration / $weekdayCount, 1);
            $avgWeekend = round($weekendDuration / $weekendCount, 1);
            $difference = round($avgWeekend - $avgWeekday, 1);
            
            if ($difference > 2) {
                echo '<br>📊 Anda tidur ' . $difference . 'h lebih lama di weekend. Ini normal untuk recovery!';
            } elseif ($difference < 0) {
                echo '<br>📊 Tidur weekend Anda lebih pendek dari weekday. Manfaatkan weekend untuk istirahat lebih!';
            }
        }
    }
    
    echo '</div>';
    echo '</div>';
}
?>