<?php
// export.php - Advanced Data Export System

session_start();

function loadAllSleepData() {
    $allData = [];
    $dataFiles = glob('data/*.json');
    
    foreach ($dataFiles as $file) {
        $monthData = json_decode(file_get_contents($file), true) ?: [];
        $allData = array_merge($allData, $monthData);
    }
    
    usort($allData, function($a, $b) {
        return strtotime($a['date']) - strtotime($b['date']);
    });
    
    return $allData;
}

function exportToCSV($data, $filename) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    $output = fopen('php://output', 'w');
    
    // CSV Headers
    fputcsv($output, [
        'Tanggal', 'Jenis Tidur', 'Jam Tidur', 'Jam Bangun', 'Durasi (Jam)', 
        'Siklus', 'Kualitas', 'Aktivitas Kerja', 'Lokasi', 'Catatan', 
        'Hari dalam Minggu', 'Bulan', 'Tahun'
    ]);
    
    // Data rows
    foreach ($data as $entry) {
        $date = $entry['date'];
        $dayOfWeek = date('l', strtotime($date));
        $month = date('F', strtotime($date));
        $year = date('Y', strtotime($date));
        
        if (!empty($entry['sleepSessions'])) {
            foreach ($entry['sleepSessions'] as $session) {
                fputcsv($output, [
                    $date,
                    ucfirst($session['type']),
                    $session['bedTime'],
                    $session['wakeTime'],
                    $session['duration'],
                    $session['cycles'],
                    $session['quality'],
                    ucfirst($entry['workActivity'] ?? ''),
                    ucfirst($entry['location'] ?? ''),
                    $entry['notes'] ?? '',
                    $dayOfWeek,
                    $month,
                    $year
                ]);
            }
        } else {
            // Entry without sleep sessions
            fputcsv($output, [
                $date, '', '', '', '', '', '',
                ucfirst($entry['workActivity'] ?? ''),
                ucfirst($entry['location'] ?? ''),
                $entry['notes'] ?? '',
                $dayOfWeek, $month, $year
            ]);
        }
    }
    
    fclose($output);
    exit;
}

function exportToJSON($data, $filename) {
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

function generateSummaryReport($data) {
    $report = [
        'generated_at' => date('Y-m-d H:i:s'),
        'generated_for' => 'Rizaldi',
        'total_days' => 0,
        'total_sessions' => 0,
        'total_sleep_hours' => 0,
        'average_daily_hours' => 0,
        'date_range' => [
            'start' => '',
            'end' => ''
        ],
        'sleep_patterns' => [
            'by_type' => [],
            'by_quality' => [],
            'by_weekday' => [],
            'by_month' => []
        ],
        'activity_correlation' => [],
        'recommendations' => []
    ];
    
    if (empty($data)) {
        return $report;
    }
    
    $report['date_range']['start'] = $data[0]['date'];
    $report['date_range']['end'] = end($data)['date'];
    
    $weekdayHours = array_fill(0, 7, 0);
    $weekdayCounts = array_fill(0, 7, 0);
    $monthlyData = [];
    $typeData = [];
    $qualityData = [];
    $activityData = [];
    
    foreach ($data as $entry) {
        if (!empty($entry['sleepSessions'])) {
            $report['total_days']++;
            $dayOfWeek = date('w', strtotime($entry['date']));
            $month = date('Y-m', strtotime($entry['date']));
            
            $dailyTotal = 0;
            foreach ($entry['sleepSessions'] as $session) {
                $report['total_sessions']++;
                $report['total_sleep_hours'] += $session['duration'];
                $dailyTotal += $session['duration'];
                
                // Collect pattern data
                $typeData[$session['type']] = ($typeData[$session['type']] ?? 0) + 1;
                $qualityData[$session['quality']] = ($qualityData[$session['quality']] ?? 0) + 1;
            }
            
            $weekdayHours[$dayOfWeek] += $dailyTotal;
            $weekdayCounts[$dayOfWeek]++;
            
            if (!isset($monthlyData[$month])) {
                $monthlyData[$month] = ['hours' => 0, 'days' => 0];
            }
            $monthlyData[$month]['hours'] += $dailyTotal;
            $monthlyData[$month]['days']++;
            
            // Activity correlation
            if (!empty($entry['workActivity'])) {
                if (!isset($activityData[$entry['workActivity']])) {
                    $activityData[$entry['workActivity']] = ['hours' => 0, 'count' => 0];
                }
                $activityData[$entry['workActivity']]['hours'] += $dailyTotal;
                $activityData[$entry['workActivity']]['count']++;
            }
        }
    }
    
    if ($report['total_days'] > 0) {
        $report['average_daily_hours'] = round($report['total_sleep_hours'] / $report['total_days'], 2);
    }
    
    // Patterns
    $report['sleep_patterns']['by_type'] = $typeData;
    $report['sleep_patterns']['by_quality'] = $qualityData;
    
    $weekdays = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
    for ($i = 0; $i < 7; $i++) {
        $avg = $weekdayCounts[$i] > 0 ? round($weekdayHours[$i] / $weekdayCounts[$i], 2) : 0;
        $report['sleep_patterns']['by_weekday'][$weekdays[$i]] = [
            'average_hours' => $avg,
            'total_days' => $weekdayCounts[$i]
        ];
    }
    
    foreach ($monthlyData as $month => $monthInfo) {
        $report['sleep_patterns']['by_month'][$month] = [
            'average_hours' => round($monthInfo['hours'] / $monthInfo['days'], 2),
            'total_hours' => round($monthInfo['hours'], 2),
            'active_days' => $monthInfo['days']
        ];
    }
    
    // Activity correlation
    foreach ($activityData as $activity => $actInfo) {
        $report['activity_correlation'][$activity] = [
            'average_hours' => round($actInfo['hours'] / $actInfo['count'], 2),
            'total_days' => $actInfo['count']
        ];
    }
    
    // Simple recommendations
    if ($report['average_daily_hours'] < 7) {
        $report['recommendations'][] = 'Tingkatkan durasi tidur harian (saat ini ' . $report['average_daily_hours'] . 'h, disarankan 7-9h)';
    }
    
    $qualityPercent = 0;
    if (isset($qualityData['Excellent'])) $qualityPercent += $qualityData['Excellent'];
    if (isset($qualityData['Good'])) $qualityPercent += $qualityData['Good'];
    $qualityPercent = $report['total_sessions'] > 0 ? ($qualityPercent / $report['total_sessions']) * 100 : 0;
    
    if ($qualityPercent < 70) {
        $report['recommendations'][] = 'Tingkatkan kualitas tidur (saat ini ' . round($qualityPercent) . '% sesi berkualitas baik)';
    }
    
    return $report;
}

// Handle export requests
if (isset($_GET['action'])) {
    $allData = loadAllSleepData();
    
    switch ($_GET['action']) {
        case 'csv':
            $filename = 'sleep_data_rizaldi_' . date('Y-m-d') . '.csv';
            exportToCSV($allData, $filename);
            break;
            
        case 'json':
            $filename = 'sleep_data_rizaldi_' . date('Y-m-d') . '.json';
            exportToJSON($allData, $filename);
            break;
            
        case 'summary':
            $summary = generateSummaryReport($allData);
            $filename = 'sleep_summary_rizaldi_' . date('Y-m-d') . '.json';
            exportToJSON($summary, $filename);
            break;
            
        case 'backup_all':
            // Create a complete backup ZIP
            $zipFile = 'sleep_backup_rizaldi_' . date('Y-m-d_H-i-s') . '.zip';
            $zip = new ZipArchive();
            
            if ($zip->open($zipFile, ZipArchive::CREATE) === TRUE) {
                // Add all data files
                $dataFiles = glob('data/*.json');
                foreach ($dataFiles as $file) {
                    $zip->addFile($file, basename($file));
                }
                
                // Add backup files
                $backupFiles = glob('backup/*.json');
                foreach ($backupFiles as $file) {
                    $zip->addFile($file, 'backup/' . basename($file));
                }
                
                // Add summary report
                $summary = generateSummaryReport($allData);
                $zip->addFromString('summary_report.json', json_encode($summary, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
                
                $zip->close();
                
                header('Content-Type: application/zip');
                header('Content-Disposition: attachment; filename="' . $zipFile . '"');
                header('Content-Length: ' . filesize($zipFile));
                readfile($zipFile);
                unlink($zipFile); // Clean up temp file
                exit;
            }
            break;
    }
}

$allData = loadAllSleepData();
$summary = generateSummaryReport($allData);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Export Data - Sleep Tracker Rizaldi</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 30px;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            color: #333;
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .nav-links {
            text-align: center;
            margin-bottom: 30px;
        }

        .nav-links a {
            display: inline-block;
            padding: 10px 20px;
            margin: 0 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            text-decoration: none;
            border-radius: 25px;
            transition: all 0.3s ease;
        }

        .nav-links a:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        .export-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .export-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            border: 1px solid rgba(102, 126, 234, 0.1);
            text-align: center;
        }

        .export-card h3 {
            color: #667eea;
            margin-bottom: 15px;
            font-size: 1.3em;
        }

        .export-icon {
            font-size: 3em;
            margin-bottom: 15px;
            color: #667eea;
        }

        .export-description {
            color: #666;
            margin-bottom: 20px;
            line-height: 1.5;
        }

        .export-btn {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            padding: 12px 25px;
            border: none;
            border-radius: 25px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .export-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
        }

        .export-btn.secondary {
            background: linear-gradient(45deg, #6c757d, #545b62);
        }

        .export-btn.success {
            background: linear-gradient(45deg, #28a745, #20c997);
        }

        .export-btn.danger {
            background: linear-gradient(45deg, #dc3545, #fd7e14);
        }

        .summary-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            border: 1px solid rgba(102, 126, 234, 0.1);
            margin-bottom: 30px;
        }

        .summary-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }

        .stat-item {
            text-align: center;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 10px;
        }

        .stat-number {
            font-size: 1.8em;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #666;
            font-size: 0.9em;
        }

        .data-preview {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
            max-height: 300px;
            overflow-y: auto;
        }

        .data-preview h4 {
            margin-bottom: 15px;
            color: #667eea;
        }

        .data-preview pre {
            background: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 12px;
            overflow-x: auto;
            border: 1px solid #e9ecef;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: #666;
        }

        .empty-state i {
            font-size: 3em;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        .format-info {
            background: #e3f2fd;
            border: 1px solid #bbdefb;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
            font-size: 0.9em;
        }

        .format-info ul {
            margin: 10px 0;
            padding-left: 20px;
        }

        .recommendations {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 10px;
            padding: 15px;
            margin-top: 20px;
        }

        .recommendations h4 {
            color: #856404;
            margin-bottom: 10px;
        }

        .recommendations ul {
            margin: 0;
            padding-left: 20px;
        }

        .recommendations li {
            color: #856404;
            margin-bottom: 5px;
        }

        @media (max-width: 768px) {
            .export-grid {
                grid-template-columns: 1fr;
            }
            
            .summary-stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-download"></i> Export Data</h1>
            <p>Download dan Backup Data Sleep Tracker Rizaldi</p>
        </div>

        <div class="nav-links">
            <a href="index.php"><i class="fas fa-home"></i> Dashboard</a>
            <a href="analytics.php"><i class="fas fa-chart-bar"></i> Analytics</a>
            <a href="export.php"><i class="fas fa-download"></i> Export Data</a>
        </div>

        <?php if (empty($allData)): ?>
            <div class="empty-state">
                <i class="fas fa-download"></i>
                <h2>Belum Ada Data untuk Diekspor</h2>
                <p>Mulai input data tidur di dashboard untuk dapat mengekspornya</p>
                <a href="index.php" style="display: inline-block; margin-top: 20px; padding: 15px 30px; background: linear-gradient(45deg, #667eea, #764ba2); color: white; text-decoration: none; border-radius: 25px;">
                    <i class="fas fa-plus"></i> Input Data Tidur
                </a>
            </div>
        <?php else: ?>
            <!-- Data Summary -->
            <div class="summary-card">
                <h3><i class="fas fa-chart-pie"></i> Ringkasan Data yang Akan Diekspor</h3>
                <div class="summary-stats">
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $summary['total_days']; ?></div>
                        <div class="stat-label">Hari Tercatat</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $summary['total_sessions']; ?></div>
                        <div class="stat-label">Total Sesi</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo round($summary['total_sleep_hours'], 1); ?>h</div>
                        <div class="stat-label">Total Jam Tidur</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $summary['average_daily_hours']; ?>h</div>
                        <div class="stat-label">Rata-rata Harian</div>
                    </div>
                </div>
                
                <div style="text-align: center; color: #666;">
                    <i class="fas fa-calendar"></i> 
                    Periode: <?php echo date('d M Y', strtotime($summary['date_range']['start'])); ?> - 
                    <?php echo date('d M Y', strtotime($summary['date_range']['end'])); ?>
                </div>

                <?php if (!empty($summary['recommendations'])): ?>
                    <div class="recommendations">
                        <h4><i class="fas fa-lightbulb"></i> Rekomendasi Berdasarkan Data:</h4>
                        <ul>
                            <?php foreach ($summary['recommendations'] as $rec): ?>
                                <li><?php echo $rec; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Export Options -->
            <div class="export-grid">
                <!-- CSV Export -->
                <div class="export-card">
                    <div class="export-icon">
                        <i class="fas fa-file-csv"></i>
                    </div>
                    <h3>Export ke CSV</h3>
                    <div class="export-description">
                        Format spreadsheet yang kompatibel dengan Excel, Google Sheets, dan aplikasi lainnya. 
                        Ideal untuk analisis data lebih lanjut.
                    </div>
                    <a href="?action=csv" class="export-btn">
                        <i class="fas fa-download"></i> Download CSV
                    </a>
                    
                    <div class="format-info">
                        <strong>Kolom yang disertakan:</strong>
                        <ul>
                            <li>Tanggal, Jenis Tidur, Jam Tidur/Bangun</li>
                            <li>Durasi, Siklus, Kualitas</li>
                            <li>Aktivitas Kerja, Lokasi, Catatan</li>
                            <li>Hari, Bulan, Tahun</li>
                        </ul>
                    </div>
                </div>

                <!-- JSON Export -->
                <div class="export-card">
                    <div class="export-icon">
                        <i class="fas fa-file-code"></i>
                    </div>
                    <h3>Export ke JSON</h3>
                    <div class="export-description">
                        Format data mentah lengkap yang dapat digunakan untuk backup, migrasi, 
                        atau integrasi dengan aplikasi lain.
                    </div>
                    <a href="?action=json" class="export-btn secondary">
                        <i class="fas fa-download"></i> Download JSON
                    </a>
                    
                    <div class="format-info">
                        <strong>Kegunaan:</strong>
                        <ul>
                            <li>Backup data lengkap</li>
                            <li>Migrasi ke aplikasi lain</li>
                            <li>Import ulang ke sistem</li>
                            <li>Analisis dengan tools custom</li>
                        </ul>
                    </div>
                </div>

                <!-- Summary Report -->
                <div class="export-card">
                    <div class="export-icon">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <h3>Laporan Ringkasan</h3>
                    <div class="export-description">
                        Laporan analitik lengkap dengan statistik, pola tidur, dan rekomendasi 
                        dalam format yang mudah dibaca.
                    </div>
                    <a href="?action=summary" class="export-btn success">
                        <i class="fas fa-download"></i> Download Laporan
                    </a>
                    
                    <div class="format-info">
                        <strong>Isi laporan:</strong>
                        <ul>
                            <li>Statistik komprehensif</li>
                            <li>Pola tidur per hari/bulan</li>
                            <li>Korelasi aktivitas</li>
                            <li>Rekomendasi personal</li>
                        </ul>
                    </div>
                </div>

                <!-- Complete Backup -->
                <div class="export-card">
                    <div class="export-icon">
                        <i class="fas fa-archive"></i>
                    </div>
                    <h3>Backup Lengkap</h3>
                    <div class="export-description">
                        Archive ZIP berisi semua data, backup historis, dan laporan ringkasan. 
                        Backup paling komprehensif untuk arsip jangka panjang.
                    </div>
                    <a href="?action=backup_all" class="export-btn danger">
                        <i class="fas fa-download"></i> Download Backup ZIP
                    </a>
                    
                    <div class="format-info">
                        <strong>Isi backup:</strong>
                        <ul>
                            <li>Semua file data JSON</li>
                            <li>File backup historis</li>
                            <li>Laporan ringkasan</li>
                            <li>Metadata sistem</li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Data Preview -->
            <div class="summary-card">
                <h3><i class="fas fa-eye"></i> Preview Data Terbaru</h3>
                <div class="data-preview">
                    <h4>5 Entri Terakhir:</h4>
                    <pre><?php 
                        $latestData = array_slice($allData, -5);
                        echo json_encode($latestData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE); 
                    ?></pre>
                </div>
            </div>

            <!-- Export Tips -->
            <div class="summary-card">
                <h3><i class="fas fa-info-circle"></i> Tips Export Data</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                    <div>
                        <h4 style="color: #667eea; margin-bottom: 10px;">📊 Untuk Analisis:</h4>
                        <p>Gunakan format CSV untuk import ke Excel/Google Sheets dan buat grafik custom atau analisis statistik lebih mendalam.</p>
                    </div>
                    <div>
                        <h4 style="color: #667eea; margin-bottom: 10px;">💾 Untuk Backup:</h4>
                        <p>Format JSON dan ZIP backup ideal untuk menyimpan data jangka panjang dan restore di kemudian hari.</p>
                    </div>
                    <div>
                        <h4 style="color: #667eea; margin-bottom: 10px;">📱 Untuk Sharing:</h4>
                        <p>Laporan ringkasan dalam format JSON mudah dibaca dan dapat dibagikan dengan dokter atau sleep specialist.</p>
                    </div>
                    <div>
                        <h4 style="color: #667eea; margin-bottom: 10px;">🔄 Untuk Migrasi:</h4>
                        <p>File JSON dapat digunakan untuk pindah ke aplikasi sleep tracking lain atau sistem monitoring kesehatan.</p>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Auto-refresh page data every 5 minutes
        setTimeout(() => {
            location.reload();
        }, 300000);

        // Add download progress indicator
        document.querySelectorAll('.export-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const originalText = this.innerHTML;
                this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Downloading...';
                this.style.pointerEvents = 'none';
                
                setTimeout(() => {
                    this.innerHTML = originalText;
                    this.style.pointerEvents = 'auto';
                }, 3000);
            });
        });
    </script>
</body>
</html>