<?php
// api.php - REST API for Sleep Tracker

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Key');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once 'config.php';

class SleepTrackerAPI {
    
    private $validApiKeys = [
        'rizaldi_mobile_app' => 'sk_rizaldi_2025_mobile_access_key',
        'rizaldi_desktop' => 'sk_rizaldi_2025_desktop_access_key',
        'rizaldi_integration' => 'sk_rizaldi_2025_integration_key'
    ];
    
    private $rateLimits = [];
    
    public function __construct() {
        // Rate limiting check
        $this->checkRateLimit();
        
        // API Key authentication
        if (!$this->authenticate()) {
            $this->sendError(401, 'Invalid or missing API key');
        }
    }
    
    private function authenticate() {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? $_GET['api_key'] ?? null;
        
        if (!$apiKey) {
            return false;
        }
        
        return in_array($apiKey, $this->validApiKeys);
    }
    
    private function checkRateLimit() {
        $clientIP = $_SERVER['REMOTE_ADDR'];
        $currentTime = time();
        $hourKey = date('Y-m-d-H', $currentTime);
        $limitKey = $clientIP . '_' . $hourKey;
        
        // Load rate limits from temp file
        $rateLimitFile = 'temp/rate_limits.json';
        if (file_exists($rateLimitFile)) {
            $this->rateLimits = json_decode(file_get_contents($rateLimitFile), true) ?: [];
        }
        
        // Clean old entries
        $this->rateLimits = array_filter($this->rateLimits, function($timestamp) use ($currentTime) {
            return ($currentTime - $timestamp) < 3600; // Keep last hour only
        });
        
        // Check current limit
        $currentRequests = array_filter($this->rateLimits, function($timestamp, $key) use ($limitKey) {
            return strpos($key, $limitKey) === 0;
        }, ARRAY_FILTER_USE_BOTH);
        
        if (count($currentRequests) >= SleepTrackerConfig::RATE_LIMIT_REQUESTS) {
            $this->sendError(429, 'Rate limit exceeded. Try again later.');
        }
        
        // Record this request
        $this->rateLimits[$limitKey . '_' . microtime(true)] = $currentTime;
        file_put_contents($rateLimitFile, json_encode($this->rateLimits));
    }
    
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $path = str_replace('/api.php', '', $path);
        $segments = array_filter(explode('/', $path));
        
        try {
            switch ($method) {
                case 'GET':
                    $this->handleGET($segments);
                    break;
                case 'POST':
                    $this->handlePOST($segments);
                    break;
                case 'PUT':
                    $this->handlePUT($segments);
                    break;
                case 'DELETE':
                    $this->handleDELETE($segments);
                    break;
                default:
                    $this->sendError(405, 'Method not allowed');
            }
        } catch (Exception $e) {
            SleepTrackerConfig::log("API Error: " . $e->getMessage(), 'ERROR');
            $this->sendError(500, 'Internal server error: ' . $e->getMessage());
        }
    }
    
    private function handleGET($segments) {
        if (empty($segments)) {
            $this->sendResponse([
                'message' => 'Sleep Tracker API v2.0',
                'owner' => 'Rizaldi',
                'endpoints' => [
                    'GET /data' => 'Get all sleep data',
                    'GET /data/{date}' => 'Get data for specific date',
                    'GET /stats' => 'Get statistics',
                    'GET /predictions' => 'Get sleep predictions',
                    'GET /health' => 'API health check',
                    'POST /sessions' => 'Add new sleep session',
                    'PUT /sessions/{id}' => 'Update sleep session',
                    'DELETE /sessions/{id}' => 'Delete sleep session'
                ]
            ]);
            return;
        }
        
        switch ($segments[0]) {
            case 'data':
                if (isset($segments[1])) {
                    $this->getDataByDate($segments[1]);
                } else {
                    $this->getAllData();
                }
                break;
                
            case 'stats':
                $this->getStatistics();
                break;
                
            case 'predictions':
                $this->getPredictions();
                break;
                
            case 'health':
                $this->getHealthStatus();
                break;
                
            case 'export':
                $this->getExportData($segments[1] ?? 'json');
                break;
                
            default:
                $this->sendError(404, 'Endpoint not found');
        }
    }
    
    private function handlePOST($segments) {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $this->sendError(400, 'Invalid JSON input');
        }
        
        switch ($segments[0] ?? '') {
            case 'sessions':
                $this->addSleepSession($input);
                break;
                
            case 'activities':
                $this->updateDailyActivity($input);
                break;
                
            case 'import':
                $this->importData($input);
                break;
                
            default:
                $this->sendError(404, 'Endpoint not found');
        }
    }
    
    private function handlePUT($segments) {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $this->sendError(400, 'Invalid JSON input');
        }
        
        switch ($segments[0] ?? '') {
            case 'sessions':
                if (!isset($segments[1])) {
                    $this->sendError(400, 'Session ID required');
                }
                $this->updateSleepSession($segments[1], $input);
                break;
                
            default:
                $this->sendError(404, 'Endpoint not found');
        }
    }
    
    private function handleDELETE($segments) {
        switch ($segments[0] ?? '') {
            case 'sessions':
                if (!isset($segments[1])) {
                    $this->sendError(400, 'Session ID required');
                }
                $this->deleteSleepSession($segments[1]);
                break;
                
            case 'data':
                if (!isset($segments[1])) {
                    $this->sendError(400, 'Date required');
                }
                $this->deleteDayData($segments[1]);
                break;
                
            default:
                $this->sendError(404, 'Endpoint not found');
        }
    }
    
    private function getAllData() {
        $allData = $this->loadAllSleepData();
        
        $response = [
            'success' => true,
            'data' => $allData,
            'total_records' => count($allData),
            'date_range' => [
                'start' => !empty($allData) ? $allData[0]['date'] : null,
                'end' => !empty($allData) ? end($allData)['date'] : null
            ]
        ];
        
        $this->sendResponse($response);
    }
    
    private function getDataByDate($date) {
        if (!$this->validateDate($date)) {
            $this->sendError(400, 'Invalid date format. Use YYYY-MM-DD');
        }
        
        $month = substr($date, 0, 7);
        $dataFile = "data/{$month}.json";
        
        if (!file_exists($dataFile)) {
            $this->sendResponse([
                'success' => true,
                'data' => null,
                'message' => 'No data found for this date'
            ]);
            return;
        }
        
        $monthData = json_decode(file_get_contents($dataFile), true) ?: [];
        
        foreach ($monthData as $entry) {
            if ($entry['date'] === $date) {
                $this->sendResponse([
                    'success' => true,
                    'data' => $entry
                ]);
                return;
            }
        }
        
        $this->sendResponse([
            'success' => true,
            'data' => null,
            'message' => 'No data found for this date'
        ]);
    }
    
    private function getStatistics() {
        $allData = $this->loadAllSleepData();
        
        $stats = [
            'total_days' => 0,
            'total_sessions' => 0,
            'total_sleep_hours' => 0,
            'average_daily_hours' => 0,
            'sleep_quality_distribution' => [],
            'sleep_type_distribution' => [],
            'activity_correlation' => [],
            'weekly_pattern' => array_fill(0, 7, 0),
            'monthly_trends' => []
        ];
        
        $weekdayHours = array_fill(0, 7, 0);
        $weekdayCounts = array_fill(0, 7, 0);
        $monthlyData = [];
        $qualityData = [];
        $typeData = [];
        $activityData = [];
        
        foreach ($allData as $entry) {
            if (!empty($entry['sleepSessions'])) {
                $stats['total_days']++;
                $dayOfWeek = date('w', strtotime($entry['date']));
                $month = substr($entry['date'], 0, 7);
                
                $dailyTotal = 0;
                foreach ($entry['sleepSessions'] as $session) {
                    $stats['total_sessions']++;
                    $stats['total_sleep_hours'] += $session['duration'];
                    $dailyTotal += $session['duration'];
                    
                    $qualityData[$session['quality']] = ($qualityData[$session['quality']] ?? 0) + 1;
                    $typeData[$session['type']] = ($typeData[$session['type']] ?? 0) + 1;
                }
                
                $weekdayHours[$dayOfWeek] += $dailyTotal;
                $weekdayCounts[$dayOfWeek]++;
                
                if (!isset($monthlyData[$month])) {
                    $monthlyData[$month] = ['hours' => 0, 'days' => 0];
                }
                $monthlyData[$month]['hours'] += $dailyTotal;
                $monthlyData[$month]['days']++;
                
                if (!empty($entry['workActivity'])) {
                    if (!isset($activityData[$entry['workActivity']])) {
                        $activityData[$entry['workActivity']] = ['hours' => 0, 'count' => 0];
                    }
                    $activityData[$entry['workActivity']]['hours'] += $dailyTotal;
                    $activityData[$entry['workActivity']]['count']++;
                }
            }
        }
        
        if ($stats['total_days'] > 0) {
            $stats['average_daily_hours'] = round($stats['total_sleep_hours'] / $stats['total_days'], 2);
        }
        
        for ($i = 0; $i < 7; $i++) {
            $stats['weekly_pattern'][$i] = $weekdayCounts[$i] > 0 ? 
                round($weekdayHours[$i] / $weekdayCounts[$i], 2) : 0;
        }
        
        $stats['sleep_quality_distribution'] = $qualityData;
        $stats['sleep_type_distribution'] = $typeData;
        
        foreach ($activityData as $activity => $actInfo) {
            $stats['activity_correlation'][$activity] = round($actInfo['hours'] / $actInfo['count'], 2);
        }
        
        foreach ($monthlyData as $month => $monthInfo) {
            $stats['monthly_trends'][$month] = round($monthInfo['hours'] / $monthInfo['days'], 2);
        }
        
        $this->sendResponse([
            'success' => true,
            'statistics' => $stats,
            'generated_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    private function getPredictions() {
        $allData = $this->loadAllSleepData();
        
        if (count($allData) < SleepTrackerConfig::MIN_DATA_FOR_PREDICTION) {
            $this->sendResponse([
                'success' => true,
                'predictions' => [],
                'message' => 'Insufficient data for predictions. Need at least ' . 
                           SleepTrackerConfig::MIN_DATA_FOR_PREDICTION . ' days of data.'
            ]);
            return;
        }
        
        $predictions = [];
        for ($i = 1; $i <= SleepTrackerConfig::PREDICTION_DAYS; $i++) {
            $targetDate = date('Y-m-d', strtotime("+{$i} days"));
            $prediction = $this->generateSleepPrediction($allData, $targetDate);
            $predictions[] = $prediction;
        }
        
        $this->sendResponse([
            'success' => true,
            'predictions' => $predictions,
            'generated_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    private function addSleepSession($input) {
        $required = ['date', 'type', 'bedTime', 'wakeTime'];
        foreach ($required as $field) {
            if (!isset($input[$field])) {
                $this->sendError(400, "Missing required field: {$field}");
            }
        }
        
        // Validate data
        $errors = SleepTrackerConfig::validateSleepSession($input);
        if (!empty($errors)) {
            $this->sendError(400, 'Validation errors: ' . implode(', ', $errors));
        }
        
        $date = $input['date'];
        $month = substr($date, 0, 7);
        
        // Load month data
        $dataFile = "data/{$month}.json";
        $monthData = file_exists($dataFile) ? json_decode(file_get_contents($dataFile), true) : [];
        if (!$monthData) $monthData = [];
        
        // Find or create daily entry
        $dayEntry = null;
        foreach ($monthData as &$entry) {
            if ($entry['date'] === $date) {
                $dayEntry = &$entry;
                break;
            }
        }
        
        if (!$dayEntry) {
            $newEntry = [
                'date' => $date,
                'sleepSessions' => [],
                'workActivity' => '',
                'location' => '',
                'notes' => '',
                'createdAt' => date('Y-m-d H:i:s'),
                'updatedAt' => date('Y-m-d H:i:s')
            ];
            $monthData[] = $newEntry;
            $dayEntry = &$monthData[count($monthData) - 1];
        }
        
        // Calculate session data
        $duration = SleepTrackerConfig::calculateSleepDuration($input['bedTime'], $input['wakeTime']);
        $cycles = round($duration / 1.5);
        $quality = SleepTrackerConfig::getSleepQuality($cycles, $input['type']);
        
        $newSession = [
            'id' => uniqid(),
            'type' => $input['type'],
            'bedTime' => $input['bedTime'],
            'wakeTime' => $input['wakeTime'],
            'duration' => $duration,
            'cycles' => $cycles,
            'quality' => $quality,
            'createdAt' => date('Y-m-d H:i:s')
        ];
        
        $dayEntry['sleepSessions'][] = $newSession;
        $dayEntry['updatedAt'] = date('Y-m-d H:i:s');
        
        // Save data
        if (file_put_contents($dataFile, json_encode($monthData, JSON_PRETTY_PRINT))) {
            $this->sendResponse([
                'success' => true,
                'message' => 'Sleep session added successfully',
                'session' => $newSession
            ]);
        } else {
            $this->sendError(500, 'Failed to save data');
        }
    }
    
    private function getHealthStatus() {
        $health = SleepTrackerConfig::healthCheck();
        $stats = SleepTrackerConfig::getAppStats();
        $systemInfo = SleepTrackerConfig::getSystemInfo();
        
        $this->sendResponse([
            'success' => true,
            'status' => $health['status'],
            'health_checks' => $health['checks'],
            'system_info' => $systemInfo,
            'app_statistics' => $stats,
            'api_version' => '2.0',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }
    
    private function getExportData($format) {
        $allData = $this->loadAllSleepData();
        
        switch ($format) {
            case 'csv':
                $this->exportToCSV($allData);
                break;
            case 'json':
                $this->sendResponse([
                    'success' => true,
                    'data' => $allData,
                    'format' => 'json',
                    'exported_at' => date('Y-m-d H:i:s')
                ]);
                break;
            default:
                $this->sendError(400, 'Unsupported export format');
        }
    }
    
    private function loadAllSleepData() {
        $allData = [];
        $dataFiles = glob('data/*.json');
        
        foreach ($dataFiles as $file) {
            $monthData = json_decode(file_get_contents($file), true) ?: [];
            $allData = array_merge($allData, $monthData);
        }
        
        usort($allData, function($a, $b) {
            return strtotime($a['date']) - strtotime($b['date']);
        });
        
        return $allData;
    }
    
    private function generateSleepPrediction($allData, $targetDate) {
        // Simplified prediction logic for API
        $dayOfWeek = date('w', strtotime($targetDate));
        $isWeekend = ($dayOfWeek == 0 || $dayOfWeek == 6);
        
        // Get relevant historical data
        $relevantData = array_filter($allData, function($entry) use ($dayOfWeek) {
            return date('w', strtotime($entry['date'])) == $dayOfWeek;
        });
        
        $relevantData = array_slice($relevantData, -10); // Last 10 occurrences
        
        if (empty($relevantData)) {
            return [
                'date' => $targetDate,
                'bedTime' => $isWeekend ? '23:30' : '22:30',
                'wakeTime' => $isWeekend ? '08:30' : '07:00',
                'duration' => $isWeekend ? 9.0 : 8.5,
                'confidence' => 30,
                'isWeekend' => $isWeekend
            ];
        }
        
        // Calculate averages
        $totalDuration = 0;
        $bedTimes = [];
        $sessionCount = 0;
        
        foreach ($relevantData as $entry) {
            foreach ($entry['sleepSessions'] as $session) {
                if ($session['type'] === 'malam') {
                    $totalDuration += $session['duration'];
                    $bedTimes[] = $session['bedTime'];
                    $sessionCount++;
                }
            }
        }
        
        $avgDuration = $sessionCount > 0 ? $totalDuration / $sessionCount : ($isWeekend ? 9.0 : 8.0);
        $avgBedTime = $sessionCount > 0 ? $this->calculateAverageTime($bedTimes) : ($isWeekend ? '23:30' : '22:30');
        
        // Calculate wake time
        $bedTimeMinutes = $this->timeToMinutes($avgBedTime);
        $wakeTimeMinutes = ($bedTimeMinutes + ($avgDuration * 60)) % (24 * 60);
        $avgWakeTime = $this->minutesToTime($wakeTimeMinutes);
        
        $confidence = min(95, 40 + (count($relevantData) * 8));
        
        return [
            'date' => $targetDate,
            'bedTime' => $avgBedTime,
            'wakeTime' => $avgWakeTime,
            'duration' => round($avgDuration, 1),
            'cycles' => round($avgDuration / 1.5),
            'confidence' => $confidence,
            'isWeekend' => $isWeekend
        ];
    }
    
    private function calculateAverageTime($times) {
        $totalMinutes = 0;
        foreach ($times as $time) {
            $minutes = $this->timeToMinutes($time);
            // Handle times after midnight
            if ($minutes < 12 * 60) {
                $minutes += 24 * 60;
            }
            $totalMinutes += $minutes;
        }
        
        $avgMinutes = round($totalMinutes / count($times)) % (24 * 60);
        return $this->minutesToTime($avgMinutes);
    }
    
    private function timeToMinutes($time) {
        list($hours, $minutes) = explode(':', $time);
        return ($hours * 60) + $minutes;
    }
    
    private function minutesToTime($minutes) {
        $hours = floor($minutes / 60) % 24;
        $mins = $minutes % 60;
        return sprintf('%02d:%02d', $hours, $mins);
    }
    
    private function validateDate($date) {
        $d = DateTime::createFromFormat('Y-m-d', $date);
        return $d && $d->format('Y-m-d') === $date;
    }
    
    private function sendResponse($data, $statusCode = 200) {
        http_response_code($statusCode);
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
    
    private function sendError($statusCode, $message) {
        http_response_code($statusCode);
        echo json_encode([
            'success' => false,
            'error' => $message,
            'timestamp' => date('Y-m-d H:i:s')
        ], JSON_PRETTY_PRINT);
        exit;
    }
}

// Initialize and handle API request
$api = new SleepTrackerAPI();
$api->handleRequest();
?>