<?php
// analytics.php - Advanced Sleep Analytics Dashboard

session_start();

function loadAllSleepData() {
    $allData = [];
    $dataFiles = glob('data/*.json');
    
    foreach ($dataFiles as $file) {
        $monthData = json_decode(file_get_contents($file), true) ?: [];
        $allData = array_merge($allData, $monthData);
    }
    
    usort($allData, function($a, $b) {
        return strtotime($a['date']) - strtotime($b['date']);
    });
    
    return $allData;
}

function generateAdvancedAnalytics($data) {
    $analytics = [
        'overview' => [
            'totalDays' => 0,
            'totalSessions' => 0,
            'totalSleepHours' => 0,
            'avgDailyHours' => 0,
            'longestSession' => 0,
            'shortestSession' => 24,
            'mostProductiveDay' => '',
            'dataRange' => ['start' => '', 'end' => '']
        ],
        'patterns' => [
            'weekdayAvg' => [],
            'monthlyTrends' => [],
            'sleepTypeDistribution' => [],
            'qualityTrends' => [],
            'activityCorrelation' => []
        ],
        'insights' => [],
        'recommendations' => []
    ];
    
    if (empty($data)) {
        return $analytics;
    }
    
    // Overview calculations
    $analytics['overview']['dataRange']['start'] = $data[0]['date'];
    $analytics['overview']['dataRange']['end'] = end($data)['date'];
    
    $weekdayHours = array_fill(0, 7, 0);
    $weekdayCounts = array_fill(0, 7, 0);
    $monthlyData = [];
    $qualityData = [];
    $activityData = [];
    
    foreach ($data as $entry) {
        if (!empty($entry['sleepSessions'])) {
            $analytics['overview']['totalDays']++;
            $dayOfWeek = date('w', strtotime($entry['date']));
            $month = substr($entry['date'], 0, 7);
            
            $dailyTotal = 0;
            foreach ($entry['sleepSessions'] as $session) {
                $analytics['overview']['totalSessions']++;
                $analytics['overview']['totalSleepHours'] += $session['duration'];
                $dailyTotal += $session['duration'];
                
                if ($session['duration'] > $analytics['overview']['longestSession']) {
                    $analytics['overview']['longestSession'] = $session['duration'];
                }
                if ($session['duration'] < $analytics['overview']['shortestSession']) {
                    $analytics['overview']['shortestSession'] = $session['duration'];
                }
                
                // Collect data for patterns
                if (!isset($analytics['patterns']['sleepTypeDistribution'][$session['type']])) {
                    $analytics['patterns']['sleepTypeDistribution'][$session['type']] = 0;
                }
                $analytics['patterns']['sleepTypeDistribution'][$session['type']]++;
                
                if (!isset($qualityData[$session['quality']])) {
                    $qualityData[$session['quality']] = 0;
                }
                $qualityData[$session['quality']]++;
            }
            
            $weekdayHours[$dayOfWeek] += $dailyTotal;
            $weekdayCounts[$dayOfWeek]++;
            
            if (!isset($monthlyData[$month])) {
                $monthlyData[$month] = ['hours' => 0, 'days' => 0];
            }
            $monthlyData[$month]['hours'] += $dailyTotal;
            $monthlyData[$month]['days']++;
            
            // Activity correlation
            if (!empty($entry['workActivity'])) {
                if (!isset($activityData[$entry['workActivity']])) {
                    $activityData[$entry['workActivity']] = ['hours' => 0, 'count' => 0];
                }
                $activityData[$entry['workActivity']]['hours'] += $dailyTotal;
                $activityData[$entry['workActivity']]['count']++;
            }
        }
    }
    
    // Calculate averages
    if ($analytics['overview']['totalDays'] > 0) {
        $analytics['overview']['avgDailyHours'] = round($analytics['overview']['totalSleepHours'] / $analytics['overview']['totalDays'], 1);
    }
    
    // Weekday patterns
    $weekdays = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
    for ($i = 0; $i < 7; $i++) {
        $avg = $weekdayCounts[$i] > 0 ? round($weekdayHours[$i] / $weekdayCounts[$i], 1) : 0;
        $analytics['patterns']['weekdayAvg'][] = [
            'day' => $weekdays[$i],
            'average' => $avg,
            'count' => $weekdayCounts[$i]
        ];
    }
    
    // Monthly trends
    foreach ($monthlyData as $month => $monthInfo) {
        $analytics['patterns']['monthlyTrends'][] = [
            'month' => $month,
            'average' => round($monthInfo['hours'] / $monthInfo['days'], 1),
            'totalHours' => round($monthInfo['hours'], 1),
            'activeDays' => $monthInfo['days']
        ];
    }
    
    // Quality trends
    $analytics['patterns']['qualityTrends'] = $qualityData;
    
    // Activity correlation
    foreach ($activityData as $activity => $actInfo) {
        $analytics['patterns']['activityCorrelation'][] = [
            'activity' => $activity,
            'average' => round($actInfo['hours'] / $actInfo['count'], 1),
            'count' => $actInfo['count']
        ];
    }
    
    // Generate insights
    $analytics['insights'] = generateInsights($analytics);
    $analytics['recommendations'] = generateRecommendations($analytics);
    
    return $analytics;
}

function generateInsights($analytics) {
    $insights = [];
    
    // Sleep consistency analysis
    $weekdayAvgs = array_column($analytics['patterns']['weekdayAvg'], 'average');
    $weekdayStdDev = calculateStandardDeviation($weekdayAvgs);
    
    if ($weekdayStdDev < 1) {
        $insights[] = [
            'type' => 'positive',
            'icon' => '🎯',
            'title' => 'Pola Tidur Konsisten',
            'description' => 'Anda memiliki pola tidur yang sangat konsisten sepanjang minggu (variasi ' . round($weekdayStdDev, 1) . 'h)'
        ];
    } elseif ($weekdayStdDev > 2) {
        $insights[] = [
            'type' => 'warning',
            'icon' => '⚠️',
            'title' => 'Pola Tidur Tidak Konsisten',
            'description' => 'Variasi durasi tidur Anda cukup tinggi (' . round($weekdayStdDev, 1) . 'h). Cobalah untuk lebih konsisten.'
        ];
    }
    
    // Weekend vs weekday analysis
    $weekendAvg = ($analytics['patterns']['weekdayAvg'][0]['average'] + $analytics['patterns']['weekdayAvg'][6]['average']) / 2;
    $weekdayAvg = 0;
    for ($i = 1; $i <= 5; $i++) {
        $weekdayAvg += $analytics['patterns']['weekdayAvg'][$i]['average'];
    }
    $weekdayAvg /= 5;
    
    $weekendDiff = $weekendAvg - $weekdayAvg;
    
    if ($weekendDiff > 2) {
        $insights[] = [
            'type' => 'info',
            'icon' => '🛌',
            'title' => 'Weekend Sleep Debt Recovery',
            'description' => 'Anda tidur ' . round($weekendDiff, 1) . 'h lebih lama di weekend, menandakan sleep debt pada weekday.'
        ];
    } elseif ($weekendDiff < -1) {
        $insights[] = [
            'type' => 'warning',
            'icon' => '😴',
            'title' => 'Kurang Istirahat Weekend',
            'description' => 'Tidur weekend Anda lebih pendek dari weekday. Manfaatkan weekend untuk recovery!'
        ];
    }
    
    // Sleep quality analysis
    $qualityTrends = $analytics['patterns']['qualityTrends'];
    $totalSessions = array_sum($qualityTrends);
    $goodQualityPercent = 0;
    
    if (isset($qualityTrends['Excellent'])) {
        $goodQualityPercent += $qualityTrends['Excellent'];
    }
    if (isset($qualityTrends['Good'])) {
        $goodQualityPercent += $qualityTrends['Good'];
    }
    
    $goodQualityPercent = $totalSessions > 0 ? ($goodQualityPercent / $totalSessions) * 100 : 0;
    
    if ($goodQualityPercent >= 80) {
        $insights[] = [
            'type' => 'positive',
            'icon' => '⭐',
            'title' => 'Kualitas Tidur Excellent',
            'description' => round($goodQualityPercent) . '% sesi tidur Anda berkualitas baik atau excellent!'
        ];
    } elseif ($goodQualityPercent < 60) {
        $insights[] = [
            'type' => 'warning',
            'icon' => '📉',
            'title' => 'Kualitas Tidur Perlu Ditingkatkan',
            'description' => 'Hanya ' . round($goodQualityPercent) . '% sesi tidur berkualitas baik. Fokus pada durasi optimal.'
        ];
    }
    
    // Activity correlation insights
    foreach ($analytics['patterns']['activityCorrelation'] as $activityData) {
        if ($activityData['activity'] === 'berat' && $activityData['average'] < 7) {
            $insights[] = [
                'type' => 'warning',
                'icon' => '💪',
                'title' => 'Kurang Tidur Setelah Aktivitas Berat',
                'description' => 'Rata-rata tidur setelah aktivitas berat hanya ' . $activityData['average'] . 'h. Pertimbangkan tidur lebih awal.'
            ];
        }
    }
    
    return $insights;
}

function generateRecommendations($analytics) {
    $recommendations = [];
    
    // Based on average daily hours
    $avgDaily = $analytics['overview']['avgDailyHours'];
    
    if ($avgDaily < 6) {
        $recommendations[] = [
            'priority' => 'high',
            'icon' => '🚨',
            'title' => 'Tingkatkan Durasi Tidur',
            'action' => 'Tidur lebih awal atau bangun lebih siang',
            'description' => 'Rata-rata tidur Anda (' . $avgDaily . 'h) di bawah minimum yang disarankan (7-9h).'
        ];
    } elseif ($avgDaily > 10) {
        $recommendations[] = [
            'priority' => 'medium',
            'icon' => '⏰',
            'title' => 'Evaluasi Kualitas Tidur',
            'action' => 'Konsultasi dengan dokter jika perlu',
            'description' => 'Durasi tidur yang terlalu lama (' . $avgDaily . 'h) mungkin menandakan masalah kualitas tidur.'
        ];
    }
    
    // Based on consistency
    $weekdayAvgs = array_column($analytics['patterns']['weekdayAvg'], 'average');
    $stdDev = calculateStandardDeviation($weekdayAvgs);
    
    if ($stdDev > 1.5) {
        $recommendations[] = [
            'priority' => 'medium',
            'icon' => '📅',
            'title' => 'Konsistensi Pola Tidur',
            'action' => 'Tetapkan jadwal tidur yang konsisten',
            'description' => 'Variasi durasi tidur cukup tinggi. Cobalah tidur dan bangun pada jam yang sama setiap hari.'
        ];
    }
    
    // Based on sleep type distribution
    $typeDistribution = $analytics['patterns']['sleepTypeDistribution'];
    $totalSessions = array_sum($typeDistribution);
    
    if (isset($typeDistribution['malam']) && $typeDistribution['malam'] / $totalSessions < 0.6) {
        $recommendations[] = [
            'priority' => 'medium',
            'icon' => '🌙',
            'title' => 'Fokus pada Tidur Malam',
            'action' => 'Prioritaskan tidur malam yang berkualitas',
            'description' => 'Proporsi tidur malam Anda relatif rendah. Tidur malam adalah yang terpenting untuk recovery.'
        ];
    }
    
    if (isset($typeDistribution['siang']) && $typeDistribution['siang'] / $totalSessions > 0.3) {
        $recommendations[] = [
            'priority' => 'low',
            'icon' => '☀️',
            'title' => 'Evaluasi Tidur Siang',
            'action' => 'Batasi tidur siang maksimal 1-2 jam',
            'description' => 'Tidur siang yang terlalu sering/lama bisa mengganggu tidur malam.'
        ];
    }
    
    return $recommendations;
}

function calculateStandardDeviation($values) {
    $count = count($values);
    if ($count < 2) return 0;
    
    $mean = array_sum($values) / $count;
    $variance = 0;
    
    foreach ($values as $value) {
        $variance += pow($value - $mean, 2);
    }
    
    return sqrt($variance / ($count - 1));
}

$allData = loadAllSleepData();
$analytics = generateAdvancedAnalytics($allData);
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Analytics - Sleep Tracker Rizaldi</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/3.9.1/chart.min.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 30px;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            color: #333;
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .nav-links {
            text-align: center;
            margin-bottom: 30px;
        }

        .nav-links a {
            display: inline-block;
            padding: 10px 20px;
            margin: 0 10px;
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            text-decoration: none;
            border-radius: 25px;
            transition: all 0.3s ease;
        }

        .nav-links a:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        .analytics-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .analytics-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            border: 1px solid rgba(102, 126, 234, 0.1);
        }

        .analytics-card h3 {
            color: #667eea;
            margin-bottom: 20px;
            font-size: 1.4em;
            border-bottom: 2px solid #f0f0f0;
            padding-bottom: 10px;
        }

        .overview-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }

        .stat-item {
            text-align: center;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 10px;
        }

        .stat-number {
            font-size: 1.8em;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #666;
            font-size: 0.9em;
        }

        .chart-container {
            position: relative;
            height: 300px;
            margin-top: 20px;
        }

        .insights-list {
            list-style: none;
        }

        .insight-item {
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 10px;
            border-left: 4px solid;
            display: flex;
            align-items: flex-start;
            gap: 15px;
        }

        .insight-item.positive {
            background: #d4edda;
            border-left-color: #28a745;
        }

        .insight-item.warning {
            background: #fff3cd;
            border-left-color: #ffc107;
        }

        .insight-item.info {
            background: #d1ecf1;
            border-left-color: #17a2b8;
        }

        .insight-icon {
            font-size: 1.5em;
            margin-top: 2px;
        }

        .insight-content h4 {
            margin-bottom: 5px;
            color: #333;
        }

        .insight-content p {
            color: #666;
            font-size: 0.95em;
            line-height: 1.4;
        }

        .recommendations-list {
            list-style: none;
        }

        .recommendation-item {
            padding: 20px;
            margin-bottom: 15px;
            border-radius: 10px;
            background: white;
            border: 1px solid #e9ecef;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
        }

        .recommendation-item.high {
            border-left: 4px solid #dc3545;
        }

        .recommendation-item.medium {
            border-left: 4px solid #ffc107;
        }

        .recommendation-item.low {
            border-left: 4px solid #28a745;
        }

        .recommendation-header {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 10px;
        }

        .recommendation-header h4 {
            color: #333;
            margin: 0;
        }

        .recommendation-action {
            background: #667eea;
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 0.9em;
            margin: 10px 0;
            display: inline-block;
        }

        .weekday-grid {
            display: grid;
            grid-template-columns: repeat(7, 1fr);
            gap: 10px;
            margin-top: 15px;
        }

        .weekday-item {
            text-align: center;
            padding: 15px 10px;
            background: #f8f9fa;
            border-radius: 8px;
        }

        .weekday-name {
            font-weight: bold;
            color: #667eea;
            margin-bottom: 5px;
        }

        .weekday-hours {
            font-size: 1.2em;
            font-weight: bold;
            color: #333;
        }

        .weekday-count {
            font-size: 0.8em;
            color: #666;
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: #666;
        }

        .empty-state i {
            font-size: 3em;
            margin-bottom: 20px;
            opacity: 0.3;
        }

        @media (max-width: 768px) {
            .analytics-grid {
                grid-template-columns: 1fr;
            }
            
            .weekday-grid {
                grid-template-columns: repeat(4, 1fr);
            }
            
            .overview-stats {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-chart-line"></i> Sleep Analytics</h1>
            <p>Analisis Mendalam Pola Tidur Rizaldi</p>
        </div>

        <div class="nav-links">
            <a href="index.php"><i class="fas fa-home"></i> Dashboard</a>
            <a href="analytics.php"><i class="fas fa-chart-bar"></i> Analytics</a>
            <a href="export.php"><i class="fas fa-download"></i> Export Data</a>
        </div>

        <?php if (empty($allData)): ?>
            <div class="empty-state">
                <i class="fas fa-chart-line"></i>
                <h2>Belum Ada Data untuk Dianalisis</h2>
                <p>Mulai input data tidur di dashboard untuk melihat analisis mendalam</p>
                <a href="index.php" style="display: inline-block; margin-top: 20px; padding: 15px 30px; background: linear-gradient(45deg, #667eea, #764ba2); color: white; text-decoration: none; border-radius: 25px;">
                    <i class="fas fa-plus"></i> Input Data Tidur
                </a>
            </div>
        <?php else: ?>
            <!-- Overview Section -->
            <div class="analytics-card" style="grid-column: 1 / -1;">
                <h3><i class="fas fa-overview"></i> Overview Data</h3>
                <div class="overview-stats">
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $analytics['overview']['totalDays']; ?></div>
                        <div class="stat-label">Hari Tercatat</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $analytics['overview']['totalSessions']; ?></div>
                        <div class="stat-label">Total Sesi</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo round($analytics['overview']['totalSleepHours'], 1); ?>h</div>
                        <div class="stat-label">Total Jam Tidur</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo $analytics['overview']['avgDailyHours']; ?>h</div>
                        <div class="stat-label">Rata-rata Harian</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo round($analytics['overview']['longestSession'], 1); ?>h</div>
                        <div class="stat-label">Sesi Terpanjang</div>
                    </div>
                    <div class="stat-item">
                        <div class="stat-number"><?php echo round($analytics['overview']['shortestSession'], 1); ?>h</div>
                        <div class="stat-label">Sesi Terpendek</div>
                    </div>
                </div>
                <div style="text-align: center; margin-top: 20px; color: #666;">
                    <small>
                        <i class="fas fa-calendar"></i> 
                        Data dari <?php echo date('d M Y', strtotime($analytics['overview']['dataRange']['start'])); ?> 
                        sampai <?php echo date('d M Y', strtotime($analytics['overview']['dataRange']['end'])); ?>
                    </small>
                </div>
            </div>

            <div class="analytics-grid">
                <!-- Weekday Pattern -->
                <div class="analytics-card">
                    <h3><i class="fas fa-calendar-week"></i> Pola Hari dalam Minggu</h3>
                    <div class="weekday-grid">
                        <?php foreach ($analytics['patterns']['weekdayAvg'] as $dayData): ?>
                            <div class="weekday-item">
                                <div class="weekday-name"><?php echo substr($dayData['day'], 0, 3); ?></div>
                                <div class="weekday-hours"><?php echo $dayData['average']; ?>h</div>
                                <div class="weekday-count"><?php echo $dayData['count']; ?> hari</div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Sleep Type Distribution -->
                <div class="analytics-card">
                    <h3><i class="fas fa-pie-chart"></i> Distribusi Jenis Tidur</h3>
                    <div class="chart-container">
                        <canvas id="sleepTypeChart"></canvas>
                    </div>
                </div>

                <!-- Quality Trends -->
                <div class="analytics-card">
                    <h3><i class="fas fa-star"></i> Distribusi Kualitas Tidur</h3>
                    <div class="chart-container">
                        <canvas id="qualityChart"></canvas>
                    </div>
                </div>

                <!-- Monthly Trends -->
                <div class="analytics-card">
                    <h3><i class="fas fa-chart-line"></i> Tren Bulanan</h3>
                    <div class="chart-container">
                        <canvas id="monthlyChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Insights Section -->
            <div class="analytics-card" style="grid-column: 1 / -1;">
                <h3><i class="fas fa-lightbulb"></i> AI Insights</h3>
                <?php if (empty($analytics['insights'])): ?>
                    <p style="text-align: center; color: #666; padding: 20px;">
                        <i class="fas fa-info-circle"></i> Belum cukup data untuk menghasilkan insights mendalam
                    </p>
                <?php else: ?>
                    <ul class="insights-list">
                        <?php foreach ($analytics['insights'] as $insight): ?>
                            <li class="insight-item <?php echo $insight['type']; ?>">
                                <div class="insight-icon"><?php echo $insight['icon']; ?></div>
                                <div class="insight-content">
                                    <h4><?php echo $insight['title']; ?></h4>
                                    <p><?php echo $insight['description']; ?></p>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>

            <!-- Recommendations Section -->
            <div class="analytics-card" style="grid-column: 1 / -1;">
                <h3><i class="fas fa-clipboard-check"></i> Rekomendasi Personal</h3>
                <?php if (empty($analytics['recommendations'])): ?>
                    <p style="text-align: center; color: #666; padding: 20px;">
                        <i class="fas fa-thumbs-up"></i> Pola tidur Anda sudah cukup baik! Pertahankan konsistensi.
                    </p>
                <?php else: ?>
                    <ul class="recommendations-list">
                        <?php foreach ($analytics['recommendations'] as $recommendation): ?>
                            <li class="recommendation-item <?php echo $recommendation['priority']; ?>">
                                <div class="recommendation-header">
                                    <span style="font-size: 1.5em;"><?php echo $recommendation['icon']; ?></span>
                                    <h4><?php echo $recommendation['title']; ?></h4>
                                    <span style="margin-left: auto; padding: 5px 10px; background: #f8f9fa; border-radius: 15px; font-size: 0.8em; color: #666;">
                                        <?php echo strtoupper($recommendation['priority']); ?>
                                    </span>
                                </div>
                                <div class="recommendation-action">
                                    <i class="fas fa-arrow-right"></i> <?php echo $recommendation['action']; ?>
                                </div>
                                <p><?php echo $recommendation['description']; ?></p>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <script>
        // Sleep Type Distribution Chart
        const sleepTypeData = <?php echo json_encode($analytics['patterns']['sleepTypeDistribution']); ?>;
        const sleepTypeCtx = document.getElementById('sleepTypeChart').getContext('2d');
        new Chart(sleepTypeCtx, {
            type: 'doughnut',
            data: {
                labels: Object.keys(sleepTypeData).map(key => key.charAt(0).toUpperCase() + key.slice(1)),
                datasets: [{
                    data: Object.values(sleepTypeData),
                    backgroundColor: ['#667eea', '#764ba2', '#a8d8f0', '#f093fb']
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Quality Distribution Chart
        const qualityData = <?php echo json_encode($analytics['patterns']['qualityTrends']); ?>;
        const qualityCtx = document.getElementById('qualityChart').getContext('2d');
        new Chart(qualityCtx, {
            type: 'bar',
            data: {
                labels: Object.keys(qualityData),
                datasets: [{
                    label: 'Jumlah Sesi',
                    data: Object.values(qualityData),
                    backgroundColor: ['#28a745', '#17a2b8', '#ffc107', '#dc3545']
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });

        // Monthly Trends Chart
        const monthlyData = <?php echo json_encode($analytics['patterns']['monthlyTrends']); ?>;
        const monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
        new Chart(monthlyCtx, {
            type: 'line',
            data: {
                labels: monthlyData.map(item => item.month),
                datasets: [{
                    label: 'Rata-rata Jam/Hari',
                    data: monthlyData.map(item => item.average),
                    borderColor: '#667eea',
                    backgroundColor: 'rgba(102, 126, 234, 0.1)',
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Jam'
                        }
                    }
                }
            }
        });
    </script>
</body>
</html>