<?php
// Timeblocking App - Single File Implementation
// Author: AI Assistant
// Version: 1.0

// Error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Configuration
define('DATA_DIR', 'data');
define('SCHEDULE_FILE', DATA_DIR . '/schedule.json');
define('LOG_FILE', DATA_DIR . '/log.txt');
define('BACKUP_PREFIX', DATA_DIR . '/schedule.json.bak.');
define('PIN_ATTEMPTS_LIMIT', 5);
define('PIN_BLOCK_TIME', 300); // 5 minutes in seconds
define('DEFAULT_PIN', '051205'); // This will be hashed on first run

// Ensure data directory exists
if (!is_dir(DATA_DIR)) {
    mkdir(DATA_DIR, 0755, true);
}

// Initialize data file if it doesn't exist
if (!file_exists(SCHEDULE_FILE)) {
    $initialData = [
        'users' => [
            'default' => [
                'pin_hash' => password_hash(DEFAULT_PIN, PASSWORD_DEFAULT),
                'settings' => [
                    'timezone' => 'Asia/Jakarta'
                ]
            ]
        ],
        'days' => []
    ];
    
    // Add sample data for 7 days
    $today = date('Y-m-d');
    for ($i = 0; $i < 7; $i++) {
        $date = date('Y-m-d', strtotime($today . ' + ' . $i . ' days'));
        $initialData['days'][$date] = generateSampleData($date, $i);
    }
    
    file_put_contents(SCHEDULE_FILE, json_encode($initialData, JSON_PRETTY_PRINT));
}

// Helper function to generate sample data
function generateSampleData($date, $dayOffset) {
    $sampleData = [];
    
    // Add a shift based on day
    $shiftTypes = ['Pagi 2 split', 'Pagi 1 split', 'Siang', 'Middle', 'Libur'];
    $shiftType = $shiftTypes[$dayOffset % 5];
    
    if ($shiftType !== 'Libur') {
        switch ($shiftType) {
            case 'Pagi 2 split':
                $sampleData[] = [
                    'id' => 'shift-1',
                    'start' => '08:00',
                    'end' => '13:00',
                    'title' => 'Shift Pagi (part 1)',
                    'type' => 'shift',
                    'tag' => 'pagi-2-split'
                ];
                $sampleData[] = [
                    'id' => 'shift-2',
                    'start' => '18:00',
                    'end' => '22:00',
                    'title' => 'Shift Pagi (part 2)',
                    'type' => 'shift',
                    'tag' => 'pagi-2-split'
                ];
                break;
            case 'Pagi 1 split':
                $sampleData[] = [
                    'id' => 'shift-1',
                    'start' => '08:00',
                    'end' => '17:00',
                    'title' => 'Shift Pagi',
                    'type' => 'shift',
                    'tag' => 'pagi-1-split'
                ];
                break;
            case 'Siang':
                $sampleData[] = [
                    'id' => 'shift-1',
                    'start' => '13:00',
                    'end' => '22:00',
                    'title' => 'Shift Siang',
                    'type' => 'shift',
                    'tag' => 'siang'
                ];
                break;
            case 'Middle':
                $sampleData[] = [
                    'id' => 'shift-1',
                    'start' => '11:00',
                    'end' => '20:00',
                    'title' => 'Shift Middle',
                    'type' => 'shift',
                    'tag' => 'middle'
                ];
                break;
        }
    }
    
    // Add some sample tasks
    $tasks = [
        ['start' => '06:00', 'end' => '07:00', 'title' => 'Olahraga Pagi', 'type' => 'task', 'tag' => 'personal'],
        ['start' => '07:00', 'end' => '07:30', 'title' => 'Sarapan', 'type' => 'task', 'tag' => 'personal'],
        ['start' => '12:00', 'end' => '13:00', 'title' => 'Istirahat Siang', 'type' => 'task', 'tag' => 'personal'],
        ['start' => '22:30', 'end' => '23:00', 'title' => 'Rencana Besok', 'type' => 'task', 'tag' => 'planning']
    ];
    
    foreach ($tasks as $task) {
        // Check if task overlaps with shift
        $overlaps = false;
        foreach ($sampleData as $item) {
            if ($item['type'] === 'shift' && timeOverlaps($task['start'], $task['end'], $item['start'], $item['end'])) {
                $overlaps = true;
                break;
            }
        }
        
        if (!$overlaps) {
            $task['id'] = 'task-' . uniqid();
            $task['notes'] = 'Catatan untuk ' . $task['title'];
            $sampleData[] = $task;
        }
    }
    
    return $sampleData;
}

// Helper function to check if time intervals overlap
function timeOverlaps($start1, $end1, $start2, $end2) {
    $t1 = strtotime($start1);
    $e1 = strtotime($end1);
    $t2 = strtotime($start2);
    $e2 = strtotime($end2);
    
    return ($t1 < $e2 && $e1 > $t2);
}

// API endpoint handling
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    $action = $_GET['action'];
    $response = ['success' => false, 'message' => 'Unknown action'];
    
    // Verify PIN for write operations
    $pinVerified = false;
    if (in_array($action, ['save', 'delete', 'import'])) {
        if (isset($_POST['pin'])) {
            $pinVerified = verifyPin($_POST['pin']);
            if (!$pinVerified) {
                $response['message'] = 'PIN salah';
                echo json_encode($response);
                exit;
            }
        } else {
            $response['message'] = 'PIN diperlukan';
            echo json_encode($response);
            exit;
        }
    }
    
    switch ($action) {
        case 'load':
            if (isset($_GET['date'])) {
                $date = $_GET['date'];
                $data = loadSchedule();
                $response['success'] = true;
                $response['data'] = isset($data['days'][$date]) ? $data['days'][$date] : [];
            } else {
                $response['message'] = 'Parameter date diperlukan';
            }
            break;
            
        case 'save':
            if (isset($_POST['date']) && isset($_POST['timeblocks'])) {
                $date = $_POST['date'];
                $timeblocks = json_decode($_POST['timeblocks'], true);
                
                if (saveTimeblocks($date, $timeblocks)) {
                    $response['success'] = true;
                    $response['message'] = 'Timeblocks berhasil disimpan';
                } else {
                    $response['message'] = 'Gagal menyimpan timeblocks';
                }
            } else {
                $response['message'] = 'Parameter date dan timeblocks diperlukan';
            }
            break;
            
        case 'delete':
            if (isset($_POST['date']) && isset($_POST['id'])) {
                $date = $_POST['date'];
                $id = $_POST['id'];
                
                if (deleteTimeblock($date, $id)) {
                    $response['success'] = true;
                    $response['message'] = 'Timeblock berhasil dihapus';
                } else {
                    $response['message'] = 'Gagal menghapus timeblock';
                }
            } else {
                $response['message'] = 'Parameter date dan id diperlukan';
            }
            break;
            
        case 'export':
            if (verifyPin($_POST['pin'])) {
                $data = loadSchedule();
                $response['success'] = true;
                $response['data'] = $data;
            } else {
                $response['message'] = 'PIN salah';
            }
            break;
            
        case 'import':
            if (isset($_FILES['importFile']) && $_FILES['importFile']['error'] === UPLOAD_ERR_OK) {
                $jsonContent = file_get_contents($_FILES['importFile']['tmp_name']);
                $importData = json_decode($jsonContent, true);
                
                if ($importData && importData($importData)) {
                    $response['success'] = true;
                    $response['message'] = 'Data berhasil diimpor';
                } else {
                    $response['message'] = 'Format data tidak valid atau gagal mengimpor';
                }
            } else {
                $response['message'] = 'File tidak valid atau terjadi kesalahan saat mengunggah';
            }
            break;
            
        case 'verifyPin':
            if (isset($_POST['pin'])) {
                $pinVerified = verifyPin($_POST['pin']);
                $response['success'] = $pinVerified;
                $response['message'] = $pinVerified ? 'PIN benar' : 'PIN salah';
            } else {
                $response['message'] = 'PIN diperlukan';
            }
            break;
    }
    
    echo json_encode($response);
    exit;
}

// Helper function to verify PIN
function verifyPin($pin) {
    // Check for too many attempts
    $attemptsFile = DATA_DIR . '/pin_attempts.json';
    $attempts = [];
    
    if (file_exists($attemptsFile)) {
        $attempts = json_decode(file_get_contents($attemptsFile), true);
    }
    
    $ip = $_SERVER['REMOTE_ADDR'];
    $now = time();
    
    // Clean old attempts
    if (isset($attempts[$ip])) {
        $attempts[$ip] = array_filter($attempts[$ip], function($timestamp) use ($now) {
            return $now - $timestamp < PIN_BLOCK_TIME;
        });
        
        // Check if blocked
        if (count($attempts[$ip]) >= PIN_ATTEMPTS_LIMIT) {
            return false;
        }
    }
    
    // Verify PIN
    $data = loadSchedule();
    $pinHash = $data['users']['default']['pin_hash'];
    $isValid = password_verify($pin, $pinHash);
    
    // Record attempt if invalid
    if (!$isValid) {
        if (!isset($attempts[$ip])) {
            $attempts[$ip] = [];
        }
        $attempts[$ip][] = $now;
        file_put_contents($attemptsFile, json_encode($attempts));
    } else {
        // Reset attempts on success
        if (isset($attempts[$ip])) {
            unset($attempts[$ip]);
            file_put_contents($attemptsFile, json_encode($attempts));
        }
    }
    
    return $isValid;
}

// Helper function to load schedule data
function loadSchedule() {
    if (!file_exists(SCHEDULE_FILE)) {
        return ['users' => [], 'days' => []];
    }
    
    $json = file_get_contents(SCHEDULE_FILE);
    return json_decode($json, true);
}

// Helper function to save timeblocks
function saveTimeblocks($date, $timeblocks) {
    // Create backup
    $backupFile = BACKUP_PREFIX . time();
    if (!copy(SCHEDULE_FILE, $backupFile)) {
        logAction('backup_failed', 'Failed to create backup before saving');
        return false;
    }
    
    // Load current data
    $data = loadSchedule();
    
    // Update timeblocks for the specified date
    $data['days'][$date] = $timeblocks;
    
    // Save with file locking to prevent race conditions
    $file = fopen(SCHEDULE_FILE, 'w');
    if (flock($file, LOCK_EX)) {
        fwrite($file, json_encode($data, JSON_PRETTY_PRINT));
        flock($file, LOCK_UN);
        fclose($file);
        
        logAction('save', "Saved timeblocks for date: $date");
        return true;
    } else {
        fclose($file);
        logAction('save_failed', "Failed to acquire lock for saving date: $date");
        return false;
    }
}

// Helper function to delete a timeblock
function deleteTimeblock($date, $id) {
    // Create backup
    $backupFile = BACKUP_PREFIX . time();
    if (!copy(SCHEDULE_FILE, $backupFile)) {
        logAction('backup_failed', 'Failed to create backup before deleting');
        return false;
    }
    
    // Load current data
    $data = loadSchedule();
    
    // Remove the timeblock with the specified ID
    if (isset($data['days'][$date])) {
        $data['days'][$date] = array_filter($data['days'][$date], function($block) use ($id) {
            return $block['id'] !== $id;
        });
        
        // Re-index array
        $data['days'][$date] = array_values($data['days'][$date]);
        
        // Save with file locking
        $file = fopen(SCHEDULE_FILE, 'w');
        if (flock($file, LOCK_EX)) {
            fwrite($file, json_encode($data, JSON_PRETTY_PRINT));
            flock($file, LOCK_UN);
            fclose($file);
            
            logAction('delete', "Deleted timeblock with ID: $id for date: $date");
            return true;
        } else {
            fclose($file);
            logAction('delete_failed', "Failed to acquire lock for deleting timeblock with ID: $id");
            return false;
        }
    }
    
    return false;
}

// Helper function to import data
function importData($importData) {
    // Create backup
    $backupFile = BACKUP_PREFIX . time();
    if (!copy(SCHEDULE_FILE, $backupFile)) {
        logAction('backup_failed', 'Failed to create backup before importing');
        return false;
    }
    
    // Save with file locking
    $file = fopen(SCHEDULE_FILE, 'w');
    if (flock($file, LOCK_EX)) {
        fwrite($file, json_encode($importData, JSON_PRETTY_PRINT));
        flock($file, LOCK_UN);
        fclose($file);
        
        logAction('import', 'Imported data from file');
        return true;
    } else {
        fclose($file);
        logAction('import_failed', 'Failed to acquire lock for importing data');
        return false;
    }
}

// Helper function to log actions
function logAction($action, $details) {
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'];
    $logEntry = "[$timestamp] [$ip] $action: $details\n";
    
    file_put_contents(LOG_FILE, $logEntry, FILE_APPEND);
}

// Helper function to get shift presets
function getShiftPresets() {
    return [
        'Pagi 2 split' => [
            ['start' => '08:00', 'end' => '13:00'],
            ['start' => '18:00', 'end' => '22:00']
        ],
        'Pagi 1 split' => [
            ['start' => '08:00', 'end' => '17:00']
        ],
        'Siang' => [
            ['start' => '13:00', 'end' => '22:00']
        ],
        'Middle' => [
            ['start' => '11:00', 'end' => '20:00']
        ],
        'Libur' => []
    ];
}

// HTML and JavaScript for the frontend
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Timeblocking App</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .analog-clock {
            width: 250px;
            height: 250px;
            position: relative;
            margin: 0 auto;
        }
        
        .clock-face {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            background-color: #fff;
            border: 2px solid #333;
            position: relative;
        }
        
        .hand {
            position: absolute;
            left: 50%;
            transform-origin: bottom center;
            bottom: 50%;
            border-radius: 10px;
        }
        
        .hour-hand {
            width: 6px;
            height: 60px;
            background-color: #333;
            margin-left: -3px;
        }
        
        .minute-hand {
            width: 4px;
            height: 80px;
            background-color: #666;
            margin-left: -2px;
        }
        
        .second-hand {
            width: 2px;
            height: 90px;
            background-color: #f00;
            margin-left: -1px;
        }
        
        .center-dot {
            position: absolute;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background-color: #333;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 10;
        }
        
        .clock-number {
            position: absolute;
            font-size: 14px;
            font-weight: bold;
        }
        
        .timeblock-overlay {
            position: absolute;
            background-color: rgba(76, 175, 80, 0.3);
            border-radius: 50%;
            z-index: 5;
        }
        
        .shift-overlay {
            position: absolute;
            background-color: rgba(158, 158, 158, 0.3);
            border-radius: 50%;
            z-index: 5;
        }
        
        .timeblock-item {
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .timeblock-item:hover {
            transform: translateX(5px);
        }
        
        .timeblock-item.shift {
            border-left: 4px solid #9e9e9e;
        }
        
        .timeblock-item.task {
            border-left: 4px solid #4caf50;
        }
        
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 5px;
            color: white;
            font-weight: bold;
            z-index: 1000;
            transform: translateX(150%);
            transition: transform 0.3s ease;
        }
        
        .notification.show {
            transform: translateX(0);
        }
        
        .notification.success {
            background-color: #4caf50;
        }
        
        .notification.error {
            background-color: #f44336;
        }
        
        .notification.info {
            background-color: #2196f3;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }
        
        .modal.show {
            display: flex;
        }
        
        .modal-content {
            background-color: white;
            padding: 20px;
            border-radius: 5px;
            width: 90%;
            max-width: 500px;
            max-height: 90vh;
            overflow-y: auto;
        }
        
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: #fff;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body class="bg-gray-100">
    <!-- Notification Container -->
    <div id="notification" class="notification"></div>
    
    <!-- PIN Verification Modal -->
    <div id="pinModal" class="modal">
        <div class="modal-content">
            <h2 class="text-xl font-bold mb-4">Verifikasi PIN</h2>
            <p class="mb-4">Masukkan PIN untuk melanjutkan:</p>
            <input type="password" id="pinInput" class="w-full p-2 border rounded mb-4" placeholder="PIN">
            <div class="flex justify-end">
                <button id="pinCancel" class="px-4 py-2 bg-gray-300 text-gray-700 rounded mr-2">Batal</button>
                <button id="pinVerify" class="px-4 py-2 bg-blue-500 text-white rounded">Verifikasi</button>
            </div>
        </div>
    </div>
    
    <!-- Timeblock Modal -->
    <div id="timeblockModal" class="modal">
        <div class="modal-content">
            <h2 class="text-xl font-bold mb-4">
                <span id="timeblockModalTitle">Tambah Timeblock</span>
            </h2>
            <form id="timeblockForm">
                <input type="hidden" id="timeblockId">
                <input type="hidden" id="timeblockDate">
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Judul</label>
                    <input type="text" id="timeblockTitle" class="w-full p-2 border rounded" required>
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Tipe</label>
                    <select id="timeblockType" class="w-full p-2 border rounded">
                        <option value="task">Tugas</option>
                        <option value="shift">Shift Kerja</option>
                    </select>
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Waktu Mulai</label>
                    <input type="time" id="timeblockStart" class="w-full p-2 border rounded" required>
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Waktu Selesai</label>
                    <input type="time" id="timeblockEnd" class="w-full p-2 border rounded" required>
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Kategori/Tag</label>
                    <input type="text" id="timeblockTag" class="w-full p-2 border rounded">
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Catatan</label>
                    <textarea id="timeblockNotes" class="w-full p-2 border rounded" rows="3"></textarea>
                </div>
                
                <div class="flex justify-end">
                    <button type="button" id="timeblockCancel" class="px-4 py-2 bg-gray-300 text-gray-700 rounded mr-2">Batal</button>
                    <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Simpan</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Shift Preset Modal -->
    <div id="shiftModal" class="modal">
        <div class="modal-content">
            <h2 class="text-xl font-bold mb-4">Pilih Shift</h2>
            <form id="shiftForm">
                <input type="hidden" id="shiftDate">
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Shift Preset</label>
                    <select id="shiftPreset" class="w-full p-2 border rounded">
                        <option value="">-- Pilih Shift --</option>
                        <option value="Pagi 2 split">Pagi 2 split</option>
                        <option value="Pagi 1 split">Pagi 1 split</option>
                        <option value="Siang">Siang</option>
                        <option value="Middle">Middle</option>
                        <option value="Libur">Libur</option>
                    </select>
                </div>
                
                <div class="mb-4">
                    <label class="block text-gray-700 text-sm font-bold mb-2">Shift Custom (opsional)</label>
                    <div id="customShifts">
                        <div class="custom-shift mb-2 flex items-center">
                            <input type="time" class="custom-shift-start p-2 border rounded mr-2" placeholder="Mulai">
                            <input type="time" class="custom-shift-end p-2 border rounded mr-2" placeholder="Selesai">
                            <button type="button" class="remove-shift px-2 py-1 bg-red-500 text-white rounded">Hapus</button>
                        </div>
                    </div>
                    <button type="button" id="addShift" class="px-4 py-2 bg-green-500 text-white rounded">Tambah Shift</button>
                </div>
                
                <div class="flex justify-end">
                    <button type="button" id="shiftCancel" class="px-4 py-2 bg-gray-300 text-gray-700 rounded mr-2">Batal</button>
                    <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Simpan</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Import/Export Modal -->
    <div id="importExportModal" class="modal">
        <div class="modal-content">
            <h2 class="text-xl font-bold mb-4">Import/Export Data</h2>
            
            <div class="mb-4">
                <h3 class="text-lg font-semibold mb-2">Export Data</h3>
                <p class="mb-2">Unduh semua data timeblocking dalam format JSON:</p>
                <button id="exportBtn" class="px-4 py-2 bg-blue-500 text-white rounded">Export</button>
            </div>
            
            <div class="mb-4">
                <h3 class="text-lg font-semibold mb-2">Import Data</h3>
                <p class="mb-2">Upload file JSON untuk mengganti semua data:</p>
                <input type="file" id="importFile" accept=".json" class="w-full p-2 border rounded">
                <button id="importBtn" class="px-4 py-2 bg-blue-500 text-white rounded mt-2">Import</button>
            </div>
            
            <div class="flex justify-end">
                <button id="importExportClose" class="px-4 py-2 bg-gray-300 text-gray-700 rounded">Tutup</button>
            </div>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="container mx-auto px-4 py-8">
        <header class="mb-8">
            <h1 class="text-3xl font-bold text-center mb-4">Timeblocking App</h1>
            <div class="flex justify-center">
                <div class="bg-white rounded-lg shadow-md p-4">
                    <div class="flex items-center space-x-4">
                        <button id="prevDay" class="px-3 py-1 bg-blue-500 text-white rounded">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                        <button id="prevWeek" class="px-3 py-1 bg-blue-500 text-white rounded">
                            <i class="fas fa-angle-double-left"></i>
                        </button>
                        <input type="date" id="currentDate" class="px-3 py-1 border rounded">
                        <button id="todayBtn" class="px-3 py-1 bg-blue-500 text-white rounded">Hari Ini</button>
                        <button id="nextWeek" class="px-3 py-1 bg-blue-500 text-white rounded">
                            <i class="fas fa-angle-double-right"></i>
                        </button>
                        <button id="nextDay" class="px-3 py-1 bg-blue-500 text-white rounded">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    </div>
                </div>
            </div>
        </header>
        
        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
            <!-- Analog Clock -->
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-xl font-bold mb-4 text-center">Jam Analog</h2>
                <div class="analog-clock">
                    <div class="clock-face">
                        <!-- Clock numbers -->
                        <div class="clock-number" style="top: 10px; left: 50%; transform: translateX(-50%);">12</div>
                        <div class="clock-number" style="top: 50%; right: 10px; transform: translateY(-50%);">3</div>
                        <div class="clock-number" style="bottom: 10px; left: 50%; transform: translateX(-50%);">6</div>
                        <div class="clock-number" style="top: 50%; left: 10px; transform: translateY(-50%);">9</div>
                        
                        <!-- Clock hands -->
                        <div id="hourHand" class="hand hour-hand"></div>
                        <div id="minuteHand" class="hand minute-hand"></div>
                        <div id="secondHand" class="hand second-hand"></div>
                        <div class="center-dot"></div>
                        
                        <!-- Timeblock overlays will be added dynamically -->
                        <div id="clockOverlays"></div>
                    </div>
                </div>
                <div class="text-center mt-4">
                    <p id="digitalTime" class="text-2xl font-mono"></p>
                </div>
            </div>
            
            <!-- Timeblocks List -->
            <div class="bg-white rounded-lg shadow-md p-6">
                <div class="flex justify-between items-center mb-4">
                    <h2 class="text-xl font-bold">Timeblocks</h2>
                    <div class="space-x-2">
                        <button id="addTimeblockBtn" class="px-3 py-1 bg-green-500 text-white rounded">
                            <i class="fas fa-plus"></i> Tambah
                        </button>
                        <button id="addShiftBtn" class="px-3 py-1 bg-blue-500 text-white rounded">
                            <i class="fas fa-briefcase"></i> Shift
                        </button>
                        <button id="copyDayBtn" class="px-3 py-1 bg-purple-500 text-white rounded">
                            <i class="fas fa-copy"></i> Salin
                        </button>
                        <button id="importExportBtn" class="px-3 py-1 bg-gray-500 text-white rounded">
                            <i class="fas fa-file-import"></i> I/E
                        </button>
                    </div>
                </div>
                
                <div id="timeblocksList" class="space-y-2 max-h-96 overflow-y-auto">
                    <!-- Timeblocks will be loaded here -->
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Global variables
        let currentDate = new Date();
        let timeblocks = [];
        let selectedTimeblock = null;
        let pinVerified = false;
        let pinCallback = null;
        
        // Initialize the app
        document.addEventListener('DOMContentLoaded', function() {
            // Set current date in the date picker
            document.getElementById('currentDate').valueAsDate = currentDate;
            
            // Load timeblocks for the current date
            loadTimeblocks();
            
            // Update clock every second
            updateClock();
            setInterval(updateClock, 1000);
            
            // Set up event listeners
            setupEventListeners();
        });
        
        // Set up event listeners
        function setupEventListeners() {
            // Navigation buttons
            document.getElementById('prevDay').addEventListener('click', function() {
                changeDate(-1);
            });
            
            document.getElementById('prevWeek').addEventListener('click', function() {
                changeDate(-7);
            });
            
            document.getElementById('nextDay').addEventListener('click', function() {
                changeDate(1);
            });
            
            document.getElementById('nextWeek').addEventListener('click', function() {
                changeDate(7);
            });
            
            document.getElementById('todayBtn').addEventListener('click', function() {
                currentDate = new Date();
                document.getElementById('currentDate').valueAsDate = currentDate;
                loadTimeblocks();
            });
            
            document.getElementById('currentDate').addEventListener('change', function() {
                currentDate = new Date(this.value);
                loadTimeblocks();
            });
            
            // Timeblock buttons
            document.getElementById('addTimeblockBtn').addEventListener('click', function() {
                openTimeblockModal();
            });
            
            document.getElementById('addShiftBtn').addEventListener('click', function() {
                openShiftModal();
            });
            
            document.getElementById('copyDayBtn').addEventListener('click', function() {
                copyDay();
            });
            
            document.getElementById('importExportBtn').addEventListener('click', function() {
                openImportExportModal();
            });
            
            // Modal buttons
            document.getElementById('pinCancel').addEventListener('click', function() {
                closePinModal();
            });
            
            document.getElementById('pinVerify').addEventListener('click', function() {
                verifyPin();
            });
            
            document.getElementById('timeblockCancel').addEventListener('click', function() {
                closeTimeblockModal();
            });
            
            document.getElementById('timeblockForm').addEventListener('submit', function(e) {
                e.preventDefault();
                saveTimeblock();
            });
            
            document.getElementById('shiftCancel').addEventListener('click', function() {
                closeShiftModal();
            });
            
            document.getElementById('shiftForm').addEventListener('submit', function(e) {
                e.preventDefault();
                saveShift();
            });
            
            document.getElementById('addShift').addEventListener('click', function() {
                addCustomShift();
            });
            
            document.getElementById('importExportClose').addEventListener('click', function() {
                closeImportExportModal();
            });
            
            document.getElementById('exportBtn').addEventListener('click', function() {
                exportData();
            });
            
            document.getElementById('importBtn').addEventListener('click', function() {
                importData();
            });
            
            // PIN input enter key
            document.getElementById('pinInput').addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    verifyPin();
                }
            });
            
            // Remove custom shift buttons (using event delegation)
            document.getElementById('customShifts').addEventListener('click', function(e) {
                if (e.target.classList.contains('remove-shift')) {
                    e.target.parentElement.remove();
                }
            });
        }
        
        // Change the current date by a number of days
        function changeDate(days) {
            currentDate.setDate(currentDate.getDate() + days);
            document.getElementById('currentDate').valueAsDate = currentDate;
            loadTimeblocks();
        }
        
        // Load timeblocks for the current date
        function loadTimeblocks() {
            const dateStr = formatDate(currentDate);
            
            // Show loading indicator
            document.getElementById('timeblocksList').innerHTML = '<div class="text-center py-4"><i class="fas fa-spinner fa-spin"></i> Memuat...</div>';
            
            // Fetch timeblocks from API
            fetch(`?action=load&date=${dateStr}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        timeblocks = data.data || [];
                        renderTimeblocks();
                        updateClockOverlays();
                    } else {
                        showNotification('Gagal memuat timeblocks: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error loading timeblocks:', error);
                    showNotification('Terjadi kesalahan saat memuat timeblocks', 'error');
                });
        }
        
        // Render timeblocks in the list
        function renderTimeblocks() {
            const container = document.getElementById('timeblocksList');
            
            if (timeblocks.length === 0) {
                container.innerHTML = '<div class="text-center py-4 text-gray-500">Tidak ada timeblock untuk hari ini</div>';
                return;
            }
            
            // Sort timeblocks by start time
            const sortedTimeblocks = [...timeblocks].sort((a, b) => {
                return a.start.localeCompare(b.start);
            });
            
            let html = '';
            sortedTimeblocks.forEach(timeblock => {
                const typeClass = timeblock.type === 'shift' ? 'shift' : 'task';
                const typeIcon = timeblock.type === 'shift' ? 'fa-briefcase' : 'fa-tasks';
                const typeColor = timeblock.type === 'shift' ? 'text-gray-600' : 'text-green-600';
                
                html += `
                    <div class="timeblock-item ${typeClass} bg-gray-50 p-3 rounded cursor-pointer" data-id="${timeblock.id}">
                        <div class="flex justify-between items-start">
                            <div class="flex-1">
                                <div class="flex items-center">
                                    <i class="fas ${typeIcon} ${typeColor} mr-2"></i>
                                    <span class="font-medium">${timeblock.start} - ${timeblock.end}</span>
                                </div>
                                <div class="mt-1">
                                    <span class="font-bold">${timeblock.title}</span>
                                    ${timeblock.tag ? `<span class="ml-2 text-xs bg-gray-200 px-2 py-1 rounded">${timeblock.tag}</span>` : ''}
                                </div>
                                ${timeblock.notes ? `<div class="text-sm text-gray-600 mt-1">${timeblock.notes}</div>` : ''}
                            </div>
                            <div class="flex space-x-1">
                                <button class="edit-timeblock px-2 py-1 text-blue-500 hover:bg-blue-100 rounded" data-id="${timeblock.id}">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="delete-timeblock px-2 py-1 text-red-500 hover:bg-red-100 rounded" data-id="${timeblock.id}">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
            
            // Add event listeners to timeblock items
            document.querySelectorAll('.timeblock-item').forEach(item => {
                item.addEventListener('click', function(e) {
                    // Don't trigger if clicking on edit/delete buttons
                    if (e.target.closest('.edit-timeblock') || e.target.closest('.delete-timeblock')) {
                        return;
                    }
                    
                    const id = this.getAttribute('data-id');
                    const timeblock = timeblocks.find(tb => tb.id === id);
                    if (timeblock) {
                        selectTimeblock(timeblock);
                    }
                });
            });
            
            // Add event listeners to edit buttons
            document.querySelectorAll('.edit-timeblock').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const id = this.getAttribute('data-id');
                    const timeblock = timeblocks.find(tb => tb.id === id);
                    if (timeblock) {
                        openTimeblockModal(timeblock);
                    }
                });
            });
            
            // Add event listeners to delete buttons
            document.querySelectorAll('.delete-timeblock').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.stopPropagation();
                    const id = this.getAttribute('data-id');
                    if (confirm('Apakah Anda yakin ingin menghapus timeblock ini?')) {
                        deleteTimeblock(id);
                    }
                });
            });
        }
        
        // Select a timeblock and update the clock
        function selectTimeblock(timeblock) {
            selectedTimeblock = timeblock;
            updateClockOverlays();
            
            // Highlight the selected timeblock
            document.querySelectorAll('.timeblock-item').forEach(item => {
                item.classList.remove('ring-2', 'ring-blue-500');
            });
            
            document.querySelector(`.timeblock-item[data-id="${timeblock.id}"]`).classList.add('ring-2', 'ring-blue-500');
        }
        
        // Update the analog clock
        function updateClock() {
            const now = new Date();
            const hours = now.getHours();
            const minutes = now.getMinutes();
            const seconds = now.getSeconds();
            
            // Calculate angles
            const hourAngle = (hours % 12) * 30 + minutes * 0.5;
            const minuteAngle = minutes * 6 + seconds * 0.1;
            const secondAngle = seconds * 6;
            
            // Update hands
            document.getElementById('hourHand').style.transform = `rotate(${hourAngle}deg)`;
            document.getElementById('minuteHand').style.transform = `rotate(${minuteAngle}deg)`;
            document.getElementById('secondHand').style.transform = `rotate(${secondAngle}deg)`;
            
            // Update digital time
            document.getElementById('digitalTime').textContent = formatTime(hours, minutes, seconds);
        }
        
        // Update clock overlays for timeblocks
        function updateClockOverlays() {
            const container = document.getElementById('clockOverlays');
            container.innerHTML = '';
            
            // Add overlay for selected timeblock
            if (selectedTimeblock) {
                const overlay = createTimeblockOverlay(selectedTimeblock);
                if (overlay) {
                    container.appendChild(overlay);
                }
            }
            
            // Add overlays for all timeblocks
            timeblocks.forEach(timeblock => {
                if (timeblock !== selectedTimeblock) {
                    const overlay = createTimeblockOverlay(timeblock, true);
                    if (overlay) {
                        container.appendChild(overlay);
                    }
                }
            });
        }
        
        // Create an overlay element for a timeblock
        function createTimeblockOverlay(timeblock, dimmed = false) {
            const startParts = timeblock.start.split(':');
            const endParts = timeblock.end.split(':');
            
            const startHour = parseInt(startParts[0]);
            const startMinute = parseInt(startParts[1]);
            const endHour = parseInt(endParts[0]);
            const endMinute = parseInt(endParts[1]);
            
            // Calculate angles
            const startAngle = (startHour % 12) * 30 + startMinute * 0.5 - 90;
            const endAngle = (endHour % 12) * 30 + endMinute * 0.5 - 90;
            
            // Calculate the size of the arc
            let arcSize = endAngle - startAngle;
            if (arcSize < 0) {
                arcSize += 360;
            }
            
            // Create the overlay
            const overlay = document.createElement('div');
            overlay.className = timeblock.type === 'shift' ? 'shift-overlay' : 'timeblock-overlay';
            
            if (dimmed) {
                overlay.style.opacity = '0.2';
            }
            
            // Calculate position and size
            const radius = 110; // Clock radius
            const centerX = 125; // Clock center X
            const centerY = 125; // Clock center Y
            
            // Create a path for the arc
            const startRad = startAngle * Math.PI / 180;
            const endRad = endAngle * Math.PI / 180;
            
            const x1 = centerX + radius * Math.cos(startRad);
            const y1 = centerY + radius * Math.sin(startRad);
            const x2 = centerX + radius * Math.cos(endRad);
            const y2 = centerY + radius * Math.sin(endRad);
            
            const largeArc = arcSize > 180 ? 1 : 0;
            
            const pathData = `M ${centerX} ${centerY} L ${x1} ${y1} A ${radius} ${radius} 0 ${largeArc} 1 ${x2} ${y2} Z`;
            
            // Create SVG element
            const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
            svg.setAttribute('width', '250');
            svg.setAttribute('height', '250');
            svg.style.position = 'absolute';
            svg.style.top = '0';
            svg.style.left = '0';
            svg.style.pointerEvents = 'none';
            
            const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
            path.setAttribute('d', pathData);
            path.setAttribute('fill', timeblock.type === 'shift' ? 'rgba(158, 158, 158, 0.3)' : 'rgba(76, 175, 80, 0.3)');
            
            svg.appendChild(path);
            overlay.appendChild(svg);
            
            return overlay;
        }
        
        // Open the timeblock modal
        function openTimeblockModal(timeblock = null) {
            const modal = document.getElementById('timeblockModal');
            const title = document.getElementById('timeblockModalTitle');
            
            if (timeblock) {
                title.textContent = 'Edit Timeblock';
                document.getElementById('timeblockId').value = timeblock.id;
                document.getElementById('timeblockTitle').value = timeblock.title;
                document.getElementById('timeblockType').value = timeblock.type;
                document.getElementById('timeblockStart').value = timeblock.start;
                document.getElementById('timeblockEnd').value = timeblock.end;
                document.getElementById('timeblockTag').value = timeblock.tag || '';
                document.getElementById('timeblockNotes').value = timeblock.notes || '';
            } else {
                title.textContent = 'Tambah Timeblock';
                document.getElementById('timeblockForm').reset();
                document.getElementById('timeblockId').value = '';
            }
            
            document.getElementById('timeblockDate').value = formatDate(currentDate);
            modal.classList.add('show');
        }
        
        // Close the timeblock modal
        function closeTimeblockModal() {
            document.getElementById('timeblockModal').classList.remove('show');
        }
        
        // Save a timeblock
        function saveTimeblock() {
            const id = document.getElementById('timeblockId').value;
            const date = document.getElementById('timeblockDate').value;
            const title = document.getElementById('timeblockTitle').value;
            const type = document.getElementById('timeblockType').value;
            const start = document.getElementById('timeblockStart').value;
            const end = document.getElementById('timeblockEnd').value;
            const tag = document.getElementById('timeblockTag').value;
            const notes = document.getElementById('timeblockNotes').value;
            
            // Validate input
            if (!title || !start || !end) {
                showNotification('Judul, waktu mulai, dan waktu selesai harus diisi', 'error');
                return;
            }
            
            // Check for time conflicts
            const hasConflict = timeblocks.some(tb => {
                if (tb.id === id) return false; // Skip the current timeblock if editing
                
                return timeOverlaps(start, end, tb.start, tb.end);
            });
            
            if (hasConflict) {
                showNotification('Timeblock bentrok dengan timeblock lain', 'error');
                return;
            }
            
            // Create or update the timeblock
            let timeblock;
            if (id) {
                // Update existing timeblock
                timeblock = timeblocks.find(tb => tb.id === id);
                timeblock.title = title;
                timeblock.type = type;
                timeblock.start = start;
                timeblock.end = end;
                timeblock.tag = tag;
                timeblock.notes = notes;
            } else {
                // Create new timeblock
                timeblock = {
                    id: 'tb-' + Date.now(),
                    title,
                    type,
                    start,
                    end,
                    tag,
                    notes
                };
                timeblocks.push(timeblock);
            }
            
            // Save to server
            saveTimeblocksToServer(date, timeblocks);
            closeTimeblockModal();
        }
        
        // Open the shift modal
        function openShiftModal() {
            const modal = document.getElementById('shiftModal');
            document.getElementById('shiftDate').value = formatDate(currentDate);
            document.getElementById('shiftForm').reset();
            
            // Reset custom shifts
            document.getElementById('customShifts').innerHTML = `
                <div class="custom-shift mb-2 flex items-center">
                    <input type="time" class="custom-shift-start p-2 border rounded mr-2" placeholder="Mulai">
                    <input type="time" class="custom-shift-end p-2 border rounded mr-2" placeholder="Selesai">
                    <button type="button" class="remove-shift px-2 py-1 bg-red-500 text-white rounded">Hapus</button>
                </div>
            `;
            
            modal.classList.add('show');
        }
        
        // Close the shift modal
        function closeShiftModal() {
            document.getElementById('shiftModal').classList.remove('show');
        }
        
        // Add a custom shift input
        function addCustomShift() {
            const container = document.getElementById('customShifts');
            const shiftDiv = document.createElement('div');
            shiftDiv.className = 'custom-shift mb-2 flex items-center';
            shiftDiv.innerHTML = `
                <input type="time" class="custom-shift-start p-2 border rounded mr-2" placeholder="Mulai">
                <input type="time" class="custom-shift-end p-2 border rounded mr-2" placeholder="Selesai">
                <button type="button" class="remove-shift px-2 py-1 bg-red-500 text-white rounded">Hapus</button>
            `;
            container.appendChild(shiftDiv);
        }
        
        // Save shift
        function saveShift() {
            const date = document.getElementById('shiftDate').value;
            const preset = document.getElementById('shiftPreset').value;
            
            // Get shift presets
            const shiftPresets = {
                'Pagi 2 split': [
                    {start: '08:00', end: '13:00'},
                    {start: '18:00', end: '22:00'}
                ],
                'Pagi 1 split': [
                    {start: '08:00', end: '17:00'}
                ],
                'Siang': [
                    {start: '13:00', end: '22:00'}
                ],
                'Middle': [
                    {start: '11:00', end: '20:00'}
                ],
                'Libur': []
            };
            
            let newShifts = [];
            
            // Use preset if selected
            if (preset && shiftPresets[preset]) {
                newShifts = shiftPresets[preset].map((shift, index) => ({
                    id: 'shift-' + Date.now() + '-' + index,
                    title: `Shift ${preset}` + (shiftPresets[preset].length > 1 ? ` (part ${index + 1})` : ''),
                    type: 'shift',
                    start: shift.start,
                    end: shift.end,
                    tag: preset.toLowerCase().replace(' ', '-')
                }));
            }
            
            // Add custom shifts
            document.querySelectorAll('.custom-shift').forEach((shiftDiv, index) => {
                const start = shiftDiv.querySelector('.custom-shift-start').value;
                const end = shiftDiv.querySelector('.custom-shift-end').value;
                
                if (start && end) {
                    newShifts.push({
                        id: 'shift-' + Date.now() + '-' + index,
                        title: 'Shift Custom',
                        type: 'shift',
                        start,
                        end,
                        tag: 'custom'
                    });
                }
            });
            
            // Remove existing shifts
            timeblocks = timeblocks.filter(tb => tb.type !== 'shift');
            
            // Add new shifts
            timeblocks = [...timeblocks, ...newShifts];
            
            // Save to server
            saveTimeblocksToServer(date, timeblocks);
            closeShiftModal();
        }
        
        // Copy timeblocks from previous day
        function copyDay() {
            const prevDate = new Date(currentDate);
            prevDate.setDate(prevDate.getDate() - 1);
            const prevDateStr = formatDate(prevDate);
            
            // Show loading indicator
            showNotification('Menyalin timeblocks dari hari sebelumnya...', 'info');
            
            // Fetch timeblocks from previous day
            fetch(`?action=load&date=${prevDateStr}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const prevTimeblocks = data.data || [];
                        
                        // Create new timeblocks with new IDs
                        const newTimeblocks = prevTimeblocks.map(tb => ({
                            ...tb,
                            id: tb.type === 'shift' ? 'shift-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9) : 'tb-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9)
                        }));
                        
                        // Save to server
                        saveTimeblocksToServer(formatDate(currentDate), newTimeblocks);
                        showNotification('Timeblocks berhasil disalin', 'success');
                    } else {
                        showNotification('Gagal menyalin timeblocks: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error copying timeblocks:', error);
                    showNotification('Terjadi kesalahan saat menyalin timeblocks', 'error');
                });
        }
        
        // Delete a timeblock
        function deleteTimeblock(id) {
            requirePin(() => {
                const date = formatDate(currentDate);
                
                // Show loading indicator
                showNotification('Menghapus timeblock...', 'info');
                
                // Send delete request
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('date', date);
                formData.append('id', id);
                formData.append('pin', document.getElementById('pinInput').value);
                
                fetch('?action=delete', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Remove from local array
                        timeblocks = timeblocks.filter(tb => tb.id !== id);
                        renderTimeblocks();
                        updateClockOverlays();
                        showNotification('Timeblock berhasil dihapus', 'success');
                    } else {
                        showNotification('Gagal menghapus timeblock: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error deleting timeblock:', error);
                    showNotification('Terjadi kesalahan saat menghapus timeblock', 'error');
                });
            });
        }
        
        // Save timeblocks to server
        function saveTimeblocksToServer(date, blocks) {
            requirePin(() => {
                // Show loading indicator
                showNotification('Menyimpan timeblocks...', 'info');
                
                // Send save request
                const formData = new FormData();
                formData.append('action', 'save');
                formData.append('date', date);
                formData.append('timeblocks', JSON.stringify(blocks));
                formData.append('pin', document.getElementById('pinInput').value);
                
                fetch('?action=save', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        timeblocks = blocks;
                        renderTimeblocks();
                        updateClockOverlays();
                        showNotification('Timeblocks berhasil disimpan', 'success');
                    } else {
                        showNotification('Gagal menyimpan timeblocks: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error saving timeblocks:', error);
                    showNotification('Terjadi kesalahan saat menyimpan timeblocks', 'error');
                });
            });
        }
        
        // Open import/export modal
        function openImportExportModal() {
            document.getElementById('importExportModal').classList.add('show');
        }
        
        // Close import/export modal
        function closeImportExportModal() {
            document.getElementById('importExportModal').classList.remove('show');
        }
        
        // Export data
        function exportData() {
            requirePin(() => {
                // Show loading indicator
                showNotification('Mengekspor data...', 'info');
                
                // Send export request
                const formData = new FormData();
                formData.append('action', 'export');
                formData.append('pin', document.getElementById('pinInput').value);
                
                fetch('?action=export', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Create and download the file
                        const dataStr = JSON.stringify(data.data, null, 2);
                        const dataUri = 'data:application/json;charset=utf-8,'+ encodeURIComponent(dataStr);
                        
                        const exportFileDefaultName = `timeblocks_${formatDate(new Date())}.json`;
                        
                        const linkElement = document.createElement('a');
                        linkElement.setAttribute('href', dataUri);
                        linkElement.setAttribute('download', exportFileDefaultName);
                        linkElement.click();
                        
                        showNotification('Data berhasil diekspor', 'success');
                    } else {
                        showNotification('Gagal mengekspor data: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error exporting data:', error);
                    showNotification('Terjadi kesalahan saat mengekspor data', 'error');
                });
            });
        }
        
        // Import data
        function importData() {
            const fileInput = document.getElementById('importFile');
            
            if (!fileInput.files.length) {
                showNotification('Pilih file untuk diimpor', 'error');
                return;
            }
            
            requirePin(() => {
                // Show loading indicator
                showNotification('Mengimpor data...', 'info');
                
                // Send import request
                const formData = new FormData();
                formData.append('action', 'import');
                formData.append('importFile', fileInput.files[0]);
                formData.append('pin', document.getElementById('pinInput').value);
                
                fetch('?action=import', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification('Data berhasil diimpor', 'success');
                        // Reload current timeblocks
                        loadTimeblocks();
                    } else {
                        showNotification('Gagal mengimpor data: ' + data.message, 'error');
                    }
                })
                .catch(error => {
                    console.error('Error importing data:', error);
                    showNotification('Terjadi kesalahan saat mengimpor data', 'error');
                });
            });
        }
        
        // Require PIN verification
        function requirePin(callback) {
            if (pinVerified) {
                callback();
                return;
            }
            
            pinCallback = callback;
            document.getElementById('pinModal').classList.add('show');
            document.getElementById('pinInput').focus();
        }
        
        // Verify PIN
        function verifyPin() {
            const pin = document.getElementById('pinInput').value;
            
            if (!pin) {
                showNotification('PIN harus diisi', 'error');
                return;
            }
            
            // Show loading indicator
            showNotification('Memverifikasi PIN...', 'info');
            
            // Send verification request
            const formData = new FormData();
            formData.append('action', 'verifyPin');
            formData.append('pin', pin);
            
            fetch('?action=verifyPin', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    pinVerified = true;
                    closePinModal();
                    
                    if (pinCallback) {
                        pinCallback();
                        pinCallback = null;
                    }
                    
                    showNotification('PIN berhasil diverifikasi', 'success');
                } else {
                    showNotification('PIN salah: ' + data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error verifying PIN:', error);
                showNotification('Terjadi kesalahan saat memverifikasi PIN', 'error');
            });
        }
        
        // Close PIN modal
        function closePinModal() {
            document.getElementById('pinModal').classList.remove('show');
            document.getElementById('pinInput').value = '';
        }
        
        // Show notification
        function showNotification(message, type = 'info') {
            const notification = document.getElementById('notification');
            notification.textContent = message;
            notification.className = 'notification ' + type;
            notification.classList.add('show');
            
            setTimeout(() => {
                notification.classList.remove('show');
            }, 3000);
        }
        
        // Check if two time intervals overlap
        function timeOverlaps(start1, end1, start2, end2) {
            const t1 = new Date('2000-01-01T' + start1);
            const e1 = new Date('2000-01-01T' + end1);
            const t2 = new Date('2000-01-01T' + start2);
            const e2 = new Date('2000-01-01T' + end2);
            
            return (t1 < e2 && e1 > t2);
        }
        
        // Format date as YYYY-MM-DD
        function formatDate(date) {
            const year = date.getFullYear();
            const month = String(date.getMonth() + 1).padStart(2, '0');
            const day = String(date.getDate()).padStart(2, '0');
            
            return `${year}-${month}-${day}`;
        }
        
        // Format time as HH:MM:SS
        function formatTime(hours, minutes, seconds) {
            return `${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
        }
    </script>
</body>
</html>