<?php
require_once 'config.php';

// Check if token is provided
if (!isset($_GET['token']) || empty($_GET['token'])) {
    die("Invalid share link.");
}

$token = sanitizeInput($_GET['token']);

// Get the shared note
$stmt = $pdo->prepare("SELECT n.*, u.username FROM notes n 
                       JOIN users u ON n.user_id = u.id 
                       WHERE n.share_token = ?");
$stmt->execute([$token]);
$note = $stmt->fetch();

if (!$note) {
    die("Note not found or share link has expired.");
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SuNote - Shared Note</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #fbbc04;
            --secondary-color: #f1f3f4;
            --text-color: #202124;
            --border-color: #e0e0e0;
            --hover-color: #f5f5f5;
            --shadow-light: 0 1px 3px rgba(0, 0, 0, 0.1);
            --shadow-medium: 0 2px 10px rgba(0, 0, 0, 0.15);
            --accent-color: #1a73e8;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Roboto', Arial, sans-serif;
        }
        
        body {
            background-color: #f9f9f9;
            color: var(--text-color);
            line-height: 1.6;
        }
        
        .container {
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px 0;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--border-color);
            background-color: #fff;
            position: sticky;
            top: 0;
            z-index: 10;
        }
        
        .header h1 {
            color: var(--primary-color);
            display: flex;
            align-items: center;
            font-size: 1.8rem;
        }
        
        .header h1 i {
            margin-right: 10px;
        }
        
        .shared-info {
            background-color: #fff;
            padding: 15px;
            border-radius: 12px;
            margin-bottom: 20px;
            font-size: 15px;
            box-shadow: var(--shadow-light);
            border: 1px solid var(--border-color);
            display: flex;
            flex-wrap: wrap;
            align-items: center;
            justify-content: space-between;
        }
        
        .shared-info-details {
            flex: 1;
            min-width: 200px;
        }
        
        .shared-info p {
            margin: 5px 0;
        }
        
        .shared-info strong {
            font-weight: 500;
            margin-right: 5px;
        }
        
        .shared-info-actions {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-top: 10px;
        }
        
        .btn {
            background-color: var(--primary-color);
            color: #fff;
            border: none;
            padding: 10px 16px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            text-decoration: none;
        }
        
        .btn i {
            margin-right: 8px;
        }
        
        .btn:hover {
            background-color: #f9a602;
            transform: translateY(-1px);
            box-shadow: var(--shadow-light);
        }
        
        .btn:active {
            transform: translateY(0);
        }
        
        .btn-light {
            background-color: var(--secondary-color);
            color: var(--text-color);
        }
        
        .btn-light:hover {
            background-color: #e4e6e7;
        }
        
        .note-container {
            background-color: #fff;
            padding: 25px;
            border-radius: 12px;
            box-shadow: var(--shadow-light);
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }
        
        .note-title {
            font-size: 24px;
            font-weight: 500;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--border-color);
        }
        
        .note-content {
            white-space: pre-wrap;
            word-break: break-word;
            font-size: 16px;
            line-height: 1.7;
        }
        
        .copy-content-btn {
            position: absolute;
            top: 15px;
            right: 15px;
            background: none;
            border: none;
            color: #5f6368;
            font-size: 18px;
            cursor: pointer;
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            transition: all 0.2s ease;
        }
        
        .copy-content-btn:hover {
            background-color: var(--secondary-color);
            color: var(--text-color);
        }
        
        .back-link {
            display: inline-flex;
            align-items: center;
            margin-top: 10px;
            color: var(--accent-color);
            text-decoration: none;
            font-weight: 500;
        }
        
        .back-link i {
            margin-right: 8px;
        }
        
        .back-link:hover {
            text-decoration: underline;
        }
        
        /* Toast notification */
        .toast {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background-color: var(--text-color);
            color: #fff;
            padding: 12px 20px;
            border-radius: 8px;
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.2);
            z-index: 2000;
            opacity: 0;
            transform: translateY(20px);
            transition: opacity 0.3s ease, transform 0.3s ease;
        }
        
        .toast.show {
            opacity: 1;
            transform: translateY(0);
        }
        
        /* Loading spinner */
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(255, 255, 255, 0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 2000;
            opacity: 1;
            visibility: visible;
        }
        
        .loading-overlay.hide {
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0s linear 0.3s;
        }
        
        .spinner {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            border: 5px solid var(--secondary-color);
            border-top-color: var(--primary-color);
            animation: spin 1s infinite linear;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .container {
                padding: 15px;
            }
            
            .note-title {
                font-size: 20px;
            }
            
            .shared-info {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .shared-info-actions {
                margin-top: 15px;
                width: 100%;
            }
            
            .note-container {
                padding: 15px;
            }
        }
    </style>
</head>
<body>
    <!-- Loading overlay -->
    <div class="loading-overlay" id="loading-overlay">
        <div class="spinner"></div>
    </div>
    
    <!-- Toast notification -->
    <div class="toast" id="toast"></div>
    
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-sticky-note"></i> SuNote</h1>
        </div>
        
        <div class="shared-info">
            <div class="shared-info-details">
                <p><strong><i class="fas fa-user"></i> Shared by:</strong> <?php echo htmlspecialchars($note['username']); ?></p>
                <p><strong><i class="far fa-clock"></i> Last updated:</strong> <?php echo date('F j, Y, g:i a', strtotime($note['updated_at'])); ?></p>
            </div>
            <div class="shared-info-actions">
                <button id="copy-all-content" class="btn btn-light">
                    <i class="far fa-copy"></i> Copy All
                </button>
                <a href="index.php" class="btn">
                    <i class="fas fa-home"></i> Go to SuNote
                </a>
            </div>
        </div>
        
        <div class="note-container" style="position: relative;">
            <button id="copy-content-btn" class="copy-content-btn" title="Copy to clipboard">
                <i class="far fa-copy"></i>
            </button>
            <h1 class="note-title"><?php echo htmlspecialchars($note['title']); ?></h1>
            <div class="note-content"><?php echo nl2br(htmlspecialchars($note['content'])); ?></div>
        </div>
    </div>
    
    <script>
        // Loading overlay
        const loadingOverlay = document.getElementById('loading-overlay');
        window.addEventListener('load', function() {
            setTimeout(function() {
                loadingOverlay.classList.add('hide');
            }, 500);
        });
        
        // Toast notification
        const toast = document.getElementById('toast');
        function showToast(message, duration = 3000) {
            toast.textContent = message;
            toast.classList.add('show');
            setTimeout(() => {
                toast.classList.remove('show');
            }, duration);
        }
        
        // Copy content button functionality
        const copyContentBtn = document.getElementById('copy-content-btn');
        const noteContent = document.querySelector('.note-content');
        
        copyContentBtn.addEventListener('click', function() {
            const tempInput = document.createElement('textarea');
            tempInput.value = noteContent.innerText;
            document.body.appendChild(tempInput);
            tempInput.select();
            document.execCommand('copy');
            document.body.removeChild(tempInput);
            
            showToast('Content copied to clipboard!');
        });
        
        // Copy all button (title + content)
        const copyAllBtn = document.getElementById('copy-all-content');
        const noteTitle = document.querySelector('.note-title');
        
        copyAllBtn.addEventListener('click', function() {
            const tempInput = document.createElement('textarea');
            tempInput.value = noteTitle.innerText + '\n\n' + noteContent.innerText;
            document.body.appendChild(tempInput);
            tempInput.select();
            document.execCommand('copy');
            document.body.removeChild(tempInput);
            
            showToast('Title and content copied to clipboard!');
        });
    </script>
</body>
</html>