<?php
session_start();

// Konfigurasi database JSON
$jsonFile = 'soal_database.json';

// Inisialisasi database jika belum ada
if (!file_exists($jsonFile)) {
    $initialData = [
        'questions' => [],
        'events' => [],
        'admins' => [
            [
                'id' => uniqid(),
                'username' => 'rizaldi',
                'password' => password_hash('admin123', PASSWORD_DEFAULT),
                'created_at' => date('Y-m-d H:i:s'),
                'is_super_admin' => true
            ]
        ]
    ];
    file_put_contents($jsonFile, json_encode($initialData, JSON_PRETTY_PRINT));
}

// Load database
$database = json_decode(file_get_contents($jsonFile), true);

// Pastikan struktur database lengkap
if (!isset($database['events'])) {
    $database['events'] = [];
}

if (!isset($database['admins'])) {
    $database['admins'] = [
        [
            'id' => uniqid(),
            'username' => 'rizaldi',
            'password' => password_hash('admin123', PASSWORD_DEFAULT),
            'created_at' => date('Y-m-d H:i:s'),
            'is_super_admin' => true
        ]
    ];
    file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
}

// Cek Autentikasi
$authenticated = false;
$current_admin = null;

if (isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true && isset($_SESSION['admin_id'])) {
    foreach ($database['admins'] as $admin) {
        if ($admin['id'] === $_SESSION['admin_id']) {
            $authenticated = true;
            $current_admin = $admin;
            break;
        }
    }
}

// Proses Login
if (isset($_POST['login'])) {
    $login_username = $_POST['username'];
    $login_password = $_POST['password'];
    $found = false;
    
    foreach ($database['admins'] as $admin) {
        if ($admin['username'] === $login_username && password_verify($login_password, $admin['password'])) {
            $_SESSION['logged_in'] = true;
            $_SESSION['admin_id'] = $admin['id'];
            
            // Proses "Ingat Saya"
            if (isset($_POST['remember_me']) && $_POST['remember_me'] == 1) {
                // Set cookie selama 30 hari
                setcookie('remember_user', 'true', time() + (86400 * 30), "/");
                setcookie('admin_id', $admin['id'], time() + (86400 * 30), "/");
            }
            
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
        }
    }
    
    if (!$found) {
        $login_error = "Username atau password salah!";
    }
}

// Cek cookie "Ingat Saya"
if (!$authenticated && isset($_COOKIE['remember_user']) && $_COOKIE['remember_user'] === 'true' && isset($_COOKIE['admin_id'])) {
    foreach ($database['admins'] as $admin) {
        if ($admin['id'] === $_COOKIE['admin_id']) {
            $_SESSION['logged_in'] = true;
            $_SESSION['admin_id'] = $admin['id'];
            $authenticated = true;
            $current_admin = $admin;
            break;
        }
    }
}

// Proses Logout
if (isset($_GET['logout'])) {
    session_unset();
    session_destroy();
    
    // Hapus cookie
    setcookie('remember_user', '', time() - 3600, '/');
    setcookie('admin_id', '', time() - 3600, '/');
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Daftar jenjang pendidikan dengan kode baru
$educationLevels = [
    'tk' => 'TK',
    'sd' => 'SD',
    'sm' => 'SMP',
    'ma' => 'SMA',
    'pt' => 'Perguruan Tinggi',
    'gr' => 'Guru',
    'um' => 'Umum'
];

// Daftar jenis soal
$questionTypes = [
    'pilgan' => 'Pilihan Ganda',
    'essay' => 'Essay',
    'campuran' => 'Campuran'
];

// Fungsi untuk menghasilkan kode soal unik dengan kode jenjang baru
function generateQuestionCode($data) {
    // Format: [Jenjang][JenisSoal][JumlahSoal][Bulan][Tahun][Random3Digit]
    $levelCode = strtoupper($data['education_level']);
    $typeCode = substr(strtoupper($data['question_type']), 0, 1);
    $questionCount = str_pad($data['total_questions'], 2, '0', STR_PAD_LEFT);
    $month = str_pad($data['month'], 2, '0', STR_PAD_LEFT);
    $year = substr($data['year'], -2);
    $randomDigits = rand(100, 999);
    
    return $levelCode . $typeCode . $questionCount . $month . $year . $randomDigits;
}

// Handler untuk menambah soal baru
if (isset($_POST['add_question']) && $authenticated) {
    $newQuestion = [
        'id' => uniqid(),
        'gform_link' => !empty($_POST['gform_link']) ? $_POST['gform_link'] : '#',
        'solution_link' => !empty($_POST['solution_link']) ? $_POST['solution_link'] : '#',
        'subject' => $_POST['subject'],
        'education_level' => $_POST['education_level'],
        'question_type' => $_POST['question_type'],
        'total_questions' => (int)$_POST['total_questions'],
        'multiple_choice_options' => $_POST['question_type'] == 'pilgan' || $_POST['question_type'] == 'campuran' ? (int)$_POST['multiple_choice_options'] : 0,
        'month' => (int)$_POST['month'],
        'year' => (int)$_POST['year'],
        'created_at' => date('Y-m-d H:i:s'),
        'links_pending' => empty($_POST['gform_link']) || empty($_POST['solution_link']) ? true : false
    ];
    
    // Generate kode soal
    $newQuestion['code'] = generateQuestionCode($newQuestion);
    
    // Tambahkan ke database
    $database['questions'][] = $newQuestion;
    file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
    
    // Set pesan sukses
    $_SESSION['message'] = "Soal berhasil ditambahkan dengan kode: " . $newQuestion['code'] . 
                          (empty($_POST['gform_link']) || empty($_POST['solution_link']) ? 
                          " (Link soal dan/atau pembahasan masih kosong, silakan update nanti)" : "");
    $_SESSION['message_type'] = "success";
    
    // Redirect untuk mencegah re-submit form
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handler untuk mengedit soal
if (isset($_POST['edit_question']) && $authenticated) {
    $editId = $_POST['edit_id'];
    $found = false;
    
    foreach ($database['questions'] as $key => $question) {
        if ($question['id'] == $editId) {
            $updatedQuestion = [
                'id' => $editId,
                'code' => $question['code'], // CHANGE: Preserve the original code instead of generating a new one
                'gform_link' => $_POST['gform_link'],
                'solution_link' => $_POST['solution_link'],
                'subject' => $_POST['subject'],
                'education_level' => $_POST['education_level'],
                'question_type' => $_POST['question_type'],
                'total_questions' => (int)$_POST['total_questions'],
                'multiple_choice_options' => $_POST['question_type'] == 'pilgan' || $_POST['question_type'] == 'campuran' ? (int)$_POST['multiple_choice_options'] : 0,
                'month' => (int)$_POST['month'],
                'year' => (int)$_POST['year'],
                'created_at' => $question['created_at'],
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            // Update data
            $database['questions'][$key] = $updatedQuestion;
            $found = true;
            break;
        }
    }
    
    if ($found) {
        file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
        $_SESSION['message'] = "Soal berhasil diperbarui!";
        $_SESSION['message_type'] = "success";
    } else {
        $_SESSION['message'] = "Soal tidak ditemukan!";
        $_SESSION['message_type'] = "danger";
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handler untuk menghapus soal
if (isset($_POST['delete_question']) && $authenticated) {
    $deleteId = $_POST['delete_id'];
    $found = false;
    
    foreach ($database['questions'] as $key => $question) {
        if ($question['id'] == $deleteId) {
            // Hapus soal
            unset($database['questions'][$key]);
            $found = true;
            break;
        }
    }
    
    // Reindex array
    $database['questions'] = array_values($database['questions']);
    
    if ($found) {
        file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
        $_SESSION['message'] = "Soal berhasil dihapus!";
        $_SESSION['message_type'] = "success";
    } else {
        $_SESSION['message'] = "Soal tidak ditemukan!";
        $_SESSION['message_type'] = "danger";
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handler untuk menambah event
if (isset($_POST['add_event']) && $authenticated) {
    $newEvent = [
        'id' => uniqid(),
        'name' => $_POST['event_name'],
        'organizer' => $_POST['event_organizer'],
        'education_levels' => isset($_POST['event_education_levels']) ? $_POST['event_education_levels'] : [],
        'question_ids' => isset($_POST['event_questions']) ? $_POST['event_questions'] : [],
        'event_date' => $_POST['event_date'],
        'event_time' => $_POST['event_time'],
        'created_at' => date('Y-m-d H:i:s'),
        'created_by' => $_SESSION['admin_id']
    ];
    
    // Tambahkan ke database
    $database['events'][] = $newEvent;
    file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
    
    // Set pesan sukses
    $_SESSION['message'] = "Event \"{$newEvent['name']}\" berhasil ditambahkan!";
    $_SESSION['message_type'] = "success";
    
    // Redirect untuk mencegah re-submit form
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handler untuk menghapus event
if (isset($_POST['delete_event']) && $authenticated) {
    $deleteId = $_POST['delete_event_id'];
    $found = false;
    
    foreach ($database['events'] as $key => $event) {
        if ($event['id'] == $deleteId) {
            // Hapus event
            unset($database['events'][$key]);
            $found = true;
            break;
        }
    }
    
    // Reindex array
    $database['events'] = array_values($database['events']);
    
    if ($found) {
        file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
        $_SESSION['message'] = "Event berhasil dihapus!";
        $_SESSION['message_type'] = "success";
    } else {
        $_SESSION['message'] = "Event tidak ditemukan!";
        $_SESSION['message_type'] = "danger";
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// Handler untuk menambah admin baru (hanya super admin)
if (isset($_POST['add_admin']) && $authenticated && isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']) {
    $username = trim($_POST['admin_username']);
    $password = $_POST['admin_password'];
    
    // Cek apakah username sudah ada
    $username_exists = false;
    foreach ($database['admins'] as $admin) {
        if ($admin['username'] === $username) {
            $username_exists = true;
            break;
        }
    }
    
    if ($username_exists) {
        $_SESSION['message'] = "Username sudah digunakan. Silakan pilih username lain.";
        $_SESSION['message_type'] = "danger";
    } else {
        $newAdmin = [
            'id' => uniqid(),
            'username' => $username,
            'password' => password_hash($password, PASSWORD_DEFAULT),
            'created_at' => date('Y-m-d H:i:s'),
            'is_super_admin' => isset($_POST['is_super_admin']) ? true : false,
            'created_by' => $_SESSION['admin_id']
        ];
        
        // Tambahkan ke database
        $database['admins'][] = $newAdmin;
        file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
        
        $_SESSION['message'] = "Admin baru berhasil ditambahkan!";
        $_SESSION['message_type'] = "success";
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF'] . "?tab=admin");
    exit;
}

// Handler untuk mengubah password
if (isset($_POST['change_password']) && $authenticated) {
    $admin_id = $_SESSION['admin_id'];
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    
    if ($new_password !== $confirm_password) {
        $_SESSION['message'] = "Password baru dan konfirmasi password tidak cocok!";
        $_SESSION['message_type'] = "danger";
    } else {
        $found = false;
        
        foreach ($database['admins'] as $key => $admin) {
            if ($admin['id'] === $admin_id) {
                // Verifikasi password saat ini
                if (password_verify($current_password, $admin['password'])) {
                    // Update password
                    $database['admins'][$key]['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                    $database['admins'][$key]['updated_at'] = date('Y-m-d H:i:s');
                    $found = true;
                } else {
                    $_SESSION['message'] = "Password saat ini tidak valid!";
                    $_SESSION['message_type'] = "danger";
                }
                break;
            }
        }
        
        if ($found) {
            file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
            $_SESSION['message'] = "Password berhasil diubah!";
            $_SESSION['message_type'] = "success";
        }
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF'] . "?tab=admin");
    exit;
}

// Handler untuk menghapus admin
if (isset($_POST['delete_admin']) && $authenticated && isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']) {
    $delete_admin_id = $_POST['delete_admin_id'];
    $found = false;
    
    // Jangan izinkan menghapus diri sendiri
    if ($delete_admin_id === $_SESSION['admin_id']) {
        $_SESSION['message'] = "Anda tidak dapat menghapus akun admin Anda sendiri!";
        $_SESSION['message_type'] = "danger";
    } else {
        foreach ($database['admins'] as $key => $admin) {
            if ($admin['id'] === $delete_admin_id) {
                // Hapus admin
                unset($database['admins'][$key]);
                $found = true;
                break;
            }
        }
        
        // Reindex array
        $database['admins'] = array_values($database['admins']);
        
        if ($found) {
            file_put_contents($jsonFile, json_encode($database, JSON_PRETTY_PRINT));
            $_SESSION['message'] = "Admin berhasil dihapus!";
            $_SESSION['message_type'] = "success";
        } else {
            $_SESSION['message'] = "Admin tidak ditemukan!";
            $_SESSION['message_type'] = "danger";
        }
    }
    
    // Redirect
    header("Location: " . $_SERVER['PHP_SELF'] . "?tab=admin");
    exit;
}

// Filter dan pencarian
$filteredQuestions = $database['questions'];

// Filter berdasarkan jenjang
if (isset($_GET['level']) && !empty($_GET['level'])) {
    $filteredQuestions = array_filter($filteredQuestions, function($q) {
        return $q['education_level'] == $_GET['level'];
    });
}

// Filter berdasarkan jenis soal
if (isset($_GET['type']) && !empty($_GET['type'])) {
    $filteredQuestions = array_filter($filteredQuestions, function($q) {
        return $q['question_type'] == $_GET['type'];
    });
}

// Pencarian berdasarkan kode
if (isset($_GET['search']) && !empty($_GET['search'])) {
    $search = strtolower($_GET['search']);
    $filteredQuestions = array_filter($filteredQuestions, function($q) use ($search) {
        return strpos(strtolower($q['code']), $search) !== false;
    });
}

// Urutkan berdasarkan waktu terbaru
usort($filteredQuestions, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Dapatkan semua event untuk ditampilkan
$events = $database['events'];
usort($events, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Mendapatkan bulan dan tahun saat ini untuk default form
$currentMonth = date('n');
$currentYear = date('Y');

// Menentukan tab aktif
$activeTab = 'questions';
if (isset($_GET['tab'])) {
    if ($_GET['tab'] === 'events') {
        $activeTab = 'events';
    } elseif ($_GET['tab'] === 'admin' && $authenticated && isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']) {
        $activeTab = 'admin';
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pengelola Soal dan Pembahasan</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #3498db;
            --secondary-color: #2ecc71;
            --accent-color: #f39c12;
            --dark-color: #2c3e50;
            --light-color: #ecf0f1;
        }
        
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .navbar {
            background-color: var(--dark-color);
        }
        
        .card {
            border: none;
            box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-5px);
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .btn-success {
            background-color: var(--secondary-color);
            border-color: var(--secondary-color);
        }
        
        .btn-warning {
            background-color: var(--accent-color);
            border-color: var(--accent-color);
        }
        
        .badge {
            font-weight: 500;
            padding: 0.5em 0.8em;
        }
        
        .table th {
            background-color: var(--dark-color);
            color: white;
        }
        
        .question-code {
            font-family: 'Courier New', monospace;
            font-weight: bold;
            letter-spacing: 1px;
            font-size: 1.1em;
            padding: 0.25rem 0.5rem;
            background-color: #f8f9fa;
            border-radius: 4px;
            border: 1px solid #dee2e6;
            display: inline-block;
        }
        
        .copy-btn {
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .copy-btn:hover {
            transform: scale(1.1);
            color: var(--primary-color);
        }
        
        .action-buttons .btn {
            padding: 0.25rem 0.5rem;
            margin-right: 0.25rem;
        }
        
        .filter-form select,
        .filter-form input {
            max-width: 200px;
        }
        
        .login-form {
            max-width: 400px;
            margin: 100px auto;
        }
        
        /* Improve tab navigation appearance */
        .nav-tabs .nav-link {
            color: var(--dark-color);
            font-weight: 500;
            transition: all 0.3s ease;
            border-radius: 0;
            border-top: 3px solid transparent;
        }
        
        .nav-tabs .nav-link:hover {
            color: var(--primary-color);
            border-top: 3px solid var(--primary-color);
            background-color: rgba(52, 152, 219, 0.05);
        }
        
        .nav-tabs .nav-link.active {
            color: var(--primary-color);
            border-top: 3px solid var(--primary-color);
            font-weight: 600;
            border-bottom: 1px solid white;
            background-color: rgba(52, 152, 219, 0.1);
        }
        
        /* Animation for copy success */
        @keyframes fadeInOut {
            0% { opacity: 0; }
            10% { opacity: 1; }
            90% { opacity: 1; }
            100% { opacity: 0; }
        }
        
        .toast {
            animation: fadeInOut 3s ease forwards;
        }
    </style>
</head>
<body>
    <?php if (!$authenticated): ?>
    <!-- Login Form -->
    <div class="container">
        <div class="login-form card">
            <div class="card-body p-5">
                <h3 class="card-title text-center mb-4">
                    <i class="fas fa-book-open me-2"></i>
                    Pengelola Soal dan Pembahasan
                </h3>
                
                <?php if (isset($login_error)): ?>
                    <div class="alert alert-danger" role="alert">
                        <?= $login_error ?>
                    </div>
                <?php endif; ?>
                
                <form action="" method="POST">
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" class="form-control" id="password" name="password" required>
                    </div>
                    <div class="mb-3 form-check">
                        <input type="checkbox" class="form-check-input" id="remember_me" name="remember_me" value="1">
                        <label class="form-check-label" for="remember_me">Ingat Saya</label>
                    </div>
                    <div class="d-grid">
                        <button type="submit" name="login" class="btn btn-primary">Login</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <?php else: ?>
    
    <!-- Main Application -->
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark mb-4">
        <div class="container">
            <a class="navbar-brand" href="#">
                <i class="fas fa-book-open me-2"></i>
                Pengelola Soal dan Pembahasan
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="#" data-bs-toggle="modal" data-bs-target="#addQuestionModal">
                            <i class="fas fa-plus-circle me-1"></i> Tambah Soal
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="#" data-bs-toggle="modal" data-bs-target="#addEventModal">
                            <i class="fas fa-calendar-plus me-1"></i> Tambah Event
                        </a>
                    </li>
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user-circle me-1"></i> <?= htmlspecialchars($current_admin['username']) ?>
                        </a>
                        <ul class="dropdown-menu" aria-labelledby="navbarDropdown">
                            <?php if (isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']): ?>
                                <li><a class="dropdown-item" href="?tab=admin"><i class="fas fa-users-cog me-1"></i> Kelola Admin</a></li>
                                <li><hr class="dropdown-divider"></li>
                            <?php endif; ?>
                            <li><a class="dropdown-item" href="#" data-bs-toggle="modal" data-bs-target="#changePasswordModal"><i class="fas fa-key me-1"></i> Ubah Password</a></li>
                            <li><a class="dropdown-item" href="?logout=1"><i class="fas fa-sign-out-alt me-1"></i> Logout</a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mb-5">
        <!-- Notification Messages -->
        <?php if (isset($_SESSION['message'])): ?>
            <div class="alert alert-<?= $_SESSION['message_type'] ?> alert-dismissible fade show" role="alert">
                <?= $_SESSION['message'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
            <?php unset($_SESSION['message']); unset($_SESSION['message_type']); ?>
        <?php endif; ?>
        
        <!-- Tabs for Navigation -->
        <ul class="nav nav-tabs mb-4" id="myTab">
            <li class="nav-item">
                <a class="nav-link <?= $activeTab === 'questions' ? 'active' : '' ?>" href="?tab=questions">
                    <i class="fas fa-list me-1"></i> Daftar Soal
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?= $activeTab === 'events' ? 'active' : '' ?>" href="?tab=events">
                    <i class="fas fa-calendar-alt me-1"></i> Daftar Event
                </a>
            </li>
            <?php if (isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']): ?>
                <li class="nav-item">
                    <a class="nav-link <?= $activeTab === 'admin' ? 'active' : '' ?>" href="?tab=admin">
                        <i class="fas fa-users-cog me-1"></i> Kelola Admin
                    </a>
                </li>
            <?php endif; ?>
        </ul>
        
        <!-- Tab Content -->
        <div class="tab-content" id="myTabContent">
            <!-- Questions Tab -->
            <div class="tab-pane fade <?= $activeTab === 'questions' ? 'show active' : '' ?>" id="questions-tab-pane" role="tabpanel" aria-labelledby="questions-tab" tabindex="0">
                <!-- Filter and Search -->
                <div class="card mb-4">
                    <div class="card-body">
                        <h5 class="card-title mb-3">
                            <i class="fas fa-filter me-2"></i> Filter dan Pencarian
                        </h5>
                        <form action="" method="GET" class="row g-3 filter-form">
                            <input type="hidden" name="tab" value="questions">
                            <div class="col-md-3">
                                <label for="level" class="form-label">Jenjang</label>
                                <select name="level" id="level" class="form-select">
                                    <option value="">Semua Jenjang</option>
                                    <?php foreach ($educationLevels as $key => $value): ?>
                                        <option value="<?= $key ?>" <?= isset($_GET['level']) && $_GET['level'] == $key ? 'selected' : '' ?>>
                                            <?= $value ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="type" class="form-label">Jenis Soal</label>
                                <select name="type" id="type" class="form-select">
                                    <option value="">Semua Jenis</option>
                                    <?php foreach ($questionTypes as $key => $value): ?>
                                        <option value="<?= $key ?>" <?= isset($_GET['type']) && $_GET['type'] == $key ? 'selected' : '' ?>>
                                            <?= $value ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="search" class="form-label">Kode Soal</label>
                                <input type="text" name="search" id="search" class="form-control" placeholder="Cari kode soal..." value="<?= isset($_GET['search']) ? $_GET['search'] : '' ?>">
                            </div>
                            <div class="col-md-3 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary me-2">
                                    <i class="fas fa-search me-1"></i> Cari
                                </button>
                                <a href="?tab=questions" class="btn btn-secondary">
                                    <i class="fas fa-redo me-1"></i> Reset
                                </a>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Questions Table -->
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-3">
                            <i class="fas fa-list me-2"></i> Daftar Soal
                            <span class="badge bg-primary float-end"><?= count($filteredQuestions) ?> Soal</span>
                        </h5>
                        
                        <?php if (empty($filteredQuestions)): ?>
                            <div class="alert alert-info" role="alert">
                                <i class="fas fa-info-circle me-2"></i> Belum ada soal yang tersedia atau tidak ada soal yang sesuai dengan filter.
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th width="5%">No</th>
                                            <th width="25%">Kode Soal</th>
                                            <th width="12%">Jenjang</th>
                                            <th width="13%">Jenis Soal</th>
                                            <th width="20%">Informasi</th>
                                            <th width="15%">Link</th>
                                            <th width="10%">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $no = 1; foreach ($filteredQuestions as $question): ?>
                                            <tr>
                                                <td><?= $no++ ?></td>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <span class="question-code"><?= $question['code'] ?></span>
                                                        <button type="button" class="btn btn-sm btn-outline-secondary ms-2 copy-btn" data-code="<?= $question['code'] ?>" title="Salin kode">
                                                            <i class="fas fa-copy"></i>
                                                        </button>
                                                    </div>
                                                </td>
                                                <td>
                                                    <?= $educationLevels[$question['education_level']] ?>
                                                </td>
                                                <td>
                                                    <?= $questionTypes[$question['question_type']] ?>
                                                </td>
                                                <td>
                                                    <small>
                                                        <div><strong>Mata Pelajaran:</strong> <?= isset($question['subject']) ? $question['subject'] : '<span class="text-muted"><i>Tidak ada</i></span>' ?></div>
                                                        <div><strong>Jumlah Soal:</strong> <?= $question['total_questions'] ?></div>
                                                        <?php if ($question['question_type'] == 'pilgan' || $question['question_type'] == 'campuran'): ?>
                                                            <div><strong>Pilihan Ganda:</strong> <?= $question['multiple_choice_options'] ?> opsi</div>
                                                        <?php endif; ?>
                                                        <div><strong>Dibuat (Bulan/Tahun):</strong> <?= date('M Y', strtotime($question['month'] . '/1/' . $question['year'])) ?></div>
                                                        <div><strong>Tanggal Dibuat:</strong> <?= date('d-m-Y H:i', strtotime($question['created_at'])) ?></div>
                                                    </small>
                                                </td>
                                                <td>
    <?php if (isset($question['links_pending']) && $question['links_pending']): ?>
        <div class="alert alert-warning py-1 px-2 mb-1">
            <small><i class="fas fa-exclamation-triangle me-1"></i> Links pending</small>
        </div>
    <?php endif; ?>
    <?php if ($question['gform_link'] != '#'): ?>
        <a href="<?= $question['gform_link'] ?>" target="_blank" class="btn btn-sm btn-primary mb-1 w-100">
            <i class="fas fa-file-alt me-1"></i> Soal
        </a>
    <?php else: ?>
        <button class="btn btn-sm btn-outline-primary mb-1 w-100" disabled>
            <i class="fas fa-file-alt me-1"></i> Link Soal Kosong
        </button>
    <?php endif; ?>
    
    <?php if ($question['solution_link'] != '#'): ?>
        <a href="<?= $question['solution_link'] ?>" target="_blank" class="btn btn-sm btn-success w-100">
            <i class="fas fa-lightbulb me-1"></i> Pembahasan
        </a>
    <?php else: ?>
        <button class="btn btn-sm btn-outline-success w-100" disabled>
            <i class="fas fa-lightbulb me-1"></i> Link Pembahasan Kosong
        </button>
    <?php endif; ?>
</td>
                                                <td class="action-buttons">
                                                    <button type="button" class="btn btn-warning edit-btn" 
                                                            data-id="<?= $question['id'] ?>"
                                                            data-gform="<?= $question['gform_link'] ?>"
                                                            data-solution="<?= $question['solution_link'] ?>"
                                                            data-subject="<?= isset($question['subject']) ? $question['subject'] : '' ?>"
                                                            data-level="<?= $question['education_level'] ?>"
                                                            data-type="<?= $question['question_type'] ?>"
                                                            data-total="<?= $question['total_questions'] ?>"
                                                            data-options="<?= $question['multiple_choice_options'] ?>"
                                                            data-month="<?= $question['month'] ?>"
                                                            data-year="<?= $question['year'] ?>"
                                                            data-bs-toggle="modal" data-bs-target="#editQuestionModal">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-danger delete-btn"
                                                            data-id="<?= $question['id'] ?>"
                                                            data-code="<?= $question['code'] ?>"
                                                            data-bs-toggle="modal" data-bs-target="#deleteQuestionModal">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Events Tab -->
            <div class="tab-pane fade <?= $activeTab === 'events' ? 'show active' : '' ?>" id="events-tab-pane" role="tabpanel" aria-labelledby="events-tab" tabindex="0">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-3">
                            <i class="fas fa-calendar-alt me-2"></i> Daftar Event
                            <span class="badge bg-primary float-end"><?= count($events) ?> Event</span>
                        </h5>
                        
                        <?php if (empty($events)): ?>
                            <div class="alert alert-info" role="alert">
                                <i class="fas fa-info-circle me-2"></i> Belum ada event yang tersedia.
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th width="5%">No</th>
                                            <th width="20%">Nama Event</th>
                                            <th width="15%">Penyelenggara</th>
                                            <th width="15%">Mata Pelajaran</th>
                                            <th width="20%">Jenjang & Soal</th>
                                            <th width="15%">Waktu Pelaksanaan</th>
                                            <th width="10%">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php $no = 1; foreach ($events as $event): ?>
                                            <tr>
                                                <td><?= $no++ ?></td>
                                                <td><?= $event['name'] ?></td>
                                                <td><?= $event['organizer'] ?></td>
                                                <td>
                                                    <?php 
                                                    // Tampilkan mapel dari soal yang digunakan
                                                    $subjects = [];
                                                    if (!empty($event['question_ids'])) {
                                                        foreach ($event['question_ids'] as $qid) {
                                                            foreach ($database['questions'] as $q) {
                                                                if ($q['id'] == $qid && isset($q['subject']) && !in_array($q['subject'], $subjects)) {
                                                                    $subjects[] = $q['subject'];
                                                                }
                                                            }
                                                        }
                                                    }
                                                    echo !empty($subjects) ? implode(', ', $subjects) : '<span class="text-muted"><i>Tidak ada mapel</i></span>';
                                                    ?>
                                                </td>
                                                <td>
                                                    <small>
                                                        <div><strong>Jenjang:</strong>
                                                            <?php 
                                                            $levelNames = [];
                                                            foreach ($event['education_levels'] as $level) {
                                                                $levelNames[] = $educationLevels[$level];
                                                            }
                                                            echo implode(', ', $levelNames);
                                                            ?>
                                                        </div>
                                                        <div class="mt-1"><strong>Soal yang Digunakan:</strong></div>
                                                        <ul class="mb-0 ps-3">
                                                            <?php 
                                                            if (!empty($event['question_ids'])) {
                                                                foreach ($event['question_ids'] as $qid) {
                                                                    $found = false;
                                                                    foreach ($database['questions'] as $q) {
                                                                        if ($q['id'] == $qid) {
                                                                            echo "<li>{$q['code']}</li>";
                                                                            $found = true;
                                                                            break;
                                                                        }
                                                                    }
                                                                    if (!$found) {
                                                                        echo "<li class='text-muted'><i>Soal tidak tersedia</i></li>";
                                                                    }
                                                                }
                                                            } else {
                                                                echo "<li class='text-muted'><i>Tidak ada soal</i></li>";
                                                            }
                                                            ?>
                                                        </ul>
                                                    </small>
                                                </td>
                                                <td>
                                                    <div><i class="far fa-calendar me-1"></i> <?= date('d-m-Y', strtotime($event['event_date'])) ?></div>
                                                    <div><i class="far fa-clock me-1"></i> <?= $event['event_time'] ?></div>
                                                </td>
                                                <td>
                                                    <button type="button" class="btn btn-danger delete-event-btn"
                                                            data-id="<?= $event['id'] ?>"
                                                            data-name="<?= $event['name'] ?>"
                                                            data-bs-toggle="modal" data-bs-target="#deleteEventModal">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <!-- Admin Tab -->
            <?php if (isset($current_admin['is_super_admin']) && $current_admin['is_super_admin']): ?>
                <div class="tab-pane fade <?= $activeTab === 'admin' ? 'show active' : '' ?>" id="admin-tab-pane" role="tabpanel" aria-labelledby="admin-tab" tabindex="0">
                    <!-- Add Admin Card -->
                    <div class="card mb-4">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">
                                <i class="fas fa-user-plus me-2"></i> Tambah Admin Baru
                            </h5>
                        </div>
                        <div class="card-body">
                            <form action="" method="POST">
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <label for="admin_username" class="form-label">Username</label>
                                        <input type="text" class="form-control" id="admin_username" name="admin_username" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label for="admin_password" class="form-label">Password</label>
                                        <input type="password" class="form-control" id="admin_password" name="admin_password" required>
                                    </div>
                                    <div class="col-md-4 d-flex align-items-end">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="is_super_admin" name="is_super_admin" value="1">
                                            <label class="form-check-label" for="is_super_admin">
                                                Super Admin
                                            </label>
                                        </div>
                                    </div>
                                </div>
                                <div class="mt-3">
                                    <button type="submit" name="add_admin" class="btn btn-primary">
                                        <i class="fas fa-plus-circle me-1"></i> Tambah Admin
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Admin List Card -->
                    <div class="card">
                        <div class="card-header bg-dark text-white">
                            <h5 class="mb-0">
                                <i class="fas fa-users-cog me-2"></i> Daftar Admin
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th width="5%">No</th>
                                            <th width="20%">Username</th>
                                            <th width="15%">Status</th>
                                            <th width="20%">Dibuat Pada</th>
                                            <th width="20%">Dibuat Oleh</th>
                                            <th width="10%">Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php 
                                        $no = 1; 
                                        foreach ($database['admins'] as $admin): 
                                            $creator = 'Sistem';
                                            if (isset($admin['created_by'])) {
                                                foreach ($database['admins'] as $a) {
                                                    if ($a['id'] === $admin['created_by']) {
                                                        $creator = $a['username'];
                                                        break;
                                                    }
                                                }
                                            }
                                        ?>
                                            <tr>
                                                <td><?= $no++ ?></td>
                                                <td><?= $admin['username'] ?></td>
                                                <td>
                                                    <?php if (isset($admin['is_super_admin']) && $admin['is_super_admin']): ?>
                                                        <span class="badge bg-danger">Super Admin</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-primary">Admin</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td><?= date('d-m-Y H:i', strtotime($admin['created_at'])) ?></td>
                                                <td><?= $creator ?></td>
                                                <td>
                                                    <?php if ($admin['id'] !== $_SESSION['admin_id']): ?>
                                                        <button type="button" class="btn btn-danger btn-sm delete-admin-btn"
                                                                data-id="<?= $admin['id'] ?>"
                                                                data-username="<?= $admin['username'] ?>"
                                                                data-bs-toggle="modal" data-bs-target="#deleteAdminModal">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    <?php else: ?>
                                                        <span class="badge bg-info">Akun Aktif</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

<!-- Add Question Modal -->
<div class="modal fade" id="addQuestionModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title">
                    <i class="fas fa-plus-circle me-2"></i> Tambah Soal Baru
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="" method="POST">
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i> Link Google Form dan Pembahasan dapat dikosongkan dan diisi nanti melalui fitur Edit.
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="gform_link" class="form-label">Link Google Form <span class="text-muted">(Opsional)</span></label>
                            <input type="url" class="form-control" name="gform_link" id="gform_link" placeholder="https://forms.google.com/...">
                        </div>
                        <div class="col-md-6">
                            <label for="solution_link" class="form-label">Link Pembahasan <span class="text-muted">(Opsional)</span></label>
                            <input type="url" class="form-control" name="solution_link" id="solution_link" placeholder="https://docs.google.com/...">
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <label for="subject" class="form-label">Mata Pelajaran</label>
                            <input type="text" class="form-control" name="subject" id="subject" required placeholder="Matematika, IPA, dll">
                        </div>
                        <div class="col-md-4">
                            <label for="education_level" class="form-label">Jenjang Pendidikan</label>
                            <select class="form-select" name="education_level" id="education_level" required>
                                <option value="" selected disabled>Pilih Jenjang</option>
                                <?php foreach ($educationLevels as $key => $value): ?>
                                    <option value="<?= $key ?>"><?= $value ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="question_type" class="form-label">Jenis Soal</label>
                            <select class="form-select" name="question_type" id="question_type" required>
                                <option value="" selected disabled>Pilih Jenis Soal</option>
                                <?php foreach ($questionTypes as $key => $value): ?>
                                    <option value="<?= $key ?>"><?= $value ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="total_questions" class="form-label">Jumlah Soal</label>
                            <input type="number" class="form-control" name="total_questions" id="total_questions" required min="1" placeholder="40">
                        </div>
                        <div class="col-md-6">
                            <label for="multiple_choice_options" class="form-label">Jumlah Pilihan Ganda</label>
                            <input type="number" class="form-control" name="multiple_choice_options" id="multiple_choice_options" min="2" placeholder="5">
                            <small class="text-muted">*Khusus untuk tipe soal pilihan ganda</small>
                        </div>
                    </div>
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="month" class="form-label">Bulan Pembuatan</label>
                            <select class="form-select" name="month" id="month" required>
                                <?php for ($i = 1; $i <= 12; $i++): ?>
                                    <option value="<?= $i ?>" <?= $i == $currentMonth ? 'selected' : '' ?>><?= date('F', mktime(0, 0, 0, $i, 1)) ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label for="year" class="form-label">Tahun Pembuatan</label>
                            <select class="form-select" name="year" id="year" required>
                                <?php for ($i = 2030; $i >= 2020; $i--): ?>
                                    <option value="<?= $i ?>" <?= $i == $currentYear ? 'selected' : '' ?>><?= $i ?></option>
                                <?php endfor; ?>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                    <button type="submit" name="add_question" class="btn btn-primary">Simpan</button>
                </div>
            </form>
        </div>
    </div>
</div>

    <!-- Edit Question Modal -->
    <div class="modal fade" id="editQuestionModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-warning">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i> Edit Soal
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <input type="hidden" name="edit_id" id="edit_id">
                    <div class="modal-body">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="edit_gform_link" class="form-label">Link Google Form</label>
                                <input type="url" class="form-control" name="gform_link" id="edit_gform_link" required>
                            </div>
                            <div class="col-md-6">
                                <label for="edit_solution_link" class="form-label">Link Pembahasan</label>
                                <input type="url" class="form-control" name="solution_link" id="edit_solution_link" required>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label for="edit_subject" class="form-label">Mata Pelajaran</label>
                                <input type="text" class="form-control" name="subject" id="edit_subject" required placeholder="Matematika, IPA, dll">
                            </div>
                            <div class="col-md-4">
                                <label for="edit_education_level" class="form-label">Jenjang Pendidikan</label>
                                <select class="form-select" name="education_level" id="edit_education_level" required>
                                    <?php foreach ($educationLevels as $key => $value): ?>
                                        <option value="<?= $key ?>"><?= $value ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label for="edit_question_type" class="form-label">Jenis Soal</label>
                                <select class="form-select" name="question_type" id="edit_question_type" required>
                                    <?php foreach ($questionTypes as $key => $value): ?>
                                        <option value="<?= $key ?>"><?= $value ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="edit_total_questions" class="form-label">Jumlah Soal</label>
                                <input type="number" class="form-control" name="total_questions" id="edit_total_questions" required min="1">
                            </div>
                            <div class="col-md-6">
                                <label for="edit_multiple_choice_options" class="form-label">Jumlah Pilihan Ganda</label>
                                <input type="number" class="form-control" name="multiple_choice_options" id="edit_multiple_choice_options" min="2">
                                <small class="text-muted">*Khusus untuk tipe soal pilihan ganda</small>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="edit_month" class="form-label">Bulan Pembuatan</label>
                                <select class="form-select" name="month" id="edit_month" required>
                                    <?php for ($i = 1; $i <= 12; $i++): ?>
                                        <option value="<?= $i ?>"><?= date('F', mktime(0, 0, 0, $i, 1)) ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="edit_year" class="form-label">Tahun Pembuatan</label>
                                <select class="form-select" name="year" id="edit_year" required>
                                    <?php for ($i = 2030; $i >= 2020; $i--): ?>
                                        <option value="<?= $i ?>"><?= $i ?></option>
                                    <?php endfor; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="edit_question" class="btn btn-warning">Perbarui</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Question Modal -->
    <div class="modal fade" id="deleteQuestionModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-trash me-2"></i> Hapus Soal
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <input type="hidden" name="delete_id" id="delete_id">
                    <div class="modal-body">
                        <p>Apakah Anda yakin ingin menghapus soal dengan kode: <strong id="delete_code"></strong>?</p>
                        <p class="text-danger">Peringatan: Tindakan ini tidak dapat dibatalkan!</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="delete_question" class="btn btn-danger">Hapus</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Add Event Modal -->
    <div class="modal fade" id="addEventModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-calendar-plus me-2"></i> Tambah Event Baru
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <div class="modal-body">
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="event_name" class="form-label">Nama Event</label>
                                <input type="text" class="form-control" name="event_name" id="event_name" required>
                            </div>
                            <div class="col-md-6">
                                <label for="event_organizer" class="form-label">Penyelenggara</label>
                                <input type="text" class="form-control" name="event_organizer" id="event_organizer" required>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-12">
                                <label class="form-label">Jenjang Pendidikan</label>
                                <div class="card p-2">
                                    <?php foreach ($educationLevels as $key => $value): ?>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" name="event_education_levels[]" value="<?= $key ?>" id="level_<?= $key ?>">
                                            <label class="form-check-label" for="level_<?= $key ?>"><?= $value ?></label>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="event_date" class="form-label">Tanggal Pelaksanaan</label>
                                <input type="date" class="form-control" name="event_date" id="event_date" required>
                            </div>
                            <div class="col-md-6">
                                <label for="event_time" class="form-label">Waktu Pelaksanaan</label>
                                <input type="time" class="form-control" name="event_time" id="event_time" required>
                            </div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-12">
                                <label class="form-label">Pilih Soal untuk Event</label>
                                <div class="card p-2" style="max-height: 200px; overflow-y: auto;">
                                    <?php if (empty($database['questions'])): ?>
                                        <p class="text-muted mb-0">Belum ada soal yang tersedia.</p>
                                    <?php else: ?>
                                        <?php foreach ($database['questions'] as $q): ?>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" name="event_questions[]" value="<?= $q['id'] ?>" id="question_<?= $q['id'] ?>">
                                                <label class="form-check-label" for="question_<?= $q['id'] ?>">
                                                    <span class="question-code"><?= $q['code'] ?></span> - 
                                                    <?= isset($q['subject']) ? $q['subject'] : '<i>Tidak ada mapel</i>' ?> - 
                                                    <?= $educationLevels[$q['education_level']] ?> - 
                                                    <?= $questionTypes[$q['question_type']] ?>
                                                </label>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="add_event" class="btn btn-primary">Simpan</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Event Modal -->
    <div class="modal fade" id="deleteEventModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-trash me-2"></i> Hapus Event
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <input type="hidden" name="delete_event_id" id="delete_event_id">
                    <div class="modal-body">
                        <p>Apakah Anda yakin ingin menghapus event "<strong id="delete_event_name"></strong>"?</p>
                        <p class="text-danger">Peringatan: Tindakan ini tidak dapat dibatalkan!</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="delete_event" class="btn btn-danger">Hapus</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Change Password Modal -->
    <div class="modal fade" id="changePasswordModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-key me-2"></i> Ubah Password
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="current_password" class="form-label">Password Saat Ini</label>
                            <input type="password" class="form-control" id="current_password" name="current_password" required>
                        </div>
                        <div class="mb-3">
                            <label for="new_password" class="form-label">Password Baru</label>
                            <input type="password" class="form-control" id="new_password" name="new_password" required>
                        </div>
                        <div class="mb-3">
                            <label for="confirm_password" class="form-label">Konfirmasi Password Baru</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="change_password" class="btn btn-primary">Ubah Password</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Admin Modal -->
    <div class="modal fade" id="deleteAdminModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">
                        <i class="fas fa-trash me-2"></i> Hapus Admin
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="" method="POST">
                    <input type="hidden" name="delete_admin_id" id="delete_admin_id">
                    <div class="modal-body">
                        <p>Apakah Anda yakin ingin menghapus admin "<strong id="delete_admin_username"></strong>"?</p>
                        <p class="text-danger">Peringatan: Tindakan ini tidak dapat dibatalkan!</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" name="delete_admin" class="btn btn-danger">Hapus</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white text-center py-3 mt-5">
        <div class="container">
            <p class="mb-0">© <?= date('Y') ?> Pengelola Soal dan Pembahasan | Dibuat dengan <i class="fas fa-heart text-danger"></i></p>
        </div>
    </footer>
    <?php endif; ?>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.2.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // For add form
            const questionTypeSelect = document.getElementById('question_type');
            const multipleChoiceOptions = document.getElementById('multiple_choice_options');
            
            if (questionTypeSelect) {
                questionTypeSelect.addEventListener('change', function() {
                    if (this.value === 'pilgan' || this.value === 'campuran') {
                        multipleChoiceOptions.parentElement.style.display = 'block';
                        multipleChoiceOptions.required = true;
                    } else {
                        multipleChoiceOptions.parentElement.style.display = 'none';
                        multipleChoiceOptions.required = false;
                        multipleChoiceOptions.value = '';
                    }
                });
            }
            
            // For edit form
            const editQuestionTypeSelect = document.getElementById('edit_question_type');
            const editMultipleChoiceOptions = document.getElementById('edit_multiple_choice_options');
            
            if (editQuestionTypeSelect) {
                editQuestionTypeSelect.addEventListener('change', function() {
                    if (this.value === 'pilgan' || this.value === 'campuran') {
                        editMultipleChoiceOptions.parentElement.style.display = 'block';
                        editMultipleChoiceOptions.required = true;
                    } else {
                        editMultipleChoiceOptions.parentElement.style.display = 'none';
                        editMultipleChoiceOptions.required = false;
                        editMultipleChoiceOptions.value = '';
                    }
                });
            }
            
            // Set up edit buttons
            const editButtons = document.querySelectorAll('.edit-btn');
            editButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    document.getElementById('edit_id').value = this.dataset.id;
                    document.getElementById('edit_gform_link').value = this.dataset.gform;
                    document.getElementById('edit_solution_link').value = this.dataset.solution;
                    document.getElementById('edit_subject').value = this.dataset.subject || '';
                    document.getElementById('edit_education_level').value = this.dataset.level;
                    document.getElementById('edit_question_type').value = this.dataset.type;
                    document.getElementById('edit_total_questions').value = this.dataset.total;
                    document.getElementById('edit_multiple_choice_options').value = this.dataset.options;
                    document.getElementById('edit_month').value = this.dataset.month;
                    document.getElementById('edit_year').value = this.dataset.year;
                    
                    // Show/hide multiple choice options
                    if (this.dataset.type === 'pilgan' || this.dataset.type === 'campuran') {
                        editMultipleChoiceOptions.parentElement.style.display = 'block';
                        editMultipleChoiceOptions.required = true;
                    } else {
                        editMultipleChoiceOptions.parentElement.style.display = 'none';
                        editMultipleChoiceOptions.required = false;
                    }
                });
            });
            
            // Set up delete buttons
            const deleteButtons = document.querySelectorAll('.delete-btn');
            deleteButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    document.getElementById('delete_id').value = this.dataset.id;
                    document.getElementById('delete_code').textContent = this.dataset.code;
                });
            });
            
            // Set up delete event buttons
            const deleteEventButtons = document.querySelectorAll('.delete-event-btn');
            deleteEventButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    document.getElementById('delete_event_id').value = this.dataset.id;
                    document.getElementById('delete_event_name').textContent = this.dataset.name;
                });
            });
            
            // Set up delete admin buttons
            const deleteAdminButtons = document.querySelectorAll('.delete-admin-btn');
            deleteAdminButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    document.getElementById('delete_admin_id').value = this.dataset.id;
                    document.getElementById('delete_admin_username').textContent = this.dataset.username;
                });
            });
            
            // Set default date for event
            const eventDateInput = document.getElementById('event_date');
            if (eventDateInput) {
                const today = new Date();
                const year = today.getFullYear();
                const month = String(today.getMonth() + 1).padStart(2, '0');
                const day = String(today.getDate()).padStart(2, '0');
                eventDateInput.value = `${year}-${month}-${day}`;
            }
            
            // Set default time for event
            const eventTimeInput = document.getElementById('event_time');
            if (eventTimeInput) {
                eventTimeInput.value = '08:00';
            }
            
            // Fix the tab navigation behavior
            // Since we converted the tabs to regular links, we need to remove
            // any remaining tab navigation code that might conflict
            const navTabs = document.querySelectorAll('#myTab .nav-link');
            navTabs.forEach(function(tab) {
                // Remove any data-bs-toggle attributes
                tab.removeAttribute('data-bs-toggle');
                // Remove role attributes
                tab.removeAttribute('role');
            });
            
            // Set up copy buttons for question codes
            const copyButtons = document.querySelectorAll('.copy-btn');
            copyButtons.forEach(function(button) {
                button.addEventListener('click', function() {
                    const codeText = this.dataset.code;
                    
                    // Create a temporary textarea to copy from
                    const textarea = document.createElement('textarea');
                    textarea.value = codeText;
                    textarea.style.position = 'fixed';  // Avoid scrolling to bottom
                    document.body.appendChild(textarea);
                    textarea.select();
                    document.execCommand('copy');
                    document.body.removeChild(textarea);
                    
                    // Show feedback (change icon temporarily)
                    const originalHTML = this.innerHTML;
                    const originalTitle = this.title;
                    this.innerHTML = '<i class="fas fa-check text-success"></i>';
                    this.title = "Tersalin!";
                    
                    // Show a toast notification
                    const toast = document.createElement('div');
                    toast.className = 'position-fixed bottom-0 end-0 p-3';
                    toast.style.zIndex = '11';
                    toast.innerHTML = `
                        <div class="toast show" role="alert" aria-live="assertive" aria-atomic="true">
                            <div class="toast-header bg-success text-white">
                                <i class="fas fa-check me-2"></i>
                                <strong class="me-auto">Sukses</strong>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast" aria-label="Close"></button>
                            </div>
                            <div class="toast-body">
                                Kode soal <strong>${codeText}</strong> berhasil disalin!
                            </div>
                        </div>
                    `;
                    document.body.appendChild(toast);
                    
                    // Reset the button after 1.5 seconds
                    setTimeout(() => {
                        this.innerHTML = originalHTML;
                        this.title = originalTitle;
                    }, 1500);
                    
                    // Remove the toast after 3 seconds
                    setTimeout(() => {
                        document.body.removeChild(toast);
                    }, 3000);
                });
            });
        });
    </script>
</body>
</html>