<?php
session_start();

// Definisi file JSON untuk penyimpanan data
define('USERS_FILE', 'data/users.json');
define('HOLIDAYS_FILE', 'data/holidays.json');
define('INSTAGRAM_FILE', 'data/instagram.json');

// Membuat direktori data jika belum ada
if (!file_exists('data')) {
    mkdir('data', 0777, true);
}

// Inisialisasi file JSON jika belum ada
if (!file_exists(USERS_FILE)) {
    $defaultUsers = [
        [
            'username' => 'rizaldi',
            'password' => 'admin123',
            'name' => 'Administrator'
        ]
    ];
    file_put_contents(USERS_FILE, json_encode($defaultUsers, JSON_PRETTY_PRINT));
}

if (!file_exists(HOLIDAYS_FILE)) {
    file_put_contents(HOLIDAYS_FILE, json_encode([], JSON_PRETTY_PRINT));
}

// Memastikan semua hari besar memiliki property skipped
function ensureSkippedProperty() {
    $holidays = getData(HOLIDAYS_FILE);
    $updated = false;
    
    foreach ($holidays as &$holiday) {
        if (!isset($holiday['skipped'])) {
            $holiday['skipped'] = false;
            $updated = true;
        }
    }
    
    if ($updated) {
        saveData(HOLIDAYS_FILE, $holidays);
    }
}

// Jalankan fungsi untuk update struktur data yang ada
ensureSkippedProperty();

if (!file_exists(INSTAGRAM_FILE)) {
    file_put_contents(INSTAGRAM_FILE, json_encode([], JSON_PRETTY_PRINT));
}

// Fungsi untuk mendapatkan data dari file JSON
function getData($file) {
    return json_decode(file_get_contents($file), true);
}

// Fungsi untuk menyimpan data ke file JSON
function saveData($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

// Fungsi untuk autentikasi pengguna
function authenticate($username, $password) {
    $users = getData(USERS_FILE);
    foreach ($users as $user) {
        if ($user['username'] === $username && $user['password'] === $password) {
            return $user;
        }
    }
    return false;
}

// Fungsi untuk mendapatkan semua hari besar
function getHolidays() {
    return getData(HOLIDAYS_FILE);
}

// Fungsi untuk mendapatkan semua akun Instagram
function getInstagramAccounts() {
    return getData(INSTAGRAM_FILE);
}

// Fungsi untuk menambahkan hari besar baru
function addHoliday($name, $date, $description, $category = '') {
    $holidays = getHolidays();
    $newHoliday = [
        'id' => uniqid(),
        'name' => $name,
        'date' => $date,
        'description' => $description,
        'category' => $category,
        'poster_created' => false,
        'instagram_posts' => [],
        'skipped' => false
    ];
    $holidays[] = $newHoliday;
    saveData(HOLIDAYS_FILE, $holidays);
    return $newHoliday;
}

// Fungsi untuk menambahkan akun Instagram baru
function addInstagramAccount($name, $username) {
    $accounts = getInstagramAccounts();
    $newAccount = [
        'id' => uniqid(),
        'name' => $name,
        'username' => $username
    ];
    $accounts[] = $newAccount;
    saveData(INSTAGRAM_FILE, $accounts);
    return $newAccount;
}

// Fungsi untuk memperbarui status poster dan posting Instagram
function updateHolidayStatus($holidayId, $posterCreated, $instagramPosts, $skip = false) {
    $holidays = getHolidays();
    foreach ($holidays as &$holiday) {
        if ($holiday['id'] === $holidayId) {
            $holiday['poster_created'] = $posterCreated;
            $holiday['instagram_posts'] = $instagramPosts;
            if ($skip) {
                $holiday['skipped'] = true;
            }
            break;
        }
    }
    saveData(HOLIDAYS_FILE, $holidays);
}

// Proses login
if (isset($_POST['login'])) {
    $username = $_POST['username'];
    $password = $_POST['password'];
    $user = authenticate($username, $password);
    if ($user) {
        $_SESSION['user'] = $user;
        header('Location: index.php');
        exit;
    } else {
        $loginError = 'Username atau password salah!';
    }
}

// Proses logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: index.php');
    exit;
}

// Proses tambah hari besar
if (isset($_POST['add_holiday'])) {
    $name = $_POST['name'];
    $date = $_POST['date'];
    $description = $_POST['description'];
    $category = isset($_POST['category']) ? $_POST['category'] : '';
    addHoliday($name, $date, $description, $category);
    header('Location: index.php?page=holidays');
    exit;
}

// Proses tambah akun Instagram
if (isset($_POST['add_instagram'])) {
    $name = $_POST['name'];
    $username = $_POST['username'];
    addInstagramAccount($name, $username);
    header('Location: index.php?page=instagram');
    exit;
}

// Proses update status hari besar
if (isset($_POST['update_holiday_status'])) {
    $holidayId = $_POST['holiday_id'];
    $posterCreated = isset($_POST['poster_created']) ? true : false;
    $instagramPosts = isset($_POST['instagram_posts']) ? $_POST['instagram_posts'] : [];
    updateHolidayStatus($holidayId, $posterCreated, $instagramPosts);
    header('Location: index.php?page=holidays');
    exit;
}

// Proses skip hari besar
if (isset($_POST['skip_holiday'])) {
    $holidayId = $_POST['holiday_id'];
    updateHolidayStatus($holidayId, false, [], true);
    header('Location: index.php?page=holidays');
    exit;
}

// Proses unskip hari besar
if (isset($_POST['unskip_holiday'])) {
    $holidayId = $_POST['holiday_id'];
    $holidays = getHolidays();
    
    foreach ($holidays as &$holiday) {
        if ($holiday['id'] === $holidayId) {
            $holiday['skipped'] = false;
            break;
        }
    }
    
    saveData(HOLIDAYS_FILE, $holidays);
    header('Location: index.php?page=holidays' . (isset($_GET['view']) && $_GET['view'] === 'skipped' ? '&view=skipped' : ''));
    exit;
}

// Mendapatkan halaman yang aktif
$page = isset($_GET['page']) ? $_GET['page'] : 'dashboard';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aplikasi Manajemen Hari Besar dan Posting Instagram</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --secondary-color: #3f37c9;
            --accent-color: #4895ef;
            --success-color: #4cc9f0;
            --light-color: #f8f9fa;
            --dark-color: #212529;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f0f2f5;
            min-height: 100vh;
        }
        
        .sidebar {
            background: var(--primary-color);
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            min-height: 100vh;
            color: white;
            transition: all 0.3s;
        }
        
        .sidebar-header {
            padding: 20px;
            background: rgba(0, 0, 0, 0.1);
        }
        
        .sidebar-menu {
            padding: 0;
            list-style: none;
        }
        
        .sidebar-menu li {
            padding: 10px 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .sidebar-menu li a {
            color: white;
            text-decoration: none;
            display: block;
        }
        
        .sidebar-menu li:hover {
            background: rgba(255, 255, 255, 0.1);
        }
        
        .sidebar-menu li.active {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .content {
            padding: 20px;
        }
        
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            border: none;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }
        
        .card-header {
            border-radius: 15px 15px 0 0 !important;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--accent-color) 100%);
            color: white;
            font-weight: bold;
        }
        
        .btn-primary {
            background: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .btn-primary:hover {
            background: var(--secondary-color);
            border-color: var(--secondary-color);
        }
        
        /* Perbaikan untuk tombol light */
        .btn-light {
            background: #ffffff;
            border-color: #d0d0d0;
            color: #505050;
        }
        
        .btn-light:hover {
            background: #f0f0f0;
            border-color: #c0c0c0;
            color: #303030;
        }
        
        .login-container {
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--accent-color) 100%);
        }
        
        .login-card {
            min-width: 350px;
            border-radius: 15px;
        }
        
        .table-hover tbody tr:hover {
            background-color: rgba(67, 97, 238, 0.1);
        }
        
        .badge-custom {
            background-color: var(--accent-color);
            color: white;
            padding: 5px 10px;
            border-radius: 10px;
        }
        
        .avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background-color: var(--primary-color);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        
        .stats-card {
            border-left: 5px solid var(--primary-color);
        }
        
        .navbar {
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            background-color: white;
        }
        
        /* Perbaikan untuk modal - gaya baru */
        .custom-modal {
            display: none;
            position: fixed;
            z-index: 1050;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            overflow: auto;
            background-color: rgba(0, 0, 0, 0.5);
            animation: fadeIn 0.3s;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .custom-modal-content {
            background-color: #fefefe;
            margin: 10% auto;
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
            width: 60%;
            animation: slideDown 0.3s;
            position: relative;
        }
        
        @keyframes slideDown {
            from { margin-top: 5%; opacity: 0; }
            to { margin-top: 10%; opacity: 1; }
        }
        
        .custom-modal-header {
            padding-bottom: 15px;
            border-bottom: 1px solid #dee2e6;
            margin-bottom: 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .custom-modal-title {
            margin: 0;
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--dark-color);
        }
        
        .custom-modal-close {
            background: transparent;
            border: none;
            font-size: 1.5rem;
            cursor: pointer;
            color: #aaa;
        }
        
        .custom-modal-close:hover {
            color: #333;
        }
        
        .custom-modal-body {
            padding: 15px 0;
        }
        
        .custom-modal-footer {
            padding-top: 15px;
            border-top: 1px solid #dee2e6;
            margin-top: 15px;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        
        .action-buttons {
            display: flex;
            gap: 6px;
        }
        
        .btn-skip {
            background-color: #dc3545;
            color: white;
            border: none;
        }
        
        .btn-skip:hover {
            background-color: #bb2d3b;
            color: white;
        }
        
        .btn-unskip {
            background-color: #198754;
            color: white;
            border: none;
        }
        
        .btn-unskip:hover {
            background-color: #157347;
            color: white;
        }
        
        @media (max-width: 768px) {
            .sidebar {
                min-height: auto;
            }
            
            .custom-modal-content {
                width: 90%;
            }
        }
    </style>
</head>
<body>
    <?php if (!isset($_SESSION['user'])): ?>
    <!-- Halaman Login -->
    <div class="login-container">
        <div class="card login-card">
            <div class="card-body p-4">
                <h2 class="text-center mb-4">Login Aplikasi</h2>
                <h4 class="text-center text-muted mb-4">Manajemen Hari Besar & Instagram</h4>
                
                <?php if (isset($loginError)): ?>
                <div class="alert alert-danger"><?php echo $loginError; ?></div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="fas fa-user"></i></span>
                            <input type="text" class="form-control" id="username" name="username" required>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <div class="input-group">
                            <span class="input-group-text"><i class="fas fa-lock"></i></span>
                            <input type="password" class="form-control" id="password" name="password" required>
                        </div>
                    </div>
                    <div class="d-grid gap-2">
                        <button type="submit" name="login" class="btn btn-primary">Masuk</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <?php else: ?>
    <!-- Halaman Dashboard -->
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar collapse">
                <div class="sidebar-header">
                    <h3>Hari Besar App</h3>
                </div>
                <div class="d-flex flex-column p-3">
                    <div class="d-flex align-items-center mb-3">
                        <div class="avatar me-2">
                            <?php echo substr($_SESSION['user']['name'], 0, 1); ?>
                        </div>
                        <div>
                            <div class="fw-bold"><?php echo $_SESSION['user']['name']; ?></div>
                            <small>Administrator</small>
                        </div>
                    </div>
                </div>
                <ul class="sidebar-menu">
                    <li class="<?php echo $page === 'dashboard' ? 'active' : ''; ?>">
                        <a href="index.php?page=dashboard"><i class="fas fa-tachometer-alt me-2"></i> Dashboard</a>
                    </li>
                    <li class="<?php echo $page === 'holidays' ? 'active' : ''; ?>">
                        <a href="index.php?page=holidays"><i class="fas fa-calendar-alt me-2"></i> Hari Besar</a>
                    </li>
                    <li class="<?php echo $page === 'instagram' ? 'active' : ''; ?>">
                        <a href="index.php?page=instagram"><i class="fab fa-instagram me-2"></i> Akun Instagram</a>
                    </li>
                    <li class="<?php echo $page === 'reports' ? 'active' : ''; ?>">
                        <a href="index.php?page=reports"><i class="fas fa-chart-bar me-2"></i> Laporan</a>
                    </li>
                    <li>
                        <a href="index.php?logout=1"><i class="fas fa-sign-out-alt me-2"></i> Logout</a>
                    </li>
                </ul>
            </div>
            
            <!-- Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <?php
                            switch ($page) {
                                case 'dashboard':
                                    echo 'Dashboard';
                                    break;
                                case 'holidays':
                                    echo 'Manajemen Hari Besar';
                                    break;
                                case 'instagram':
                                    echo 'Manajemen Akun Instagram';
                                    break;
                                case 'reports':
                                    echo 'Laporan';
                                    break;
                                default:
                                    echo 'Dashboard';
                            }
                        ?>
                    </h1>
                </div>
                
                <?php if ($page === 'dashboard'): ?>
                <!-- Dashboard Content -->
                <div class="row">
                                                        <div class="col-md-4">
                        <div class="card stats-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted">Total Hari Besar</h6>
                                        <?php
                                            $holidays = getHolidays();
                                            $activeHolidays = array_filter($holidays, function($holiday) {
                                                return !(isset($holiday['skipped']) && $holiday['skipped']);
                                            });
                                            $skippedCount = count($holidays) - count($activeHolidays);
                                        ?>
                                        <h2 class="mb-0"><?php echo count($activeHolidays); ?></h2>
                                        <?php if ($skippedCount > 0): ?>
                                        <small class="text-muted">(<?php echo $skippedCount; ?> di-skip)</small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="fs-1 text-primary">
                                        <i class="fas fa-calendar-alt"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stats-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted">Poster Dibuat</h6>
                                        <?php
                                            $holidays = getHolidays();
                                            $posterCreated = array_filter($holidays, function($holiday) {
                                                return $holiday['poster_created'] === true;
                                            });
                                        ?>
                                        <h2 class="mb-0"><?php echo count($posterCreated); ?></h2>
                                    </div>
                                    <div class="fs-1 text-success">
                                        <i class="fas fa-image"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stats-card">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div>
                                        <h6 class="card-title text-muted">Akun Instagram</h6>
                                        <h2 class="mb-0"><?php echo count(getInstagramAccounts()); ?></h2>
                                    </div>
                                    <div class="fs-1 text-danger">
                                        <i class="fab fa-instagram"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row mt-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                Hari Besar Mendatang
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Tanggal</th>
                                                <th>Status Poster</th>
                                                <th>Kategori</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $holidays = getHolidays();
                                            
                                            // Filter yang tidak di-skip
                                            $activeHolidays = array_filter($holidays, function($holiday) {
                                                return !(isset($holiday['skipped']) && $holiday['skipped']);
                                            });
                                            
                                            // Urutkan berdasarkan tanggal
                                            usort($activeHolidays, function($a, $b) {
                                                return strtotime($a['date']) - strtotime($b['date']);
                                            });
                                            
                                            // Ambil 5 hari besar terdekat
                                            $upcomingHolidays = array_slice($activeHolidays, 0, 5);
                                            
                                            foreach ($upcomingHolidays as $holiday):
                                            ?>
                                            <tr>
                                                <td><?php echo $holiday['name']; ?></td>
                                                <td><?php echo date('d M Y', strtotime($holiday['date'])); ?></td>
                                                <td>
                                                    <?php if ($holiday['poster_created']): ?>
                                                    <span class="badge bg-success">Sudah Dibuat</span>
                                                    <?php else: ?>
                                                    <span class="badge bg-warning text-dark">Belum Dibuat</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if (isset($holiday['category']) && $holiday['category']): ?>
                                                    <span class="badge bg-<?php echo $holiday['category'] === 'Nasional' ? 'danger' : 'primary'; ?>"><?php echo $holiday['category']; ?></span>
                                                    <?php else: ?>
                                                    <span class="badge bg-secondary">-</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            
                                            <?php if (empty($upcomingHolidays)): ?>
                                            <tr>
                                                <td colspan="3" class="text-center">Belum ada data hari besar.</td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                Akun Instagram
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Username</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $instagramAccounts = getInstagramAccounts();
                                            foreach ($instagramAccounts as $account):
                                            ?>
                                            <tr>
                                                <td><?php echo $account['name']; ?></td>
                                                <td>@<?php echo $account['username']; ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                            
                                            <?php if (empty($instagramAccounts)): ?>
                                            <tr>
                                                <td colspan="2" class="text-center">Belum ada data akun Instagram.</td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($page === 'holidays'): ?>
                <!-- Hari Besar Content -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <span>Tambah Hari Besar Baru</span>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <div class="row">
                                        <div class="col-md-3">
                                            <div class="mb-3">
                                                <label for="name" class="form-label">Nama Hari Besar</label>
                                                <input type="text" class="form-control" id="name" name="name" required>
                                            </div>
                                        </div>
                                        <div class="col-md-3">
                                            <div class="mb-3">
                                                <label for="date" class="form-label">Tanggal</label>
                                                <input type="date" class="form-control" id="date" name="date" required>
                                            </div>
                                        </div>
                                        <div class="col-md-3">
                                            <div class="mb-3">
                                                <label for="description" class="form-label">Deskripsi</label>
                                                <input type="text" class="form-control" id="description" name="description">
                                            </div>
                                        </div>
                                        <div class="col-md-3">
                                            <div class="mb-3">
                                                <label for="category" class="form-label">Kategori</label>
                                                <select class="form-select" id="category" name="category">
                                                    <option value="">Pilih Kategori</option>
                                                    <option value="Nasional">Nasional</option>
                                                    <option value="Internasional">Internasional</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                    <button type="submit" name="add_holiday" class="btn btn-primary">
                                        <i class="fas fa-plus me-1"></i> Tambah Hari Besar
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <span>
                                    <?php if (isset($_GET['view']) && $_GET['view'] === 'skipped'): ?>
                                        Daftar Hari Besar yang Di-skip
                                    <?php else: ?>
                                        Daftar Hari Besar
                                    <?php endif; ?>
                                </span>
                                <div>
                                    <?php if (isset($_GET['view']) && $_GET['view'] === 'skipped'): ?>
                                        <a href="index.php?page=holidays" class="btn btn-sm btn-light">
                                            <i class="fas fa-eye me-1"></i> Lihat Semua Hari Besar
                                        </a>
                                    <?php else: ?>
                                        <a href="index.php?page=holidays&view=skipped" class="btn btn-sm btn-light me-2">
                                            <i class="fas fa-eye-slash me-1"></i> Lihat Hari Besar yang Di-skip
                                        </a>
                                        <a href="index.php?page=holidays" class="btn btn-sm btn-light">
                                            <i class="fas fa-sync-alt me-1"></i> Refresh
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Tanggal</th>
                                                <th>Deskripsi</th>
                                                <th>Status Poster</th>
                                                <th>Posting Instagram</th>
                                                <th>Aksi</th>
                                                <th>Kategori</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $holidays = getHolidays();
                                            $instagramAccounts = getInstagramAccounts();
                                            $viewSkipped = isset($_GET['view']) && $_GET['view'] === 'skipped';
                                            
                                            // Urutkan berdasarkan tanggal
                                            usort($holidays, function($a, $b) {
                                                return strtotime($a['date']) - strtotime($b['date']);
                                            });
                                            
                                            foreach ($holidays as $holiday):
                                                // Tampilkan sesuai dengan filter
                                                if ($viewSkipped) {
                                                    // Tampilkan hanya yang di-skip
                                                    if (!(isset($holiday['skipped']) && $holiday['skipped'])) {
                                                        continue;
                                                    }
                                                } else {
                                                    // Tampilkan yang tidak di-skip
                                                    if (isset($holiday['skipped']) && $holiday['skipped']) {
                                                        continue;
                                                    }
                                                }
                                            ?>
                                            <tr>
                                                <td><?php echo $holiday['name']; ?></td>
                                                <td><?php echo date('d M Y', strtotime($holiday['date'])); ?></td>
                                                <td><?php echo $holiday['description']; ?></td>
                                                <td>
                                                    <?php if ($holiday['poster_created']): ?>
                                                    <span class="badge bg-success">Sudah Dibuat</span>
                                                    <?php else: ?>
                                                    <span class="badge bg-warning text-dark">Belum Dibuat</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php
                                                    if (!empty($holiday['instagram_posts'])) {
                                                        foreach ($holiday['instagram_posts'] as $postId) {
                                                            foreach ($instagramAccounts as $account) {
                                                                if ($account['id'] === $postId) {
                                                                    echo '<span class="badge bg-info me-1">@' . $account['username'] . '</span>';
                                                                }
                                                            }
                                                        }
                                                    } else {
                                                        echo '<span class="badge bg-secondary">Belum ada</span>';
                                                    }
                                                    ?>
                                                </td>
                                                <td>
                                                    <?php if (isset($holiday['category'])): ?>
                                                    <span class="badge bg-<?php echo $holiday['category'] === 'Nasional' ? 'danger' : 'primary'; ?>"><?php echo $holiday['category']; ?></span>
                                                    <?php else: ?>
                                                    <span class="badge bg-secondary">Tidak dikategorikan</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="action-buttons">
                                                        <button type="button" class="btn btn-sm btn-primary" onclick="showCustomModal('<?php echo $holiday['id']; ?>')">
                                                            <i class="fas fa-edit"></i> Update
                                                        </button>
                                                        
                                                        <?php if (!(isset($holiday['skipped']) && $holiday['skipped'])): ?>
                                                        <form id="skip-form-<?php echo $holiday['id']; ?>" method="POST" action="" style="display:inline;">
                                                            <input type="hidden" id="holiday-id-<?php echo $holiday['id']; ?>" name="holiday_id" value="">
                                                            <button type="button" class="btn btn-sm btn-skip" onclick="skipHoliday('<?php echo $holiday['id']; ?>')">
                                                                <i class="fas fa-eye-slash"></i> Skip
                                                            </button>
                                                            <input type="hidden" name="skip_holiday" value="1">
                                                        </form>
                                                        <?php else: ?>
                                                        <form id="unskip-form-<?php echo $holiday['id']; ?>" method="POST" action="" style="display:inline;">
                                                            <input type="hidden" id="holiday-id-unskip-<?php echo $holiday['id']; ?>" name="holiday_id" value="">
                                                            <button type="button" class="btn btn-sm btn-unskip" onclick="unskipHoliday('<?php echo $holiday['id']; ?>')">
                                                                <i class="fas fa-eye"></i> Unskip
                                                            </button>
                                                            <input type="hidden" name="unskip_holiday" value="1">
                                                        </form>
                                                        <?php endif; ?>
                                                    </div>
                                                    
                                                    <!-- Modal Update Status (Kustom) -->
                                                    <div id="customModal-<?php echo $holiday['id']; ?>" class="custom-modal">
                                                        <div class="custom-modal-content">
                                                            <div class="custom-modal-header">
                                                                <h5 class="custom-modal-title">Update Status: <?php echo $holiday['name']; ?></h5>
                                                                <button type="button" class="custom-modal-close" onclick="closeCustomModal('<?php echo $holiday['id']; ?>')">&times;</button>
                                                            </div>
                                                            <form method="POST" action="">
                                                                <div class="custom-modal-body">
                                                                    <input type="hidden" name="holiday_id" value="<?php echo $holiday['id']; ?>">
                                                                    
                                                                    <div class="mb-3 form-check">
                                                                        <input type="checkbox" class="form-check-input" id="posterCreated<?php echo $holiday['id']; ?>" name="poster_created" <?php echo $holiday['poster_created'] ? 'checked' : ''; ?>>
                                                                        <label class="form-check-label" for="posterCreated<?php echo $holiday['id']; ?>">Poster sudah dibuat</label>
                                                                    </div>
                                                                    
                                                                    <div class="mb-3">
                                                                        <label class="form-label">Posting Instagram</label>
                                                                        <?php foreach ($instagramAccounts as $account): ?>
                                                                        <div class="form-check">
                                                                            <input type="checkbox" class="form-check-input" id="instagram<?php echo $holiday['id']; ?>_<?php echo $account['id']; ?>" name="instagram_posts[]" value="<?php echo $account['id']; ?>" <?php echo in_array($account['id'], $holiday['instagram_posts']) ? 'checked' : ''; ?>>
                                                                            <label class="form-check-label" for="instagram<?php echo $holiday['id']; ?>_<?php echo $account['id']; ?>">
                                                                                <?php echo $account['name']; ?> (@<?php echo $account['username']; ?>)
                                                                            </label>
                                                                        </div>
                                                                        <?php endforeach; ?>
                                                                        
                                                                        <?php if (empty($instagramAccounts)): ?>
                                                                        <div class="alert alert-info mt-2">
                                                                            Belum ada akun Instagram. Tambahkan di menu Akun Instagram.
                                                                        </div>
                                                                        <?php endif; ?>
                                                                    </div>
                                                                </div>
                                                                <div class="custom-modal-footer">
                                                                    <button type="button" class="btn btn-secondary" onclick="closeCustomModal('<?php echo $holiday['id']; ?>')">Tutup</button>
                                                                    <button type="submit" name="update_holiday_status" class="btn btn-primary">Simpan Perubahan</button>
                                                                </div>
                                                            </form>
                                                        </div>
                                                    </div>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                            
                                            <?php if (empty($holidays)): ?>
                                            <tr>
                                                <td colspan="6" class="text-center">Belum ada data hari besar.</td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($page === 'instagram'): ?>
                <!-- Instagram Content -->
                <div class="row mb-4">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <span>Tambah Akun Instagram Baru</span>
                            </div>
                            <div class="card-body">
                                <form method="POST" action="">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="name" class="form-label">Nama Akun</label>
                                                <input type="text" class="form-control" id="name" name="name" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="username" class="form-label">Username Instagram</label>
                                                <div class="input-group">
                                                    <span class="input-group-text">@</span>
                                                    <input type="text" class="form-control" id="username" name="username" required>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <button type="submit" name="add_instagram" class="btn btn-primary">
                                        <i class="fas fa-plus me-1"></i> Tambah Akun Instagram
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header">
                                Daftar Akun Instagram
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Nama</th>
                                                <th>Username</th>
                                                <th>Total Posting</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $instagramAccounts = getInstagramAccounts();
                                            $holidays = getHolidays();
                                            
                                            foreach ($instagramAccounts as $account):
                                                // Hitung total posting
                                                $totalPosts = 0;
                                                foreach ($holidays as $holiday) {
                                                    if (in_array($account['id'], $holiday['instagram_posts'])) {
                                                        $totalPosts++;
                                                    }
                                                }
                                            ?>
                                            <tr>
                                                <td><?php echo $account['name']; ?></td>
                                                <td><a href="https://instagram.com/<?php echo $account['username']; ?>" target="_blank" class="text-decoration-none">@<?php echo $account['username']; ?></a></td>
                                                <td><span class="badge bg-primary"><?php echo $totalPosts; ?> posting</span></td>
                                            </tr>
                                            <?php endforeach; ?>
                                            
                                            <?php if (empty($instagramAccounts)): ?>
                                            <tr>
                                                <td colspan="3" class="text-center">Belum ada data akun Instagram.</td>
                                            </tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <?php elseif ($page === 'reports'): ?>
                <!-- Reports Content -->
                <div class="row">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-header">
                                Laporan Hari Besar
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card mb-3">
                                            <div class="card-body">
                                                <h5 class="card-title">Status Pembuatan Poster</h5>
                                                <?php
                                                $holidays = getHolidays();
                                                $totalHolidays = count($holidays);
                                                $posterCreated = 0;
                                                
                                                foreach ($holidays as $holiday) {
                                                    if ($holiday['poster_created']) {
                                                        $posterCreated++;
                                                    }
                                                }
                                                
                                                $posterCreatedPercentage = $totalHolidays > 0 ? round(($posterCreated / $totalHolidays) * 100) : 0;
                                                ?>
                                                <div class="progress mt-3">
                                                    <div class="progress-bar bg-success" role="progressbar" style="width: <?php echo $posterCreatedPercentage; ?>%" aria-valuenow="<?php echo $posterCreatedPercentage; ?>" aria-valuemin="0" aria-valuemax="100"><?php echo $posterCreatedPercentage; ?>%</div>
                                                </div>
                                                <div class="d-flex justify-content-between mt-2">
                                                    <span>Total: <?php echo $totalHolidays; ?></span>
                                                    <span>Dibuat: <?php echo $posterCreated; ?></span>
                                                    <span>Belum: <?php echo $totalHolidays - $posterCreated; ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card mb-3">
                                            <div class="card-body">
                                                <h5 class="card-title">Akun Instagram Paling Aktif</h5>
                                                <?php
                                                $instagramAccounts = getInstagramAccounts();
                                                $accountPostCounts = [];
                                                
                                                foreach ($instagramAccounts as $account) {
                                                    $postCount = 0;
                                                    foreach ($holidays as $holiday) {
                                                        if (in_array($account['id'], $holiday['instagram_posts'])) {
                                                            $postCount++;
                                                        }
                                                    }
                                                    $accountPostCounts[$account['username']] = $postCount;
                                                }
                                                
                                                arsort($accountPostCounts);
                                                ?>
                                                <div class="table-responsive">
                                                    <table class="table table-sm">
                                                        <thead>
                                                            <tr>
                                                                <th>Username</th>
                                                                <th>Jumlah Posting</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($accountPostCounts as $username => $count): ?>
                                                            <tr>
                                                                <td>@<?php echo $username; ?></td>
                                                                <td><span class="badge bg-primary"><?php echo $count; ?></span></td>
                                                            </tr>
                                                            <?php endforeach; ?>
                                                            
                                                            <?php if (empty($accountPostCounts)): ?>
                                                            <tr>
                                                                <td colspan="2" class="text-center">Belum ada data posting.</td>
                                                            </tr>
                                                            <?php endif; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-12">
                                        <div class="card">
                                            <div class="card-body">
                                                <h5 class="card-title">Kalender Hari Besar</h5>
                                                <div class="table-responsive">
                                                    <table class="table table-bordered">
                                                        <thead>
                                                            <tr>
                                                                <th>Bulan</th>
                                                                <th>Hari Besar</th>
                                                                <th>Status</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php
                                                            $holidays = getHolidays();
                                                            $holidaysByMonth = [];
                                                            
                                                            foreach ($holidays as $holiday) {
                                                                $month = date('F', strtotime($holiday['date']));
                                                                if (!isset($holidaysByMonth[$month])) {
                                                                    $holidaysByMonth[$month] = [];
                                                                }
                                                                $holidaysByMonth[$month][] = $holiday;
                                                            }
                                                            
                                                            $months = [
                                                                'January', 'February', 'March', 'April',
                                                                'May', 'June', 'July', 'August',
                                                                'September', 'October', 'November', 'December'
                                                            ];
                                                            
                                                            foreach ($months as $month):
                                                                $monthHolidays = isset($holidaysByMonth[$month]) ? $holidaysByMonth[$month] : [];
                                                            ?>
                                                            <tr>
                                                                <td><?php echo $month; ?></td>
                                                                <td>
                                                                    <?php
                                                                    if (!empty($monthHolidays)) {
                                                                        foreach ($monthHolidays as $holiday) {
                                                                            echo '<div class="mb-1">' . date('d', strtotime($holiday['date'])) . ' - ' . $holiday['name'] . '</div>';
                                                                        }
                                                                    } else {
                                                                        echo 'Tidak ada';
                                                                    }
                                                                    ?>
                                                                </td>
                                                                <td>
                                                                    <?php
                                                                    if (!empty($monthHolidays)) {
                                                                        foreach ($monthHolidays as $holiday) {
                                                                            if ($holiday['poster_created']) {
                                                                                echo '<div class="mb-1"><span class="badge bg-success">Poster Dibuat</span></div>';
                                                                            } else {
                                                                                echo '<div class="mb-1"><span class="badge bg-warning text-dark">Belum Dibuat</span></div>';
                                                                            }
                                                                        }
                                                                    } else {
                                                                        echo '-';
                                                                    }
                                                                    ?>
                                                                </td>
                                                            </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Script untuk menangani modal kustom
        document.addEventListener('DOMContentLoaded', function() {
            // Fungsi untuk menampilkan modal kustom
            window.showCustomModal = function(id) {
                document.getElementById('customModal-' + id).style.display = 'block';
                document.body.style.overflow = 'hidden';
            };
            
            // Fungsi untuk menutup modal kustom
            window.closeCustomModal = function(id) {
                document.getElementById('customModal-' + id).style.display = 'none';
                document.body.style.overflow = 'auto';
            };
            
            // Proses skip hari besar
            window.skipHoliday = function(id) {
                document.getElementById('holiday-id-' + id).value = id;
                document.getElementById('skip-form-' + id).submit();
            };
            
            // Proses unskip hari besar
            window.unskipHoliday = function(id) {
                document.getElementById('holiday-id-unskip-' + id).value = id;
                document.getElementById('unskip-form-' + id).submit();
            };
            
            // Menutup modal saat mengklik di luar modal
            window.addEventListener('click', function(event) {
                document.querySelectorAll('.custom-modal').forEach(function(modal) {
                    if (event.target === modal) {
                        modal.style.display = 'none';
                        document.body.style.overflow = 'auto';
                    }
                });
            });
        });
    </script>
</body>
</html>