<?php
require_once 'config.php';
require_once 'functions.php';

// Mendapatkan pengaturan aplikasi
$settings = getSettings();

// Mengatur tema
$theme = $settings['theme'];
if ($theme === 'auto') {
    $theme = isset($_SERVER['HTTP_SEC_CH_PREFERS_COLOR_SCHEME']) && 
             $_SERVER['HTTP_SEC_CH_PREFERS_COLOR_SCHEME'] === 'dark' ? 'dark' : 'light';
}

// Mendapatkan data untuk laporan
$businesses = getAllBusinesses();
$personal = getPersonalFinance();
$transactions = getAllTransactions();

// Mengelompokkan transaksi berdasarkan status
$activeTransactions = array_filter($transactions, function($t) {
    return $t['status'] === 'active';
});
$paidTransactions = array_filter($transactions, function($t) {
    return $t['status'] === 'paid';
});

// Mengelompokkan berdasarkan tipe
$debtTransactions = array_filter($transactions, function($t) {
    return $t['type'] === 'debt';
});
$receivableTransactions = array_filter($transactions, function($t) {
    return $t['type'] === 'receivable';
});

// Menghitung total
$totalActiveDebt = array_reduce(
    array_filter($activeTransactions, function($t) { return $t['type'] === 'debt'; }),
    function($carry, $t) { return $carry + $t['amount']; },
    0
);

$totalActiveReceivable = array_reduce(
    array_filter($activeTransactions, function($t) { return $t['type'] === 'receivable'; }),
    function($carry, $t) { return $carry + $t['amount']; },
    0
);

$totalPaidDebt = array_reduce(
    array_filter($paidTransactions, function($t) { return $t['type'] === 'debt'; }),
    function($carry, $t) { return $carry + $t['amount']; },
    0
);

$totalPaidReceivable = array_reduce(
    array_filter($paidTransactions, function($t) { return $t['type'] === 'receivable'; }),
    function($carry, $t) { return $carry + $t['amount']; },
    0
);

// Mendapatkan transaksi jatuh tempo (overdue)
$overdueTransactions = array_filter($activeTransactions, function($t) {
    $dueDate = new DateTime($t['due_date']);
    $now = new DateTime();
    return $dueDate < $now;
});

// Mendapatkan summary keuangan
$summary = getFinancialSummary();
?>
<!DOCTYPE html>
<html lang="id" data-theme="<?php echo $theme; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Meta tags untuk mencegah pengindeksan oleh mesin pencari -->
    <meta name="robots" content="noindex, nofollow">
    <meta name="googlebot" content="noindex, nofollow">
    <meta name="bingbot" content="noindex, nofollow">
    <meta name="yandex" content="none">
    <link rel="icon" href="https://app.dezki.biz.id/linkpic/assets/uploads/67dc5f2646a8d.png">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laporan - <?php echo $settings['app_name']; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script src="assets/js/app.js" defer></script>
</head>
<body>
    <div class="app-container">
        <!-- Sidebar -->
        <aside class="sidebar">
            <div class="sidebar-header">
                <h1><?php echo $settings['app_name']; ?></h1>
            </div>
            <nav class="sidebar-nav">
                <ul>
                    <li><a href="index.php"><i class="icon icon-dashboard"></i> Dashboard</a></li>
                    <li><a href="businesses.php"><i class="icon icon-business"></i> Bisnis</a></li>
                    <li><a href="transactions.php"><i class="icon icon-transaction"></i> Transaksi</a></li>
                    <li class="active"><a href="reports.php"><i class="icon icon-report"></i> Laporan</a></li>
                    <li><a href="settings.php"><i class="icon icon-settings"></i> Pengaturan</a></li>
                </ul>
            </nav>
        </aside>

        <!-- Main Content -->
        <main class="main-content">
            <header class="content-header">
                <h2>Laporan Keuangan</h2>
                <div class="header-actions">
                    <button onclick="window.print();" class="btn btn-secondary">
                        <i class="icon icon-print"></i> Cetak Laporan
                    </button>
                </div>
            </header>

            <!-- Ringkasan Utang Piutang -->
            <div class="card">
                <div class="card-header">
                    <h3>Ringkasan Utang Piutang</h3>
                </div>
                <div class="card-body">
                    <div class="report-summary">
                        <div class="summary-box">
                            <div class="summary-title">Total Utang Aktif</div>
                            <div class="summary-value debt"><?php echo formatCurrency($totalActiveDebt); ?></div>
                        </div>
                        <div class="summary-box">
                            <div class="summary-title">Total Piutang Aktif</div>
                            <div class="summary-value receivable"><?php echo formatCurrency($totalActiveReceivable); ?></div>
                        </div>
                        <div class="summary-box">
                            <div class="summary-title">Selisih</div>
                            <div class="summary-value <?php echo $totalActiveReceivable > $totalActiveDebt ? 'positive' : 'negative'; ?>">
                                <?php echo formatCurrency($totalActiveReceivable - $totalActiveDebt); ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Perbandingan Entitas -->
            <div class="card">
                <div class="card-header">
                    <h3>Perbandingan Antar Entitas</h3>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Entitas</th>
                                    <th>Saldo</th>
                                    <th>Total Utang</th>
                                    <th>Total Piutang</th>
                                    <th>Selisih</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td><strong>Pribadi</strong></td>
                                    <td><?php echo formatCurrency($summary['personal']['balance']); ?></td>
                                    <td><?php echo formatCurrency($summary['personal']['total_debt']); ?></td>
                                    <td><?php echo formatCurrency($summary['personal']['total_receivable']); ?></td>
                                    <td class="<?php echo $summary['personal']['total_receivable'] > $summary['personal']['total_debt'] ? 'positive' : 'negative'; ?>">
                                        <?php echo formatCurrency($summary['personal']['total_receivable'] - $summary['personal']['total_debt']); ?>
                                    </td>
                                </tr>
                                <?php foreach ($summary['businesses'] as $business): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($business['name']); ?></td>
                                    <td><?php echo formatCurrency($business['balance']); ?></td>
                                    <td><?php echo formatCurrency($business['total_debt']); ?></td>
                                    <td><?php echo formatCurrency($business['total_receivable']); ?></td>
                                    <td class="<?php echo $business['total_receivable'] > $business['total_debt'] ? 'positive' : 'negative'; ?>">
                                        <?php echo formatCurrency($business['total_receivable'] - $business['total_debt']); ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Transaksi Jatuh Tempo -->
            <div class="card">
                <div class="card-header">
                    <h3>Transaksi Jatuh Tempo (Terlambat)</h3>
                </div>
                <div class="card-body">
                    <?php if (empty($overdueTransactions)): ?>
                    <div class="empty-state">
                        <p>Tidak ada transaksi yang telah melewati jatuh tempo.</p>
                    </div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Deskripsi</th>
                                    <th>Dari</th>
                                    <th>Ke</th>
                                    <th>Jumlah</th>
                                    <th>Jatuh Tempo</th>
                                    <th>Tipe</th>
                                    <th>Terlambat</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($overdueTransactions as $t): ?>
                                <tr>
                                    <td>
                                        <a href="transactions.php?action=view&id=<?php echo $t['id']; ?>">
                                            <?php echo htmlspecialchars($t['description']); ?>
                                        </a>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($t['from']['type'] === 'personal') {
                                            echo 'Pribadi';
                                        } else if ($t['from']['type'] === 'business') {
                                            $b = getBusinessById($t['from']['id']);
                                            echo $b ? htmlspecialchars($b['name']) : 'Bisnis #' . $t['from']['id'];
                                        } else {
                                            echo htmlspecialchars($t['from']['id']); // External name
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($t['to']['type'] === 'personal') {
                                            echo 'Pribadi';
                                        } else if ($t['to']['type'] === 'business') {
                                            $b = getBusinessById($t['to']['id']);
                                            echo $b ? htmlspecialchars($b['name']) : 'Bisnis #' . $t['to']['id'];
                                        } else {
                                            echo htmlspecialchars($t['to']['id']); // External name
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo formatCurrency($t['amount']); ?></td>
                                    <td><?php echo formatDate($t['due_date']); ?></td>
                                    <td>
                                        <?php if ($t['type'] === 'debt'): ?>
                                        <span class="badge badge-danger">Utang</span>
                                        <?php else: ?>
                                        <span class="badge badge-success">Piutang</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php
                                        $dueDate = new DateTime($t['due_date']);
                                        $now = new DateTime();
                                        $interval = $now->diff($dueDate);
                                        echo '<span class="badge badge-danger">' . $interval->days . ' hari</span>';
                                        ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Riwayat Transaksi yang Sudah Lunas -->
            <div class="card">
                <div class="card-header">
                    <h3>Riwayat Transaksi Lunas</h3>
                </div>
                <div class="card-body">
                    <?php if (empty($paidTransactions)): ?>
                    <div class="empty-state">
                        <p>Belum ada transaksi yang lunas.</p>
                    </div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Tanggal</th>
                                    <th>Deskripsi</th>
                                    <th>Dari</th>
                                    <th>Ke</th>
                                    <th>Jumlah</th>
                                    <th>Tipe</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php 
                                // Urutkan berdasarkan tanggal (terbaru duluan)
                                usort($paidTransactions, function($a, $b) {
                                    return strtotime($b['date']) - strtotime($a['date']);
                                });
                                
                                foreach (array_slice($paidTransactions, 0, 10) as $t): // Tampilkan 10 teratas
                                ?>
                                <tr>
                                    <td><?php echo formatDate($t['date']); ?></td>
                                    <td>
                                        <a href="transactions.php?action=view&id=<?php echo $t['id']; ?>">
                                            <?php echo htmlspecialchars($t['description']); ?>
                                        </a>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($t['from']['type'] === 'personal') {
                                            echo 'Pribadi';
                                        } else if ($t['from']['type'] === 'business') {
                                            $b = getBusinessById($t['from']['id']);
                                            echo $b ? htmlspecialchars($b['name']) : 'Bisnis #' . $t['from']['id'];
                                        } else {
                                            echo htmlspecialchars($t['from']['id']); // External name
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <?php 
                                        if ($t['to']['type'] === 'personal') {
                                            echo 'Pribadi';
                                        } else if ($t['to']['type'] === 'business') {
                                            $b = getBusinessById($t['to']['id']);
                                            echo $b ? htmlspecialchars($b['name']) : 'Bisnis #' . $t['to']['id'];
                                        } else {
                                            echo htmlspecialchars($t['to']['id']); // External name
                                        }
                                        ?>
                                    </td>
                                    <td><?php echo formatCurrency($t['amount']); ?></td>
                                    <td>
                                        <?php if ($t['type'] === 'debt'): ?>
                                        <span class="badge badge-danger">Utang</span>
                                        <?php else: ?>
                                        <span class="badge badge-success">Piutang</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php if (count($paidTransactions) > 10): ?>
                        <div class="text-center mt-3">
                            <a href="transactions.php?filter_status=paid" class="btn btn-secondary">
                                Lihat Semua Transaksi Lunas
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</body>
</html>