<?php
// functions.php - Fungsi CRUD untuk mengelola data

require_once 'config.php';

// ===== FUNGSI PENGELOLAAN BISNIS =====

// Mendapatkan semua bisnis
function getAllBusinesses() {
    $data = readJsonFile(BUSINESSES_FILE);
    return $data['businesses'] ?? [];
}

// Mendapatkan bisnis berdasarkan ID
function getBusinessById($id) {
    $businesses = getAllBusinesses();
    foreach ($businesses as $business) {
        if ($business['id'] === $id) {
            return $business;
        }
    }
    return null;
}

// Menambahkan bisnis baru
function addBusiness($name, $description, $initialBalance = 0) {
    $data = readJsonFile(BUSINESSES_FILE);
    $newBusiness = [
        'id' => generateId('bis'),
        'name' => $name,
        'description' => $description,
        'balance' => $initialBalance,
        'created_at' => date('c'),
        'updated_at' => date('c')
    ];
    
    $data['businesses'][] = $newBusiness;
    writeJsonFile(BUSINESSES_FILE, $data);
    
    return $newBusiness;
}

// Mengedit bisnis
function updateBusiness($id, $data) {
    $businessesData = readJsonFile(BUSINESSES_FILE);
    $updated = false;
    
    foreach ($businessesData['businesses'] as $key => $business) {
        if ($business['id'] === $id) {
            $businessesData['businesses'][$key] = array_merge($business, $data);
            $businessesData['businesses'][$key]['updated_at'] = date('c');
            $updated = true;
            break;
        }
    }
    
    if ($updated) {
        writeJsonFile(BUSINESSES_FILE, $businessesData);
        return true;
    }
    
    return false;
}

// Menghapus bisnis
function deleteBusiness($id) {
    $businessesData = readJsonFile(BUSINESSES_FILE);
    $businesses = $businessesData['businesses'];
    
    foreach ($businesses as $key => $business) {
        if ($business['id'] === $id) {
            unset($businesses[$key]);
            $businessesData['businesses'] = array_values($businesses); // Reindexing array
            writeJsonFile(BUSINESSES_FILE, $businessesData);
            return true;
        }
    }
    
    return false;
}

// ===== FUNGSI PENGELOLAAN KEUANGAN PRIBADI =====

// Mendapatkan data keuangan pribadi
function getPersonalFinance() {
    return readJsonFile(PERSONAL_FILE);
}

// Memperbarui saldo pribadi
function updatePersonalBalance($newBalance) {
    $personalData = getPersonalFinance();
    $personalData['balance'] = $newBalance;
    $personalData['updated_at'] = date('c');
    return writeJsonFile(PERSONAL_FILE, $personalData);
}

// ===== FUNGSI PENGELOLAAN TRANSAKSI =====

// Mendapatkan semua transaksi
function getAllTransactions() {
    $data = readJsonFile(TRANSACTIONS_FILE);
    return $data['transactions'] ?? [];
}

// Mendapatkan transaksi berdasarkan ID
function getTransactionById($id) {
    $transactions = getAllTransactions();
    foreach ($transactions as $transaction) {
        if ($transaction['id'] === $id) {
            return $transaction;
        }
    }
    return null;
}

// Menambahkan transaksi baru
function addTransaction($type, $amount, $description, $from, $to, $dueDate = null) {
    $data = readJsonFile(TRANSACTIONS_FILE);
    $now = date('c');
    
    if ($dueDate === null) {
        // Default due date: 30 days from now
        $dueDateTime = new DateTime();
        $dueDateTime->modify('+30 days');
        $dueDate = $dueDateTime->format('c');
    }
    
    $newTransaction = [
        'id' => generateId('trx'),
        'date' => $now,
        'type' => $type, // debt atau receivable
        'amount' => $amount,
        'description' => $description,
        'from' => $from,
        'to' => $to,
        'status' => 'active',
        'due_date' => $dueDate,
        'payment_history' => []
    ];
    
    $data['transactions'][] = $newTransaction;
    writeJsonFile(TRANSACTIONS_FILE, $data);
    
    // Update saldo
    updateBalanceForTransaction($newTransaction);
    
    return $newTransaction;
}

// Update saldo saat transaksi terjadi
function updateBalanceForTransaction($transaction) {
    // Jika transaksi adalah utang (debt)
    if ($transaction['type'] === 'debt') {
        // Jika dari personal ke bisnis
        if ($transaction['from']['type'] === 'personal' && $transaction['to']['type'] === 'business') {
            // Kurangi saldo personal
            $personalData = getPersonalFinance();
            $newBalance = $personalData['balance'] - $transaction['amount'];
            updatePersonalBalance($newBalance);
            
            // Tambah saldo bisnis
            $business = getBusinessById($transaction['to']['id']);
            if ($business) {
                $newBusinessBalance = $business['balance'] + $transaction['amount'];
                updateBusiness($business['id'], ['balance' => $newBusinessBalance]);
            }
        }
        // Jika dari bisnis ke personal
        else if ($transaction['from']['type'] === 'business' && $transaction['to']['type'] === 'personal') {
            // Kurangi saldo bisnis
            $business = getBusinessById($transaction['from']['id']);
            if ($business) {
                $newBusinessBalance = $business['balance'] - $transaction['amount'];
                updateBusiness($business['id'], ['balance' => $newBusinessBalance]);
            }
            
            // Tambah saldo personal
            $personalData = getPersonalFinance();
            $newBalance = $personalData['balance'] + $transaction['amount'];
            updatePersonalBalance($newBalance);
        }
        // Jika dari bisnis ke bisnis
        else if ($transaction['from']['type'] === 'business' && $transaction['to']['type'] === 'business') {
            // Kurangi saldo bisnis pengirim
            $fromBusiness = getBusinessById($transaction['from']['id']);
            if ($fromBusiness) {
                $newFromBalance = $fromBusiness['balance'] - $transaction['amount'];
                updateBusiness($fromBusiness['id'], ['balance' => $newFromBalance]);
            }
            
            // Tambah saldo bisnis penerima
            $toBusiness = getBusinessById($transaction['to']['id']);
            if ($toBusiness) {
                $newToBalance = $toBusiness['balance'] + $transaction['amount'];
                updateBusiness($toBusiness['id'], ['balance' => $newToBalance]);
            }
        }
    }
    // Jika transaksi adalah piutang (receivable)
    else if ($transaction['type'] === 'receivable') {
        // Logika untuk piutang, inversnya dari logika utang
        // (Tidak diimplementasikan secara detail di sini untuk menyederhanakan kode)
    }
}

// Memperbarui status transaksi
function updateTransactionStatus($id, $status) {
    $transactionsData = readJsonFile(TRANSACTIONS_FILE);
    $updated = false;
    
    foreach ($transactionsData['transactions'] as $key => $transaction) {
        if ($transaction['id'] === $id) {
            $transactionsData['transactions'][$key]['status'] = $status;
            $updated = true;
            break;
        }
    }
    
    if ($updated) {
        writeJsonFile(TRANSACTIONS_FILE, $transactionsData);
        return true;
    }
    
    return false;
}

// Menambahkan pembayaran ke transaksi
function addPaymentToTransaction($id, $amount, $note = '') {
    $transactionsData = readJsonFile(TRANSACTIONS_FILE);
    $updated = false;
    
    foreach ($transactionsData['transactions'] as $key => $transaction) {
        if ($transaction['id'] === $id) {
            $payment = [
                'date' => date('c'),
                'amount' => $amount,
                'note' => $note
            ];
            
            $transactionsData['transactions'][$key]['payment_history'][] = $payment;
            
            // Jika total pembayaran >= jumlah transaksi, ubah status menjadi 'paid'
            $totalPaid = 0;
            foreach ($transactionsData['transactions'][$key]['payment_history'] as $p) {
                $totalPaid += $p['amount'];
            }
            
            if ($totalPaid >= $transaction['amount']) {
                $transactionsData['transactions'][$key]['status'] = 'paid';
            }
            
            $updated = true;
            break;
        }
    }
    
    if ($updated) {
        writeJsonFile(TRANSACTIONS_FILE, $transactionsData);
        return true;
    }
    
    return false;
}

// ===== FUNGSI FILTER DAN PENCARIAN =====

// Filter transaksi berdasarkan status
function filterTransactionsByStatus($status) {
    $transactions = getAllTransactions();
    return array_filter($transactions, function($transaction) use ($status) {
        return $transaction['status'] === $status;
    });
}

// Filter transaksi berdasarkan tipe (debt/receivable)
function filterTransactionsByType($type) {
    $transactions = getAllTransactions();
    return array_filter($transactions, function($transaction) use ($type) {
        return $transaction['type'] === $type;
    });
}

// Filter transaksi berdasarkan entitas (bisnis atau pribadi)
function filterTransactionsByEntity($entityType, $entityId = null) {
    $transactions = getAllTransactions();
    return array_filter($transactions, function($transaction) use ($entityType, $entityId) {
        // Cek apakah entitas ini ada di 'from' atau 'to'
        $fromMatch = $transaction['from']['type'] === $entityType && 
                    ($entityId === null || $transaction['from']['id'] === $entityId);
        $toMatch = $transaction['to']['type'] === $entityType && 
                  ($entityId === null || $transaction['to']['id'] === $entityId);
        
        return $fromMatch || $toMatch;
    });
}

// Mendapatkan transaksi jatuh tempo
function getDueTransactions($days = null) {
    $settings = getSettings();
    $days = $days ?? $settings['reminder_days'];
    
    $transactions = getAllTransactions();
    $now = new DateTime();
    $dueDateLimit = (new DateTime())->modify("+{$days} days");
    
    return array_filter($transactions, function($transaction) use ($now, $dueDateLimit) {
        if ($transaction['status'] !== 'active') {
            return false;
        }
        
        $dueDate = new DateTime($transaction['due_date']);
        return $dueDate >= $now && $dueDate <= $dueDateLimit;
    });
}

// ===== FUNGSI LAPORAN DAN ANALISIS =====

// Mendapatkan total utang berdasarkan entitas
function getTotalDebtByEntity($entityType, $entityId = null) {
    $transactions = filterTransactionsByType('debt');
    $total = 0;
    
    foreach ($transactions as $transaction) {
        if ($transaction['status'] === 'active' && 
            $transaction['from']['type'] === $entityType && 
            ($entityId === null || $transaction['from']['id'] === $entityId)) {
            $total += $transaction['amount'];
        }
    }
    
    return $total;
}

// Mendapatkan total piutang berdasarkan entitas
function getTotalReceivableByEntity($entityType, $entityId = null) {
    $transactions = filterTransactionsByType('receivable');
    $total = 0;
    
    foreach ($transactions as $transaction) {
        if ($transaction['status'] === 'active' && 
            $transaction['from']['type'] === $entityType && 
            ($entityId === null || $transaction['from']['id'] === $entityId)) {
            $total += $transaction['amount'];
        }
    }
    
    return $total;
}

// Mendapatkan summary keuangan untuk semua entitas
function getFinancialSummary() {
    $businesses = getAllBusinesses();
    $personal = getPersonalFinance();
    $summary = [
        'personal' => [
            'balance' => $personal['balance'],
            'total_debt' => getTotalDebtByEntity('personal'),
            'total_receivable' => getTotalReceivableByEntity('personal')
        ],
        'businesses' => []
    ];
    
    foreach ($businesses as $business) {
        $summary['businesses'][] = [
            'id' => $business['id'],
            'name' => $business['name'],
            'balance' => $business['balance'],
            'total_debt' => getTotalDebtByEntity('business', $business['id']),
            'total_receivable' => getTotalReceivableByEntity('business', $business['id'])
        ];
    }
    
    return $summary;
}