<?php
/**
 * Sistem Pengelola Event Olimpiade - Complete Version
 * Main Application File with Full Integration
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/calendar.php';
require_once __DIR__ . '/analysis.php';

// Validasi environment
$envErrors = Config::validateEnvironment();
if (!empty($envErrors)) {
    die('<h2>Environment Error</h2><ul><li>' . implode('</li><li>', $envErrors) . '</li></ul>');
}

// Inisialisasi variabel
$error = '';
$success = '';
$events = [];
$currentView = $_GET['view'] ?? 'generator';
$currentMonth = $_GET['month'] ?? date('Y-m');
$editEventId = $_GET['edit'] ?? null;
$detailEventId = $_GET['detail'] ?? null;

// Load events jika sudah login
if (Auth::isLoggedIn()) {
    $events = EventManager::loadEvents();
}

// Proses form submissions
if ($_POST) {
    // Validasi CSRF untuk form yang memerlukan keamanan ekstra
    $securedActions = ['simpan_event', 'hapus_event'];
    $currentAction = '';
    foreach ($securedActions as $action) {
        if (isset($_POST[$action])) {
            $currentAction = $action;
            break;
        }
    }
    
    if ($currentAction && !Config::validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Token keamanan tidak valid. Silakan refresh halaman.';
    } else {
        // Proses berbagai aksi
        if (isset($_POST['login'])) {
            $pin = Validator::sanitizeInput($_POST['pin']);
            if (Auth::login($pin)) {
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } else {
                $error = Config::getMessage('login_failed');
            }
        } 
        elseif (Auth::isLoggedIn()) {
            // Aksi yang memerlukan login
            if (isset($_POST['logout'])) {
                Auth::logout();
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
            
            if (isset($_POST['cari_slot_otomatis'])) {
                $inputData = Validator::sanitizeInput($_POST);
                $validationErrors = Validator::validateEventData($inputData);
                
                if (!empty($validationErrors)) {
                    $error = implode('<br>', $validationErrors);
                } else {
                    $optimalSlots = EventManager::findOptimalSlots($events);
                    $_SESSION['slot_optimal'] = $optimalSlots;
                    $_SESSION['data_form'] = $inputData;
                    
                    if (empty($optimalSlots)) {
                        $error = Config::getMessage('no_optimal_slot');
                    }
                }
            }
            
            if (isset($_POST['pilih_slot'])) {
                if (isset($_POST['slot_index'], $_SESSION['slot_optimal'], $_SESSION['data_form'])) {
                    $slotIndex = (int)$_POST['slot_index'];
                    if (isset($_SESSION['slot_optimal'][$slotIndex])) {
                        $selectedSlot = $_SESSION['slot_optimal'][$slotIndex];
                        $previewData = EventManager::calculateTimeline($_SESSION['data_form'], $selectedSlot);
                        if ($previewData) {
                            $previewData = array_merge($_SESSION['data_form'], $previewData, ['slot_terpilih' => $selectedSlot]);
                            $_SESSION['data_preview'] = $previewData;
                            unset($_SESSION['slot_optimal']);
                        } else {
                            $error = 'Terjadi kesalahan saat menghitung timeline';
                        }
                    } else {
                        $error = 'Slot yang dipilih tidak valid';
                    }
                } else {
                    $error = 'Data slot tidak lengkap';
                }
            }
            
            if (isset($_POST['generate_preview'])) {
                $inputData = Validator::sanitizeInput($_POST);
                $validationErrors = Validator::validateEventData($inputData);
                
                if (!empty($validationErrors)) {
                    $error = implode('<br>', $validationErrors);
                } else {
                    $previewData = EventManager::calculateTimeline($inputData);
                    if ($previewData) {
                        $previewData = array_merge($inputData, $previewData);
                        $_SESSION['data_preview'] = $previewData;
                    } else {
                        $error = 'Terjadi kesalahan saat menghitung timeline';
                    }
                }
            }
            
            if (isset($_POST['simpan_event'])) {
                if (isset($_SESSION['data_preview'])) {
                    $eventData = $_SESSION['data_preview'];
                    $eventData['id'] = $_POST['event_id'] ?: uniqid();
                    $eventData['dibuat_pada'] = date(DATETIME_FORMAT);
                    
                    // Pastikan field yang diperlukan ada
                    $eventData['nama_lomba'] = $eventData['namaLomba'];
                    $eventData['singkatan_lomba'] = $eventData['singkatanLomba'];
                    
                    $conflicts = EventManager::checkConflicts($eventData, $events, $_POST['event_id']);
                    
                    if (!empty($conflicts) && !isset($_POST['paksa_simpan'])) {
                        $warningConflicts = $conflicts;
                        $eventForEdit = $eventData;
                    } else {
                        if ($_POST['event_id']) {
                            // Update existing event
                            foreach ($events as &$event) {
                                if ($event['id'] == $_POST['event_id']) {
                                    $event = array_merge($event, $eventData);
                                    break;
                                }
                            }
                        } else {
                            // Add new event
                            $events[] = $eventData;
                        }
                        
                        if (EventManager::saveEvents($events)) {
                            unset($_SESSION['data_preview'], $_SESSION['slot_optimal'], $_SESSION['data_form']);
                            $success = Config::getMessage('event_saved');
                            $events = EventManager::loadEvents(); // Reload events
                        } else {
                            $error = 'Gagal menyimpan event ke database';
                        }
                    }
                } else {
                    $error = 'Data preview tidak ditemukan. Silakan buat preview terlebih dahulu.';
                }
            }
            
            if (isset($_POST['hapus_event'])) {
                $eventId = Validator::sanitizeInput($_POST['event_id']);
                $events = array_filter($events, function($event) use ($eventId) {
                    return $event['id'] != $eventId;
                });
                
                if (EventManager::saveEvents($events)) {
                    $success = Config::getMessage('event_deleted');
                    $events = EventManager::loadEvents(); // Reload events
                } else {
                    $error = 'Gagal menghapus event';
                }
            }
        }
    }
}

// Load data untuk edit jika diperlukan
$eventForEdit = null;
if ($editEventId) {
    $eventForEdit = array_filter($events, function($event) use ($editEventId) {
        return $event['id'] == $editEventId;
    });
    $eventForEdit = reset($eventForEdit);
    unset($_SESSION['data_preview'], $_SESSION['slot_optimal']); // Clear preview saat edit
}

// Load data untuk detail jika diperlukan
$detailEvent = null;
if ($detailEventId) {
    $detailEvent = array_filter($events, function($event) use ($detailEventId) {
        return $event['id'] == $detailEventId;
    });
    $detailEvent = reset($detailEvent);
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= APP_NAME ?></title>
    <meta name="description" content="Sistem manajemen event olimpiade dengan penjadwalan otomatis dan analisis beban kerja">
    <meta name="author" content="<?= APP_NAME ?>">
    
    <!-- Security headers -->
    <meta http-equiv="X-Content-Type-Options" content="nosniff">
    <meta http-equiv="X-Frame-Options" content="DENY">
    <meta http-equiv="X-XSS-Protection" content="1; mode=block">
    
    <?= CalendarStyles::getCSS() ?>
    <?= AnalysisStyles::getCSS() ?>
    
    <style>
        /* Base styles */
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            background: #f5f5f5; 
            line-height: 1.6;
            color: #374151;
        }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        
        /* Header */
        .header { 
            background: linear-gradient(135deg, #2563eb, #1d4ed8); 
            color: white; 
            padding: 20px; 
            border-radius: 8px; 
            margin-bottom: 20px; 
            display: flex;
            align-items: center;
            justify-content: space-between;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .header h1 {
            font-size: 24px;
            margin-bottom: 5px;
        }
        
        .header small {
            opacity: 0.9;
            font-size: 14px;
        }
        
        /* Navigation */
        .nav { 
            display: flex; 
            gap: 10px; 
            margin-bottom: 20px; 
            flex-wrap: wrap; 
        }
        .nav a { 
            background: #3b82f6; 
            color: white; 
            padding: 12px 20px; 
            border-radius: 6px; 
            text-decoration: none; 
            transition: all 0.3s; 
            font-weight: 500;
            border: 2px solid transparent;
        }
        .nav a.active { 
            background: #1d4ed8; 
            border-color: #1e40af;
            transform: translateY(-1px); 
        }
        .nav a:hover { 
            background: #2563eb; 
            transform: translateY(-1px); 
        }
        
        /* Cards */
        .card { 
            background: white; 
            padding: 20px; 
            border-radius: 8px; 
            box-shadow: 0 2px 8px rgba(0,0,0,0.1); 
            margin-bottom: 20px; 
            border: 1px solid #e5e7eb;
        }
        
        .card h2, .card h3 {
            color: #1f2937;
            margin-bottom: 15px;
        }
        
        /* Forms */
        .form-group { margin-bottom: 15px; }
        .form-group label { 
            display: block; 
            margin-bottom: 5px; 
            font-weight: 600; 
            color: #374151; 
        }
        .form-group input, .form-group select, .form-group textarea { 
            width: 100%; 
            padding: 12px; 
            border: 1px solid #d1d5db; 
            border-radius: 6px; 
            font-size: 14px; 
            transition: all 0.3s;
            background: white;
        }
        .form-group input:focus, .form-group select:focus, .form-group textarea:focus { 
            outline: none; 
            border-color: #3b82f6; 
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1); 
        }
        .form-group small {
            color: #6b7280;
            font-size: 12px;
            margin-top: 5px;
            display: block;
        }
        
        /* Buttons */
        .btn { 
            background: #3b82f6; 
            color: white; 
            padding: 12px 20px; 
            border: none; 
            border-radius: 6px; 
            cursor: pointer; 
            margin-right: 10px; 
            margin-bottom: 10px;
            font-weight: 500; 
            transition: all 0.3s; 
            text-decoration: none;
            display: inline-block;
            font-size: 14px;
            border: 2px solid transparent;
        }
        .btn:hover { 
            background: #2563eb; 
            transform: translateY(-1px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
        }
        .btn-danger { background: #dc2626; }
        .btn-danger:hover { background: #b91c1c; }
        .btn-success { background: #16a34a; }
        .btn-success:hover { background: #15803d; }
        .btn-warning { background: #ea580c; }
        .btn-warning:hover { background: #c2410c; }
        .btn-secondary { background: #6b7280; }
        .btn-secondary:hover { background: #4b5563; }
        
        /* Alerts */
        .alert { 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 20px; 
            border: 1px solid;
            display: flex;
            align-items: flex-start;
            gap: 10px;
        }
        .alert-success { 
            background: #dcfce7; 
            color: #166534; 
            border-color: #bbf7d0; 
        }
        .alert-danger { 
            background: #fef2f2; 
            color: #991b1b; 
            border-color: #fecaca; 
        }
        .alert-warning { 
            background: #fefbeb; 
            color: #92400e; 
            border-color: #fed7aa; 
        }
        .alert-info {
            background: #dbeafe;
            color: #1e40af;
            border-color: #bfdbfe;
        }
        
        /* Login form */
        .form-login { 
            max-width: 400px; 
            margin: 100px auto; 
        }
        
        /* Grid layouts */
        .baris-form { 
            display: grid; 
            grid-template-columns: 1fr 1fr; 
            gap: 15px; 
        }
        .grup-radio { 
            display: flex; 
            gap: 15px; 
            align-items: center; 
            flex-wrap: wrap;
        }
        .grup-radio label { 
            margin-bottom: 0; 
            cursor: pointer; 
            display: flex;
            align-items: center;
            gap: 5px;
            font-weight: normal;
        }
        
        /* Special sections */
        .auto-info { 
            background: #f0f9ff; 
            border: 1px solid #0ea5e9; 
            padding: 15px; 
            border-radius: 6px; 
            margin: 15px 0; 
        }
        .auto-info h4 {
            color: #0c4a6e;
            margin-bottom: 10px;
        }
        .auto-info p {
            margin-bottom: 5px;
            font-size: 14px;
        }
        
        .preview-card { 
            background: linear-gradient(135deg, #dbeafe, #e0f2fe); 
            border: 1px solid #3b82f6; 
            padding: 20px; 
            border-radius: 8px; 
            margin: 20px 0; 
        }
        
        .timeline-section { 
            border-left: 4px solid #3b82f6; 
            padding-left: 15px; 
            margin-bottom: 15px; 
            background: rgba(255, 255, 255, 0.7);
            padding: 15px;
            border-radius: 0 6px 6px 0;
        }
        .timeline-section h4 { 
            color: #1d4ed8; 
            margin-bottom: 8px; 
            font-size: 16px; 
        }
        .timeline-section.hijau { border-left-color: #16a34a; }
        .timeline-section.hijau h4 { color: #15803d; }
        .timeline-section.orange { border-left-color: #ea580c; }
        .timeline-section.orange h4 { color: #c2410c; }
        .timeline-section.ungu { border-left-color: #7c3aed; }
        .timeline-section.ungu h4 { color: #6d28d9; }
        .timeline-section.kuning { border-left-color: #eab308; }
        .timeline-section.kuning h4 { color: #ca8a04; }
        
        .slot-card { 
            background: #f8fafc; 
            border: 2px solid #e2e8f0; 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 15px; 
            cursor: pointer; 
            transition: all 0.3s; 
        }
        .slot-card:hover { 
            background: #e2e8f0; 
            border-color: #64748b; 
            transform: translateY(-1px); 
        }
        .slot-card.selected { 
            background: #dbeafe; 
            border-color: #3b82f6; 
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1); 
        }
        
        /* App info */
        .app-info {
            position: fixed;
            bottom: 10px;
            right: 10px;
            background: rgba(0,0,0,0.8);
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 1000;
        }
        
        /* Navigation month */
        .nav-bulan {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding: 15px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .nav-bulan h2 {
            flex: 1;
            text-align: center;
            margin: 0;
            color: #1f2937;
        }
        
        /* Event list */
        .daftar-event {
            display: grid;
            gap: 15px;
        }
        .item-event {
            background: white;
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid #3b82f6;
            transition: all 0.3s;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .item-event:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .item-event h3 {
            color: #1f2937;
            margin-bottom: 10px;
        }
        .item-event p {
            margin-bottom: 5px;
            color: #4b5563;
        }
        
        .flex {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        /* Responsive design */
        @media (max-width: 768px) {
            .container { padding: 10px; }
            .baris-form { grid-template-columns: 1fr; }
            .header { 
                flex-direction: column; 
                gap: 15px; 
                text-align: center; 
            }
            .nav { justify-content: center; }
            .nav-bulan {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
            .flex {
                flex-direction: column;
                align-items: stretch;
            }
            .item-event .flex {
                flex-direction: column;
            }
        }
        
        /* Loading animation */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3b82f6;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Status indicators */
        .status-indicator {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }
        
        /* Tooltips */
        [title] {
            position: relative;
            cursor: help;
        }
    </style>
</head>
<body>

<?php if (!Auth::isLoggedIn()): ?>
<div class="container">
    <div class="form-login card">
        <h2>🔐 Masuk <?= APP_NAME ?></h2>
        <?php if ($error): ?>
            <div class="alert alert-danger">❌ <?= $error ?></div>
        <?php endif; ?>
        <form method="POST" autocomplete="off">
            <div class="form-group">
                <label for="pin">PIN Akses: *</label>
                <input type="password" name="pin" id="pin" required placeholder="Masukkan PIN" autofocus autocomplete="current-password">
                <small>Masukkan PIN yang telah dikonfigurasi untuk mengakses sistem</small>
            </div>
            <button type="submit" name="login" class="btn">🚀 Masuk Sistem</button>
        </form>
        
        <div style="margin-top: 25px; padding-top: 20px; border-top: 1px solid #e5e7eb;">
            <h4 style="color: #374151; margin-bottom: 10px;">🎯 Fitur Sistem:</h4>
            <ul style="margin-left: 20px; color: #6b7280; font-size: 14px;">
                <li>Pencarian slot optimal otomatis</li>
                <li>Timeline event terintegrasi dan komprehensif</li>
                <li>Analisis beban kerja real-time</li>
                <li>Deteksi konflik jadwal pintar</li>
                <li>Kalender visual interaktif</li>
                <li>Backup dan restore data</li>
                <li>Export data ke berbagai format</li>
            </ul>
            
            <div style="margin-top: 15px; font-size: 12px; color: #9ca3af;">
                <strong>Sistem Keamanan:</strong> CSRF Protection, Session Management, Input Validation
            </div>
        </div>
    </div>
</div>
<?php else: ?>

<div class="container">
    <div class="header">
        <div>
            <h1>🏆 <?= APP_NAME ?></h1>
            <small>
                Versi <?= APP_VERSION ?> | 
                <?= count($events) ?> Event Aktif | 
                Timezone: <?= DEFAULT_TIMEZONE ?>
            </small>
        </div>
        <div style="text-align: right;">
            <div style="margin-bottom: 10px; font-size: 14px; opacity: 0.9;">
                👤 Sesi aktif sejak <?= date('H:i', $_SESSION['login_time']) ?>
            </div>
            <form method="POST" style="display: inline;">
                <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
                <button type="submit" name="logout" class="btn btn-danger" onclick="return confirm('Yakin ingin keluar?')">🚪 Keluar</button>
            </form>
        </div>
    </div>

    <div class="nav">
        <a href="?view=generator" class="<?= $currentView == 'generator' ? 'active' : '' ?>">🎯 Buat Event</a>
        <a href="?view=list" class="<?= $currentView == 'list' ? 'active' : '' ?>">📋 Daftar Event</a>
        <a href="?view=calendar" class="<?= $currentView == 'calendar' ? 'active' : '' ?>">📅 Kalender</a>
        <a href="?view=analysis" class="<?= $currentView == 'analysis' ? 'active' : '' ?>">📊 Analisis</a>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success">✅ <?= $success ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">❌ <?= $error ?></div>
    <?php endif; ?>

    <?php if (isset($warningConflicts)): ?>
        <div class="alert alert-warning">
            <div>
                <h3>⚠️ Peringatan Konflik Jadwal</h3>
                <?php foreach ($warningConflicts as $conflict): ?>
                    <p>• <?= $conflict['message'] ?></p>
                <?php endforeach; ?>
                
                <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #f59e0b;">
                    <p><strong>Tetap simpan event ini?</strong></p>
                    <form method="POST" style="display: inline;">
                        <?php foreach ($_POST as $key => $value): ?>
                            <input type="hidden" name="<?= htmlspecialchars($key) ?>" value="<?= htmlspecialchars($value) ?>">
                        <?php endforeach; ?>
                        <input type="hidden" name="paksa_simpan" value="1">
                        <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
                        <button type="submit" class="btn btn-warning">⚠️ Ya, Tetap Simpan</button>
                        <button type="button" onclick="window.location.reload()" class="btn btn-secondary">❌ Batal</button>
                    </form>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <?php if ($currentView == 'generator'): ?>
    <div class="card">
        <h2><?= $editEventId ? '✏️ Ubah Event' : '🎯 Buat Event Baru' ?></h2>
        
        <?php if (!$editEventId && !isset($_SESSION['slot_optimal']) && !isset($_SESSION['data_preview'])): ?>
        <div class="auto-info">
            <h4>🤖 Sistem Penjadwalan Otomatis</h4>
            <p>• <strong>Persiapan:</strong> <?= DEFAULT_PREPARATION_DAYS ?> hari sebelum pendaftaran dibuka</p>
            <p>• <strong>Durasi Optimal:</strong> <?= MIN_REGISTRATION_DAYS ?>-<?= MAX_REGISTRATION_DAYS ?> hari pendaftaran (berakhir Jumat)</p>
            <p>• <strong>Pelaksanaan:</strong> Selalu di hari Sabtu setelah pendaftaran berakhir</p>
            <p>• <strong>Kapasitas:</strong> Maksimal <?= MAX_EVENTS_PER_MONTH ?> event per bulan</p>
            <p>• <strong>Jarak Aman:</strong> Minimal <?= MIN_DAYS_BETWEEN_EVENTS ?> hari antar pelaksanaan</p>
            <p>• <strong>Timeline Otomatis:</strong> Semua fase dari pendaftaran hingga selesai dihitung otomatis</p>
        </div>
        <?php endif; ?>
        
        <form method="POST" id="formEvent" autocomplete="off">
            <input type="hidden" name="event_id" value="<?= htmlspecialchars($editEventId) ?>">
            <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
            
            <div class="baris-form">
                <div class="form-group">
                    <label for="namaLomba">Nama Lomba: *</label>
                    <input type="text" name="namaLomba" id="namaLomba" required 
                           value="<?= htmlspecialchars($eventForEdit['nama_lomba'] ?? $_SESSION['data_form']['namaLomba'] ?? $_SESSION['data_preview']['namaLomba'] ?? '') ?>" 
                           placeholder="Contoh: Olimpiade Sains Nasional"
                           autocomplete="off">
                </div>
                
                <div class="form-group">
                    <label for="singkatanLomba">Singkatan Lomba: *</label>
                    <input type="text" name="singkatanLomba" id="singkatanLomba" required 
                           value="<?= htmlspecialchars($eventForEdit['singkatan_lomba'] ?? $_SESSION['data_form']['singkatanLomba'] ?? $_SESSION['data_preview']['singkatanLomba'] ?? '') ?>" 
                           placeholder="Contoh: OSN"
                           autocomplete="off">
                </div>
            </div>
            
            <div class="form-group">
                <label for="penyelenggara">Penyelenggara: *</label>
                <input type="text" name="penyelenggara" id="penyelenggara" required 
                       value="<?= htmlspecialchars($eventForEdit['penyelenggara'] ?? $_SESSION['data_form']['penyelenggara'] ?? $_SESSION['data_preview']['penyelenggara'] ?? '') ?>" 
                       placeholder="Contoh: Kementerian Pendidikan dan Kebudayaan"
                       autocomplete="organization">
            </div>
            
            <div class="form-group">
                <label>Pengaturan Susulan:</label>
                <div class="grup-radio">
                    <label>
                        <input type="radio" name="susulan" value="ya" 
                               <?= ($eventForEdit['susulan'] ?? $_SESSION['data_form']['susulan'] ?? $_SESSION['data_preview']['susulan'] ?? '') == 'ya' ? 'checked' : '' ?>>
                        Ada Susulan
                    </label>
                    <label>
                        <input type="radio" name="susulan" value="tidak" 
                               <?= ($eventForEdit['susulan'] ?? $_SESSION['data_form']['susulan'] ?? $_SESSION['data_preview']['susulan'] ?? 'tidak') == 'tidak' ? 'checked' : '' ?>>
                        Tidak Ada Susulan
                    </label>
                    <select name="hariSusulan" style="width: auto; margin-left: 10px;">
                        <option value="minggu" <?= ($eventForEdit['hari_susulan'] ?? $_SESSION['data_form']['hariSusulan'] ?? $_SESSION['data_preview']['hariSusulan'] ?? '') == 'minggu' ? 'selected' : '' ?>>Hari Minggu</option>
                        <option value="senin" <?= ($eventForEdit['hari_susulan'] ?? $_SESSION['data_form']['hariSusulan'] ?? $_SESSION['data_preview']['hariSusulan'] ?? '') == 'senin' ? 'selected' : '' ?>>Hari Senin</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label>Jenis Event:</label>
                <div class="grup-radio">
                    <label>
                        <input type="radio" name="babakFinal" value="ya" 
                               <?= ($eventForEdit['babak_final'] ?? $_SESSION['data_form']['babakFinal'] ?? $_SESSION['data_preview']['babakFinal'] ?? '') == 'ya' ? 'checked' : '' ?>>
                        Event dengan Babak Final
                    </label>
                    <label>
                        <input type="radio" name="babakFinal" value="tidak" 
                               <?= ($eventForEdit['babak_final'] ?? $_SESSION['data_form']['babakFinal'] ?? $_SESSION['data_preview']['babakFinal'] ?? 'tidak') == 'tidak' ? 'checked' : '' ?>>
                        Event Reguler
                    </label>
                </div>
                <small>Event dengan babak final memiliki timeline yang berbeda untuk fase penghargaan</small>
            </div>
            
            <?php if ($editEventId): ?>
            <div class="baris-form">
                <div class="form-group">
                    <label for="tanggalMulai">Tanggal Mulai Pendaftaran: *</label>
                    <input type="date" name="tanggalMulai" id="tanggalMulai" 
                           value="<?= $eventForEdit['mulai_pendaftaran'] ?? '' ?>" required>
                    <small>Ubah tanggal dengan hati-hati untuk menghindari konflik dengan event lain</small>
                </div>
                
                <div class="form-group">
                    <label for="durasiPendaftaran">Durasi Pendaftaran (Hari): *</label>
                    <input type="number" name="durasiPendaftaran" id="durasiPendaftaran" 
                           value="<?= $eventForEdit['durasi_pendaftaran'] ?? 30 ?>" 
                           min="<?= MIN_REGISTRATION_DAYS ?>" max="<?= MAX_REGISTRATION_DAYS ?>" required>
                    <small><?= MIN_REGISTRATION_DAYS ?>-<?= MAX_REGISTRATION_DAYS ?> hari adalah durasi optimal untuk pendaftaran</small>
                </div>
            </div>
            <button type="submit" name="generate_preview" class="btn">📋 Buat Preview Timeline</button>
            <a href="?view=generator" class="btn btn-secondary">❌ Batal Edit</a>
            <?php else: ?>
            <button type="submit" name="cari_slot_otomatis" class="btn btn-success">🔍 Cari Slot Otomatis</button>
            <a href="?view=list" class="btn btn-secondary">📋 Lihat Event Existing</a>
            <?php endif; ?>
        </form>
        
        <?php if (isset($_SESSION['slot_optimal']) && !empty($_SESSION['slot_optimal'])): ?>
        <div class="card" style="margin-top: 25px; border: 2px solid #16a34a;">
            <h3>🎯 Slot Optimal Ditemukan</h3>
            <p>Berdasarkan <?= count($events) ?> event existing, sistem menemukan <?= count($_SESSION['slot_optimal']) ?> slot terbaik:</p>
            
            <form method="POST" id="formSlot">
                <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
                <?php foreach ($_SESSION['slot_optimal'] as $index => $slot): ?>
                <div class="slot-card" onclick="selectSlot(<?= $index ?>)" id="card_<?= $index ?>">
                    <div style="display: flex; align-items: center;">
                        <input type="radio" name="slot_index" value="<?= $index ?>" id="slot_<?= $index ?>" style="margin-right: 15px;" required>
                        <div style="flex: 1;">
                            <strong>Opsi <?= $index + 1 ?>:</strong> Pelaksanaan <strong><?= DateHelper::formatIndonesian($slot['pelaksanaan']) ?></strong><br>
                            <div style="margin-top: 8px; color: #6b7280; font-size: 14px;">
                                📅 <strong>Pendaftaran:</strong> <?= DateHelper::formatDateRange($slot['mulai_pendaftaran'], $slot['selesai_pendaftaran']) ?> (<?= $slot['durasi_pendaftaran'] ?> hari)<br>
                                📊 <strong>Beban Bulan:</strong> <?= $slot['event_bulan_ini'] ?>/<?= MAX_EVENTS_PER_MONTH ?> event di <?= $slot['bulan'] ?><br>
                                ⭐ <strong>Skor Prioritas:</strong> <?= $slot['skor'] ?> poin
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
                
                <div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #e5e7eb;">
                    <button type="submit" name="pilih_slot" class="btn btn-success">✅ Pilih Slot & Buat Preview</button>
                    <a href="?view=generator" class="btn btn-secondary">🔄 Cari Ulang</a>
                    <a href="?view=analysis" class="btn btn-secondary">📊 Lihat Analisis Beban</a>
                </div>
            </form>
        </div>
        <?php elseif (isset($_SESSION['slot_optimal']) && empty($_SESSION['slot_optimal'])): ?>
        <div class="alert alert-warning">
            <div>
                <h4>😔 Tidak Ada Slot Optimal Tersedia</h4>
                <p>Sistem tidak menemukan slot yang memenuhi kriteria optimal dalam 90 hari ke depan:</p>
                <ul style="margin: 10px 0 15px 20px;">
                    <li>Maksimal <?= MAX_EVENTS_PER_MONTH ?> event per bulan (distribusi merata)</li>
                    <li>Jarak minimal <?= MIN_DAYS_BETWEEN_EVENTS ?> hari antar pelaksanaan</li>
                    <li>Pelaksanaan di hari Sabtu setelah pendaftaran berakhir Jumat</li>
                    <li>Durasi pendaftaran optimal <?= MIN_REGISTRATION_DAYS ?>-<?= MAX_REGISTRATION_DAYS ?> hari</li>
                </ul>
                
                <div style="background: white; padding: 15px; border-radius: 6px; margin-top: 15px;">
                    <h5 style="color: #0c4a6e; margin-bottom: 10px;">💡 Rekomendasi Solusi:</h5>
                    <ol style="margin-left: 20px; color: #374151;">
                        <li><strong>Review Event Existing:</strong> Periksa apakah ada event yang bisa digeser atau dibatalkan</li>
                        <li><strong>Perpanjang Periode:</strong> Coba cari slot untuk bulan berikutnya</li>
                        <li><strong>Fleksibilitas Kriteria:</strong> Pertimbangkan durasi pendaftaran yang berbeda</li>
                        <li><strong>Analisis Mendalam:</strong> Gunakan menu Analisis untuk melihat distribusi beban</li>
                    </ol>
                </div>
                
                <div style="margin-top: 15px;">
                    <a href="?view=generator" class="btn">🔄 Coba Lagi</a>
                    <a href="?view=list" class="btn btn-secondary">📋 Review Event Existing</a>
                    <a href="?view=analysis" class="btn btn-secondary">📊 Analisis Beban Kerja</a>
                </div>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['data_preview'])): 
            $preview = $_SESSION['data_preview'];
        ?>
        <div class="preview-card">
            <h3 style="color: #1d4ed8; margin-bottom: 20px;">📅 Preview Timeline Event Lengkap</h3>
            
            <?php if (isset($preview['slot_terpilih'])): ?>
            <div style="background: rgba(16, 185, 129, 0.15); padding: 15px; border-radius: 6px; margin-bottom: 20px; border: 1px solid #16a34a;">
                <strong>🎯 Slot Terpilih:</strong> Opsi dengan <?= $preview['slot_terpilih']['event_bulan_ini'] ?> event di <?= $preview['slot_terpilih']['bulan'] ?> 
                (Skor: <?= $preview['slot_terpilih']['skor'] ?> poin)
            </div>
            <?php endif; ?>
            
            <div class="timeline-section">
                <h4>ℹ️ Informasi Event</h4>
                <p><strong>Nama Lomba:</strong> <?= htmlspecialchars($preview['namaLomba']) ?> (<?= htmlspecialchars($preview['singkatanLomba']) ?>)</p>
                <p><strong>Penyelenggara:</strong> <?= htmlspecialchars($preview['penyelenggara']) ?></p>
                <p><strong>Jenis Event:</strong> <?= $preview['babak_final'] == 'ya' ? 'Event dengan Babak Final' : 'Event Reguler' ?></p>
            </div>
            
            <div class="timeline-section hijau">
                <h4>📝 Tahap Pendaftaran</h4>
                <p><strong>Periode Pendaftaran:</strong> <?= DateHelper::formatDateRange($preview['mulai_pendaftaran'], $preview['selesai_pendaftaran']) ?></p>
                <small style="color: #15803d;">Durasi: <?= $preview['durasi_pendaftaran'] ?> hari (berakhir hari <?= (new DateTime($preview['selesai_pendaftaran']))->format('l') ?>)</small>
            </div>
            
            <div class="timeline-section orange">
                <h4>🏆 Tahap Pelaksanaan</h4>
                <p><strong><?= $preview['babak_final'] == 'ya' ? 'Babak Penyisihan' : 'Pelaksanaan Utama' ?>:</strong> <?= DateHelper::formatIndonesian($preview['pelaksanaan']) ?></p>
                <?php if ($preview['susulan'] == 'ya'): ?>
                    <p><strong>Hari Susulan:</strong> <?= DateHelper::formatIndonesian($preview['tanggal_susulan']) ?> (<?= ucfirst($preview['hari_susulan']) ?>)</p>
                <?php endif; ?>
                <p><strong>Pengumuman Hasil:</strong> <?= DateHelper::formatIndonesian($preview['pengumuman']) ?></p>
                <small style="color: #c2410c;">Pengumuman otomatis <?= TIMELINE_CONFIG['announcement_delay_days'] ?> hari setelah pelaksanaan<?= $preview['susulan'] == 'ya' ? '/susulan' : '' ?></small>
            </div>
            
            <?php if ($preview['babak_final'] == 'ya'): ?>
            <div class="timeline-section ungu">
                <h4>🥇 Tahap Babak Final</h4>
                <p><strong>Daftar Ulang Finalis:</strong> <?= DateHelper::formatDateRange($preview['daftar_ulang_mulai'], $preview['daftar_ulang_selesai']) ?></p>
                <p><strong>Pelaksanaan Final:</strong> <?= DateHelper::formatIndonesian($preview['tanggal_final']) ?></p>
                <p><strong>Pengumuman Pemenang:</strong> <?= DateHelper::formatIndonesian($preview['pengumuman_final']) ?></p>
                <small style="color: #6d28d9;">Final dilaksanakan hari Sabtu setelah periode daftar ulang berakhir</small>
            </div>
            
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan & Penyelesaian</h4>
                <p><strong>Akses Penghargaan:</strong> <?= DateHelper::formatDateRange($preview['akses_penghargaan_mulai'], $preview['akses_penghargaan_selesai']) ?></p>
                <p><strong>Sapu Bersih & Penutupan:</strong> <?= DateHelper::formatIndonesian($preview['sapu_bersih']) ?></p>
                <small style="color: #ca8a04;">Fase terakhir untuk menyelesaikan semua administrasi event</small>
            </div>
            <?php else: ?>
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan & Administrasi</h4>
                <p><strong>Klaim Penghargaan Gelombang 1:</strong> <?= DateHelper::formatDateRange($preview['klaim_gel1_mulai'], $preview['klaim_gel1_selesai']) ?></p>
                <p><strong>Form Umpan Balik:</strong> <?= DateHelper::formatIndonesian($preview['form_umpan_balik']) ?></p>
                <p><strong>Jeda Administrasi 1:</strong> <?= DateHelper::formatDateRange($preview['jeda1_mulai'], $preview['jeda1_selesai']) ?></p>
                <p><strong>Klaim Penghargaan Gelombang 2:</strong> <?= DateHelper::formatDateRange($preview['klaim_gel2_mulai'], $preview['klaim_gel2_selesai']) ?></p>
                <p><strong>Jeda Administrasi 2:</strong> <?= DateHelper::formatDateRange($preview['jeda2_mulai'], $preview['jeda2_selesai']) ?></p>
                <p><strong>Sapu Bersih & Penutupan:</strong> <?= DateHelper::formatIndonesian($preview['sapu_bersih']) ?></p>
                <small style="color: #ca8a04;">Sistem dua gelombang untuk memastikan semua peserta mendapat penghargaan</small>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 25px; padding-top: 20px; border-top: 2px solid #1d4ed8; background: rgba(255, 255, 255, 0.8); border-radius: 6px; padding: 20px;">
                <div style="margin-bottom: 15px;">
                    <strong>📊 Ringkasan Timeline:</strong><br>
                    <small style="color: #6b7280;">
                        Total durasi event: <?= round((strtotime($preview['sapu_bersih']) - strtotime($preview['mulai_pendaftaran'])) / (60 * 60 * 24)) ?> hari |
                        Fase aktif: <?= $preview['babak_final'] == 'ya' ? '7 tahap' : '10 tahap' ?> |
                        Pelaksanaan di: <?= (new DateTime($preview['pelaksanaan']))->format('l, d F Y') ?>
                    </small>
                </div>
                
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="event_id" value="<?= htmlspecialchars($editEventId) ?>">
                    <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
                    <button type="submit" name="simpan_event" class="btn btn-success" onclick="return confirm('Yakin ingin menyimpan event dengan timeline ini?')">💾 Simpan Event</button>
                </form>
                <button type="button" onclick="window.location.href='?view=generator<?= $editEventId ? '&edit=' . $editEventId : '' ?>'" class="btn btn-secondary">✏️ Edit Konfigurasi</button>
                <a href="?view=calendar&month=<?= substr($preview['pelaksanaan'], 0, 7) ?>" class="btn btn-secondary">📅 Lihat di Kalender</a>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php elseif ($currentView == 'list'): ?>
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
            <h2>📋 Daftar Event (<?= count($events) ?>)</h2>
            <div>
                <a href="?view=generator" class="btn btn-success">🎯 Tambah Event Baru</a>
                <?php if (!empty($events)): ?>
                <a href="backup.php?action=export&key=<?= LOGIN_PIN ?>" class="btn btn-secondary">📊 Export CSV</a>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (empty($events)): ?>
            <div style="text-align: center; color: #6b7280; padding: 60px 20px;">
                <div style="font-size: 48px; margin-bottom: 20px;">📋</div>
                <h3 style="color: #374151; margin-bottom: 10px;">Belum Ada Event</h3>
                <p style="margin-bottom: 25px;">Mulai buat event pertama Anda dengan sistem penjadwalan otomatis</p>
                <a href="?view=generator" class="btn btn-success">🎯 Buat Event Pertama</a>
            </div>
        <?php else: ?>
            <!-- Event statistics -->
            <div style="background: #f8fafc; padding: 15px; border-radius: 6px; margin-bottom: 20px; border: 1px solid #e2e8f0;">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; text-align: center;">
                    <?php
                    $statusCount = ['upcoming' => 0, 'active' => 0, 'pre_execution' => 0, 'completed' => 0];
                    $today = date('Y-m-d');
                    foreach ($events as $event) {
                        if ($today < $event['mulai_pendaftaran']) $statusCount['upcoming']++;
                        elseif ($today <= $event['selesai_pendaftaran']) $statusCount['active']++;
                        elseif ($today <= $event['pelaksanaan']) $statusCount['pre_execution']++;
                        else $statusCount['completed']++;
                    }
                    ?>
                    <div><strong><?= $statusCount['upcoming'] ?></strong><br><small>Akan Datang</small></div>
                    <div><strong><?= $statusCount['active'] ?></strong><br><small>Pendaftaran Aktif</small></div>
                    <div><strong><?= $statusCount['pre_execution'] ?></strong><br><small>Menuju Pelaksanaan</small></div>
                    <div><strong><?= $statusCount['completed'] ?></strong><br><small>Selesai</small></div>
                </div>
            </div>

            <div class="daftar-event">
                <?php foreach ($events as $event): ?>
                <div class="item-event">
                    <div class="flex">
                        <div style="flex: 1;">
                            <h3><?= htmlspecialchars($event['nama_lomba'] ?? $event['namaLomba']) ?> (<?= htmlspecialchars($event['singkatan_lomba'] ?? $event['singkatanLomba']) ?>)</h3>
                            <p><strong>📍 Penyelenggara:</strong> <?= htmlspecialchars($event['penyelenggara']) ?></p>
                            <p><strong>🏆 Pelaksanaan:</strong> <?= DateHelper::formatIndonesian($event['pelaksanaan']) ?></p>
                            <p><strong>📝 Pendaftaran:</strong> <?= DateHelper::formatDateRange($event['mulai_pendaftaran'], $event['selesai_pendaftaran']) ?></p>
                            <p><strong>📊 Status:</strong> 
                                <?php
                                $today = date('Y-m-d');
                                if ($today < $event['mulai_pendaftaran']) {
                                    echo '<span style="color: #3b82f6; background: #dbeafe; padding: 2px 8px; border-radius: 12px; font-size: 12px;">🔵 Akan Datang</span>';
                                } elseif ($today <= $event['selesai_pendaftaran']) {
                                    echo '<span style="color: #16a34a; background: #dcfce7; padding: 2px 8px; border-radius: 12px; font-size: 12px;">🟢 Pendaftaran Aktif</span>';
                                } elseif ($today <= $event['pelaksanaan']) {
                                    echo '<span style="color: #ea580c; background: #fed7aa; padding: 2px 8px; border-radius: 12px; font-size: 12px;">🟡 Menuju Pelaksanaan</span>';
                                } else {
                                    echo '<span style="color: #6b7280; background: #f3f4f6; padding: 2px 8px; border-radius: 12px; font-size: 12px;">⚪ Selesai</span>';
                                }
                                ?>
                            </p>
                            <?php if (isset($event['dibuat_pada'])): ?>
                            <p><small style="color: #9ca3af;">Dibuat: <?= date('d/m/Y H:i', strtotime($event['dibuat_pada'])) ?></small></p>
                            <?php endif; ?>
                        </div>
                        <div style="text-align: right;">
                            <div style="margin-bottom: 10px;">
                                <a href="?view=list&detail=<?= $event['id'] ?>" class="btn btn-secondary" title="Lihat detail timeline lengkap">👁️ Detail</a>
                            </div>
                            <div style="margin-bottom: 10px;">
                                <a href="?view=generator&edit=<?= $event['id'] ?>" class="btn" title="Edit konfigurasi event">✏️ Edit</a>
                            </div>
                            <div>
                                <form method="POST" style="display: inline;" onsubmit="return confirm('Yakin ingin menghapus event \'<?= htmlspecialchars($event['nama_lomba'] ?? $event['namaLomba']) ?>\'?\n\nData yang dihapus tidak dapat dikembalikan.')">
                                    <input type="hidden" name="event_id" value="<?= $event['id'] ?>">
                                    <input type="hidden" name="csrf_token" value="<?= Config::generateCSRFToken() ?>">
                                    <button type="submit" name="hapus_event" class="btn btn-danger" title="Hapus event permanen">🗑️ Hapus</button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- Detail Event Modal-like Display -->
    <?php if ($detailEventId && $detailEvent): ?>
    <div class="card" style="border: 2px solid #3b82f6;">
        <h3>📋 Detail Timeline: <?= htmlspecialchars($detailEvent['nama_lomba'] ?? $detailEvent['namaLomba']) ?></h3>
        <div style="background: #f8fafc; padding: 25px; border-radius: 8px; margin-top: 15px;">
            <div class="timeline-section">
                <h4>ℹ️ Informasi Event</h4>
                <p><strong>Nama Lengkap:</strong> <?= htmlspecialchars($detailEvent['nama_lomba'] ?? $detailEvent['namaLomba']) ?></p>
                <p><strong>Singkatan:</strong> <?= htmlspecialchars($detailEvent['singkatan_lomba'] ?? $detailEvent['singkatanLomba']) ?></p>
                <p><strong>Penyelenggara:</strong> <?= htmlspecialchars($detailEvent['penyelenggara']) ?></p>
                <p><strong>Jenis Event:</strong> <?= ($detailEvent['babak_final'] ?? $detailEvent['babakFinal'] ?? 'tidak') == 'ya' ? 'Event dengan Babak Final' : 'Event Reguler' ?></p>
                <?php if (isset($detailEvent['dibuat_pada'])): ?>
                <p><strong>Dibuat:</strong> <?= DateHelper::formatIndonesian(substr($detailEvent['dibuat_pada'], 0, 10)) ?> pukul <?= substr($detailEvent['dibuat_pada'], 11, 5) ?></p>
                <?php endif; ?>
            </div>
            
            <div class="timeline-section hijau">
                <h4>📝 Tahap Pendaftaran</h4>
                <p><strong>Periode:</strong> <?= DateHelper::formatDateRange($detailEvent['mulai_pendaftaran'], $detailEvent['selesai_pendaftaran']) ?></p>
                <small style="color: #15803d;">Durasi: <?= $detailEvent['durasi_pendaftaran'] ?? 'N/A' ?> hari</small>
            </div>
            
            <div class="timeline-section orange">
                <h4>🏆 Tahap Pelaksanaan</h4>
                <p><strong><?= ($detailEvent['babak_final'] ?? $detailEvent['babakFinal'] ?? 'tidak') == 'ya' ? 'Babak Penyisihan' : 'Pelaksanaan Utama' ?>:</strong> <?= DateHelper::formatIndonesian($detailEvent['pelaksanaan']) ?></p>
                <?php if (($detailEvent['susulan'] ?? 'tidak') == 'ya' && isset($detailEvent['tanggal_susulan'])): ?>
                    <p><strong>Hari Susulan:</strong> <?= DateHelper::formatIndonesian($detailEvent['tanggal_susulan']) ?></p>
                <?php endif; ?>
                <p><strong>Pengumuman:</strong> <?= DateHelper::formatIndonesian($detailEvent['pengumuman']) ?></p>
            </div>
            
            <?php if (($detailEvent['babak_final'] ?? $detailEvent['babakFinal'] ?? 'tidak') == 'ya' && isset($detailEvent['tanggal_final'])): ?>
            <div class="timeline-section ungu">
                <h4>🥇 Tahap Babak Final</h4>
                <p><strong>Daftar Ulang Finalis:</strong> <?= DateHelper::formatDateRange($detailEvent['daftar_ulang_mulai'], $detailEvent['daftar_ulang_selesai']) ?></p>
                <p><strong>Pelaksanaan Final:</strong> <?= DateHelper::formatIndonesian($detailEvent['tanggal_final']) ?></p>
                <p><strong>Pengumuman Pemenang:</strong> <?= DateHelper::formatIndonesian($detailEvent['pengumuman_final']) ?></p>
            </div>
            
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan</h4>
                <p><strong>Akses Penghargaan:</strong> <?= DateHelper::formatDateRange($detailEvent['akses_penghargaan_mulai'], $detailEvent['akses_penghargaan_selesai']) ?></p>
                <p><strong>Sapu Bersih:</strong> <?= DateHelper::formatIndonesian($detailEvent['sapu_bersih']) ?></p>
            </div>
            <?php elseif (isset($detailEvent['klaim_gel1_mulai'])): ?>
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan & Administrasi</h4>
                <p><strong>Klaim Gelombang 1:</strong> <?= DateHelper::formatDateRange($detailEvent['klaim_gel1_mulai'], $detailEvent['klaim_gel1_selesai']) ?></p>
                <p><strong>Form Umpan Balik:</strong> <?= DateHelper::formatIndonesian($detailEvent['form_umpan_balik']) ?></p>
                <p><strong>Jeda Administrasi 1:</strong> <?= DateHelper::formatDateRange($detailEvent['jeda1_mulai'], $detailEvent['jeda1_selesai']) ?></p>
                <p><strong>Klaim Gelombang 2:</strong> <?= DateHelper::formatDateRange($detailEvent['klaim_gel2_mulai'], $detailEvent['klaim_gel2_selesai']) ?></p>
                <p><strong>Jeda Administrasi 2:</strong> <?= DateHelper::formatDateRange($detailEvent['jeda2_mulai'], $detailEvent['jeda2_selesai']) ?></p>
                <p><strong>Sapu Bersih:</strong> <?= DateHelper::formatIndonesian($detailEvent['sapu_bersih']) ?></p>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 25px; padding-top: 20px; border-top: 1px solid #d1d5db;">
                <a href="?view=list" class="btn btn-secondary">← Kembali ke Daftar</a>
                <a href="?view=generator&edit=<?= $detailEvent['id'] ?>" class="btn">✏️ Edit Event</a>
                <a href="?view=calendar&month=<?= substr($detailEvent['pelaksanaan'], 0, 7) ?>" class="btn btn-secondary">📅 Lihat di Kalender</a>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <?php elseif ($currentView == 'calendar'): ?>
    <div class="card">
        <?php 
        $calendarNav = CalendarView::getNavigation($currentMonth);
        $calendarStats = CalendarView::getStatistics($events, $currentMonth);
        ?>
        
        <div class="nav-bulan">
            <a href="?view=calendar&month=<?= $calendarNav['prev_month'] ?>" class="btn">← Bulan Sebelumnya</a>
            <h2>📅 Kalender Event - <?= $calendarNav['current_month_name'] ?></h2>
            <a href="?view=calendar&month=<?= $calendarNav['next_month'] ?>" class="btn">Bulan Berikutnya →</a>
        </div>
        
        <!-- Calendar Statistics -->
        <div class="calendar-stats">
            <h4>📊 Statistik Bulan <?= $calendarNav['current_month_name'] ?></h4>
            <div class="stats-grid">
                <div class="stat-item">
                    <div class="stat-value"><?= $calendarStats['total_events'] ?>/<?= $calendarStats['max_events_per_month'] ?></div>
                    <div class="stat-label">Event Bulan Ini</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value" style="color: <?= $calendarStats['capacity_percentage'] >= 100 ? '#dc2626' : ($calendarStats['capacity_percentage'] >= 67 ? '#ea580c' : '#16a34a') ?>"><?= $calendarStats['capacity_percentage'] ?>%</div>
                    <div class="stat-label">Kapasitas Terpakai</div>
                </div>
                <div class="stat-item">
                    <div class="stat-value"><?= count(array_filter($events, function($e) { return date('Y-m') == substr($e['pelaksanaan'], 0, 7); })) ?></div>
                    <div class="stat-label">Event Hari Ini</div>
                </div>
            </div>
            
            <?php if (!empty($calendarStats['events_this_month'])): ?>
            <div class="month-events-list">
                <h5 style="margin-bottom: 10px;">📋 Event Bulan Ini:</h5>
                <?php foreach ($calendarStats['events_this_month'] as $monthEvent): ?>
                <div class="month-event-item">
                    <div class="event-info">
                        <div class="event-name"><?= htmlspecialchars($monthEvent['name']) ?> (<?= htmlspecialchars($monthEvent['short_name']) ?>)</div>
                        <div class="event-date"><?= DateHelper::formatIndonesian($monthEvent['execution_date']) ?></div>
                    </div>
                    <div class="event-status" style="background: <?= $monthEvent['status']['color'] ?>20; color: <?= $monthEvent['status']['color'] ?>;">
                        <?= $monthEvent['status']['label'] ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Calendar Display -->
        <?= CalendarView::generateCalendar($events, $currentMonth) ?>
        
        <!-- Quick Actions -->
        <div style="margin-top: 20px; text-align: center; padding: 15px; background: #f8fafc; border-radius: 6px;">
            <a href="?view=generator" class="btn btn-success">🎯 Tambah Event</a>
            <a href="?view=analysis&month=<?= $currentMonth ?>" class="btn btn-secondary">📊 Analisis Bulan Ini</a>
            <a href="?view=calendar&month=<?= date('Y-m') ?>" class="btn btn-secondary">📅 Bulan Ini</a>
        </div>
    </div>

    <?php elseif ($currentView == 'analysis'): ?>
    <div class="card">
        <div class="nav-bulan">
            <a href="?view=analysis&month=<?= date('Y-m', strtotime($currentMonth . '-01 -1 month')) ?>" class="btn">← Bulan Sebelumnya</a>
            <h2>📊 Analisis Beban Kerja - <?= (new DateTime($currentMonth . '-01'))->format('F Y') ?></h2>
            <a href="?view=analysis&month=<?= date('Y-m', strtotime($currentMonth . '-01 +1 month')) ?>" class="btn">Bulan Berikutnya →</a>
        </div>
        
        <?= AnalysisView::generateWorkloadAnalysis($events, $currentMonth) ?>
        
        <!-- Quick Actions -->
        <div style="margin-top: 25px; text-align: center; padding: 20px; background: #f8fafc; border-radius: 8px; border: 1px solid #e2e8f0;">
            <h4 style="margin-bottom: 15px; color: #374151;">🛠️ Aksi Cepat</h4>
            <a href="?view=generator" class="btn btn-success">🎯 Tambah Event Baru</a>
            <a href="?view=calendar&month=<?= $currentMonth ?>" class="btn btn-secondary">📅 Lihat Kalender</a>
            <a href="?view=analysis&month=<?= date('Y-m') ?>" class="btn btn-secondary">📊 Bulan Ini</a>
            <a href="backup.php?action=create&key=<?= LOGIN_PIN ?>" class="btn btn-secondary" onclick="return confirm('Buat backup data sebelum melakukan perubahan besar?')">💾 Backup Data</a>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- App Info Footer -->
<div class="app-info">
    <?= APP_NAME ?> v<?= APP_VERSION ?> | PHP <?= PHP_VERSION ?>
</div>

<!-- JavaScript -->
<script>
// Slot selection functionality
function selectSlot(index) {
    // Clear all selections
    document.querySelectorAll('.slot-card').forEach(card => {
        card.classList.remove('selected');
    });
    
    // Select the clicked slot
    const radioButton = document.getElementById('slot_' + index);
    const card = document.getElementById('card_' + index);
    
    if (radioButton && card) {
        radioButton.checked = true;
        card.classList.add('selected');
    }
}

// DOM Ready functionality
document.addEventListener('DOMContentLoaded', function() {
    // Auto-select first slot if none selected
    const firstSlot = document.getElementById('slot_0');
    if (firstSlot && !document.querySelector('input[name="slot_index"]:checked')) {
        selectSlot(0);
    }
    
    // Enhanced form validation
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('input[required], select[required]');
            let hasError = false;
            let firstErrorField = null;
            
            requiredFields.forEach(field => {
                const value = field.type === 'radio' ? 
                    form.querySelector(`input[name="${field.name}"]:checked`) !== null :
                    field.value.trim() !== '';
                    
                if (!value) {
                    field.style.borderColor = '#dc2626';
                    field.style.backgroundColor = '#fef2f2';
                    hasError = true;
                    if (!firstErrorField) firstErrorField = field;
                } else {
                    field.style.borderColor = '#d1d5db';
                    field.style.backgroundColor = 'white';
                }
            });
            
            if (hasError) {
                e.preventDefault();
                alert('❌ Mohon lengkapi semua field yang diperlukan!');
                if (firstErrorField) {
                    firstErrorField.focus();
                    firstErrorField.scrollIntoView({ behavior: 'smooth', block: 'center' });
                }
                return false;
            }
            
            // Show loading for heavy operations
            const heavyActions = ['cari_slot_otomatis', 'generate_preview', 'simpan_event'];
            const isHeavyAction = heavyActions.some(action => form.querySelector(`[name="${action}"]`));
            
            if (isHeavyAction) {
                const submitBtn = form.querySelector('button[type="submit"]');
                if (submitBtn) {
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<span class="loading"></span> Memproses...';
                    submitBtn.disabled = true;
                    
                    // Re-enable after timeout (fallback)
                    setTimeout(() => {
                        submitBtn.innerHTML = originalText;
                        submitBtn.disabled = false;
                    }, 10000);
                }
            }
        });
    });
    
    // Auto-save form data to prevent loss
    const formInputs = document.querySelectorAll('input[type="text"], input[type="number"], select, textarea');
    formInputs.forEach(input => {
        // Load saved data
        const savedValue = localStorage.getItem('form_' + input.name);
        if (savedValue && !input.value) {
            input.value = savedValue;
        }
        
        // Save on change
        input.addEventListener('change', function() {
            localStorage.setItem('form_' + this.name, this.value);
        });
    });
    
    // Clear saved form data on successful submission
    const successAlert = document.querySelector('.alert-success');
    if (successAlert) {
        formInputs.forEach(input => {
            localStorage.removeItem('form_' + input.name);
        });
    }
    
    // Enhanced tooltips for better UX
    const tooltipElements = document.querySelectorAll('[title]');
    tooltipElements.forEach(element => {
        element.addEventListener('mouseenter', function() {
            this.style.position = 'relative';
        });
    });
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        // Ctrl/Cmd + S to trigger save (prevent default browser save)
        if ((e.ctrlKey || e.metaKey) && e.key === 's') {
            e.preventDefault();
            const saveButton = document.querySelector('button[name="simpan_event"]');
            if (saveButton && !saveButton.disabled) {
                saveButton.click();
            }
        }
        
        // Ctrl/Cmd + N for new event
        if ((e.ctrlKey || e.metaKey) && e.key === 'n') {
            e.preventDefault();
            window.location.href = '?view=generator';
        }
        
        // Escape to cancel/go back
        if (e.key === 'Escape') {
            const cancelButton = document.querySelector('.btn-secondary');
            if (cancelButton) {
                cancelButton.click();
            }
        }
    });
    
    // Calendar interaction enhancements
    const calendarDays = document.querySelectorAll('.kalender-hari');
    calendarDays.forEach(day => {
        day.addEventListener('click', function() {
            const eventDots = this.querySelectorAll('.event-dot');
            if (eventDots.length > 0) {
                // Show event details in a simple way
                const eventTitles = Array.from(eventDots).map(dot => dot.getAttribute('title')).join('\n• ');
                if (eventTitles) {
                    alert('📅 Event pada tanggal ini:\n\n• ' + eventTitles);
                }
            }
        });
    });
    
    // Auto-refresh for active registration events
    const hasActiveEvents = document.querySelector('.alert-success, .alert-info');
    if (hasActiveEvents && window.location.search.includes('view=list')) {
        // Refresh every 5 minutes to update status
        setTimeout(() => {
            window.location.reload();
        }, 300000);
    }
    
    // Smooth scrolling for internal links
    const internalLinks = document.querySelectorAll('a[href^="#"]');
    internalLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({ behavior: 'smooth' });
            }
        });
    });
    
    // Form field enhancements
    const dateInputs = document.querySelectorAll('input[type="date"]');
    dateInputs.forEach(input => {
        // Set minimum date to today for new events
        if (!input.value) {
            input.min = new Date().toISOString().split('T')[0];
        }
        
        // Validate date selection
        input.addEventListener('change', function() {
            const selectedDate = new Date(this.value);
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            
            if (selectedDate < today) {
                alert('⚠️ Tanggal tidak boleh di masa lalu');
                this.value = '';
                this.focus();
            }
        });
    });
    
    // Duration input validation
    const durationInputs = document.querySelectorAll('input[name="durasiPendaftaran"]');
    durationInputs.forEach(input => {
        input.addEventListener('input', function() {
            const value = parseInt(this.value);
            const min = parseInt(this.min);
            const max = parseInt(this.max);
            
            if (value < min || value > max) {
                this.style.borderColor = '#dc2626';
                this.title = `Durasi harus antara ${min}-${max} hari untuk hasil optimal`;
            } else {
                this.style.borderColor = '#16a34a';
                this.title = 'Durasi pendaftaran optimal';
            }
        });
    });
    
    // Enhanced radio button interactions
    const radioGroups = document.querySelectorAll('input[type="radio"]');
    radioGroups.forEach(radio => {
        radio.addEventListener('change', function() {
            // Clear any previous error styling on radio group
            const groupName = this.name;
            const allRadios = document.querySelectorAll(`input[name="${groupName}"]`);
            allRadios.forEach(r => {
                const parent = r.closest('.grup-radio');
                if (parent) {
                    parent.style.borderColor = '';
                    parent.style.backgroundColor = '';
                }
            });
        });
    });
});

// Utility functions
function formatNumber(num) {
    return new Intl.NumberFormat('id-ID').format(num);
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type}`;
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.zIndex = '9999';
    notification.style.maxWidth = '400px';
    notification.innerHTML = message;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        notification.remove();
    }, 5000);
    
    // Click to dismiss
    notification.addEventListener('click', () => {
        notification.remove();
    });
}

// Performance monitoring
if ('performance' in window) {
    window.addEventListener('load', function() {
        const loadTime = performance.timing.loadEventEnd - performance.timing.navigationStart;
        if (loadTime > 3000) {
            console.warn('⚠️ Halaman membutuhkan waktu lama untuk dimuat:', loadTime + 'ms');
        }
    });
}

// Error handling
window.addEventListener('error', function(e) {
    console.error('❌ JavaScript Error:', e.error);
    // Could send to logging service in production
});

// Unload warning for unsaved changes
let hasUnsavedChanges = false;
const formElements = document.querySelectorAll('input, select, textarea');
formElements.forEach(element => {
    element.addEventListener('change', () => {
        hasUnsavedChanges = true;
    });
});

document.addEventListener('submit', () => {
    hasUnsavedChanges = false;
});

window.addEventListener('beforeunload', function(e) {
    if (hasUnsavedChanges) {
        e.preventDefault();
        e.returnValue = 'Anda memiliki perubahan yang belum disimpan. Yakin ingin meninggalkan halaman?';
        return e.returnValue;
    }
});

// Print functionality
function printCalendar() {
    const printWindow = window.open('', '_blank');
    const calendarContent = document.querySelector('.kalender').outerHTML;
    const styles = Array.from(document.querySelectorAll('style')).map(style => style.outerHTML).join('');
    
    printWindow.document.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>Kalender Event - <?= APP_NAME ?></title>
            ${styles}
            <style>
                body { font-family: Arial, sans-serif; padding: 20px; }
                .kalender { page-break-inside: avoid; }
                @media print {
                    .btn { display: none; }
                    .nav { display: none; }
                }
            </style>
        </head>
        <body>
            <h2>📅 Kalender Event - ${new Date().getFullYear()}</h2>
            ${calendarContent}
            <p style="margin-top: 20px; font-size: 12px; color: #666;">
                Dicetak dari <?= APP_NAME ?> pada ${new Date().toLocaleString('id-ID')}
            </p>
        </body>
        </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    setTimeout(() => {
        printWindow.print();
    }, 500);
}

// Export functionality
function exportEventData(format = 'json') {
    const events = <?= json_encode($events) ?>;
    
    if (format === 'json') {
        const dataStr = JSON.stringify(events, null, 2);
        const dataBlob = new Blob([dataStr], {type: 'application/json'});
        const url = URL.createObjectURL(dataBlob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `events_${new Date().toISOString().split('T')[0]}.json`;
        link.click();
        URL.revokeObjectURL(url);
    } else if (format === 'csv') {
        // Redirect to server-side CSV export
        window.location.href = `backup.php?action=export&key=<?= LOGIN_PIN ?>`;
    }
}

// Service worker registration for offline capability (if needed)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        // Uncomment if you have a service worker
        // navigator.serviceWorker.register('/sw.js')
        //     .then(registration => console.log('SW registered'))
        //     .catch(registrationError => console.log('SW registration failed'));
    });
}
</script>

<?php endif; ?>
</body>
</html>