<?php
/**
 * Sistem Pengelola Event Olimpiade - Standalone Version
 * All-in-One file untuk kemudahan deployment
 */

// Error reporting untuk debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Start session
session_start();

// ============================================
// CONFIGURATION
// ============================================

// Konfigurasi dasar
define('APP_NAME', 'Sistem Pengelola Event Olimpiade');
define('APP_VERSION', '1.0.0');
define('LOGIN_PIN', '051205'); // GANTI INI!
define('DEFAULT_TIMEZONE', 'Asia/Jakarta');
define('DEBUG_MODE', true); // Set false untuk production

// Konfigurasi event
define('MIN_REGISTRATION_DAYS', 25);
define('MAX_REGISTRATION_DAYS', 45);
define('MAX_EVENTS_PER_MONTH', 3);
define('MIN_DAYS_BETWEEN_EVENTS', 12);
define('DEFAULT_PREPARATION_DAYS', 5);

// Konfigurasi file
define('DATA_DIR', __DIR__ . '/data');
define('DATA_FILE', DATA_DIR . '/events.json');

// Set timezone
date_default_timezone_set(DEFAULT_TIMEZONE);

// ============================================
// UTILITY FUNCTIONS
// ============================================

/**
 * Initialize data directory and file
 */
function initializeData() {
    if (!is_dir(DATA_DIR)) {
        mkdir(DATA_DIR, 0755, true);
    }
    
    if (!file_exists(DATA_FILE)) {
        $initialData = [
            'events' => [],
            'created_at' => date('Y-m-d H:i:s'),
            'version' => APP_VERSION
        ];
        file_put_contents(DATA_FILE, json_encode($initialData, JSON_PRETTY_PRINT));
    }
}

/**
 * Load events from JSON file
 */
function loadEvents() {
    initializeData();
    
    if (!file_exists(DATA_FILE)) {
        return [];
    }
    
    $data = file_get_contents(DATA_FILE);
    $decoded = json_decode($data, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logError('JSON decode error: ' . json_last_error_msg());
        return [];
    }
    
    return $decoded['events'] ?? [];
}

/**
 * Save events to JSON file
 */
function saveEvents($events) {
    initializeData();
    
    $data = [
        'events' => $events,
        'updated_at' => date('Y-m-d H:i:s'),
        'version' => APP_VERSION
    ];
    
    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        logError('JSON encode error: ' . json_last_error_msg());
        return false;
    }
    
    $result = file_put_contents(DATA_FILE, $json);
    
    if ($result === false) {
        logError('Failed to write to data file: ' . DATA_FILE);
        return false;
    }
    
    return true;
}

/**
 * Check if user is logged in
 */
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * Process login
 */
function processLogin($pin) {
    if ($pin === LOGIN_PIN) {
        $_SESSION['logged_in'] = true;
        $_SESSION['login_time'] = time();
        return true;
    }
    return false;
}

/**
 * Logout user
 */
function logout() {
    session_destroy();
}

/**
 * Sanitize input
 */
function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Validate event data
 */
function validateEventData($data) {
    $errors = [];
    
    if (empty($data['namaLomba'])) {
        $errors[] = 'Nama lomba harus diisi';
    }
    
    if (empty($data['singkatanLomba'])) {
        $errors[] = 'Singkatan lomba harus diisi';
    }
    
    if (empty($data['penyelenggara'])) {
        $errors[] = 'Penyelenggara harus diisi';
    }
    
    if (!empty($data['tanggalMulai'])) {
        $date = DateTime::createFromFormat('Y-m-d', $data['tanggalMulai']);
        if (!$date || $date->format('Y-m-d') !== $data['tanggalMulai']) {
            $errors[] = 'Format tanggal mulai tidak valid';
        }
    }
    
    if (!empty($data['durasiPendaftaran'])) {
        $duration = (int)$data['durasiPendaftaran'];
        if ($duration < MIN_REGISTRATION_DAYS || $duration > MAX_REGISTRATION_DAYS) {
            $errors[] = 'Durasi pendaftaran harus antara ' . MIN_REGISTRATION_DAYS . '-' . MAX_REGISTRATION_DAYS . ' hari';
        }
    }
    
    return $errors;
}

/**
 * Calculate event timeline
 */
function calculateTimeline($inputData, $selectedSlot = null) {
    if ($selectedSlot) {
        $startRegistration = new DateTime($selectedSlot['mulai_pendaftaran']);
        $registrationDuration = $selectedSlot['durasi_pendaftaran'];
    } else {
        $startRegistration = new DateTime($inputData['tanggalMulai']);
        $registrationDuration = (int)$inputData['durasiPendaftaran'];
    }
    
    $hasSubstitute = ($inputData['susulan'] ?? 'tidak') === 'ya';
    $substituteDay = $inputData['hariSusulan'] ?? 'minggu';
    $hasFinal = ($inputData['babakFinal'] ?? 'tidak') === 'ya';
    
    // Calculate end registration date
    $endRegistration = clone $startRegistration;
    $endRegistration->modify('+' . ($registrationDuration - 1) . ' hari');
    
    // Calculate execution date (must be Saturday)
    $execution = clone $endRegistration;
    $execution->modify('+1 hari');
    
    // Ensure it falls on Saturday
    while ($execution->format('w') != 6) {
        $execution->modify('+1 hari');
    }
    
    // Calculate substitute date
    $substituteDate = null;
    if ($hasSubstitute) {
        $substituteDate = clone $execution;
        if ($substituteDay === 'minggu') {
            $substituteDate->modify('+1 hari');
        } else if ($substituteDay === 'senin') {
            $substituteDate->modify('+2 hari');
        }
    }
    
    // Calculate announcement date
    $announcement = $substituteDate ? clone $substituteDate : clone $execution;
    $announcement->modify('+3 hari');
    
    $timeline = [
        'mulai_pendaftaran' => $startRegistration->format('Y-m-d'),
        'selesai_pendaftaran' => $endRegistration->format('Y-m-d'),
        'pelaksanaan' => $execution->format('Y-m-d'),
        'tanggal_susulan' => $substituteDate ? $substituteDate->format('Y-m-d') : null,
        'pengumuman' => $announcement->format('Y-m-d'),
        'durasi_pendaftaran' => $registrationDuration,
        'susulan' => $hasSubstitute ? 'ya' : 'tidak',
        'hari_susulan' => $substituteDay,
        'babak_final' => $hasFinal ? 'ya' : 'tidak'
    ];
    
    // Add additional phases based on event type
    if ($hasFinal) {
        $timeline = array_merge($timeline, calculateFinalPhases($announcement));
    } else {
        $timeline = array_merge($timeline, calculateRegularPhases($announcement));
    }
    
    return $timeline;
}

/**
 * Calculate phases for events with finals
 */
function calculateFinalPhases($announcement) {
    $finalRegistrationStart = clone $announcement;
    $finalRegistrationStart->modify('+1 hari');
    
    $finalRegistrationEnd = clone $finalRegistrationStart;
    $finalRegistrationEnd->modify('+10 hari');
    
    $finalDate = clone $finalRegistrationEnd;
    $finalDate->modify('+1 hari');
    while ($finalDate->format('w') != 6) {
        $finalDate->modify('+1 hari');
    }
    
    $finalAnnouncement = clone $finalDate;
    $finalAnnouncement->modify('+3 hari');
    
    $rewardAccessStart = clone $finalAnnouncement;
    $rewardAccessStart->modify('+1 hari');
    
    $rewardAccessEnd = clone $rewardAccessStart;
    $rewardAccessEnd->modify('+10 hari');
    
    $cleanup = clone $rewardAccessEnd;
    $cleanup->modify('+5 hari');
    
    return [
        'daftar_ulang_mulai' => $finalRegistrationStart->format('Y-m-d'),
        'daftar_ulang_selesai' => $finalRegistrationEnd->format('Y-m-d'),
        'tanggal_final' => $finalDate->format('Y-m-d'),
        'pengumuman_final' => $finalAnnouncement->format('Y-m-d'),
        'akses_penghargaan_mulai' => $rewardAccessStart->format('Y-m-d'),
        'akses_penghargaan_selesai' => $rewardAccessEnd->format('Y-m-d'),
        'sapu_bersih' => $cleanup->format('Y-m-d')
    ];
}

/**
 * Calculate phases for regular events
 */
function calculateRegularPhases($announcement) {
    $claim1Start = clone $announcement;
    $claim1Start->modify('+1 hari');
    
    $claim1End = clone $claim1Start;
    $claim1End->modify('+10 hari');
    
    $feedbackForm = clone $claim1End;
    $feedbackForm->modify('+1 hari');
    
    $break1Start = clone $feedbackForm;
    $break1Start->modify('+1 hari');
    
    $break1End = clone $break1Start;
    $break1End->modify('+5 hari');
    
    $claim2Start = clone $break1End;
    $claim2Start->modify('+1 hari');
    
    $claim2End = clone $claim2Start;
    $claim2End->modify('+14 hari');
    
    $break2Start = clone $claim2End;
    $break2Start->modify('+1 hari');
    
    $break2End = clone $break2Start;
    $break2End->modify('+5 hari');
    
    $cleanup = clone $break2End;
    $cleanup->modify('+1 hari');
    
    return [
        'klaim_gel1_mulai' => $claim1Start->format('Y-m-d'),
        'klaim_gel1_selesai' => $claim1End->format('Y-m-d'),
        'form_umpan_balik' => $feedbackForm->format('Y-m-d'),
        'jeda1_mulai' => $break1Start->format('Y-m-d'),
        'jeda1_selesai' => $break1End->format('Y-m-d'),
        'klaim_gel2_mulai' => $claim2Start->format('Y-m-d'),
        'klaim_gel2_selesai' => $claim2End->format('Y-m-d'),
        'jeda2_mulai' => $break2Start->format('Y-m-d'),
        'jeda2_selesai' => $break2End->format('Y-m-d'),
        'sapu_bersih' => $cleanup->format('Y-m-d')
    ];
}

/**
 * Find optimal slots for new events
 */
function findOptimalSlots($existingEvents) {
    $today = new DateTime();
    $startRegistration = clone $today;
    $startRegistration->modify('+' . DEFAULT_PREPARATION_DAYS . ' days');
    
    $slotOptions = [];
    
    // Try to find optimal slots for next 90 days
    for ($i = 0; $i < 90; $i += 7) {
        $testStart = clone $startRegistration;
        $testStart->modify("+$i days");
        
        // Try different registration durations
        for ($duration = MIN_REGISTRATION_DAYS; $duration <= MAX_REGISTRATION_DAYS; $duration++) {
            $testEnd = clone $testStart;
            $testEnd->modify('+' . ($duration - 1) . ' days');
            
            // Must end on Friday
            if ($testEnd->format('w') != 5) continue;
            
            // Execution must be on Saturday
            $testExecution = clone $testEnd;
            $testExecution->modify('+1 day');
            
            // Check conflicts
            $conflicts = checkConflicts([
                'pelaksanaan' => $testExecution->format('Y-m-d')
            ], $existingEvents);
            
            if (empty($conflicts)) {
                // Check monthly distribution
                $executionMonth = $testExecution->format('Y-m');
                $monthEvents = array_filter($existingEvents, function($event) use ($executionMonth) {
                    return substr($event['pelaksanaan'], 0, 7) == $executionMonth;
                });
                
                if (count($monthEvents) < MAX_EVENTS_PER_MONTH) {
                    $slotOptions[] = [
                        'mulai_pendaftaran' => $testStart->format('Y-m-d'),
                        'durasi_pendaftaran' => $duration,
                        'selesai_pendaftaran' => $testEnd->format('Y-m-d'),
                        'pelaksanaan' => $testExecution->format('Y-m-d'),
                        'bulan' => $testExecution->format('F Y'),
                        'event_bulan_ini' => count($monthEvents),
                        'skor' => (MAX_EVENTS_PER_MONTH - count($monthEvents)) * 10 + (45 - $i)
                    ];
                    
                    if (count($slotOptions) >= 3) break 2;
                }
            }
        }
    }
    
    // Sort by score (highest first)
    usort($slotOptions, function($a, $b) {
        return $b['skor'] - $a['skor'];
    });
    
    return array_slice($slotOptions, 0, 3);
}

/**
 * Check for schedule conflicts
 */
function checkConflicts($newEvent, $existingEvents, $excludeId = null) {
    $conflicts = [];
    $newExecutionTimestamp = strtotime($newEvent['pelaksanaan']);
    
    foreach ($existingEvents as $event) {
        if ($excludeId && $event['id'] == $excludeId) continue;
        
        $existingExecutionTimestamp = strtotime($event['pelaksanaan']);
        $daysDifference = abs(($newExecutionTimestamp - $existingExecutionTimestamp) / (60 * 60 * 24));
        
        // Check minimum distance between events
        if ($daysDifference < MIN_DAYS_BETWEEN_EVENTS) {
            $conflicts[] = [
                'type' => 'distance',
                'event' => $event,
                'days' => $daysDifference,
                'message' => sprintf(
                    "Jarak hanya %.0f hari dengan %s (minimal %d hari)",
                    $daysDifference,
                    $event['nama_lomba'],
                    MIN_DAYS_BETWEEN_EVENTS
                )
            ];
        }
        
        // Check same week
        $newWeek = date('Y-W', $newExecutionTimestamp);
        $existingWeek = date('Y-W', $existingExecutionTimestamp);
        if ($newWeek == $existingWeek) {
            $conflicts[] = [
                'type' => 'same_week',
                'event' => $event,
                'message' => sprintf(
                    "Pelaksanaan di minggu yang sama dengan %s",
                    $event['nama_lomba']
                )
            ];
        }
    }
    
    return $conflicts;
}

/**
 * Format date to Indonesian
 */
function formatIndonesian($date) {
    $months = [
        1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April',
        5 => 'Mei', 6 => 'Juni', 7 => 'Juli', 8 => 'Agustus',
        9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
    ];
    
    $days = [
        'Sunday' => 'Minggu', 'Monday' => 'Senin', 'Tuesday' => 'Selasa',
        'Wednesday' => 'Rabu', 'Thursday' => 'Kamis', 'Friday' => 'Jumat', 'Saturday' => 'Sabtu'
    ];
    
    $dateObj = new DateTime($date);
    $dayName = $days[$dateObj->format('l')];
    $day = $dateObj->format('d');
    $monthName = $months[(int)$dateObj->format('m')];
    $year = $dateObj->format('Y');
    
    return "$dayName, $day $monthName $year";
}

/**
 * Format date range
 */
function formatDateRange($startDate, $endDate) {
    $start = new DateTime($startDate);
    $end = new DateTime($endDate);
    
    if ($start->format('Y-m') === $end->format('Y-m')) {
        return formatIndonesian($startDate) . ' - ' . $end->format('d') . ' ' . formatIndonesian($endDate);
    } else {
        return formatIndonesian($startDate) . ' - ' . formatIndonesian($endDate);
    }
}

/**
 * Log error for debugging
 */
function logError($message) {
    if (DEBUG_MODE) {
        error_log('[' . date('Y-m-d H:i:s') . '] ' . $message);
    }
}

// ============================================
// MAIN APPLICATION LOGIC
// ============================================

// Initialize variables
$error = '';
$success = '';
$events = [];
$currentView = $_GET['view'] ?? 'generator';
$editEventId = $_GET['edit'] ?? null;
$detailEventId = $_GET['detail'] ?? null;

// Load events if logged in
if (isLoggedIn()) {
    $events = loadEvents();
}

// Process form submissions
if ($_POST) {
    if (isset($_POST['login'])) {
        $pin = sanitizeInput($_POST['pin']);
        if (processLogin($pin)) {
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } else {
            $error = 'PIN salah!';
        }
    } 
    elseif (isLoggedIn()) {
        if (isset($_POST['logout'])) {
            logout();
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
        
        if (isset($_POST['cari_slot_otomatis'])) {
            $inputData = sanitizeInput($_POST);
            $validationErrors = validateEventData($inputData);
            
            if (!empty($validationErrors)) {
                $error = implode('<br>', $validationErrors);
            } else {
                $optimalSlots = findOptimalSlots($events);
                $_SESSION['slot_optimal'] = $optimalSlots;
                $_SESSION['data_form'] = $inputData;
                
                if (empty($optimalSlots)) {
                    $error = 'Tidak ditemukan slot optimal untuk jadwal ini';
                }
            }
        }
        
        if (isset($_POST['pilih_slot'])) {
            if (isset($_POST['slot_index'], $_SESSION['slot_optimal'], $_SESSION['data_form'])) {
                $slotIndex = (int)$_POST['slot_index'];
                if (isset($_SESSION['slot_optimal'][$slotIndex])) {
                    $selectedSlot = $_SESSION['slot_optimal'][$slotIndex];
                    $previewData = calculateTimeline($_SESSION['data_form'], $selectedSlot);
                    if ($previewData) {
                        $previewData = array_merge($_SESSION['data_form'], $previewData, ['slot_terpilih' => $selectedSlot]);
                        $_SESSION['data_preview'] = $previewData;
                        unset($_SESSION['slot_optimal']);
                    } else {
                        $error = 'Terjadi kesalahan saat menghitung timeline';
                    }
                } else {
                    $error = 'Slot yang dipilih tidak valid';
                }
            } else {
                $error = 'Data slot tidak lengkap';
            }
        }
        
        if (isset($_POST['generate_preview'])) {
            $inputData = sanitizeInput($_POST);
            $validationErrors = validateEventData($inputData);
            
            if (!empty($validationErrors)) {
                $error = implode('<br>', $validationErrors);
            } else {
                $previewData = calculateTimeline($inputData);
                if ($previewData) {
                    $previewData = array_merge($inputData, $previewData);
                    $_SESSION['data_preview'] = $previewData;
                } else {
                    $error = 'Terjadi kesalahan saat menghitung timeline';
                }
            }
        }
        
        if (isset($_POST['simpan_event'])) {
            if (isset($_SESSION['data_preview'])) {
                $eventData = $_SESSION['data_preview'];
                $eventData['id'] = $_POST['event_id'] ?: uniqid();
                $eventData['dibuat_pada'] = date('Y-m-d H:i:s');
                
                // Ensure required fields exist
                $eventData['nama_lomba'] = $eventData['namaLomba'];
                $eventData['singkatan_lomba'] = $eventData['singkatanLomba'];
                
                $conflicts = checkConflicts($eventData, $events, $_POST['event_id']);
                
                if (!empty($conflicts) && !isset($_POST['paksa_simpan'])) {
                    $warningConflicts = $conflicts;
                    $eventForEdit = $eventData;
                } else {
                    if ($_POST['event_id']) {
                        // Update existing event
                        foreach ($events as &$event) {
                            if ($event['id'] == $_POST['event_id']) {
                                $event = array_merge($event, $eventData);
                                break;
                            }
                        }
                    } else {
                        // Add new event
                        $events[] = $eventData;
                    }
                    
                    if (saveEvents($events)) {
                        unset($_SESSION['data_preview'], $_SESSION['slot_optimal'], $_SESSION['data_form']);
                        $success = 'Event berhasil disimpan!';
                        $events = loadEvents(); // Reload events
                    } else {
                        $error = 'Gagal menyimpan event ke database';
                    }
                }
            } else {
                $error = 'Data preview tidak ditemukan. Silakan buat preview terlebih dahulu.';
            }
        }
        
        if (isset($_POST['hapus_event'])) {
            $eventId = sanitizeInput($_POST['event_id']);
            $events = array_filter($events, function($event) use ($eventId) {
                return $event['id'] != $eventId;
            });
            
            if (saveEvents($events)) {
                $success = 'Event berhasil dihapus!';
                $events = loadEvents(); // Reload events
            } else {
                $error = 'Gagal menghapus event';
            }
        }
    }
}

// Load data for editing if needed
$eventForEdit = null;
if ($editEventId) {
    $eventForEdit = array_filter($events, function($event) use ($editEventId) {
        return $event['id'] == $editEventId;
    });
    $eventForEdit = reset($eventForEdit);
    unset($_SESSION['data_preview'], $_SESSION['slot_optimal']); // Clear preview when editing
}

// Load data for detail if needed
$detailEvent = null;
if ($detailEventId) {
    $detailEvent = array_filter($events, function($event) use ($detailEventId) {
        return $event['id'] == $detailEventId;
    });
    $detailEvent = reset($detailEvent);
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= APP_NAME ?></title>
    <meta name="description" content="Sistem manajemen event olimpiade dengan penjadwalan otomatis">
    
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            background: #f5f5f5; 
            line-height: 1.6;
            color: #374151;
        }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        
        /* Header */
        .header { 
            background: linear-gradient(135deg, #2563eb, #1d4ed8); 
            color: white; 
            padding: 20px; 
            border-radius: 8px; 
            margin-bottom: 20px; 
            display: flex;
            align-items: center;
            justify-content: space-between;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        /* Navigation */
        .nav { 
            display: flex; 
            gap: 10px; 
            margin-bottom: 20px; 
            flex-wrap: wrap; 
        }
        .nav a { 
            background: #3b82f6; 
            color: white; 
            padding: 12px 20px; 
            border-radius: 6px; 
            text-decoration: none; 
            transition: all 0.3s; 
            font-weight: 500;
        }
        .nav a.active, .nav a:hover { 
            background: #1d4ed8; 
            transform: translateY(-1px); 
        }
        
        /* Cards */
        .card { 
            background: white; 
            padding: 20px; 
            border-radius: 8px; 
            box-shadow: 0 2px 8px rgba(0,0,0,0.1); 
            margin-bottom: 20px; 
        }
        
        /* Forms */
        .form-group { margin-bottom: 15px; }
        .form-group label { 
            display: block; 
            margin-bottom: 5px; 
            font-weight: 600; 
            color: #374151; 
        }
        .form-group input, .form-group select { 
            width: 100%; 
            padding: 12px; 
            border: 1px solid #d1d5db; 
            border-radius: 6px; 
            font-size: 14px; 
            transition: all 0.3s;
        }
        .form-group input:focus, .form-group select:focus { 
            outline: none; 
            border-color: #3b82f6; 
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1); 
        }
        
        /* Buttons */
        .btn { 
            background: #3b82f6; 
            color: white; 
            padding: 12px 20px; 
            border: none; 
            border-radius: 6px; 
            cursor: pointer; 
            margin-right: 10px; 
            margin-bottom: 10px;
            font-weight: 500; 
            transition: all 0.3s; 
            text-decoration: none;
            display: inline-block;
        }
        .btn:hover { 
            background: #2563eb; 
            transform: translateY(-1px);
        }
        .btn-danger { background: #dc2626; }
        .btn-danger:hover { background: #b91c1c; }
        .btn-success { background: #16a34a; }
        .btn-success:hover { background: #15803d; }
        .btn-secondary { background: #6b7280; }
        .btn-secondary:hover { background: #4b5563; }
        .btn-warning { background: #ea580c; }
        .btn-warning:hover { background: #c2410c; }
        
        /* Alerts */
        .alert { 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 20px; 
            border: 1px solid;
        }
        .alert-success { 
            background: #dcfce7; 
            color: #166534; 
            border-color: #bbf7d0; 
        }
        .alert-danger { 
            background: #fef2f2; 
            color: #991b1b; 
            border-color: #fecaca; 
        }
        .alert-warning { 
            background: #fefbeb; 
            color: #92400e; 
            border-color: #fed7aa; 
        }
        
        /* Login form */
        .form-login { max-width: 400px; margin: 100px auto; }
        
        /* Grid layouts */
        .baris-form { 
            display: grid; 
            grid-template-columns: 1fr 1fr; 
            gap: 15px; 
        }
        .grup-radio { 
            display: flex; 
            gap: 15px; 
            align-items: center; 
            flex-wrap: wrap;
        }
        .grup-radio label { 
            margin-bottom: 0; 
            cursor: pointer; 
            display: flex;
            align-items: center;
            gap: 5px;
            font-weight: normal;
        }
        
        /* Special sections */
        .auto-info { 
            background: #f0f9ff; 
            border: 1px solid #0ea5e9; 
            padding: 15px; 
            border-radius: 6px; 
            margin: 15px 0; 
        }
        
        .preview-card { 
            background: linear-gradient(135deg, #dbeafe, #e0f2fe); 
            border: 1px solid #3b82f6; 
            padding: 20px; 
            border-radius: 8px; 
            margin: 20px 0; 
        }
        
        .timeline-section { 
            border-left: 4px solid #3b82f6; 
            padding-left: 15px; 
            margin-bottom: 15px; 
            background: rgba(255, 255, 255, 0.7);
            padding: 15px;
            border-radius: 0 6px 6px 0;
        }
        .timeline-section h4 { 
            color: #1d4ed8; 
            margin-bottom: 8px; 
            font-size: 16px; 
        }
        .timeline-section.hijau { border-left-color: #16a34a; }
        .timeline-section.hijau h4 { color: #15803d; }
        .timeline-section.orange { border-left-color: #ea580c; }
        .timeline-section.orange h4 { color: #c2410c; }
        .timeline-section.ungu { border-left-color: #7c3aed; }
        .timeline-section.ungu h4 { color: #6d28d9; }
        .timeline-section.kuning { border-left-color: #eab308; }
        .timeline-section.kuning h4 { color: #ca8a04; }
        
        .slot-card { 
            background: #f8fafc; 
            border: 2px solid #e2e8f0; 
            padding: 15px; 
            border-radius: 6px; 
            margin-bottom: 15px; 
            cursor: pointer; 
            transition: all 0.3s; 
        }
        .slot-card:hover { 
            background: #e2e8f0; 
            border-color: #64748b; 
            transform: translateY(-1px); 
        }
        .slot-card.selected { 
            background: #dbeafe; 
            border-color: #3b82f6; 
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1); 
        }
        
        /* Event list */
        .daftar-event {
            display: grid;
            gap: 15px;
        }
        .item-event {
            background: white;
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid #3b82f6;
            transition: all 0.3s;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .item-event:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        
        .flex {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        /* Responsive design */
        @media (max-width: 768px) {
            .container { padding: 10px; }
            .baris-form { grid-template-columns: 1fr; }
            .header { 
                flex-direction: column; 
                gap: 15px; 
                text-align: center; 
            }
            .nav { justify-content: center; }
            .flex {
                flex-direction: column;
                align-items: stretch;
            }
            .item-event .flex {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>

<?php if (!isLoggedIn()): ?>
<div class="container">
    <div class="form-login card">
        <h2>🔐 Masuk <?= APP_NAME ?></h2>
        <?php if ($error): ?>
            <div class="alert alert-danger">❌ <?= $error ?></div>
        <?php endif; ?>
        <form method="POST">
            <div class="form-group">
                <label for="pin">PIN Akses:</label>
                <input type="password" name="pin" id="pin" required placeholder="Masukkan PIN" autofocus>
            </div>
            <button type="submit" name="login" class="btn">🚀 Masuk Sistem</button>
        </form>
        
        <div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #e5e7eb; font-size: 12px; color: #6b7280;">
            <strong>Informasi Sistem:</strong><br>
            Version: <?= APP_VERSION ?><br>
            Debug Mode: <?= DEBUG_MODE ? 'ON' : 'OFF' ?><br>
            Data File: <?= file_exists(DATA_FILE) ? 'Available' : 'Not Found' ?>
        </div>
    </div>
</div>
<?php else: ?>

<div class="container">
    <div class="header">
        <div>
            <h1>🏆 <?= APP_NAME ?></h1>
            <small>Versi <?= APP_VERSION ?> | <?= count($events) ?> Event Aktif</small>
        </div>
        <div>
            <form method="POST" style="display: inline;">
                <button type="submit" name="logout" class="btn btn-danger">🚪 Keluar</button>
            </form>
        </div>
    </div>

    <div class="nav">
        <a href="?view=generator" class="<?= $currentView == 'generator' ? 'active' : '' ?>">🎯 Buat Event</a>
        <a href="?view=list" class="<?= $currentView == 'list' ? 'active' : '' ?>">📋 Daftar Event</a>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success">✅ <?= $success ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">❌ <?= $error ?></div>
    <?php endif; ?>

    <?php if (isset($warningConflicts)): ?>
        <div class="alert alert-warning">
            <h3>⚠️ Peringatan Konflik Jadwal</h3>
            <?php foreach ($warningConflicts as $conflict): ?>
                <p>• <?= $conflict['message'] ?></p>
            <?php endforeach; ?>
            
            <p style="margin-top: 15px;"><strong>Tetap simpan event ini?</strong></p>
            <form method="POST" style="display: inline;">
                <?php foreach ($_POST as $key => $value): ?>
                    <input type="hidden" name="<?= htmlspecialchars($key) ?>" value="<?= htmlspecialchars($value) ?>">
                <?php endforeach; ?>
                <input type="hidden" name="paksa_simpan" value="1">
                <button type="submit" class="btn btn-warning">⚠️ Ya, Tetap Simpan</button>
                <button type="button" onclick="window.location.reload()" class="btn btn-secondary">❌ Batal</button>
            </form>
        </div>
    <?php endif; ?>

    <?php if ($currentView == 'generator'): ?>
    <div class="card">
        <h2><?= $editEventId ? '✏️ Ubah Event' : '🎯 Buat Event Baru' ?></h2>
        
        <?php if (!$editEventId && !isset($_SESSION['slot_optimal']) && !isset($_SESSION['data_preview'])): ?>
        <div class="auto-info">
            <h4>🤖 Sistem Penjadwalan Otomatis</h4>
            <p>• <strong>Persiapan:</strong> <?= DEFAULT_PREPARATION_DAYS ?> hari sebelum pendaftaran</p>
            <p>• <strong>Durasi Optimal:</strong> <?= MIN_REGISTRATION_DAYS ?>-<?= MAX_REGISTRATION_DAYS ?> hari (berakhir Jumat)</p>
            <p>• <strong>Pelaksanaan:</strong> Selalu di hari Sabtu</p>
            <p>• <strong>Kapasitas:</strong> Maksimal <?= MAX_EVENTS_PER_MONTH ?> event per bulan</p>
            <p>• <strong>Jarak Aman:</strong> Minimal <?= MIN_DAYS_BETWEEN_EVENTS ?> hari antar event</p>
        </div>
        <?php endif; ?>
        
        <form method="POST" id="formEvent">
            <input type="hidden" name="event_id" value="<?= htmlspecialchars($editEventId) ?>">
            
            <div class="baris-form">
                <div class="form-group">
                    <label for="namaLomba">Nama Lomba: *</label>
                    <input type="text" name="namaLomba" id="namaLomba" required 
                           value="<?= htmlspecialchars($eventForEdit['nama_lomba'] ?? $_SESSION['data_form']['namaLomba'] ?? $_SESSION['data_preview']['namaLomba'] ?? '') ?>" 
                           placeholder="Contoh: Olimpiade Sains Nasional">
                </div>
                
                <div class="form-group">
                    <label for="singkatanLomba">Singkatan Lomba: *</label>
                    <input type="text" name="singkatanLomba" id="singkatanLomba" required 
                           value="<?= htmlspecialchars($eventForEdit['singkatan_lomba'] ?? $_SESSION['data_form']['singkatanLomba'] ?? $_SESSION['data_preview']['singkatanLomba'] ?? '') ?>" 
                           placeholder="Contoh: OSN">
                </div>
            </div>
            
            <div class="form-group">
                <label for="penyelenggara">Penyelenggara: *</label>
                <input type="text" name="penyelenggara" id="penyelenggara" required 
                       value="<?= htmlspecialchars($eventForEdit['penyelenggara'] ?? $_SESSION['data_form']['penyelenggara'] ?? $_SESSION['data_preview']['penyelenggara'] ?? '') ?>" 
                       placeholder="Contoh: Kementerian Pendidikan dan Kebudayaan">
            </div>
            
            <div class="form-group">
                <label>Ada Susulan:</label>
                <div class="grup-radio">
                    <label>
                        <input type="radio" name="susulan" value="ya" 
                               <?= ($eventForEdit['susulan'] ?? $_SESSION['data_form']['susulan'] ?? $_SESSION['data_preview']['susulan'] ?? '') == 'ya' ? 'checked' : '' ?>>
                        Ya
                    </label>
                    <label>
                        <input type="radio" name="susulan" value="tidak" 
                               <?= ($eventForEdit['susulan'] ?? $_SESSION['data_form']['susulan'] ?? $_SESSION['data_preview']['susulan'] ?? 'tidak') == 'tidak' ? 'checked' : '' ?>>
                        Tidak
                    </label>
                    <select name="hariSusulan" style="width: auto; margin-left: 10px;">
                        <option value="minggu" <?= ($eventForEdit['hari_susulan'] ?? $_SESSION['data_form']['hariSusulan'] ?? $_SESSION['data_preview']['hariSusulan'] ?? '') == 'minggu' ? 'selected' : '' ?>>Minggu</option>
                        <option value="senin" <?= ($eventForEdit['hari_susulan'] ?? $_SESSION['data_form']['hariSusulan'] ?? $_SESSION['data_preview']['hariSusulan'] ?? '') == 'senin' ? 'selected' : '' ?>>Senin</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label>Ada Babak Final:</label>
                <div class="grup-radio">
                    <label>
                        <input type="radio" name="babakFinal" value="ya" 
                               <?= ($eventForEdit['babak_final'] ?? $_SESSION['data_form']['babakFinal'] ?? $_SESSION['data_preview']['babakFinal'] ?? '') == 'ya' ? 'checked' : '' ?>>
                        Ya
                    </label>
                    <label>
                        <input type="radio" name="babakFinal" value="tidak" 
                               <?= ($eventForEdit['babak_final'] ?? $_SESSION['data_form']['babakFinal'] ?? $_SESSION['data_preview']['babakFinal'] ?? 'tidak') == 'tidak' ? 'checked' : '' ?>>
                        Tidak
                    </label>
                </div>
            </div>
            
            <?php if ($editEventId): ?>
            <div class="baris-form">
                <div class="form-group">
                    <label for="tanggalMulai">Tanggal Mulai Pendaftaran: *</label>
                    <input type="date" name="tanggalMulai" id="tanggalMulai" 
                           value="<?= $eventForEdit['mulai_pendaftaran'] ?? '' ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="durasiPendaftaran">Durasi Pendaftaran (Hari): *</label>
                    <input type="number" name="durasiPendaftaran" id="durasiPendaftaran" 
                           value="<?= $eventForEdit['durasi_pendaftaran'] ?? 30 ?>" 
                           min="<?= MIN_REGISTRATION_DAYS ?>" max="<?= MAX_REGISTRATION_DAYS ?>" required>
                </div>
            </div>
            <button type="submit" name="generate_preview" class="btn">📋 Buat Preview Timeline</button>
            <a href="?view=generator" class="btn btn-secondary">❌ Batal</a>
            <?php else: ?>
            <button type="submit" name="cari_slot_otomatis" class="btn btn-success">🔍 Cari Slot Otomatis</button>
            <?php endif; ?>
        </form>
        
        <?php if (isset($_SESSION['slot_optimal']) && !empty($_SESSION['slot_optimal'])): ?>
        <div class="card" style="margin-top: 20px;">
            <h3>🎯 Slot Optimal Ditemukan</h3>
            <p>Berikut adalah <?= count($_SESSION['slot_optimal']) ?> slot terbaik:</p>
            
            <form method="POST" id="formSlot">
                <?php foreach ($_SESSION['slot_optimal'] as $index => $slot): ?>
                <div class="slot-card" onclick="selectSlot(<?= $index ?>)" id="card_<?= $index ?>">
                    <div style="display: flex; align-items: center;">
                        <input type="radio" name="slot_index" value="<?= $index ?>" id="slot_<?= $index ?>" style="margin-right: 10px;" required>
                        <div style="flex: 1;">
                            <strong>Opsi <?= $index + 1 ?>:</strong> Pelaksanaan <strong><?= formatIndonesian($slot['pelaksanaan']) ?></strong><br>
                            <small style="color: #6b7280;">
                                📅 Pendaftaran: <?= formatDateRange($slot['mulai_pendaftaran'], $slot['selesai_pendaftaran']) ?> (<?= $slot['durasi_pendaftaran'] ?> hari)<br>
                                📊 Event di <?= $slot['bulan'] ?>: <?= $slot['event_bulan_ini'] ?>/<?= MAX_EVENTS_PER_MONTH ?> event
                            </small>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
                
                <div style="margin-top: 15px;">
                    <button type="submit" name="pilih_slot" class="btn btn-success">✅ Pilih Slot & Buat Preview</button>
                    <a href="?view=generator" class="btn btn-secondary">🔄 Cari Ulang</a>
                </div>
            </form>
        </div>
        <?php elseif (isset($_SESSION['slot_optimal']) && empty($_SESSION['slot_optimal'])): ?>
        <div class="alert alert-warning">
            <h4>😔 Tidak Ada Slot Optimal</h4>
            <p>Tidak ditemukan slot yang cocok dalam 90 hari ke depan.</p>
            <a href="?view=generator" class="btn">🔄 Coba Lagi</a>
        </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['data_preview'])): 
            $preview = $_SESSION['data_preview'];
        ?>
        <div class="preview-card">
            <h3 style="color: #1d4ed8; margin-bottom: 20px;">📅 Preview Timeline Event</h3>
            
            <div class="timeline-section">
                <h4>ℹ️ Informasi Event</h4>
                <p><strong>Nama:</strong> <?= htmlspecialchars($preview['namaLomba']) ?> (<?= htmlspecialchars($preview['singkatanLomba']) ?>)</p>
                <p><strong>Penyelenggara:</strong> <?= htmlspecialchars($preview['penyelenggara']) ?></p>
            </div>
            
            <div class="timeline-section hijau">
                <h4>📝 Tahap Pendaftaran</h4>
                <p><strong>Periode:</strong> <?= formatDateRange($preview['mulai_pendaftaran'], $preview['selesai_pendaftaran']) ?></p>
                <small>Durasi: <?= $preview['durasi_pendaftaran'] ?> hari</small>
            </div>
            
            <div class="timeline-section orange">
                <h4>🏆 Tahap Pelaksanaan</h4>
                <p><strong><?= $preview['babak_final'] == 'ya' ? 'Penyisihan' : 'Pelaksanaan' ?>:</strong> <?= formatIndonesian($preview['pelaksanaan']) ?></p>
                <?php if ($preview['susulan'] == 'ya'): ?>
                    <p><strong>Susulan:</strong> <?= formatIndonesian($preview['tanggal_susulan']) ?></p>
                <?php endif; ?>
                <p><strong>Pengumuman:</strong> <?= formatIndonesian($preview['pengumuman']) ?></p>
            </div>
            
            <?php if ($preview['babak_final'] == 'ya'): ?>
            <div class="timeline-section ungu">
                <h4>🥇 Tahap Final</h4>
                <p><strong>Daftar Ulang Finalis:</strong> <?= formatDateRange($preview['daftar_ulang_mulai'], $preview['daftar_ulang_selesai']) ?></p>
                <p><strong>Babak Final:</strong> <?= formatIndonesian($preview['tanggal_final']) ?></p>
                <p><strong>Pengumuman Pemenang:</strong> <?= formatIndonesian($preview['pengumuman_final']) ?></p>
            </div>
            
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan</h4>
                <p><strong>Akses Penghargaan:</strong> <?= formatDateRange($preview['akses_penghargaan_mulai'], $preview['akses_penghargaan_selesai']) ?></p>
                <p><strong>Sapu Bersih:</strong> <?= formatIndonesian($preview['sapu_bersih']) ?></p>
            </div>
            <?php else: ?>
            <div class="timeline-section kuning">
                <h4>🎁 Tahap Penghargaan</h4>
                <p><strong>Klaim Gelombang 1:</strong> <?= formatDateRange($preview['klaim_gel1_mulai'], $preview['klaim_gel1_selesai']) ?></p>
                <p><strong>Form Umpan Balik:</strong> <?= formatIndonesian($preview['form_umpan_balik']) ?></p>
                <p><strong>Jeda 1:</strong> <?= formatDateRange($preview['jeda1_mulai'], $preview['jeda1_selesai']) ?></p>
                <p><strong>Klaim Gelombang 2:</strong> <?= formatDateRange($preview['klaim_gel2_mulai'], $preview['klaim_gel2_selesai']) ?></p>
                <p><strong>Jeda 2:</strong> <?= formatDateRange($preview['jeda2_mulai'], $preview['jeda2_selesai']) ?></p>
                <p><strong>Sapu Bersih:</strong> <?= formatIndonesian($preview['sapu_bersih']) ?></p>
            </div>
            <?php endif; ?>
            
            <div style="margin-top: 20px; padding-top: 15px; border-top: 1px solid #ddd;">
                <form method="POST" style="display: inline;">
                    <input type="hidden" name="event_id" value="<?= htmlspecialchars($editEventId) ?>">
                    <button type="submit" name="simpan_event" class="btn btn-success">💾 Simpan Event</button>
                </form>
                <button type="button" onclick="window.location.href='?view=generator<?= $editEventId ? '&edit=' . $editEventId : '' ?>'" class="btn btn-secondary">✏️ Edit</button>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php elseif ($currentView == 'list'): ?>
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
            <h2>📋 Daftar Event (<?= count($events) ?>)</h2>
            <a href="?view=generator" class="btn btn-success">🎯 Tambah Event</a>
        </div>
        
        <?php if (empty($events)): ?>
            <div style="text-align: center; color: #6b7280; padding: 40px;">
                <h3>Belum Ada Event</h3>
                <p>Mulai buat event pertama Anda</p>
                <a href="?view=generator" class="btn btn-success">🎯 Buat Event Pertama</a>
            </div>
        <?php else: ?>
            <div class="daftar-event">
                <?php foreach ($events as $event): ?>
                <div class="item-event">
                    <div class="flex">
                        <div style="flex: 1;">
                            <h3><?= htmlspecialchars($event['nama_lomba'] ?? $event['namaLomba']) ?> (<?= htmlspecialchars($event['singkatan_lomba'] ?? $event['singkatanLomba']) ?>)</h3>
                            <p><strong>Penyelenggara:</strong> <?= htmlspecialchars($event['penyelenggara']) ?></p>
                            <p><strong>Pelaksanaan:</strong> <?= formatIndonesian($event['pelaksanaan']) ?></p>
                            <p><strong>Pendaftaran:</strong> <?= formatDateRange($event['mulai_pendaftaran'], $event['selesai_pendaftaran']) ?></p>
                            <p><strong>Status:</strong> 
                                <?php
                                $today = date('Y-m-d');
                                if ($today < $event['mulai_pendaftaran']) {
                                    echo '<span style="color: #3b82f6;">🔵 Akan Datang</span>';
                                } elseif ($today <= $event['selesai_pendaftaran']) {
                                    echo '<span style="color: #16a34a;">🟢 Pendaftaran Aktif</span>';
                                } elseif ($today <= $event['pelaksanaan']) {
                                    echo '<span style="color: #ea580c;">🟡 Menuju Pelaksanaan</span>';
                                } else {
                                    echo '<span style="color: #6b7280;">⚪ Selesai</span>';
                                }
                                ?>
                            </p>
                        </div>
                        <div>
                            <a href="?view=list&detail=<?= $event['id'] ?>" class="btn btn-secondary">👁️ Detail</a>
                            <a href="?view=generator&edit=<?= $event['id'] ?>" class="btn">✏️ Edit</a>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Yakin hapus event ini?')">
                                <input type="hidden" name="event_id" value="<?= $event['id'] ?>">
                                <button type="submit" name="hapus_event" class="btn btn-danger">🗑️ Hapus</button>
                            </form>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <?php if ($detailEventId && $detailEvent): ?>
    <div class="card">
        <h3>📋 Detail Timeline: <?= htmlspecialchars($detailEvent['nama_lomba'] ?? $detailEvent['namaLomba']) ?></h3>
        <div style="background: #f8fafc; padding: 20px; border-radius: 8px; margin-top: 15px;">
            <!-- Detail timeline content here -->
            <div class="timeline-section">
                <h4>ℹ️ Informasi Event</h4>
                <p><strong>Nama:</strong> <?= htmlspecialchars($detailEvent['nama_lomba'] ?? $detailEvent['namaLomba']) ?></p>
                <p><strong>Singkatan:</strong> <?= htmlspecialchars($detailEvent['singkatan_lomba'] ?? $detailEvent['singkatanLomba']) ?></p>
                <p><strong>Penyelenggara:</strong> <?= htmlspecialchars($detailEvent['penyelenggara']) ?></p>
            </div>
            
            <div style="margin-top: 20px;">
                <a href="?view=list" class="btn btn-secondary">← Kembali</a>
                <a href="?view=generator&edit=<?= $detailEvent['id'] ?>" class="btn">✏️ Edit Event</a>
            </div>
        </div>
    </div>
    <?php endif; ?>
    <?php endif; ?>
</div>

<script>
function selectSlot(index) {
    document.querySelectorAll('.slot-card').forEach(card => {
        card.classList.remove('selected');
    });
    
    const radioButton = document.getElementById('slot_' + index);
    const card = document.getElementById('card_' + index);
    
    if (radioButton && card) {
        radioButton.checked = true;
        card.classList.add('selected');
    }
}

document.addEventListener('DOMContentLoaded', function() {
    // Auto-select first slot if available
    const firstSlot = document.getElementById('slot_0');
    if (firstSlot && !document.querySelector('input[name="slot_index"]:checked')) {
        selectSlot(0);
    }
    
    // Form validation
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('input[required]');
            let hasError = false;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    field.style.borderColor = '#dc2626';
                    hasError = true;
                } else {
                    field.style.borderColor = '#d1d5db';
                }
            });
            
            if (hasError) {
                e.preventDefault();
                alert('Mohon lengkapi semua field yang diperlukan!');
                return false;
            }
        });
    });
});
</script>

<?php endif; ?>

<!-- Debug info (hanya jika DEBUG_MODE aktif) -->
<?php if (DEBUG_MODE && isLoggedIn()): ?>
<div style="position: fixed; bottom: 10px; right: 10px; background: rgba(0,0,0,0.8); color: white; padding: 10px; border-radius: 4px; font-size: 12px; max-width: 300px;">
    <strong>Debug Info:</strong><br>
    Events Count: <?= count($events) ?><br>
    Data File: <?= file_exists(DATA_FILE) ? 'EXISTS' : 'NOT FOUND' ?><br>
    Data Dir Writable: <?= is_writable(DATA_DIR) ? 'YES' : 'NO' ?><br>
    PHP Version: <?= PHP_VERSION ?><br>
    Session ID: <?= session_id() ?>
</div>
<?php endif; ?>

</body>
</html>