<?php
/**
 * Deployment and Setup Script
 * Script untuk setup awal dan deployment sistem
 */

// Hanya bisa dijalankan via command line atau dengan parameter khusus
if (php_sapi_name() !== 'cli' && (!isset($_GET['setup_key']) || $_GET['setup_key'] !== 'setup_olimpiade_2025')) {
    die('❌ Script ini hanya bisa dijalankan melalui command line atau dengan setup key yang benar');
}

echo "🚀 Sistem Pengelola Event Olimpiade - Deployment Script\n";
echo "=================================================\n\n";

class Deployer {
    
    private $errors = [];
    private $warnings = [];
    private $success = [];
    
    public function __construct() {
        echo "📋 Memulai proses deployment...\n\n";
    }
    
    /**
     * Run full deployment process
     */
    public function deploy() {
        echo "🔍 1. Checking system requirements...\n";
        $this->checkSystemRequirements();
        
        echo "\n📁 2. Setting up directories...\n";
        $this->setupDirectories();
        
        echo "\n🔧 3. Configuring permissions...\n";
        $this->setupPermissions();
        
        echo "\n📄 4. Creating configuration files...\n";
        $this->setupConfigFiles();
        
        echo "\n🛡️ 5. Setting up security...\n";
        $this->setupSecurity();
        
        echo "\n📊 6. Initializing database...\n";
        $this->initializeDatabase();
        
        echo "\n🧪 7. Running tests...\n";
        $this->runTests();
        
        echo "\n📋 8. Generating report...\n";
        $this->generateReport();
        
        return empty($this->errors);
    }
    
    /**
     * Check system requirements
     */
    private function checkSystemRequirements() {
        // PHP Version
        if (version_compare(PHP_VERSION, '7.4.0', '<')) {
            $this->errors[] = "PHP 7.4.0 atau lebih tinggi diperlukan. Versi saat ini: " . PHP_VERSION;
        } else {
            $this->success[] = "✅ PHP Version: " . PHP_VERSION;
        }
        
        // Required extensions
        $requiredExtensions = ['json', 'session', 'date', 'zip', 'fileinfo'];
        foreach ($requiredExtensions as $ext) {
            if (!extension_loaded($ext)) {
                $this->errors[] = "Ekstensi PHP '$ext' tidak tersedia";
            } else {
                $this->success[] = "✅ Extension $ext: tersedia";
            }
        }
        
        // Optional but recommended extensions
        $recommendedExtensions = ['mbstring', 'curl', 'openssl'];
        foreach ($recommendedExtensions as $ext) {
            if (!extension_loaded($ext)) {
                $this->warnings[] = "⚠️ Extension $ext: direkomendasikan tapi tidak tersedia";
            } else {
                $this->success[] = "✅ Extension $ext: tersedia";
            }
        }
        
        // Memory limit
        $memoryLimit = ini_get('memory_limit');
        $memoryBytes = $this->parseBytes($memoryLimit);
        if ($memoryBytes < 64 * 1024 * 1024) { // 64MB
            $this->warnings[] = "⚠️ Memory limit rendah: $memoryLimit (direkomendasikan: 64M+)";
        } else {
            $this->success[] = "✅ Memory limit: $memoryLimit";
        }
        
        // Max execution time
        $maxExecutionTime = ini_get('max_execution_time');
        if ($maxExecutionTime > 0 && $maxExecutionTime < 30) {
            $this->warnings[] = "⚠️ Max execution time rendah: {$maxExecutionTime}s (direkomendasikan: 30s+)";
        } else {
            $this->success[] = "✅ Max execution time: " . ($maxExecutionTime == 0 ? 'unlimited' : $maxExecutionTime . 's');
        }
        
        // Web server detection
        $webServer = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
        echo "🌐 Web Server: $webServer\n";
    }
    
    /**
     * Setup required directories
     */
    private function setupDirectories() {
        $directories = [
            'data' => 'Direktori penyimpanan data utama',
            'backup' => 'Direktori backup otomatis',
            'logs' => 'Direktori log sistem',
            'tmp' => 'Direktori temporary files',
            'uploads' => 'Direktori upload files (opsional)'
        ];
        
        foreach ($directories as $dir => $description) {
            if (!is_dir($dir)) {
                if (mkdir($dir, 0755, true)) {
                    $this->success[] = "✅ Created directory: $dir ($description)";
                } else {
                    $this->errors[] = "❌ Failed to create directory: $dir";
                }
            } else {
                $this->success[] = "✅ Directory exists: $dir";
            }
            
            // Create .htaccess for security
            $htaccessPath = $dir . '/.htaccess';
            if (!file_exists($htaccessPath)) {
                $htaccessContent = "Order Allow,Deny\nDeny from all\n";
                if (file_put_contents($htaccessPath, $htaccessContent)) {
                    $this->success[] = "✅ Created security .htaccess in $dir";
                }
            }
        }
    }
    
    /**
     * Setup file permissions
     */
    private function setupPermissions() {
        $permissions = [
            '.' => 0755,
            'data' => 0755,
            'backup' => 0755,
            'logs' => 0755,
            'tmp' => 0755
        ];
        
        foreach ($permissions as $path => $perm) {
            if (is_dir($path)) {
                if (chmod($path, $perm)) {
                    $this->success[] = "✅ Set permissions for $path: " . decoct($perm);
                } else {
                    $this->warnings[] = "⚠️ Failed to set permissions for $path";
                }
            }
        }
        
        // Check if directories are writable
        $writableCheck = ['data', 'backup', 'logs', 'tmp'];
        foreach ($writableCheck as $dir) {
            if (is_writable($dir)) {
                $this->success[] = "✅ Directory $dir is writable";
            } else {
                $this->errors[] = "❌ Directory $dir is not writable";
            }
        }
    }
    
    /**
     * Setup configuration files
     */
    private function setupConfigFiles() {
        // Check if config.php exists
        if (!file_exists('config.php')) {
            $this->errors[] = "❌ config.php tidak ditemukan. Silakan buat dari template.";
            return;
        }
        
        // Validate config
        try {
            require_once 'config.php';
            $this->success[] = "✅ config.php valid dan terbaca";
            
            // Check if default PIN is still being used
            if (LOGIN_PIN === '051205') {
                $this->warnings[] = "⚠️ PENTING: Ganti PIN default di config.php untuk keamanan!";
            } else {
                $this->success[] = "✅ PIN login telah diubah dari default";
            }
            
            // Check debug mode
            if (DEBUG_MODE === true) {
                $this->warnings[] = "⚠️ Debug mode aktif - nonaktifkan untuk production";
            } else {
                $this->success[] = "✅ Debug mode: nonaktif (production ready)";
            }
            
        } catch (Exception $e) {
            $this->errors[] = "❌ Error loading config.php: " . $e->getMessage();
        }
        
        // Create sample .env file if not exists
        if (!file_exists('.env.example')) {
            $envExample = "# Environment Configuration Example\n";
            $envExample .= "# Copy to .env and modify as needed\n\n";
            $envExample .= "LOGIN_PIN=your_secure_pin_here\n";
            $envExample .= "DEBUG_MODE=false\n";
            $envExample .= "DEFAULT_TIMEZONE=Asia/Jakarta\n";
            $envExample .= "MAX_EVENTS_PER_MONTH=3\n";
            
            if (file_put_contents('.env.example', $envExample)) {
                $this->success[] = "✅ Created .env.example file";
            }
        }
    }
    
    /**
     * Setup security measures
     */
    private function setupSecurity() {
        // Check .htaccess
        if (!file_exists('.htaccess')) {
            $this->warnings[] = "⚠️ .htaccess tidak ditemukan - pertimbangkan untuk menggunakannya";
        } else {
            $this->success[] = "✅ .htaccess file exists";
            
            // Check for security headers
            $htaccessContent = file_get_contents('.htaccess');
            $securityChecks = [
                'X-Frame-Options' => 'Header.*X-Frame-Options',
                'X-Content-Type-Options' => 'Header.*X-Content-Type-Options',
                'X-XSS-Protection' => 'Header.*X-XSS-Protection'
            ];
            
            foreach ($securityChecks as $header => $pattern) {
                if (preg_match("/$pattern/i", $htaccessContent)) {
                    $this->success[] = "✅ Security header configured: $header";
                } else {
                    $this->warnings[] = "⚠️ Security header missing: $header";
                }
            }
        }
        
        // Check file permissions for sensitive files
        $sensitiveFiles = ['config.php', 'functions.php', 'backup.php'];
        foreach ($sensitiveFiles as $file) {
            if (file_exists($file)) {
                $perms = fileperms($file) & 0777;
                if ($perms > 0644) {
                    $this->warnings[] = "⚠️ File $file memiliki permission terlalu permissive: " . decoct($perms);
                } else {
                    $this->success[] = "✅ File $file has secure permissions: " . decoct($perms);
                }
            }
        }
        
        // Check for backup files that should be removed
        $dangerousFiles = ['*.bak', '*.backup', '*.old', '*.tmp', '*~'];
        foreach ($dangerousFiles as $pattern) {
            $files = glob($pattern);
            if (!empty($files)) {
                $this->warnings[] = "⚠️ Backup files found: " . implode(', ', $files) . " (consider removing)";
            }
        }
    }
    
    /**
     * Initialize database/data files
     */
    private function initializeDatabase() {
        try {
            // Initialize Config class
            if (class_exists('Config')) {
                Config::init();
                $this->success[] = "✅ Config system initialized";
            }
            
            // Check data file
            $dataFile = 'data/events.json';
            if (!file_exists($dataFile)) {
                $initialData = [
                    'events' => [],
                    'created_at' => date('Y-m-d H:i:s'),
                    'version' => '1.0.0'
                ];
                
                if (file_put_contents($dataFile, json_encode($initialData, JSON_PRETTY_PRINT))) {
                    $this->success[] = "✅ Created initial data file: $dataFile";
                } else {
                    $this->errors[] = "❌ Failed to create data file: $dataFile";
                }
            } else {
                // Validate existing data file
                $data = json_decode(file_get_contents($dataFile), true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $eventCount = count($data['events'] ?? []);
                    $this->success[] = "✅ Data file valid with $eventCount events";
                } else {
                    $this->errors[] = "❌ Data file corrupted: " . json_last_error_msg();
                }
            }
            
        } catch (Exception $e) {
            $this->errors[] = "❌ Database initialization error: " . $e->getMessage();
        }
    }
    
    /**
     * Run basic functionality tests
     */
    private function runTests() {
        $tests = [
            'Config loading' => function() {
                return class_exists('Config') && defined('APP_NAME');
            },
            'EventManager class' => function() {
                return class_exists('EventManager');
            },
            'Auth class' => function() {
                return class_exists('Auth');
            },
            'DateHelper class' => function() {
                return class_exists('DateHelper');
            },
            'Data directory writable' => function() {
                return is_writable('data');
            },
            'JSON functions' => function() {
                $test = ['test' => 'data'];
                return json_encode($test) && json_decode(json_encode($test), true);
            },
            'Session functionality' => function() {
                if (session_status() === PHP_SESSION_NONE) {
                    session_start();
                }
                $_SESSION['test'] = 'deployment_test';
                return isset($_SESSION['test']);
            }
        ];
        
        foreach ($tests as $testName => $testFunction) {
            try {
                if ($testFunction()) {
                    $this->success[] = "✅ Test passed: $testName";
                } else {
                    $this->errors[] = "❌ Test failed: $testName";
                }
            } catch (Exception $e) {
                $this->errors[] = "❌ Test error ($testName): " . $e->getMessage();
            }
        }
    }
    
    /**
     * Generate deployment report
     */
    private function generateReport() {
        $report = [];
        $report[] = "DEPLOYMENT REPORT - " . date('Y-m-d H:i:s');
        $report[] = str_repeat("=", 50);
        $report[] = "";
        
        // System info
        $report[] = "SYSTEM INFORMATION:";
        $report[] = "- PHP Version: " . PHP_VERSION;
        $report[] = "- Server: " . ($_SERVER['SERVER_SOFTWARE'] ?? 'Unknown');
        $report[] = "- OS: " . PHP_OS;
        $report[] = "- Memory Limit: " . ini_get('memory_limit');
        $report[] = "- Max Execution Time: " . ini_get('max_execution_time') . 's';
        $report[] = "";
        
        // Success items
        if (!empty($this->success)) {
            $report[] = "SUCCESSFUL ITEMS (" . count($this->success) . "):";
            foreach ($this->success as $item) {
                $report[] = "  $item";
            }
            $report[] = "";
        }
        
        // Warnings
        if (!empty($this->warnings)) {
            $report[] = "WARNINGS (" . count($this->warnings) . "):";
            foreach ($this->warnings as $item) {
                $report[] = "  $item";
            }
            $report[] = "";
        }
        
        // Errors
        if (!empty($this->errors)) {
            $report[] = "ERRORS (" . count($this->errors) . "):";
            foreach ($this->errors as $item) {
                $report[] = "  $item";
            }
            $report[] = "";
        }
        
        // Status
        $status = empty($this->errors) ? "SUCCESS" : "FAILED";
        $report[] = "DEPLOYMENT STATUS: $status";
        $report[] = "";
        
        // Next steps
        $report[] = "NEXT STEPS:";
        if (empty($this->errors)) {
            $report[] = "  1. Access the application via web browser";
            $report[] = "  2. Login using the configured PIN";
            $report[] = "  3. Create your first event";
            $report[] = "  4. Setup regular backups";
            $report[] = "  5. Configure SSL/HTTPS for production";
        } else {
            $report[] = "  1. Fix all errors listed above";
            $report[] = "  2. Re-run deployment script";
            $report[] = "  3. Check server logs for additional info";
        }
        $report[] = "";
        
        // Save report
        $reportContent = implode("\n", $report);
        $reportFile = 'deployment_report_' . date('Y-m-d_H-i-s') . '.txt';
        
        if (file_put_contents($reportFile, $reportContent)) {
            echo "📋 Deployment report saved: $reportFile\n\n";
        }
        
        // Display summary
        echo $reportContent;
    }
    
    /**
     * Helper function to parse bytes
     */
    private function parseBytes($val) {
        $val = trim($val);
        $last = strtolower($val[strlen($val)-1]);
        $val = (int)$val;
        switch($last) {
            case 'g': $val *= 1024;
            case 'm': $val *= 1024;
            case 'k': $val *= 1024;
        }
        return $val;
    }
}

// Command line interface
if (php_sapi_name() === 'cli') {
    $deployer = new Deployer();
    $success = $deployer->deploy();
    
    echo "\n" . str_repeat("=", 50) . "\n";
    if ($success) {
        echo "🎉 DEPLOYMENT BERHASIL!\n";
        echo "✅ Sistem siap digunakan.\n";
        echo "🌐 Akses melalui browser untuk memulai.\n";
        exit(0);
    } else {
        echo "❌ DEPLOYMENT GAGAL!\n";
        echo "🔧 Perbaiki error di atas dan coba lagi.\n";
        exit(1);
    }
} else {
    // Web interface
    echo "<pre>";
    $deployer = new Deployer();
    $success = $deployer->deploy();
    echo "</pre>";
    
    if ($success) {
        echo "<h2 style='color: green;'>🎉 Deployment Berhasil!</h2>";
        echo "<p><a href='index.php'>👉 Akses Aplikasi</a></p>";
    } else {
        echo "<h2 style='color: red;'>❌ Deployment Gagal!</h2>";
        echo "<p>Perbaiki error di atas dan coba lagi.</p>";
    }
}
?>