<?php
/**
 * Calendar View Component
 * Komponen untuk menampilkan kalender event
 */

class CalendarView {
    
    /**
     * Generate calendar HTML
     */
    public static function generateCalendar($events, $targetMonth) {
        $firstDay = new DateTime($targetMonth . '-01');
        $lastDay = new DateTime($firstDay->format('Y-m-t'));
        
        // Start calendar from last Sunday of previous month
        $calendarStart = clone $firstDay;
        $calendarStart->modify('last sunday');
        
        // End calendar at next Saturday after last day
        $calendarEnd = clone $lastDay;
        $calendarEnd->modify('next saturday');
        
        // Prepare event data per date
        $eventsPerDate = self::organizeEventsByDate($events);
        
        $html = '<div class="kalender">';
        
        // Header days
        $dayNames = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
        foreach ($dayNames as $dayName) {
            $html .= '<div class="kalender-header">' . $dayName . '</div>';
        }
        
        // Calendar days
        $currentDate = clone $calendarStart;
        while ($currentDate <= $calendarEnd) {
            $html .= self::generateDayCell($currentDate, $targetMonth, $eventsPerDate);
            $currentDate->modify('+1 day');
        }
        
        $html .= '</div>';
        
        // Legend
        $html .= self::generateLegend();
        
        return $html;
    }
    
    /**
     * Organize events by date for calendar display
     */
    private static function organizeEventsByDate($events) {
        $eventsPerDate = [];
        
        foreach ($events as $event) {
            // Registration start
            if (!isset($eventsPerDate[$event['mulai_pendaftaran']])) {
                $eventsPerDate[$event['mulai_pendaftaran']] = [];
            }
            $eventsPerDate[$event['mulai_pendaftaran']][] = [
                'type' => 'registration_start',
                'intensity' => 'medium',
                'event' => $event,
                'title' => 'Mulai Pendaftaran: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
            ];
            
            // Registration end
            if (!isset($eventsPerDate[$event['selesai_pendaftaran']])) {
                $eventsPerDate[$event['selesai_pendaftaran']] = [];
            }
            $eventsPerDate[$event['selesai_pendaftaran']][] = [
                'type' => 'registration_end',
                'intensity' => 'medium',
                'event' => $event,
                'title' => 'Akhir Pendaftaran: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
            ];
            
            // Execution
            if (!isset($eventsPerDate[$event['pelaksanaan']])) {
                $eventsPerDate[$event['pelaksanaan']] = [];
            }
            $eventsPerDate[$event['pelaksanaan']][] = [
                'type' => 'execution',
                'intensity' => 'high',
                'event' => $event,
                'title' => 'Pelaksanaan: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
            ];
            
            // Substitute (if exists)
            if (!empty($event['tanggal_susulan'])) {
                if (!isset($eventsPerDate[$event['tanggal_susulan']])) {
                    $eventsPerDate[$event['tanggal_susulan']] = [];
                }
                $eventsPerDate[$event['tanggal_susulan']][] = [
                    'type' => 'substitute',
                    'intensity' => 'high',
                    'event' => $event,
                    'title' => 'Susulan: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
                ];
            }
            
            // Announcement
            if (!isset($eventsPerDate[$event['pengumuman']])) {
                $eventsPerDate[$event['pengumuman']] = [];
            }
            $eventsPerDate[$event['pengumuman']][] = [
                'type' => 'announcement',
                'intensity' => 'medium',
                'event' => $event,
                'title' => 'Pengumuman: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
            ];
            
            // Final phase dates (if exists)
            if (!empty($event['tanggal_final'])) {
                if (!isset($eventsPerDate[$event['tanggal_final']])) {
                    $eventsPerDate[$event['tanggal_final']] = [];
                }
                $eventsPerDate[$event['tanggal_final']][] = [
                    'type' => 'final',
                    'intensity' => 'high',
                    'event' => $event,
                    'title' => 'Final: ' . ($event['singkatan_lomba'] ?? $event['singkatanLomba'])
                ];
            }
        }
        
        return $eventsPerDate;
    }
    
    /**
     * Generate individual day cell
     */
    private static function generateDayCell($date, $targetMonth, $eventsPerDate) {
        $dateString = $date->format('Y-m-d');
        $isCurrentMonth = $date->format('Y-m') === $targetMonth;
        $isToday = $dateString === date('Y-m-d');
        $dayEvents = $eventsPerDate[$dateString] ?? [];
        
        $classes = ['kalender-hari'];
        if (!$isCurrentMonth) $classes[] = 'bulan-lain';
        if ($isToday) $classes[] = 'hari-ini';
        
        $html = '<div class="' . implode(' ', $classes) . '">';
        $html .= '<div class="tanggal-label">' . $date->format('j') . '</div>';
        
        // Add event dots
        if (!empty($dayEvents)) {
            $html .= '<div class="event-dots">';
            foreach ($dayEvents as $eventItem) {
                $intensityClass = 'event-' . $eventItem['intensity'];
                $typeClass = 'event-type-' . $eventItem['type'];
                $title = htmlspecialchars($eventItem['title']);
                
                $html .= '<div class="event-dot ' . $intensityClass . ' ' . $typeClass . '" ' .
                        'title="' . $title . '" ' .
                        'data-event-id="' . $eventItem['event']['id'] . '">' .
                        '</div>';
            }
            $html .= '</div>';
        }
        
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Generate legend
     */
    private static function generateLegend() {
        $html = '<div class="calendar-legend" style="margin-top: 20px;">';
        $html .= '<h4>📌 Keterangan:</h4>';
        $html .= '<div class="legend-items">';
        
        $legends = [
            ['class' => 'event-low', 'label' => 'Aktivitas Rendah (Pendaftaran berjalan)'],
            ['class' => 'event-medium', 'label' => 'Aktivitas Sedang (Mulai/Akhir Pendaftaran, Pengumuman)'],
            ['class' => 'event-high', 'label' => 'Aktivitas Tinggi (Pelaksanaan, Susulan, Final)']
        ];
        
        foreach ($legends as $legend) {
            $html .= '<p style="display: flex; align-items: center; margin-bottom: 8px;">';
            $html .= '<span class="event-dot ' . $legend['class'] . '" style="margin-right: 8px;"></span>';
            $html .= $legend['label'];
            $html .= '</p>';
        }
        
        $html .= '</div>';
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Get calendar navigation
     */
    public static function getNavigation($currentMonth) {
        $prevMonth = date('Y-m', strtotime($currentMonth . '-01 -1 month'));
        $nextMonth = date('Y-m', strtotime($currentMonth . '-01 +1 month'));
        $monthName = (new DateTime($currentMonth . '-01'))->format('F Y');
        
        return [
            'prev_month' => $prevMonth,
            'next_month' => $nextMonth,
            'current_month_name' => $monthName
        ];
    }
    
    /**
     * Get calendar statistics
     */
    public static function getStatistics($events, $targetMonth) {
        $monthEvents = array_filter($events, function($event) use ($targetMonth) {
            return substr($event['pelaksanaan'], 0, 7) === $targetMonth;
        });
        
        $stats = [
            'total_events' => count($monthEvents),
            'max_events_per_month' => MAX_EVENTS_PER_MONTH,
            'capacity_percentage' => round((count($monthEvents) / MAX_EVENTS_PER_MONTH) * 100),
            'events_this_month' => []
        ];
        
        foreach ($monthEvents as $event) {
            $stats['events_this_month'][] = [
                'name' => $event['nama_lomba'] ?? $event['namaLomba'],
                'short_name' => $event['singkatan_lomba'] ?? $event['singkatanLomba'],
                'execution_date' => $event['pelaksanaan'],
                'status' => self::getEventStatus($event)
            ];
        }
        
        return $stats;
    }
    
    /**
     * Get event status
     */
    private static function getEventStatus($event) {
        $today = date('Y-m-d');
        
        if ($today < $event['mulai_pendaftaran']) {
            return ['status' => 'upcoming', 'label' => '🔵 Akan Datang', 'color' => '#3b82f6'];
        } elseif ($today <= $event['selesai_pendaftaran']) {
            return ['status' => 'registration', 'label' => '🟢 Pendaftaran Aktif', 'color' => '#16a34a'];
        } elseif ($today <= $event['pelaksanaan']) {
            return ['status' => 'pre_execution', 'label' => '🟡 Menuju Pelaksanaan', 'color' => '#ea580c'];
        } else {
            return ['status' => 'completed', 'label' => '⚪ Selesai', 'color' => '#6b7280'];
        }
    }
}

/**
 * Calendar CSS Styles
 */
class CalendarStyles {
    
    public static function getCSS() {
        return '
        <style>
        .kalender {
            display: grid;
            grid-template-columns: repeat(7, 1fr);
            gap: 2px;
            background: #e5e7eb;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
        }
        
        .kalender-header {
            background: #374151;
            color: white;
            padding: 12px 8px;
            text-align: center;
            font-weight: bold;
            font-size: 14px;
        }
        
        .kalender-hari {
            background: white;
            min-height: 100px;
            padding: 8px;
            position: relative;
            transition: background-color 0.2s;
            cursor: pointer;
        }
        
        .kalender-hari:hover {
            background: #f9fafb;
        }
        
        .kalender-hari.bulan-lain {
            background: #f9fafb;
            color: #9ca3af;
        }
        
        .kalender-hari.bulan-lain:hover {
            background: #f3f4f6;
        }
        
        .kalender-hari.hari-ini {
            background: #dbeafe;
            border: 2px solid #3b82f6;
        }
        
        .kalender-hari.hari-ini .tanggal-label {
            background: #3b82f6;
            color: white;
            border-radius: 50%;
            width: 24px;
            height: 24px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }
        
        .tanggal-label {
            font-weight: 600;
            margin-bottom: 5px;
            font-size: 14px;
        }
        
        .event-dots {
            display: flex;
            flex-wrap: wrap;
            gap: 2px;
        }
        
        .event-dot {
            width: 8px;
            height: 8px;
            border-radius: 50%;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .event-dot:hover {
            transform: scale(1.3);
        }
        
        .event-low {
            background: #16a34a;
        }
        
        .event-medium {
            background: #ea580c;
        }
        
        .event-high {
            background: #dc2626;
        }
        
        .event-type-registration_start {
            border: 2px solid #059669;
        }
        
        .event-type-registration_end {
            border: 2px solid #d97706;
        }
        
        .event-type-execution {
            border: 2px solid #b91c1c;
            width: 10px;
            height: 10px;
        }
        
        .event-type-substitute {
            border: 2px solid #7c2d12;
            width: 10px;
            height: 10px;
        }
        
        .event-type-announcement {
            border: 2px solid #9333ea;
        }
        
        .event-type-final {
            border: 2px solid #1e40af;
            width: 10px;
            height: 10px;
        }
        
        .calendar-legend {
            background: #f8fafc;
            padding: 15px;
            border-radius: 6px;
            border: 1px solid #e2e8f0;
        }
        
        .calendar-legend h4 {
            margin-bottom: 10px;
            color: #374151;
        }
        
        .legend-items p {
            margin: 0;
            font-size: 14px;
            color: #4b5563;
        }
        
        .calendar-stats {
            background: #f0f9ff;
            border: 1px solid #0ea5e9;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
        }
        
        .calendar-stats h4 {
            color: #0c4a6e;
            margin-bottom: 10px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 10px;
        }
        
        .stat-item {
            background: white;
            padding: 10px;
            border-radius: 4px;
            border-left: 4px solid #0ea5e9;
        }
        
        .stat-value {
            font-size: 18px;
            font-weight: bold;
            color: #0c4a6e;
        }
        
        .stat-label {
            font-size: 12px;
            color: #64748b;
            margin-top: 2px;
        }
        
        .month-events-list {
            margin-top: 15px;
        }
        
        .month-event-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 12px;
            background: white;
            border-radius: 4px;
            margin-bottom: 5px;
            border-left: 3px solid #0ea5e9;
        }
        
        .event-info {
            flex: 1;
        }
        
        .event-name {
            font-weight: 600;
            color: #374151;
        }
        
        .event-date {
            font-size: 12px;
            color: #6b7280;
        }
        
        .event-status {
            font-size: 12px;
            padding: 2px 6px;
            border-radius: 12px;
            white-space: nowrap;
        }
        
        @media (max-width: 768px) {
            .kalender {
                font-size: 12px;
            }
            
            .kalender-hari {
                min-height: 80px;
                padding: 4px;
            }
            
            .kalender-header {
                padding: 8px 4px;
                font-size: 12px;
            }
            
            .event-dot {
                width: 6px;
                height: 6px;
            }
            
            .stats-grid {
                grid-template-columns: 1fr;
            }
        }
        </style>';
    }
}
?>