<?php
/**
 * Backup and Restore Utility
 * Utilitas untuk backup dan restore data sistem
 */

require_once __DIR__ . '/config.php';

// Security check - hanya bisa diakses melalui command line atau dengan PIN
if (php_sapi_name() !== 'cli') {
    if (!isset($_GET['key']) || $_GET['key'] !== LOGIN_PIN) {
        die('Access denied');
    }
}

class BackupManager {
    
    private $backupDir;
    private $dataDir;
    
    public function __construct() {
        $this->backupDir = __DIR__ . '/backup';
        $this->dataDir = DATA_DIR;
        
        // Create backup directory if not exists
        if (!is_dir($this->backupDir)) {
            mkdir($this->backupDir, 0755, true);
        }
    }
    
    /**
     * Create full system backup
     */
    public function createFullBackup($description = '') {
        $timestamp = date('Y-m-d_H-i-s');
        $backupFile = $this->backupDir . '/backup_' . $timestamp . '.zip';
        
        $zip = new ZipArchive();
        if ($zip->open($backupFile, ZipArchive::CREATE) !== TRUE) {
            throw new Exception('Cannot create backup file: ' . $backupFile);
        }
        
        // Add data files
        $this->addDirectoryToZip($zip, $this->dataDir, 'data/');
        
        // Add application files (excluding sensitive ones)
        $appFiles = ['index.php', 'config.php', 'functions.php', 'calendar.php', 'analysis.php'];
        foreach ($appFiles as $file) {
            if (file_exists($file)) {
                $zip->addFile($file, 'app/' . $file);
            }
        }
        
        // Add backup metadata
        $metadata = [
            'created_at' => date(DATETIME_FORMAT),
            'description' => $description,
            'version' => APP_VERSION,
            'php_version' => PHP_VERSION,
            'files_count' => $zip->numFiles
        ];
        
        $zip->addFromString('backup_metadata.json', json_encode($metadata, JSON_PRETTY_PRINT));
        
        if ($zip->close()) {
            Config::log('Full backup created: ' . $backupFile);
            return [
                'success' => true,
                'file' => $backupFile,
                'size' => filesize($backupFile),
                'metadata' => $metadata
            ];
        } else {
            throw new Exception('Failed to create backup');
        }
    }
    
    /**
     * Create data-only backup
     */
    public function createDataBackup($description = '') {
        $timestamp = date('Y-m-d_H-i-s');
        $backupFile = $this->backupDir . '/data_backup_' . $timestamp . '.json';
        
        // Load current data
        $data = [
            'backup_info' => [
                'created_at' => date(DATETIME_FORMAT),
                'description' => $description,
                'version' => APP_VERSION,
                'type' => 'data_only'
            ],
            'events' => EventManager::loadEvents(),
            'system_info' => Config::getAppInfo()
        ];
        
        if (file_put_contents($backupFile, json_encode($data, JSON_PRETTY_PRINT))) {
            Config::log('Data backup created: ' . $backupFile);
            return [
                'success' => true,
                'file' => $backupFile,
                'size' => filesize($backupFile),
                'events_count' => count($data['events'])
            ];
        } else {
            throw new Exception('Failed to create data backup');
        }
    }
    
    /**
     * Restore from backup
     */
    public function restoreFromBackup($backupFile, $restoreType = 'data_only') {
        if (!file_exists($backupFile)) {
            throw new Exception('Backup file not found: ' . $backupFile);
        }
        
        // Create restore point first
        $restorePoint = $this->createDataBackup('Auto restore point before restore');
        
        try {
            if ($restoreType === 'full' && pathinfo($backupFile, PATHINFO_EXTENSION) === 'zip') {
                return $this->restoreFullBackup($backupFile);
            } else {
                return $this->restoreDataBackup($backupFile);
            }
        } catch (Exception $e) {
            // Restore failed, revert to restore point
            if ($restorePoint['success']) {
                $this->restoreDataBackup($restorePoint['file']);
            }
            throw $e;
        }
    }
    
    /**
     * Restore full backup from ZIP
     */
    private function restoreFullBackup($zipFile) {
        $zip = new ZipArchive();
        if ($zip->open($zipFile) !== TRUE) {
            throw new Exception('Cannot open backup file: ' . $zipFile);
        }
        
        // Extract to temporary directory first
        $tempDir = sys_get_temp_dir() . '/restore_' . uniqid();
        mkdir($tempDir, 0755, true);
        
        $zip->extractTo($tempDir);
        $zip->close();
        
        // Validate backup
        if (!file_exists($tempDir . '/backup_metadata.json')) {
            throw new Exception('Invalid backup file - metadata missing');
        }
        
        $metadata = json_decode(file_get_contents($tempDir . '/backup_metadata.json'), true);
        
        // Restore data directory
        if (is_dir($tempDir . '/data')) {
            $this->copyDirectory($tempDir . '/data', $this->dataDir);
        }
        
        // Clean up
        $this->removeDirectory($tempDir);
        
        Config::log('Full backup restored: ' . $zipFile);
        return [
            'success' => true,
            'metadata' => $metadata,
            'restored_at' => date(DATETIME_FORMAT)
        ];
    }
    
    /**
     * Restore data backup from JSON
     */
    private function restoreDataBackup($jsonFile) {
        $data = json_decode(file_get_contents($jsonFile), true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON backup file');
        }
        
        if (!isset($data['events'])) {
            throw new Exception('Invalid backup format - events data missing');
        }
        
        // Restore events
        if (EventManager::saveEvents($data['events'])) {
            Config::log('Data backup restored: ' . $jsonFile);
            return [
                'success' => true,
                'events_count' => count($data['events']),
                'backup_info' => $data['backup_info'] ?? [],
                'restored_at' => date(DATETIME_FORMAT)
            ];
        } else {
            throw new Exception('Failed to save restored data');
        }
    }
    
    /**
     * List available backups
     */
    public function listBackups() {
        $backups = [];
        $files = glob($this->backupDir . '/*');
        
        foreach ($files as $file) {
            if (is_file($file)) {
                $backup = [
                    'filename' => basename($file),
                    'filepath' => $file,
                    'size' => filesize($file),
                    'created_at' => date(DATETIME_FORMAT, filemtime($file)),
                    'type' => $this->getBackupType($file)
                ];
                
                // Try to get metadata
                if ($backup['type'] === 'full') {
                    $backup['metadata'] = $this->getZipMetadata($file);
                } elseif ($backup['type'] === 'data') {
                    $backup['metadata'] = $this->getJsonMetadata($file);
                }
                
                $backups[] = $backup;
            }
        }
        
        // Sort by creation time (newest first)
        usort($backups, function($a, $b) {
            return strtotime($b['created_at']) - strtotime($a['created_at']);
        });
        
        return $backups;
    }
    
    /**
     * Clean old backups
     */
    public function cleanOldBackups($keepDays = 30, $keepMinimum = 5) {
        $backups = $this->listBackups();
        $cutoffTime = time() - ($keepDays * 24 * 60 * 60);
        $deleted = 0;
        
        // Skip the minimum number of newest backups
        $backupsToCheck = array_slice($backups, $keepMinimum);
        
        foreach ($backupsToCheck as $backup) {
            if (filemtime($backup['filepath']) < $cutoffTime) {
                if (unlink($backup['filepath'])) {
                    $deleted++;
                    Config::log('Old backup deleted: ' . $backup['filename']);
                }
            }
        }
        
        return [
            'deleted_count' => $deleted,
            'remaining_count' => count($backups) - $deleted
        ];
    }
    
    /**
     * Export events to CSV
     */
    public function exportToCSV($outputFile = null) {
        $events = EventManager::loadEvents();
        
        if (!$outputFile) {
            $outputFile = $this->backupDir . '/events_export_' . date('Y-m-d_H-i-s') . '.csv';
        }
        
        $handle = fopen($outputFile, 'w');
        if (!$handle) {
            throw new Exception('Cannot create CSV file: ' . $outputFile);
        }
        
        // CSV headers
        $headers = [
            'ID', 'Nama Lomba', 'Singkatan', 'Penyelenggara',
            'Mulai Pendaftaran', 'Selesai Pendaftaran', 'Pelaksanaan',
            'Susulan', 'Tanggal Susulan', 'Pengumuman',
            'Babak Final', 'Dibuat Pada'
        ];
        
        fputcsv($handle, $headers);
        
        // Event data
        foreach ($events as $event) {
            $row = [
                $event['id'],
                $event['nama_lomba'] ?? $event['namaLomba'],
                $event['singkatan_lomba'] ?? $event['singkatanLomba'],
                $event['penyelenggara'],
                $event['mulai_pendaftaran'],
                $event['selesai_pendaftaran'],
                $event['pelaksanaan'],
                $event['susulan'] ?? 'tidak',
                $event['tanggal_susulan'] ?? '',
                $event['pengumuman'],
                $event['babak_final'] ?? 'tidak',
                $event['dibuat_pada'] ?? ''
            ];
            
            fputcsv($handle, $row);
        }
        
        fclose($handle);
        
        Config::log('Events exported to CSV: ' . $outputFile);
        return [
            'success' => true,
            'file' => $outputFile,
            'size' => filesize($outputFile),
            'events_count' => count($events)
        ];
    }
    
    /**
     * Helper methods
     */
    
    private function addDirectoryToZip($zip, $dir, $zipPath = '') {
        if (is_dir($dir)) {
            $files = scandir($dir);
            foreach ($files as $file) {
                if ($file != '.' && $file != '..') {
                    $fullPath = $dir . '/' . $file;
                    if (is_dir($fullPath)) {
                        $this->addDirectoryToZip($zip, $fullPath, $zipPath . $file . '/');
                    } else {
                        $zip->addFile($fullPath, $zipPath . $file);
                    }
                }
            }
        }
    }
    
    private function copyDirectory($src, $dst) {
        if (!is_dir($dst)) {
            mkdir($dst, 0755, true);
        }
        
        $files = scandir($src);
        foreach ($files as $file) {
            if ($file != '.' && $file != '..') {
                $srcFile = $src . '/' . $file;
                $dstFile = $dst . '/' . $file;
                
                if (is_dir($srcFile)) {
                    $this->copyDirectory($srcFile, $dstFile);
                } else {
                    copy($srcFile, $dstFile);
                }
            }
        }
    }
    
    private function removeDirectory($dir) {
        if (is_dir($dir)) {
            $files = scandir($dir);
            foreach ($files as $file) {
                if ($file != '.' && $file != '..') {
                    $fullPath = $dir . '/' . $file;
                    if (is_dir($fullPath)) {
                        $this->removeDirectory($fullPath);
                    } else {
                        unlink($fullPath);
                    }
                }
            }
            rmdir($dir);
        }
    }
    
    private function getBackupType($file) {
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        if ($ext === 'zip') return 'full';
        if ($ext === 'json') return 'data';
        if ($ext === 'csv') return 'export';
        return 'unknown';
    }
    
    private function getZipMetadata($zipFile) {
        $zip = new ZipArchive();
        if ($zip->open($zipFile) === TRUE) {
            $metadataContent = $zip->getFromName('backup_metadata.json');
            $zip->close();
            
            if ($metadataContent) {
                return json_decode($metadataContent, true);
            }
        }
        return null;
    }
    
    private function getJsonMetadata($jsonFile) {
        $data = json_decode(file_get_contents($jsonFile), true);
        return $data['backup_info'] ?? null;
    }
}

// Command line interface
if (php_sapi_name() === 'cli') {
    $backup = new BackupManager();
    
    $command = $argv[1] ?? 'help';
    
    switch ($command) {
        case 'create':
            $description = $argv[2] ?? 'Manual backup from CLI';
            try {
                $result = $backup->createDataBackup($description);
                echo "Backup created successfully:\n";
                echo "File: " . $result['file'] . "\n";
                echo "Size: " . number_format($result['size']) . " bytes\n";
                echo "Events: " . $result['events_count'] . "\n";
            } catch (Exception $e) {
                echo "Error: " . $e->getMessage() . "\n";
                exit(1);
            }
            break;
            
        case 'create-full':
            $description = $argv[2] ?? 'Full backup from CLI';
            try {
                $result = $backup->createFullBackup($description);
                echo "Full backup created successfully:\n";
                echo "File: " . $result['file'] . "\n";
                echo "Size: " . number_format($result['size']) . " bytes\n";
            } catch (Exception $e) {
                echo "Error: " . $e->getMessage() . "\n";
                exit(1);
            }
            break;
            
        case 'list':
            $backups = $backup->listBackups();
            echo "Available backups:\n";
            echo str_pad("Filename", 30) . str_pad("Type", 10) . str_pad("Size", 12) . "Created\n";
            echo str_repeat("-", 70) . "\n";
            
            foreach ($backups as $b) {
                echo str_pad($b['filename'], 30) . 
                     str_pad($b['type'], 10) . 
                     str_pad(number_format($b['size']), 12) . 
                     $b['created_at'] . "\n";
            }
            break;
            
        case 'restore':
            $backupFile = $argv[2] ?? null;
            if (!$backupFile) {
                echo "Usage: php backup.php restore <backup_file>\n";
                exit(1);
            }
            
            try {
                $result = $backup->restoreFromBackup($backupFile);
                echo "Restore completed successfully:\n";
                echo "Events restored: " . ($result['events_count'] ?? 'N/A') . "\n";
                echo "Restored at: " . $result['restored_at'] . "\n";
            } catch (Exception $e) {
                echo "Error: " . $e->getMessage() . "\n";
                exit(1);
            }
            break;
            
        case 'export':
            try {
                $result = $backup->exportToCSV();
                echo "Export completed successfully:\n";
                echo "File: " . $result['file'] . "\n";
                echo "Size: " . number_format($result['size']) . " bytes\n";
                echo "Events: " . $result['events_count'] . "\n";
            } catch (Exception $e) {
                echo "Error: " . $e->getMessage() . "\n";
                exit(1);
            }
            break;
            
        case 'clean':
            $keepDays = (int)($argv[2] ?? 30);
            $keepMinimum = (int)($argv[3] ?? 5);
            
            try {
                $result = $backup->cleanOldBackups($keepDays, $keepMinimum);
                echo "Cleanup completed:\n";
                echo "Deleted: " . $result['deleted_count'] . " backups\n";
                echo "Remaining: " . $result['remaining_count'] . " backups\n";
            } catch (Exception $e) {
                echo "Error: " . $e->getMessage() . "\n";
                exit(1);
            }
            break;
            
        default:
            echo "Backup Manager for Sistem Pengelola Event Olimpiade\n\n";
            echo "Available commands:\n";
            echo "  create [description]     - Create data backup\n";
            echo "  create-full [description] - Create full system backup\n";
            echo "  list                     - List available backups\n";
            echo "  restore <file>           - Restore from backup\n";
            echo "  export                   - Export events to CSV\n";
            echo "  clean [days] [minimum]   - Clean old backups (default: 30 days, keep 5)\n";
            echo "  help                     - Show this help\n\n";
            echo "Examples:\n";
            echo "  php backup.php create 'Before major update'\n";
            echo "  php backup.php restore backup/data_backup_2025-01-10_14-30-00.json\n";
            echo "  php backup.php clean 7 3\n";
            break;
    }
}

// Web interface (requires PIN)
if (php_sapi_name() !== 'cli' && isset($_GET['action'])) {
    header('Content-Type: application/json');
    
    try {
        $backup = new BackupManager();
        $action = $_GET['action'];
        
        switch ($action) {
            case 'create':
                $description = $_POST['description'] ?? 'Manual backup from web';
                $result = $backup->createDataBackup($description);
                echo json_encode($result);
                break;
                
            case 'list':
                $result = $backup->listBackups();
                echo json_encode($result);
                break;
                
            case 'export':
                $result = $backup->exportToCSV();
                echo json_encode($result);
                break;
                
            default:
                throw new Exception('Unknown action');
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}
?>