<?php
/**
 * Analysis View Component
 * Komponen untuk analisis beban kerja dan statistik
 */

class AnalysisView {
    
    /**
     * Generate workload analysis
     */
    public static function generateWorkloadAnalysis($events, $targetMonth) {
        $workload = EventManager::calculateWorkload($events, $targetMonth);
        $weeks = self::generateWeeksForMonth($targetMonth);
        $monthStats = self::calculateMonthStatistics($events, $targetMonth);
        
        $html = '';
        
        // Month summary
        $html .= self::generateMonthSummary($monthStats);
        
        // Workload table
        $html .= self::generateWorkloadTable($weeks, $workload, $events, $targetMonth);
        
        // Insights and recommendations
        $html .= self::generateInsights($monthStats, $workload, $weeks);
        
        // Workload chart
        $html .= self::generateWorkloadChart($weeks, $workload);
        
        return $html;
    }
    
    /**
     * Generate weeks for target month
     */
    private static function generateWeeksForMonth($targetMonth) {
        $firstDay = new DateTime($targetMonth . '-01');
        $weeks = [];
        
        // Start from first Monday of the month (or before)
        $startWeek = clone $firstDay;
        $startWeek->modify('last monday');
        
        for ($i = 0; $i < 6; $i++) {
            $weekStart = clone $startWeek;
            $weekStart->modify("+$i weeks");
            $weekEnd = clone $weekStart;
            $weekEnd->modify('+6 days');
            $weekCode = $weekStart->format('Y-W');
            
            $weeks[$weekCode] = [
                'start' => $weekStart->format('Y-m-d'),
                'end' => $weekEnd->format('Y-m-d'),
                'week_number' => (int)$weekStart->format('W'),
                'month_coverage' => self::calculateMonthCoverage($weekStart, $weekEnd, $targetMonth)
            ];
        }
        
        return $weeks;
    }
    
    /**
     * Calculate how much of the week is in target month
     */
    private static function calculateMonthCoverage($weekStart, $weekEnd, $targetMonth) {
        $coverage = 0;
        $current = clone $weekStart;
        
        while ($current <= $weekEnd) {
            if ($current->format('Y-m') === $targetMonth) {
                $coverage++;
            }
            $current->modify('+1 day');
        }
        
        return $coverage;
    }
    
    /**
     * Calculate month statistics
     */
    private static function calculateMonthStatistics($events, $targetMonth) {
        $monthEvents = array_filter($events, function($event) use ($targetMonth) {
            return substr($event['pelaksanaan'], 0, 7) === $targetMonth;
        });
        
        $stats = [
            'total_events' => count($monthEvents),
            'max_capacity' => MAX_EVENTS_PER_MONTH,
            'capacity_used' => count($monthEvents),
            'capacity_percentage' => round((count($monthEvents) / MAX_EVENTS_PER_MONTH) * 100),
            'events' => $monthEvents,
            'event_types' => self::analyzeEventTypes($monthEvents),
            'timeline_overlap' => self::analyzeTimelineOverlap($monthEvents)
        ];
        
        return $stats;
    }
    
    /**
     * Analyze event types
     */
    private static function analyzeEventTypes($events) {
        $types = [
            'with_final' => 0,
            'with_substitute' => 0,
            'regular' => 0
        ];
        
        foreach ($events as $event) {
            if (($event['babak_final'] ?? 'tidak') === 'ya') {
                $types['with_final']++;
            } elseif (($event['susulan'] ?? 'tidak') === 'ya') {
                $types['with_substitute']++;
            } else {
                $types['regular']++;
            }
        }
        
        return $types;
    }
    
    /**
     * Analyze timeline overlap
     */
    private static function analyzeTimelineOverlap($events) {
        $overlaps = [];
        
        for ($i = 0; $i < count($events); $i++) {
            for ($j = $i + 1; $j < count($events); $j++) {
                $event1 = $events[$i];
                $event2 = $events[$j];
                
                // Check various phase overlaps
                $phases1 = self::getEventPhases($event1);
                $phases2 = self::getEventPhases($event2);
                
                foreach ($phases1 as $phase1Name => $phase1) {
                    foreach ($phases2 as $phase2Name => $phase2) {
                        if (self::datesOverlap($phase1['start'], $phase1['end'], $phase2['start'], $phase2['end'])) {
                            $overlaps[] = [
                                'event1' => $event1['singkatan_lomba'] ?? $event1['singkatanLomba'],
                                'event2' => $event2['singkatan_lomba'] ?? $event2['singkatanLomba'],
                                'phase1' => $phase1Name,
                                'phase2' => $phase2Name,
                                'overlap_days' => self::calculateOverlapDays($phase1['start'], $phase1['end'], $phase2['start'], $phase2['end'])
                            ];
                        }
                    }
                }
            }
        }
        
        return $overlaps;
    }
    
    /**
     * Get event phases
     */
    private static function getEventPhases($event) {
        $phases = [
            'pendaftaran' => [
                'start' => $event['mulai_pendaftaran'],
                'end' => $event['selesai_pendaftaran']
            ],
            'pelaksanaan' => [
                'start' => $event['pelaksanaan'],
                'end' => $event['pelaksanaan']
            ]
        ];
        
        if (!empty($event['klaim_gel1_mulai'])) {
            $phases['klaim_gel1'] = [
                'start' => $event['klaim_gel1_mulai'],
                'end' => $event['klaim_gel1_selesai']
            ];
        }
        
        if (!empty($event['klaim_gel2_mulai'])) {
            $phases['klaim_gel2'] = [
                'start' => $event['klaim_gel2_mulai'],
                'end' => $event['klaim_gel2_selesai']
            ];
        }
        
        return $phases;
    }
    
    /**
     * Check if two date ranges overlap
     */
    private static function datesOverlap($start1, $end1, $start2, $end2) {
        return strtotime($start1) <= strtotime($end2) && strtotime($end1) >= strtotime($start2);
    }
    
    /**
     * Calculate overlap days
     */
    private static function calculateOverlapDays($start1, $end1, $start2, $end2) {
        $overlapStart = max(strtotime($start1), strtotime($start2));
        $overlapEnd = min(strtotime($end1), strtotime($end2));
        
        if ($overlapStart <= $overlapEnd) {
            return ($overlapEnd - $overlapStart) / (60 * 60 * 24) + 1;
        }
        
        return 0;
    }
    
    /**
     * Generate month summary
     */
    private static function generateMonthSummary($stats) {
        $html = '<div class="analysis-summary">';
        $html .= '<h4>📊 Ringkasan Bulan Ini</h4>';
        $html .= '<div class="summary-grid">';
        
        // Total events
        $html .= '<div class="summary-item">';
        $html .= '<div class="summary-value">' . $stats['total_events'] . '/' . $stats['max_capacity'] . '</div>';
        $html .= '<div class="summary-label">Total Event</div>';
        $html .= '</div>';
        
        // Capacity percentage
        $capacityColor = $stats['capacity_percentage'] >= 100 ? '#dc2626' : 
                        ($stats['capacity_percentage'] >= 67 ? '#ea580c' : '#16a34a');
        $html .= '<div class="summary-item">';
        $html .= '<div class="summary-value" style="color: ' . $capacityColor . '">' . $stats['capacity_percentage'] . '%</div>';
        $html .= '<div class="summary-label">Kapasitas Terpakai</div>';
        $html .= '</div>';
        
        // Event types
        $html .= '<div class="summary-item">';
        $html .= '<div class="summary-value">' . $stats['event_types']['with_final'] . '</div>';
        $html .= '<div class="summary-label">Event dengan Final</div>';
        $html .= '</div>';
        
        // Timeline overlaps
        $html .= '<div class="summary-item">';
        $html .= '<div class="summary-value">' . count($stats['timeline_overlap']) . '</div>';
        $html .= '<div class="summary-label">Overlap Timeline</div>';
        $html .= '</div>';
        
        $html .= '</div>';
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Generate workload table
     */
    private static function generateWorkloadTable($weeks, $workload, $events, $targetMonth) {
        $html = '<div class="workload-table-container">';
        $html .= '<h4>📅 Analisis Beban Kerja per Minggu</h4>';
        $html .= '<table class="workload-table">';
        $html .= '<thead>';
        $html .= '<tr>';
        $html .= '<th>Minggu</th>';
        $html .= '<th>Tanggal</th>';
        $html .= '<th>Beban Kerja</th>';
        $html .= '<th>Status</th>';
        $html .= '<th>Event di Minggu Ini</th>';
        $html .= '<th>Fase Aktif</th>';
        $html .= '</tr>';
        $html .= '</thead>';
        $html .= '<tbody>';
        
        foreach ($weeks as $weekCode => $weekData) {
            $weekWorkload = $workload[$weekCode] ?? 0;
            $weekEvents = self::getWeekEvents($events, $weekData['start'], $weekData['end']);
            $activePhases = self::getActivePhases($events, $weekData['start'], $weekData['end']);
            
            $statusClass = '';
            $statusText = '';
            if ($weekWorkload > WORKLOAD_CONFIG['max_weekly_load']) {
                $statusClass = 'status-critical';
                $statusText = '🔴 Sangat Padat';
            } elseif ($weekWorkload > 4) {
                $statusClass = 'status-high';
                $statusText = '🟡 Padat';
            } elseif ($weekWorkload > 2) {
                $statusClass = 'status-medium';
                $statusText = '🟠 Sedang';
            } else {
                $statusClass = 'status-low';
                $statusText = '🟢 Ringan';
            }
            
            $html .= '<tr class="' . $statusClass . '">';
            $html .= '<td>Minggu ' . $weekData['week_number'] . '</td>';
            $html .= '<td>' . self::formatWeekRange($weekData['start'], $weekData['end']) . '</td>';
            $html .= '<td><strong>' . $weekWorkload . '</strong> poin</td>';
            $html .= '<td>' . $statusText . '</td>';
            $html .= '<td>' . self::formatWeekEvents($weekEvents) . '</td>';
            $html .= '<td>' . self::formatActivePhases($activePhases) . '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody>';
        $html .= '</table>';
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Get events in specific week
     */
    private static function getWeekEvents($events, $weekStart, $weekEnd) {
        return array_filter($events, function($event) use ($weekStart, $weekEnd) {
            $executionDate = $event['pelaksanaan'];
            return $executionDate >= $weekStart && $executionDate <= $weekEnd;
        });
    }
    
    /**
     * Get active phases in specific week
     */
    private static function getActivePhases($events, $weekStart, $weekEnd) {
        $activePhases = [];
        
        foreach ($events as $event) {
            $phases = self::getEventPhases($event);
            
            foreach ($phases as $phaseName => $phase) {
                if (self::datesOverlap($weekStart, $weekEnd, $phase['start'], $phase['end'])) {
                    $activePhases[] = [
                        'event' => $event['singkatan_lomba'] ?? $event['singkatanLomba'],
                        'phase' => $phaseName
                    ];
                }
            }
        }
        
        return $activePhases;
    }
    
    /**
     * Format week range
     */
    private static function formatWeekRange($start, $end) {
        $startDate = new DateTime($start);
        $endDate = new DateTime($end);
        
        return $startDate->format('d M') . ' - ' . $endDate->format('d M Y');
    }
    
    /**
     * Format week events
     */
    private static function formatWeekEvents($events) {
        if (empty($events)) {
            return '<span class="no-events">-</span>';
        }
        
        $html = '';
        foreach ($events as $event) {
            $shortName = $event['singkatan_lomba'] ?? $event['singkatanLomba'];
            $html .= '<span class="event-tag">' . htmlspecialchars($shortName) . '</span> ';
        }
        
        return $html;
    }
    
    /**
     * Format active phases
     */
    private static function formatActivePhases($phases) {
        if (empty($phases)) {
            return '<span class="no-phases">-</span>';
        }
        
        $html = '';
        foreach ($phases as $phase) {
            $html .= '<span class="phase-tag">' . $phase['event'] . ':' . $phase['phase'] . '</span> ';
        }
        
        return $html;
    }
    
    /**
     * Generate insights and recommendations
     */
    private static function generateInsights($stats, $workload, $weeks) {
        $html = '<div class="insights-section">';
        $html .= '<h4>💡 Insight dan Rekomendasi</h4>';
        
        // Find busiest week
        $busiestWeek = '';
        $maxWorkload = 0;
        foreach ($workload as $weekCode => $load) {
            if ($load > $maxWorkload) {
                $maxWorkload = $load;
                $busiestWeek = $weekCode;
            }
        }
        
        $html .= '<div class="insight-grid">';
        
        // Busiest week insight
        $html .= '<div class="insight-item">';
        $html .= '<h5>📈 Minggu Terpadat</h5>';
        if ($maxWorkload > 0) {
            $weekNumber = substr($busiestWeek, -2);
            $html .= '<p>Minggu ' . $weekNumber . ' dengan beban kerja <strong>' . $maxWorkload . ' poin</strong></p>';
            if ($maxWorkload > WORKLOAD_CONFIG['max_weekly_load']) {
                $html .= '<p class="recommendation">⚠️ Beban kerja berlebihan! Pertimbangkan untuk mengatur ulang jadwal.</p>';
            }
        } else {
            $html .= '<p>Tidak ada beban kerja bulan ini</p>';
        }
        $html .= '</div>';
        
        // Capacity analysis
        $html .= '<div class="insight-item">';
        $html .= '<h5>📊 Analisis Kapasitas</h5>';
        if ($stats['capacity_percentage'] >= 100) {
            $html .= '<p class="recommendation">🔴 Kapasitas sudah maksimal! Tidak bisa menambah event lagi.</p>';
        } elseif ($stats['capacity_percentage'] >= 67) {
            $html .= '<p class="recommendation">🟡 Mendekati batas kapasitas. Hati-hati menambah event baru.</p>';
        } else {
            $remaining = $stats['max_capacity'] - $stats['capacity_used'];
            $html .= '<p class="recommendation">🟢 Masih bisa menambah <strong>' . $remaining . ' event</strong> lagi.</p>';
        }
        $html .= '</div>';
        
        // Timeline overlap analysis
        $html .= '<div class="insight-item">';
        $html .= '<h5>🔄 Analisis Overlap</h5>';
        $overlapCount = count($stats['timeline_overlap']);
        if ($overlapCount > 0) {
            $html .= '<p>Ditemukan <strong>' . $overlapCount . ' overlap</strong> timeline antar event</p>';
            $html .= '<p class="recommendation">⚠️ Periksa jadwal untuk menghindari konflik sumber daya.</p>';
        } else {
            $html .= '<p class="recommendation">✅ Tidak ada overlap timeline yang bermasalah.</p>';
        }
        $html .= '</div>';
        
        // Distribution recommendation
        $html .= '<div class="insight-item">';
        $html .= '<h5>📅 Rekomendasi Distribusi</h5>';
        $highWorkloadWeeks = count(array_filter($workload, function($load) {
            return $load > WORKLOAD_CONFIG['max_weekly_load'];
        }));
        
        if ($highWorkloadWeeks > 1) {
            $html .= '<p class="recommendation">🔴 ' . $highWorkloadWeeks . ' minggu dengan beban tinggi. Pertimbangkan redistribusi event.</p>';
        } elseif ($highWorkloadWeeks == 1) {
            $html .= '<p class="recommendation">🟡 1 minggu dengan beban tinggi. Masih dalam batas toleransi.</p>';
        } else {
            $html .= '<p class="recommendation">✅ Distribusi beban kerja sudah optimal.</p>';
        }
        $html .= '</div>';
        
        $html .= '</div>';
        $html .= '</div>';
        
        return $html;
    }
    
    /**
     * Generate workload chart
     */
    private static function generateWorkloadChart($weeks, $workload) {
        $html = '<div class="workload-chart-section">';
        $html .= '<h4>📈 Grafik Beban Kerja</h4>';
        $html .= '<div class="chart-container">';
        
        $maxLoad = max(array_merge($workload, [WORKLOAD_CONFIG['max_weekly_load']]));
        $chartHeight = 200;
        
        $html .= '<div class="chart-bars">';
        foreach ($weeks as $weekCode => $weekData) {
            $load = $workload[$weekCode] ?? 0;
            $barHeight = $maxLoad > 0 ? ($load / $maxLoad) * $chartHeight : 0;
            $barColor = $load > WORKLOAD_CONFIG['max_weekly_load'] ? '#dc2626' : 
                       ($load > 4 ? '#ea580c' : '#16a34a');
            
            $html .= '<div class="chart-bar-container">';
            $html .= '<div class="chart-bar" style="height: ' . $barHeight . 'px; background-color: ' . $barColor . ';" ';
            $html .= 'title="Minggu ' . $weekData['week_number'] . ': ' . $load . ' poin"></div>';
            $html .= '<div class="chart-label">W' . $weekData['week_number'] . '</div>';
            $html .= '</div>';
        }
        $html .= '</div>';
        
        // Chart legend
        $html .= '<div class="chart-legend">';
        $html .= '<div class="legend-item"><div class="legend-color" style="background: #16a34a;"></div>Ringan (≤4)</div>';
        $html .= '<div class="legend-item"><div class="legend-color" style="background: #ea580c;"></div>Sedang (5-6)</div>';
        $html .= '<div class="legend-item"><div class="legend-color" style="background: #dc2626;"></div>Tinggi (>6)</div>';
        $html .= '</div>';
        
        $html .= '</div>';
        $html .= '</div>';
        
        return $html;
    }
}

/**
 * Analysis CSS Styles
 */
class AnalysisStyles {
    
    public static function getCSS() {
        return '
        <style>
        .analysis-summary {
            background: #f0f9ff;
            border: 1px solid #0ea5e9;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 25px;
        }
        
        .analysis-summary h4 {
            color: #0c4a6e;
            margin-bottom: 15px;
        }
        
        .summary-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
        }
        
        .summary-item {
            background: white;
            padding: 15px;
            border-radius: 6px;
            text-align: center;
            border-left: 4px solid #0ea5e9;
        }
        
        .summary-value {
            font-size: 24px;
            font-weight: bold;
            color: #0c4a6e;
            margin-bottom: 5px;
        }
        
        .summary-label {
            font-size: 12px;
            color: #64748b;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .workload-table-container {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .workload-table-container h4 {
            background: #374151;
            color: white;
            margin: 0;
            padding: 15px 20px;
        }
        
        .workload-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .workload-table th {
            background: #f9fafb;
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #374151;
            border-bottom: 2px solid #e5e7eb;
        }
        
        .workload-table td {
            padding: 12px;
            border-bottom: 1px solid #e5e7eb;
            vertical-align: top;
        }
        
        .workload-table tr.status-critical {
            background: #fef2f2;
        }
        
        .workload-table tr.status-high {
            background: #fefbeb;
        }
        
        .workload-table tr.status-medium {
            background: #fff7ed;
        }
        
        .workload-table tr.status-low {
            background: #f0fdf4;
        }
        
        .event-tag {
            display: inline-block;
            background: #e2e8f0;
            color: #475569;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            margin: 1px;
        }
        
        .phase-tag {
            display: inline-block;
            background: #dbeafe;
            color: #1e40af;
            padding: 2px 6px;
            border-radius: 10px;
            font-size: 10px;
            margin: 1px;
        }
        
        .no-events, .no-phases {
            color: #9ca3af;
            font-style: italic;
        }
        
        .insights-section {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 25px;
        }
        
        .insights-section h4 {
            color: #374151;
            margin-bottom: 15px;
        }
        
        .insight-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
        }
        
        .insight-item {
            background: #f8fafc;
            padding: 15px;
            border-radius: 6px;
            border-left: 4px solid #6366f1;
        }
        
        .insight-item h5 {
            color: #4338ca;
            margin-bottom: 10px;
            font-size: 14px;
        }
        
        .insight-item p {
            margin-bottom: 8px;
            font-size: 13px;
            color: #4b5563;
        }
        
        .recommendation {
            background: white;
            padding: 8px 12px;
            border-radius: 4px;
            border: 1px solid #d1d5db;
            font-weight: 500;
        }
        
        .workload-chart-section {
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .workload-chart-section h4 {
            color: #374151;
            margin-bottom: 15px;
        }
        
        .chart-container {
            padding: 20px;
            background: #f9fafb;
            border-radius: 6px;
        }
        
        .chart-bars {
            display: flex;
            align-items: end;
            justify-content: space-around;
            height: 220px;
            margin-bottom: 15px;
            padding: 10px;
            background: white;
            border-radius: 4px;
        }
        
        .chart-bar-container {
            display: flex;
            flex-direction: column;
            align-items: center;
            min-width: 40px;
        }
        
        .chart-bar {
            width: 30px;
            min-height: 2px;
            border-radius: 2px 2px 0 0;
            transition: all 0.3s;
            cursor: pointer;
        }
        
        .chart-bar:hover {
            opacity: 0.8;
            transform: scaleY(1.05);
        }
        
        .chart-label {
            margin-top: 8px;
            font-size: 11px;
            color: #6b7280;
            font-weight: 500;
        }
        
        .chart-legend {
            display: flex;
            justify-content: center;
            gap: 20px;
            margin-top: 15px;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 12px;
            color: #4b5563;
        }
        
        .legend-color {
            width: 12px;
            height: 12px;
            border-radius: 2px;
        }
        
        @media (max-width: 768px) {
            .summary-grid {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .insight-grid {
                grid-template-columns: 1fr;
            }
            
            .workload-table {
                font-size: 12px;
            }
            
            .workload-table th,
            .workload-table td {
                padding: 8px 4px;
            }
            
            .chart-bars {
                height: 150px;
            }
            
            .chart-legend {
                flex-direction: column;
                align-items: center;
                gap: 8px;
            }
        }
        </style>';
    }
}
?>