<?php
// Inisialisasi session
session_start();

// Fungsi untuk membaca data dari JSON file
function readJsonFile($file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        return json_decode($content, true);
    }
    return [];
}

// Fungsi untuk menyimpan data ke JSON file
function saveJsonFile($file, $data) {
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
}

// Fungsi untuk memeriksa berapa banyak kemungkinan email yang bisa dibuat
function countPossibleEmails($localPart) {
    $length = strlen($localPart);
    
    // Untuk menambahkan 2 titik, kita perlu memilih 2 posisi dari (length-1) posisi yang tersedia
    if ($length <= 2) {
        return 0; // Tidak bisa menambahkan 2 titik jika panjang kurang dari 3
    }
    
    $n = $length - 1; // Jumlah posisi yang tersedia
    $k = 2; // Jumlah titik yang akan ditambahkan
    
    // Menghitung kombinasi C(n,k)
    $result = 1;
    for ($i = 1; $i <= $k; $i++) {
        $result *= ($n - ($i - 1));
        $result /= $i;
    }
    
    return $result;
}

// Fungsi untuk menghasilkan semua kemungkinan penempatan 2 titik dalam string
function generateAllDotsPositions($localPart) {
    $length = strlen($localPart);
    $results = [];
    
    // Kita hanya bisa menempatkan titik di antara karakter (tidak di awal atau akhir)
    // dan tidak bisa menempatkan dua titik berturut-turut
    for ($i = 1; $i < $length; $i++) {
        for ($j = $i + 1; $j < $length; $j++) {
            $modified = substr($localPart, 0, $i) . '.' . 
                       substr($localPart, $i, $j - $i) . '.' . 
                       substr($localPart, $j);
            $results[] = $modified;
        }
    }
    
    return $results;
}

// Fungsi untuk menghasilkan email dengan menambahkan 2 titik
function generateEmailsWithDots($email, $count, $blacklist) {
    $parts = explode('@', $email);
    if (count($parts) != 2) {
        return [];
    }
    
    $localPart = $parts[0];
    $domain = $parts[1];
    
    // Hapus titik yang mungkin sudah ada di local part
    $localPartNoPoints = str_replace('.', '', $localPart);
    
    // Dapatkan semua kemungkinan penempatan 2 titik
    $allPossibilities = generateAllDotsPositions($localPartNoPoints);
    
    // Filter berdasarkan yang belum masuk blacklist
    $availablePossibilities = [];
    foreach ($allPossibilities as $possibility) {
        $newEmail = $possibility . '@' . $domain;
        if (!in_array($newEmail, $blacklist)) {
            $availablePossibilities[] = $newEmail;
        }
    }
    
    // Acak dan batasi jumlah sesuai yang diminta
    shuffle($availablePossibilities);
    return array_slice($availablePossibilities, 0, min($count, count($availablePossibilities)));
}

// Mendapatkan konfigurasi
$config = readJsonFile('config.json');
if (empty($config)) {
    $config = [
        'default_count' => 20
    ];
    saveJsonFile('config.json', $config);
}

// Mendapatkan blacklist
$blacklist = readJsonFile('blacklist.json');
if ($blacklist === null) {
    $blacklist = [];
}

// Mendapatkan statistik penggunaan email
$emailUsage = readJsonFile('email_usage.json');
if ($emailUsage === null) {
    $emailUsage = [];
}

// Fungsi untuk mendapatkan jumlah penggunaan email
function getEmailUsageCount($email, $emailUsage) {
    return isset($emailUsage[$email]) ? $emailUsage[$email] : 0;
}

// Handler untuk AJAX
if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
    $response = ['success' => false, 'message' => ''];
    
    // Menangani permintaan blacklist AJAX
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'blacklist_ajax') {
        $email_to_blacklist = filter_var($_POST['email_to_blacklist'], FILTER_VALIDATE_EMAIL);
        
        if ($email_to_blacklist && !in_array($email_to_blacklist, $blacklist)) {
            $blacklist[] = $email_to_blacklist;
            saveJsonFile('blacklist.json', $blacklist);
            $response = [
                'success' => true, 
                'message' => "Email $email_to_blacklist telah ditambahkan ke blacklist.",
                'blacklistCount' => count($blacklist)
            ];
        } else {
            $response['message'] = "Email tidak valid atau sudah ada dalam blacklist.";
        }
    }
    
    // Menangani permintaan copy/salin email
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'track_copy') {
        $copied_email = filter_var($_POST['email'], FILTER_VALIDATE_EMAIL);
        
        if ($copied_email) {
            // Tambah hitungan penggunaan
            if (!isset($emailUsage[$copied_email])) {
                $emailUsage[$copied_email] = 1;
            } else {
                $emailUsage[$copied_email]++;
            }
            
            saveJsonFile('email_usage.json', $emailUsage);
            $response = [
                'success' => true, 
                'message' => "Penggunaan email $copied_email telah dicatat.",
                'count' => $emailUsage[$copied_email]
            ];
        } else {
            $response['message'] = "Email tidak valid.";
        }
    }
    
    // Kirim respon JSON
    header('Content-Type: application/json');
    echo json_encode($response);
    exit;
}

// Inisialisasi variabel
$message = '';
$email = '';
$possible_count = 0;
$generated_emails = [];

// Memproses aksi yang dikirim
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = isset($_POST['action']) ? $_POST['action'] : '';
    
    // Aksi generate email
    if ($action === 'generate' && isset($_POST['email'])) {
        $email = filter_var($_POST['email'], FILTER_VALIDATE_EMAIL);
        
        if ($email) {
            $count = isset($_POST['count']) && is_numeric($_POST['count']) ? (int)$_POST['count'] : $config['default_count'];
            
            // Generate email
            $generated_emails = generateEmailsWithDots($email, $count, $blacklist);
            
            if (!empty($generated_emails)) {
                $message = "Berhasil menghasilkan " . count($generated_emails) . " email baru dengan dua titik.";
                
                // Urutkan email berdasarkan penggunaan (yang sudah digunakan ke bawah)
                usort($generated_emails, function($a, $b) use ($emailUsage) {
                    $aUsage = isset($emailUsage[$a]) ? $emailUsage[$a] : 0;
                    $bUsage = isset($emailUsage[$b]) ? $emailUsage[$b] : 0;
                    return $aUsage - $bUsage; // Yang belum digunakan di atas
                });
            } else {
                $message = "Tidak dapat menghasilkan email baru. Semua kemungkinan mungkin sudah ada dalam blacklist.";
            }
        } else {
            $message = "Email tidak valid. Silakan masukkan alamat email yang benar.";
        }
    }
    
    // Aksi tambahkan email ke blacklist
    elseif ($action === 'blacklist' && isset($_POST['email_to_blacklist'])) {
        $email_to_blacklist = filter_var($_POST['email_to_blacklist'], FILTER_VALIDATE_EMAIL);
        
        if ($email_to_blacklist && !in_array($email_to_blacklist, $blacklist)) {
            $blacklist[] = $email_to_blacklist;
            saveJsonFile('blacklist.json', $blacklist);
            
            // Hapus dari daftar email yang dihasilkan
            if (isset($generated_emails)) {
                $key = array_search($email_to_blacklist, $generated_emails);
                if ($key !== false) {
                    unset($generated_emails[$key]);
                    $generated_emails = array_values($generated_emails);
                }
            }
            
            $message = "Email $email_to_blacklist telah ditambahkan ke blacklist.";
        }
    }
    
    // Aksi hapus email dari blacklist
    elseif ($action === 'remove_blacklist' && isset($_POST['email_to_remove'])) {
        $email_to_remove = $_POST['email_to_remove'];
        
        $key = array_search($email_to_remove, $blacklist);
        if ($key !== false) {
            unset($blacklist[$key]);
            $blacklist = array_values($blacklist); // Re-index array
            saveJsonFile('blacklist.json', $blacklist);
            $message = "Email $email_to_remove telah dihapus dari blacklist.";
        }
    }
} else {
    // Jika bukan POST, cek apakah ada data yang tersimpan
    $email = isset($_COOKIE['last_email']) ? $_COOKIE['last_email'] : '';
}

// Menghitung kemungkinan email jika ada email
if (!empty($email)) {
    $parts = explode('@', $email);
    if (count($parts) == 2) {
        $localPart = $parts[0];
        $possible_count = countPossibleEmails($localPart);
    }
    
    // Simpan email di cookie untuk diingat selama 30 hari
    setcookie('last_email', $email, time() + (86400 * 30), "/");
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>DoubleDot Email Generator</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&family=Roboto+Mono:wght@400;500&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4361ee;
            --primary-darker: #3a56d4;
            --secondary-color: #7209b7;
            --accent-color: #f72585;
            --success-color: #06d6a0;
            --warning-color: #ffd166;
            --danger-color: #ef476f;
            --dark-color: #1a1a2e;
            --grey-color: #6c757d;
            --light-grey: #f8f9fa;
            --white-color: #ffffff;
            --border-radius: 12px;
            --box-shadow: 0 8px 30px rgba(0, 0, 0, 0.08);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Poppins', sans-serif;
            line-height: 1.6;
            background: linear-gradient(135deg, #f8f9fa 0%, #e4eaf1 100%);
            color: var(--dark-color);
            padding: 20px;
            min-height: 100vh;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: var(--white-color);
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            overflow: hidden;
            position: relative;
        }

        .header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--white-color);
            padding: 30px 40px;
            text-align: center;
        }

        .header h1 {
            font-weight: 700;
            font-size: 2.5rem;
            margin-bottom: 10px;
        }

        .header p {
            font-weight: 300;
            opacity: 0.9;
            max-width: 600px;
            margin: 0 auto;
        }

        .content {
            padding: 40px;
        }

        .alert {
            padding: 16px 20px;
            border-radius: 8px;
            margin-bottom: 30px;
            font-weight: 500;
            display: flex;
            align-items: center;
            animation: fadeIn 0.5s ease;
        }

        .alert i {
            margin-right: 12px;
            font-size: 1.1rem;
        }

        .alert-success {
            background-color: rgba(6, 214, 160, 0.15);
            color: #06864b;
            border-left: 4px solid var(--success-color);
        }

        .alert-warning {
            background-color: rgba(255, 209, 102, 0.15);
            color: #b16c00;
            border-left: 4px solid var(--warning-color);
        }

        .alert-danger {
            background-color: rgba(239, 71, 111, 0.15);
            color: #c21f4a;
            border-left: 4px solid var(--danger-color);
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .card {
            background-color: var(--light-grey);
            border-radius: var(--border-radius);
            padding: 30px;
            margin-bottom: 30px;
            transition: var(--transition);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }

        .card:hover {
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
            transform: translateY(-5px);
        }

        .card-title {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid rgba(0, 0, 0, 0.05);
        }

        .card-title h2 {
            font-size: 1.5rem;
            font-weight: 600;
            color: var(--secondary-color);
            margin-right: 10px;
        }

        .badge {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 6px 14px;
            border-radius: 30px;
            font-size: 14px;
            font-weight: 600;
            color: var(--white-color);
            background-color: var(--primary-color);
            margin-left: 10px;
        }

        .badge-danger {
            background-color: var(--danger-color);
        }

        .form-group {
            margin-bottom: 25px;
        }

        .form-group label {
            display: block;
            margin-bottom: 10px;
            font-weight: 500;
            color: var(--dark-color);
        }

        .form-group input {
            width: 100%;
            padding: 15px 20px;
            border: 2px solid rgba(0, 0, 0, 0.05);
            border-radius: 8px;
            font-size: 16px;
            transition: var(--transition);
            background-color: var(--white-color);
        }

        .form-group input:focus {
            border-color: var(--primary-color);
            outline: none;
            box-shadow: 0 0 0 3px rgba(67, 97, 238, 0.2);
        }

        .form-row {
            display: flex;
            gap: 20px;
            margin-bottom: 25px;
        }

        .form-col {
            flex: 1;
        }

        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 14px 28px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            border: none;
            text-decoration: none;
        }

        .btn i {
            margin-right: 8px;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--white-color);
            box-shadow: 0 4px 10px rgba(67, 97, 238, 0.3);
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, var(--primary-darker), var(--secondary-color));
            box-shadow: 0 6px 15px rgba(67, 97, 238, 0.4);
            transform: translateY(-2px);
        }

        .btn-outline {
            background: transparent;
            color: var(--primary-color);
            border: 2px solid var(--primary-color);
        }

        .btn-outline:hover {
            background-color: rgba(67, 97, 238, 0.1);
        }

        .btn-danger {
            background: linear-gradient(135deg, #ef476f, #d64161);
            color: var(--white-color);
            box-shadow: 0 4px 10px rgba(239, 71, 111, 0.3);
        }

        .btn-danger:hover {
            background: linear-gradient(135deg, #d64161, #b8304f);
            box-shadow: 0 6px 15px rgba(239, 71, 111, 0.4);
            transform: translateY(-2px);
        }

        .btn-success {
            background: linear-gradient(135deg, #06d6a0, #06a77d);
            color: var(--white-color);
            box-shadow: 0 4px 10px rgba(6, 214, 160, 0.3);
        }

        .btn-success:hover {
            background: linear-gradient(135deg, #06a77d, #06855f);
            box-shadow: 0 6px 15px rgba(6, 214, 160, 0.4);
            transform: translateY(-2px);
        }

        .btn-used {
            background: linear-gradient(135deg, #888888, #666666);
            color: var(--white-color);
            box-shadow: 0 4px 10px rgba(104, 104, 104, 0.3);
        }

        .btn-neutral {
            background-color: #f1f3f5;
            color: var(--grey-color);
        }

        .btn-neutral:hover {
            background-color: #e9ecef;
            color: var(--dark-color);
        }

        .btn-sm {
            padding: 8px 16px;
            font-size: 14px;
        }

        .info-box {
            display: flex;
            align-items: center;
            background: linear-gradient(135deg, rgba(67, 97, 238, 0.1), rgba(114, 9, 183, 0.1));
            border-left: 4px solid var(--primary-color);
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 30px;
            transition: var(--transition);
        }

        .info-box:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }

        .info-box i {
            font-size: 24px;
            color: var(--primary-color);
            margin-right: 15px;
        }

        .info-box p {
            margin: 0;
            font-weight: 500;
        }

        .info-box strong {
            color: var(--primary-color);
        }

        .email-list {
            background-color: var(--white-color);
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
        }

        .email-item {
            display: flex;
            align-items: center;
            padding: 16px 20px;
            border-bottom: 1px solid rgba(0, 0, 0, 0.05);
            transition: var(--transition);
        }

        .email-item:hover {
            background-color: rgba(67, 97, 238, 0.05);
        }

        .email-item:last-child {
            border-bottom: none;
        }

        .email-text {
            font-family: 'Roboto Mono', monospace;
            font-size: 15px;
            background-color: rgba(67, 97, 238, 0.08);
            padding: 8px 12px;
            border-radius: 6px;
            flex: 1;
            transition: var(--transition);
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            position: relative;
        }

        .email-usage {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            background-color: rgba(6, 214, 160, 0.2);
            color: #06864b;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: bold;
        }

        .email-actions {
            display: flex;
            gap: 10px;
            margin-left: 15px;
        }

        .tooltip {
            position: relative;
            display: inline-block;
        }

        .tooltip .tooltiptext {
            visibility: hidden;
            width: 120px;
            background-color: var(--dark-color);
            color: var(--white-color);
            text-align: center;
            border-radius: 6px;
            padding: 5px;
            position: absolute;
            z-index: 1;
            bottom: 125%;
            left: 50%;
            margin-left: -60px;
            opacity: 0;
            transition: opacity 0.3s;
            font-size: 14px;
            font-weight: 400;
        }

        .tooltip .tooltiptext::after {
            content: "";
            position: absolute;
            top: 100%;
            left: 50%;
            margin-left: -5px;
            border-width: 5px;
            border-style: solid;
            border-color: var(--dark-color) transparent transparent transparent;
        }

        .tooltip:hover .tooltiptext {
            visibility: visible;
            opacity: 1;
        }

        .clipboard-feedback {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 25px;
            background-color: var(--success-color);
            color: var(--white-color);
            border-radius: 8px;
            font-weight: 500;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
            z-index: 1000;
            display: none;
            animation: slideIn 0.5s ease, slideOut 0.5s ease 2.5s;
        }

        .blacklist-feedback {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 25px;
            background-color: var(--danger-color);
            color: var(--white-color);
            border-radius: 8px;
            font-weight: 500;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
            z-index: 1000;
            display: none;
            animation: slideIn 0.5s ease, slideOut 0.5s ease 2.5s;
        }

        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }

        @keyframes slideOut {
            from { transform: translateX(0); opacity: 1; }
            to { transform: translateX(100%); opacity: 0; }
        }

        .tabs {
            display: flex;
            background-color: var(--light-grey);
            border-radius: 8px;
            overflow: hidden;
            margin-bottom: 30px;
        }

        .tab {
            flex: 1;
            text-align: center;
            padding: 15px;
            cursor: pointer;
            transition: var(--transition);
            font-weight: 500;
            color: var(--grey-color);
            border-bottom: 3px solid transparent;
        }

        .tab.active {
            color: var(--primary-color);
            border-bottom-color: var(--primary-color);
            background-color: rgba(67, 97, 238, 0.08);
        }

        .tab:hover:not(.active) {
            background-color: rgba(67, 97, 238, 0.05);
            color: var(--primary-darker);
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }

        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--grey-color);
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 15px;
            opacity: 0.3;
        }

        .empty-state p {
            font-size: 1.1rem;
            font-weight: 500;
        }

        .footer {
            text-align: center;
            padding: 20px;
            background-color: var(--light-grey);
            color: var(--grey-color);
            font-size: 0.9rem;
            border-top: 1px solid rgba(0, 0, 0, 0.05);
        }

        .footer a {
            color: var(--primary-color);
            text-decoration: none;
        }

        /* Responsive Styles */
        @media (max-width: 992px) {
            .content {
                padding: 30px;
            }
            
            .form-row {
                flex-direction: column;
                gap: 15px;
            }
        }

        @media (max-width: 768px) {
            .header h1 {
                font-size: 2rem;
            }
            
            .content {
                padding: 20px;
            }
            
            .card {
                padding: 20px;
            }
            
            .email-item {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .email-text {
                width: 100%;
                margin-bottom: 15px;
                white-space: normal;
                word-break: break-all;
            }
            
            .email-actions {
                width: 100%;
                margin-left: 0;
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 10px;
            }
            
            .btn {
                width: 100%;
            }
        }

        @media (max-width: 480px) {
            .header {
                padding: 25px 20px;
            }
            
            .header h1 {
                font-size: 1.7rem;
            }
            
            .btn {
                padding: 12px 20px;
                font-size: 14px;
            }
            
            .tabs {
                flex-direction: column;
            }
            
            .tab {
                border-bottom: 1px solid rgba(0, 0, 0, 0.05);
                border-left: 3px solid transparent;
            }
            
            .tab.active {
                border-bottom-color: rgba(0, 0, 0, 0.05);
                border-left-color: var(--primary-color);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>DoubleDot <span style="font-weight: 300;">Email Generator</span></h1>
            <p>Buat banyak variasi email dengan dua titik tambahan sebelum tanda @</p>
        </div>
        
        <div class="content">
            <div id="alert-container">
                <?php if (!empty($message)): ?>
                    <?php 
                        $alertClass = strpos($message, "Berhasil") !== false ? "alert-success" : 
                                    (strpos($message, "Tidak dapat") !== false ? "alert-warning" : "alert-danger");
                        $alertIcon = strpos($message, "Berhasil") !== false ? "fa-check-circle" : 
                                    (strpos($message, "Tidak dapat") !== false ? "fa-exclamation-triangle" : "fa-times-circle");
                    ?>
                    <div class="alert <?php echo $alertClass; ?>">
                        <i class="fas <?php echo $alertIcon; ?>"></i>
                        <?php echo $message; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="tabs">
                <div class="tab active" data-tab="generator">
                    <i class="fas fa-magic"></i> Generator
                </div>
                <div class="tab" data-tab="blacklist">
                    <i class="fas fa-ban"></i> Blacklist
                    <?php if (!empty($blacklist)): ?>
                        <span class="badge badge-danger"><?php echo count($blacklist); ?></span>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="tab-content active" id="generator-tab">
                <div class="card">
                    <div class="card-title">
                        <h2>Buat Email Baru</h2>
                    </div>
                    
                    <form method="post" action="">
                        <div class="form-row">
                            <div class="form-col">
                                <div class="form-group">
                                    <label for="email"><i class="fas fa-envelope"></i> Email Asli</label>
                                    <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($email); ?>" placeholder="emailanda@domain.com" required>
                                </div>
                            </div>
                            <div class="form-col">
                                <div class="form-group">
                                    <label for="count"><i class="fas fa-hashtag"></i> Jumlah Email</label>
                                    <input type="number" id="count" name="count" min="1" max="<?php echo $possible_count > 0 ? $possible_count : 100; ?>" value="<?php echo $config['default_count']; ?>">
                                </div>
                            </div>
                        </div>
                        
                        <?php if ($possible_count > 0): ?>
                        <div class="info-box">
                            <i class="fas fa-lightbulb"></i>
                            <p>Dari email <strong><?php echo htmlspecialchars($email); ?></strong>, Anda bisa membuat maksimal <strong><?php echo $possible_count; ?></strong> variasi email dengan dua titik.</p>
                        </div>
                        <?php endif; ?>
                        
                        <div style="text-align: center;">
                            <button type="submit" name="action" value="generate" class="btn btn-primary">
                                <i class="fas fa-magic"></i> Generate Email
                            </button>
                        </div>
                    </form>
                </div>
                
                <?php if (!empty($generated_emails)): ?>
                <div class="card">
                    <div class="card-title">
                        <h2>Email yang Dihasilkan</h2>
                        <span class="badge"><?php echo count($generated_emails); ?></span>
                    </div>
                    
                    <div class="email-list">
                        <?php foreach($generated_emails as $gen_email): ?>
                        <?php 
                            $usageCount = getEmailUsageCount($gen_email, $emailUsage);
                            $usageClass = $usageCount > 0 ? 'used' : '';
                        ?>
                        <div class="email-item">
                            <div class="email-text" title="<?php echo htmlspecialchars($gen_email); ?>">
                                <i class="fas fa-at" style="margin-right: 8px; opacity: 0.6;"></i>
                                <?php echo htmlspecialchars($gen_email); ?>
                                <?php if ($usageCount > 0): ?>
                                <span class="email-usage">Digunakan <?php echo $usageCount; ?>x</span>
                                <?php endif; ?>
                            </div>
                            <div class="email-actions">
                                <div class="tooltip">
                                    <button type="button" class="btn btn-sm <?php echo $usageCount > 0 ? 'btn-used' : 'btn-success'; ?> copy-btn" data-email="<?php echo htmlspecialchars($gen_email); ?>" data-count="<?php echo $usageCount; ?>">
                                        <i class="fas fa-copy"></i> Salin<?php echo $usageCount > 0 ? " ($usageCount)" : ""; ?>
                                    </button>
                                    <span class="tooltiptext">Salin ke clipboard</span>
                                </div>
                                <div class="tooltip">
                                    <button type="button" class="btn btn-sm btn-danger blacklist-btn" data-email="<?php echo htmlspecialchars($gen_email); ?>">
                                        <i class="fas fa-ban"></i> Blacklist
                                    </button>
                                    <span class="tooltiptext">Tambahkan ke blacklist</span>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <div class="tab-content" id="blacklist-tab">
                <div class="card">
                    <div class="card-title">
                        <h2>Email dalam Blacklist</h2>
                        <?php if (!empty($blacklist)): ?>
                            <span class="badge badge-danger"><?php echo count($blacklist); ?></span>
                        <?php endif; ?>
                    </div>
                    
                    <?php if (!empty($blacklist)): ?>
                    <div class="email-list">
                        <?php foreach($blacklist as $blacklisted_email): ?>
                        <div class="email-item">
                            <div class="email-text" title="<?php echo htmlspecialchars($blacklisted_email); ?>">
                                <i class="fas fa-ban" style="margin-right: 8px; color: #ef476f; opacity: 0.6;"></i>
                                <?php echo htmlspecialchars($blacklisted_email); ?>
                                <?php 
                                    $usageCount = getEmailUsageCount($blacklisted_email, $emailUsage);
                                    if ($usageCount > 0): 
                                ?>
                                <span class="email-usage">Digunakan <?php echo $usageCount; ?>x</span>
                                <?php endif; ?>
                            </div>
                            <div class="email-actions">
                                <div class="tooltip">
                                    <button type="button" class="btn btn-sm <?php echo $usageCount > 0 ? 'btn-used' : 'btn-success'; ?> copy-btn" data-email="<?php echo htmlspecialchars($blacklisted_email); ?>" data-count="<?php echo $usageCount; ?>">
                                        <i class="fas fa-copy"></i> Salin<?php echo $usageCount > 0 ? " ($usageCount)" : ""; ?>
                                    </button>
                                    <span class="tooltiptext">Salin ke clipboard</span>
                                </div>
                                <form method="post" action="" style="display: inline;">
                                    <input type="hidden" name="email_to_remove" value="<?php echo htmlspecialchars($blacklisted_email); ?>">
                                    <div class="tooltip">
                                        <button type="submit" name="action" value="remove_blacklist" class="btn btn-sm btn-neutral">
                                            <i class="fas fa-trash-alt"></i> Hapus
                                        </button>
                                        <span class="tooltiptext">Hapus dari blacklist</span>
                                    </div>
                                </form>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-ban"></i>
                        <p>Tidak ada email dalam blacklist</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="footer">
            <p>&copy; <?php echo date('Y'); ?> DoubleDot Email Generator | Dibuat untuk membuat variasi email dengan dua titik</p>
        </div>
    </div>
    
    <div id="clipboard-feedback" class="clipboard-feedback">
        <i class="fas fa-check-circle"></i> Email berhasil disalin!
    </div>
    
    <div id="blacklist-feedback" class="blacklist-feedback">
        <i class="fas fa-ban"></i> Email berhasil ditambahkan ke blacklist!
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Tab switching
        const tabs = document.querySelectorAll('.tab');
        const tabContents = document.querySelectorAll('.tab-content');
        
        tabs.forEach(tab => {
            tab.addEventListener('click', function() {
                // Remove active class from all tabs
                tabs.forEach(t => t.classList.remove('active'));
                
                // Add active class to clicked tab
                this.classList.add('active');
                
                // Hide all tab contents
                tabContents.forEach(content => content.classList.remove('active'));
                
                // Show the corresponding tab content
                const tabId = this.getAttribute('data-tab');
                document.getElementById(tabId + '-tab').classList.add('active');
            });
        });
        
        // Copy to clipboard functionality
        const copyButtons = document.querySelectorAll('.copy-btn');
        const clipboardFeedback = document.getElementById('clipboard-feedback');
        
        copyButtons.forEach(button => {
            button.addEventListener('click', function() {
                const emailText = this.getAttribute('data-email');
                const currentCount = parseInt(this.getAttribute('data-count') || 0);
                
                // Copy to clipboard
                navigator.clipboard.writeText(emailText).then(() => {
                    // Track copy via AJAX
                    trackCopy(emailText, this);
                    
                    // Show feedback
                    clipboardFeedback.style.display = 'block';
                    
                    // Hide feedback after 3 seconds
                    setTimeout(() => {
                        clipboardFeedback.style.display = 'none';
                    }, 3000);
                }).catch(err => {
                    console.error('Could not copy text: ', err);
                    
                    // Fallback method
                    const textArea = document.createElement('textarea');
                    textArea.value = emailText;
                    document.body.appendChild(textArea);
                    textArea.select();
                    document.execCommand('copy');
                    document.body.removeChild(textArea);
                    
                    // Track copy via AJAX
                    trackCopy(emailText, this);
                    
                    // Show feedback
                    clipboardFeedback.style.display = 'block';
                    
                    // Hide feedback after 3 seconds
                    setTimeout(() => {
                        clipboardFeedback.style.display = 'none';
                    }, 3000);
                });
            });
        });
        
        // Function to track copy via AJAX
        function trackCopy(email, button) {
            // Create FormData object
            const formData = new FormData();
            formData.append('action', 'track_copy');
            formData.append('email', email);
            
            // Send AJAX request
            fetch(window.location.href, {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update button appearance
                    button.classList.remove('btn-success');
                    button.classList.add('btn-used');
                    button.setAttribute('data-count', data.count);
                    button.innerHTML = `<i class="fas fa-copy"></i> Salin (${data.count})`;
                    
                    // Update usage badge
                    const emailItem = button.closest('.email-item');
                    const emailText = emailItem.querySelector('.email-text');
                    let usageBadge = emailText.querySelector('.email-usage');
                    
                    if (!usageBadge) {
                        usageBadge = document.createElement('span');
                        usageBadge.className = 'email-usage';
                        emailText.appendChild(usageBadge);
                    }
                    
                    usageBadge.textContent = `Digunakan ${data.count}x`;
                }
            })
            .catch(error => {
                console.error('Error tracking copy:', error);
            });
        }
        
        // Blacklist with AJAX
        const blacklistButtons = document.querySelectorAll('.blacklist-btn');
        const blacklistFeedback = document.getElementById('blacklist-feedback');
        
        blacklistButtons.forEach(button => {
            button.addEventListener('click', function() {
                const email = this.getAttribute('data-email');
                
                // Create FormData object
                const formData = new FormData();
                formData.append('action', 'blacklist_ajax');
                formData.append('email_to_blacklist', email);
                
                // Send AJAX request
                fetch(window.location.href, {
                    method: 'POST',
                    body: formData,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    }
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        // Remove email item from the list
                        const emailItem = button.closest('.email-item');
                        emailItem.style.backgroundColor = 'rgba(239, 71, 111, 0.1)';
                        emailItem.style.opacity = '0.5';
                        
                        // Update the badge in blacklist tab
                        const blacklistTab = document.querySelector('.tab[data-tab="blacklist"]');
                        let badge = blacklistTab.querySelector('.badge');
                        
                        if (!badge) {
                            badge = document.createElement('span');
                            badge.className = 'badge badge-danger';
                            blacklistTab.appendChild(badge);
                        }
                        
                        badge.textContent = data.blacklistCount;
                        
                        // Show feedback
                        blacklistFeedback.style.display = 'block';
                        
                        // Hide feedback after 3 seconds
                        setTimeout(() => {
                            blacklistFeedback.style.display = 'none';
                            // Remove the email item from DOM after animation
                            emailItem.style.display = 'none';
                        }, 3000);
                    }
                })
                .catch(error => {
                    console.error('Error blacklisting email:', error);
                });
            });
        });
    });
    </script>
</body>
</html>