<?php
session_start();

// Configuration
$PIN = '051205';
$DATA_FILE = 'templates.json';

// Initialize data file if not exists
if (!file_exists($DATA_FILE)) {
    file_put_contents($DATA_FILE, json_encode([]));
}

// Load data
function loadData() {
    global $DATA_FILE;
    $content = file_get_contents($DATA_FILE);
    return json_decode($content, true) ?: [];
}

// Save data
function saveData($data) {
    global $DATA_FILE;
    return file_put_contents($DATA_FILE, json_encode($data, JSON_PRETTY_PRINT));
}

// Check if logged in
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

// Handle login
if (isset($_POST['login'])) {
    if ($_POST['pin'] === $PIN) {
        $_SESSION['logged_in'] = true;
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        $error = 'PIN salah!';
    }
}

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Handle add template
if (isset($_POST['add_template']) && isLoggedIn()) {
    $templates = loadData();
    $newTemplate = [
        'id' => time(),
        'name' => $_POST['name'],
        'category' => $_POST['category'],
        'link' => $_POST['link'],
        'description' => $_POST['description'],
        'image' => $_POST['image'] ?: '',
        'upload_date' => date('Y-m-d H:i:s')
    ];
    $templates[] = $newTemplate;
    saveData($templates);
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Handle edit template
if (isset($_POST['edit_template']) && isLoggedIn()) {
    $templates = loadData();
    foreach ($templates as &$template) {
        if ($template['id'] == $_POST['template_id']) {
            $template['name'] = $_POST['name'];
            $template['category'] = $_POST['category'];
            $template['link'] = $_POST['link'];
            $template['description'] = $_POST['description'];
            $template['image'] = $_POST['image'] ?: $template['image'];
            break;
        }
    }
    saveData($templates);
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Handle delete template
if (isset($_GET['delete']) && isLoggedIn()) {
    $templates = loadData();
    $templates = array_filter($templates, function($template) {
        return $template['id'] != $_GET['delete'];
    });
    saveData(array_values($templates));
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

// Get templates
$templates = loadData();
$categories = array_unique(array_filter(array_column($templates, 'category')));
$selectedCategory = $_GET['category'] ?? '';

// Filter templates by category
$filteredTemplates = $selectedCategory ? 
    array_filter($templates, function($template) use ($selectedCategory) {
        return $template['category'] === $selectedCategory;
    }) : $templates;

// Get template for editing
$editTemplate = null;
if (isset($_GET['edit']) && isLoggedIn()) {
    foreach ($templates as $template) {
        if ($template['id'] == $_GET['edit']) {
            $editTemplate = $template;
            break;
        }
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Template Manager - Canva Style</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background-color: #f7f8fc;
            color: #2b2d42;
        }

        /* Header Canva Style */
        .header {
            background: #ffffff;
            border-bottom: 1px solid #e5e7eb;
            padding: 16px 24px;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 1px 3px rgba(0,0,0,0.05);
        }

        .header-content {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 12px;
            font-size: 24px;
            font-weight: 700;
            color: #7c3aed;
        }

        .logo-icon {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #7c3aed, #06b6d4);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 16px;
        }

        .header-actions {
            display: flex;
            align-items: center;
            gap: 16px;
        }

        /* Login Page */
        .login-container {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }

        .login-card {
            background: white;
            padding: 48px;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.1);
            text-align: center;
            min-width: 400px;
        }

        .login-title {
            font-size: 32px;
            font-weight: 700;
            margin-bottom: 8px;
            color: #1f2937;
        }

        .login-subtitle {
            color: #6b7280;
            margin-bottom: 32px;
        }

        /* Main Content */
        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 32px 24px;
        }

        .page-title {
            font-size: 32px;
            font-weight: 700;
            color: #1f2937;
            margin-bottom: 8px;
        }

        .page-subtitle {
            color: #6b7280;
            margin-bottom: 32px;
        }

        /* Controls Bar */
        .controls-bar {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 32px;
            gap: 16px;
            flex-wrap: wrap;
        }

        .category-filters {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }

        .category-chip {
            padding: 8px 16px;
            border-radius: 20px;
            border: 1px solid #e5e7eb;
            background: white;
            color: #374151;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .category-chip:hover {
            border-color: #7c3aed;
            color: #7c3aed;
        }

        .category-chip.active {
            background: #7c3aed;
            color: white;
            border-color: #7c3aed;
        }

        /* Add Button */
        .add-btn {
            background: #7c3aed;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 12px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.2s ease;
            font-size: 14px;
        }

        .add-btn:hover {
            background: #6d28d9;
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(124,58,237,0.3);
        }

        .add-btn-icon {
            font-size: 16px;
        }

        /* Templates Grid */
        .templates-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 24px;
            margin-top: 24px;
        }

        .template-card {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.06);
            border: 1px solid #f3f4f6;
            transition: all 0.2s ease;
            cursor: pointer;
            position: relative;
        }

        .template-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.1);
            border-color: #e5e7eb;
        }

        .template-preview {
            height: 200px;
            background: linear-gradient(135deg, #f3f4f6, #e5e7eb);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            color: #9ca3af;
            position: relative;
            overflow: hidden;
        }

        .template-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            object-position: top center;
        }

        .template-overlay {
            position: absolute;
            top: 8px;
            right: 8px;
            display: flex;
            gap: 4px;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .template-card:hover .template-overlay {
            opacity: 1;
        }

        .overlay-btn {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            border: none;
            background: rgba(255,255,255,0.9);
            color: #6b7280;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            transition: all 0.2s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }

        .overlay-btn:hover {
            background: white;
            color: #374151;
            transform: scale(1.1);
        }

        .overlay-btn.edit:hover {
            color: #7c3aed;
        }

        .overlay-btn.delete:hover {
            color: #ef4444;
        }

        .template-content {
            padding: 20px;
        }

        .template-name {
            font-weight: 600;
            font-size: 16px;
            color: #1f2937;
            margin-bottom: 8px;
            line-height: 1.4;
        }

        .template-meta {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 12px;
        }

        .template-category {
            background: #f3f4f6;
            color: #6b7280;
            padding: 4px 8px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 500;
        }

        .template-date {
            color: #9ca3af;
            font-size: 12px;
        }

        .template-description {
            color: #6b7280;
            font-size: 14px;
            line-height: 1.5;
            margin-bottom: 16px;
        }

        .template-actions {
            display: flex;
            gap: 8px;
        }

        .btn-primary {
            background: #7c3aed;
            color: white;
            padding: 8px 16px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
            flex: 1;
            text-align: center;
        }

        .btn-primary:hover {
            background: #6d28d9;
        }

        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 1000;
            display: none;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .modal-overlay.show {
            display: flex;
        }

        .modal {
            background: white;
            border-radius: 16px;
            width: 100%;
            max-width: 600px;
            max-height: 90vh;
            overflow-y: auto;
            animation: modalSlideIn 0.3s ease;
        }

        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: scale(0.9) translateY(-20px);
            }
            to {
                opacity: 1;
                transform: scale(1) translateY(0);
            }
        }

        .modal-header {
            padding: 24px 24px 0;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .modal-title {
            font-size: 20px;
            font-weight: 700;
            color: #1f2937;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 24px;
            color: #9ca3af;
            cursor: pointer;
            padding: 4px;
            border-radius: 4px;
            transition: all 0.2s ease;
        }

        .modal-close:hover {
            background: #f3f4f6;
            color: #6b7280;
        }

        .modal-body {
            padding: 24px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #374151;
        }

        .form-input {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.2s ease;
            background: white;
        }

        .form-input:focus {
            outline: none;
            border-color: #7c3aed;
            box-shadow: 0 0 0 3px rgba(124,58,237,0.1);
        }

        .form-textarea {
            resize: vertical;
            min-height: 80px;
        }

        .form-select {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 14px;
            background: white;
            cursor: pointer;
        }

        .form-select:focus {
            outline: none;
            border-color: #7c3aed;
            box-shadow: 0 0 0 3px rgba(124,58,237,0.1);
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 16px;
        }

        .image-preview {
            width: 100%;
            height: 120px;
            border: 2px dashed #d1d5db;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #9ca3af;
            margin-top: 8px;
            overflow: hidden;
        }

        .image-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
            object-position: top center;
            border-radius: 6px;
        }

        .modal-actions {
            padding: 0 24px 24px;
            display: flex;
            gap: 12px;
            justify-content: flex-end;
        }

        .btn-cancel {
            background: #f3f4f6;
            color: #6b7280;
            padding: 12px 24px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .btn-cancel:hover {
            background: #e5e7eb;
        }

        .btn-submit {
            background: #7c3aed;
            color: white;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            border: none;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .btn-submit:hover {
            background: #6d28d9;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 80px 20px;
            color: #6b7280;
        }

        .empty-icon {
            font-size: 64px;
            margin-bottom: 16px;
            opacity: 0.5;
        }

        .empty-title {
            font-size: 20px;
            font-weight: 600;
            color: #374151;
            margin-bottom: 8px;
        }

        .empty-subtitle {
            color: #9ca3af;
        }

        /* Logout Button */
        .logout-btn {
            background: #f3f4f6;
            color: #6b7280;
            padding: 8px 16px;
            border-radius: 8px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
            transition: all 0.2s ease;
        }

        .logout-btn:hover {
            background: #e5e7eb;
            color: #374151;
        }

        /* Error Message */
        .error-message {
            background: #fef2f2;
            border: 1px solid #fecaca;
            color: #dc2626;
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 14px;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .main-container {
                padding: 24px 16px;
            }

            .controls-bar {
                flex-direction: column;
                align-items: stretch;
                gap: 16px;
            }

            .category-filters {
                justify-content: center;
            }

            .templates-grid {
                grid-template-columns: repeat(auto-fill, minmax(240px, 1fr));
                gap: 16px;
            }

            .login-card {
                margin: 20px;
                min-width: auto;
                padding: 32px;
            }

            .modal {
                margin: 20px;
                max-width: none;
            }

            .form-row {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 480px) {
            .header-content {
                padding: 0 16px;
            }

            .logo {
                font-size: 20px;
            }

            .page-title {
                font-size: 24px;
            }

            .templates-grid {
                grid-template-columns: 1fr;
            }
        }
    </style>
</head>
<body>
    <?php if (!isLoggedIn()): ?>
        <div class="login-container">
            <div class="login-card">
                <div class="logo" style="justify-content: center; margin-bottom: 32px;">
                    <div class="logo-icon">🎨</div>
                    <span>Template Manager</span>
                </div>
                <h1 class="login-title">Welcome Back</h1>
                <p class="login-subtitle">Enter your PIN to access your templates</p>
                
                <?php if (isset($error)): ?>
                    <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>
                
                <form method="POST">
                    <div class="form-group">
                        <input type="password" name="pin" class="form-input" placeholder="Enter your PIN" required style="text-align: center; font-size: 18px; letter-spacing: 2px;">
                    </div>
                    <button type="submit" name="login" class="btn-submit" style="width: 100%; padding: 16px;">
                        Sign In
                    </button>
                </form>
            </div>
        </div>
    <?php else: ?>
        <div class="header">
            <div class="header-content">
                <div class="logo">
                    <div class="logo-icon">🎨</div>
                    <span>Template Manager</span>
                </div>
                <div class="header-actions">
                    <a href="?logout" class="logout-btn">Sign Out</a>
                </div>
            </div>
        </div>

        <div class="main-container">
            <h1 class="page-title">Your Templates</h1>
            <p class="page-subtitle">Manage your Canva template collection</p>

            <div class="controls-bar">
                <div class="category-filters">
                    <a href="?" class="category-chip <?php echo $selectedCategory === '' ? 'active' : ''; ?>">
                        All Templates
                    </a>
                    <?php foreach ($categories as $category): ?>
                        <a href="?category=<?php echo urlencode($category); ?>" 
                           class="category-chip <?php echo $selectedCategory === $category ? 'active' : ''; ?>">
                            <?php echo htmlspecialchars($category); ?>
                        </a>
                    <?php endforeach; ?>
                </div>
                <button class="add-btn" onclick="openModal()">
                    <span class="add-btn-icon">+</span>
                    Add Template
                </button>
            </div>

            <?php if (empty($filteredTemplates)): ?>
                <div class="empty-state">
                    <div class="empty-icon">📋</div>
                    <h3 class="empty-title">No templates found</h3>
                    <p class="empty-subtitle">
                        <?php echo $selectedCategory ? 'No templates in this category yet.' : 'Start by adding your first template!'; ?>
                    </p>
                </div>
            <?php else: ?>
                <div class="templates-grid">
                    <?php foreach ($filteredTemplates as $template): ?>
                        <div class="template-card" onclick="openTemplate('<?php echo htmlspecialchars($template['link']); ?>')">
                            <div class="template-preview">
                                <?php if (!empty($template['image'])): ?>
                                    <img src="<?php echo htmlspecialchars($template['image']); ?>" alt="<?php echo htmlspecialchars($template['name']); ?>">
                                <?php else: ?>
                                    🎨
                                <?php endif; ?>
                                <div class="template-overlay">
                                    <button class="overlay-btn edit" onclick="event.stopPropagation(); editTemplate(<?php echo $template['id']; ?>)" title="Edit">
                                        ✏️
                                    </button>
                                    <button class="overlay-btn delete" onclick="event.stopPropagation(); deleteTemplate(<?php echo $template['id']; ?>)" title="Delete">
                                        🗑️
                                    </button>
                                </div>
                            </div>
                            <div class="template-content">
                                <h3 class="template-name"><?php echo htmlspecialchars($template['name']); ?></h3>
                                <div class="template-meta">
                                    <span class="template-category"><?php echo htmlspecialchars($template['category']); ?></span>
                                    <span class="template-date"><?php echo date('M d, Y', strtotime($template['upload_date'])); ?></span>
                                </div>
                                <?php if (!empty($template['description'])): ?>
                                    <p class="template-description"><?php echo htmlspecialchars($template['description']); ?></p>
                                <?php endif; ?>
                                <div class="template-actions">
                                    <a href="<?php echo htmlspecialchars($template['link']); ?>" target="_blank" class="btn-primary" onclick="event.stopPropagation();">
                                        Open in Canva
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>

        <!-- Add/Edit Template Modal -->
        <div class="modal-overlay" id="templateModal">
            <div class="modal">
                <div class="modal-header">
                    <h2 class="modal-title" id="modalTitle">Add New Template</h2>
                    <button class="modal-close" onclick="closeModal()">&times;</button>
                </div>
                <form method="POST" id="templateForm">
                    <input type="hidden" name="template_id" id="templateId">
                    <div class="modal-body">
                        <div class="form-group">
                            <label class="form-label">Template Name</label>
                            <input type="text" name="name" id="templateName" class="form-input" placeholder="Enter template name" required>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Category</label>
                                <select name="category" id="templateCategory" class="form-select" required>
                                    <option value="">Select or type new category</option>
                                    <?php foreach ($categories as $category): ?>
                                        <option value="<?php echo htmlspecialchars($category); ?>"><?php echo htmlspecialchars($category); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Or New Category</label>
                                <input type="text" id="newCategory" class="form-input" placeholder="e.g. Poster, Logo">
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Template Link</label>
                            <input type="url" name="link" id="templateLink" class="form-input" placeholder="https://www.canva.com/design/..." required>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Image URL (Optional)</label>
                            <input type="url" name="image" id="templateImage" class="form-input" placeholder="https://example.com/image.jpg" onchange="previewImage(this.value)">
                            <div class="image-preview" id="imagePreview">
                                📷 Image Preview
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="form-label">Description (Optional)</label>
                            <textarea name="description" id="templateDescription" class="form-input form-textarea" placeholder="Brief description of the template..."></textarea>
                        </div>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn-cancel" onclick="closeModal()">Cancel</button>
                        <button type="submit" name="add_template" id="submitBtn" class="btn-submit">Add Template</button>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>

    <script>
        let isEditing = false;

        function openModal() {
            resetForm();
            document.getElementById('templateModal').classList.add('show');
            document.body.style.overflow = 'hidden';
        }

        function closeModal() {
            document.getElementById('templateModal').classList.remove('show');
            document.body.style.overflow = 'auto';
            resetForm();
        }

        function resetForm() {
            isEditing = false;
            document.getElementById('modalTitle').textContent = 'Add New Template';
            document.getElementById('submitBtn').textContent = 'Add Template';
            document.getElementById('submitBtn').name = 'add_template';
            document.getElementById('templateForm').reset();
            document.getElementById('templateId').value = '';
            document.getElementById('imagePreview').innerHTML = '📷 Image Preview';
        }

        function editTemplate(id) {
            <?php if ($editTemplate): ?>
                if (<?php echo $editTemplate['id']; ?> === id) {
                    isEditing = true;
                    document.getElementById('modalTitle').textContent = 'Edit Template';
                    document.getElementById('submitBtn').textContent = 'Save Changes';
                    document.getElementById('submitBtn').name = 'edit_template';
                    
                    document.getElementById('templateId').value = '<?php echo $editTemplate['id']; ?>';
                    document.getElementById('templateName').value = '<?php echo htmlspecialchars($editTemplate['name']); ?>';
                    document.getElementById('templateLink').value = '<?php echo htmlspecialchars($editTemplate['link']); ?>';
                    document.getElementById('templateDescription').value = '<?php echo htmlspecialchars($editTemplate['description'] ?? ''); ?>';
                    document.getElementById('templateImage').value = '<?php echo htmlspecialchars($editTemplate['image'] ?? ''); ?>';
                    
                    // Set category
                    const categorySelect = document.getElementById('templateCategory');
                    const category = '<?php echo htmlspecialchars($editTemplate['category']); ?>';
                    if (category) {
                        let found = false;
                        for (let option of categorySelect.options) {
                            if (option.value === category) {
                                option.selected = true;
                                found = true;
                                break;
                            }
                        }
                        if (!found) {
                            document.getElementById('newCategory').value = category;
                        }
                    }
                    
                    // Preview image if exists
                    if ('<?php echo htmlspecialchars($editTemplate['image'] ?? ''); ?>') {
                        previewImage('<?php echo htmlspecialchars($editTemplate['image'] ?? ''); ?>');
                    }
                    
                    document.getElementById('templateModal').classList.add('show');
                    document.body.style.overflow = 'hidden';
                    return;
                }
            <?php endif; ?>
            
            // If not the current edit template, redirect to edit
            window.location.href = '?edit=' + id;
        }

        function deleteTemplate(id) {
            if (confirm('Are you sure you want to delete this template?')) {
                window.location.href = '?delete=' + id;
            }
        }

        function openTemplate(link) {
            window.open(link, '_blank');
        }

        function previewImage(url) {
            const preview = document.getElementById('imagePreview');
            if (url && url.trim()) {
                preview.innerHTML = '<img src="' + url + '" alt="Preview" onerror="this.parentElement.innerHTML=\'❌ Invalid image URL\'">';
            } else {
                preview.innerHTML = '📷 Image Preview';
            }
        }

        // Handle category selection
        document.getElementById('templateCategory').addEventListener('change', function() {
            if (this.value) {
                document.getElementById('newCategory').value = '';
            }
        });

        document.getElementById('newCategory').addEventListener('input', function() {
            if (this.value) {
                document.getElementById('templateCategory').value = '';
            }
        });

        // Update form submission to handle category
        document.getElementById('templateForm').addEventListener('submit', function(e) {
            const categorySelect = document.getElementById('templateCategory');
            const newCategory = document.getElementById('newCategory');
            
            if (!categorySelect.value && !newCategory.value) {
                e.preventDefault();
                alert('Please select or enter a category');
                return;
            }
            
            if (newCategory.value) {
                categorySelect.value = newCategory.value;
            }
        });

        // Close modal when clicking outside
        document.getElementById('templateModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });

        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });

        // Auto-open modal if editing
        <?php if ($editTemplate): ?>
            window.addEventListener('load', function() {
                editTemplate(<?php echo $editTemplate['id']; ?>);
            });
        <?php endif; ?>
    </script>
</body>
</html>