<?php
// Initialize or load app data from JSON file
$dataFile = 'data.json';

if (file_exists($dataFile)) {
    $jsonData = file_get_contents($dataFile);
    $apps = json_decode($jsonData, true);
} else {
    // Create default app data if file doesn't exist
    $apps = [
        [
            'name' => 'Google',
            'icon' => 'https://upload.wikimedia.org/wikipedia/commons/thumb/5/53/Google_%22G%22_Logo.svg/800px-Google_%22G%22_Logo.svg.png',
            'url' => 'https://google.com',
            'category' => 'Search',
            'favorite' => false
        ],
        [
            'name' => 'YouTube',
            'icon' => 'https://upload.wikimedia.org/wikipedia/commons/thumb/0/09/YouTube_full-color_icon_%282017%29.svg/2560px-YouTube_full-color_icon_%282017%29.svg.png',
            'url' => 'https://youtube.com',
            'category' => 'Entertainment',
            'favorite' => false
        ],
        [
            'name' => 'Facebook',
            'icon' => 'https://upload.wikimedia.org/wikipedia/commons/thumb/b/b8/2021_Facebook_icon.svg/2048px-2021_Facebook_icon.svg.png',
            'url' => 'https://facebook.com',
            'category' => 'Social',
            'favorite' => false
        ]
    ];
    
    // Save default data to JSON file
    file_put_contents($dataFile, json_encode($apps, JSON_PRETTY_PRINT));
}

// Update existing app records if they don't have category/favorite fields
$needsUpdate = false;
foreach ($apps as $key => $app) {
    if (!isset($app['category'])) {
        $apps[$key]['category'] = 'Other';
        $needsUpdate = true;
    }
    if (!isset($app['favorite'])) {
        $apps[$key]['favorite'] = false;
        $needsUpdate = true;
    }
}

if ($needsUpdate) {
    // Load original unsorted data to preserve order
    $unsortedApps = json_decode(file_get_contents($dataFile), true);
    foreach ($unsortedApps as $key => $app) {
        if (!isset($app['category'])) {
            $unsortedApps[$key]['category'] = 'Other';
        }
        if (!isset($app['favorite'])) {
            $unsortedApps[$key]['favorite'] = false;
        }
    }
    file_put_contents($dataFile, json_encode($unsortedApps, JSON_PRETTY_PRINT));
}

// Sort apps alphabetically by name (case-insensitive)
usort($apps, function($a, $b) {
    return strcasecmp($a['name'], $b['name']);
});

// Get unique categories
$categories = array_unique(array_column($apps, 'category'));
sort($categories);

// Handle new app addition
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['submit'])) {
        $newApp = [
            'name' => htmlspecialchars($_POST['app_name']),
            'icon' => htmlspecialchars($_POST['app_icon']),
            'url' => htmlspecialchars($_POST['app_url']),
            'category' => htmlspecialchars($_POST['app_category']),
            'favorite' => false
        ];
        
        // Add new app to array (unsorted in the file)
        $unsortedApps = json_decode(file_get_contents($dataFile), true);
        $unsortedApps[] = $newApp;
        
        // Save updated data (unsorted)
        file_put_contents($dataFile, json_encode($unsortedApps, JSON_PRETTY_PRINT));
        
        // Redirect to prevent form resubmission
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } elseif (isset($_POST['toggle_favorite'])) {
        // Toggle favorite status
        $appName = $_POST['app_name'];
        $unsortedApps = json_decode(file_get_contents($dataFile), true);
        
        foreach ($unsortedApps as $key => $app) {
            if ($app['name'] === $appName) {
                $unsortedApps[$key]['favorite'] = !$app['favorite'];
                break;
            }
        }
        
        file_put_contents($dataFile, json_encode($unsortedApps, JSON_PRETTY_PRINT));
        echo json_encode(['success' => true]);
        exit;
    }
}

// Handle category addition if it doesn't exist
if (isset($_POST['add_category'])) {
    $newCategory = htmlspecialchars($_POST['new_category']);
    if (!in_array($newCategory, $categories)) {
        $categories[] = $newCategory;
        sort($categories);
    }
    echo json_encode(['success' => true, 'categories' => $categories]);
    exit;
}

// Get favorite apps
$favoriteApps = array_filter($apps, function($app) {
    return $app['favorite'] === true;
});

// Group apps by category
$appsByCategory = [];
foreach ($apps as $app) {
    $category = $app['category'];
    if (!isset($appsByCategory[$category])) {
        $appsByCategory[$category] = [];
    }
    $appsByCategory[$category][] = $app;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>DEZKI APP - App Launcher</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #6246ea;
            --primary-gradient: linear-gradient(135deg, #6246ea, #8567ff);
            --secondary-color: #6c757d;
            --bg-light: #f8f9fa;
            --bg-dark: #121212;
            --text-light: #212529;
            --text-dark: #e9ecef;
            --card-light: #ffffff;
            --card-dark: #1e1e1e;
            --shadow-light: 0 4px 12px rgba(0, 0, 0, 0.1);
            --shadow-dark: 0 4px 12px rgba(0, 0, 0, 0.3);
            --border-radius: 16px;
            --app-icon-size: 72px;
            --app-icon-size-mobile: 65px;
            --transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
        }

        /* Dark Mode Variables */
        .dark-mode {
            --bg-color: var(--bg-dark);
            --text-color: var(--text-dark);
            --card-color: var(--card-dark);
            --shadow: var(--shadow-dark);
        }

        /* Light Mode Variables */
        .light-mode {
            --bg-color: var(--bg-light);
            --text-color: var(--text-light);
            --card-color: var(--card-light);
            --shadow: var(--shadow-light);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background: var(--bg-color);
            color: var(--text-color);
            min-height: 100vh;
            padding: 15px 10px;
            overflow-x: hidden;
            transition: var(--transition);
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding-bottom: 70px; /* Space for bottom nav */
        }

        header {
            text-align: center;
            margin-bottom: 10px;
            position: relative;
            padding: 10px 0;
        }

        .logo {
            font-size: 2.2rem;
            font-weight: 700;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            background-clip: text;
            -webkit-text-fill-color: transparent;
            display: inline-block;
            margin-bottom: 15px;
        }

        .search-bar {
            position: relative;
            max-width: 600px;
            margin: 0 auto;
        }

        .search-bar input {
            width: 100%;
            padding: 15px 20px 15px 50px;
            border: none;
            border-radius: var(--border-radius);
            background: var(--card-color);
            color: var(--text-color);
            box-shadow: var(--shadow);
            font-size: 1rem;
            transition: var(--transition);
        }

        .search-bar input::placeholder {
            padding-left: 5px; /* Move placeholder text to the right */
        }

        .search-bar input:focus {
            outline: none;
            box-shadow: 0 0 0 3px rgba(98, 70, 234, 0.3);
        }

        .search-bar .search-icon {
            position: absolute;
            left: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--secondary-color);
        }

        .content-area {
            padding-top: 10px;
        }

        .app-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(var(--app-icon-size), 1fr));
            gap: 20px;
            padding: 5px;
        }

        .app-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
            cursor: pointer;
            transition: transform 0.3s ease, opacity 0.3s ease;
            animation: fadeIn 0.5s ease-out;
            opacity: 0;
            animation-fill-mode: forwards;
            position: relative;
        }

        .app-item:hover {
            transform: translateY(-3px);
        }

        .favorite-btn {
            position: absolute;
            top: -5px;
            right: -5px;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            background: var(--card-color);
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #ccc;
            cursor: pointer;
            transition: var(--transition);
            z-index: 2;
            font-size: 0.8rem;
            box-shadow: var(--shadow);
        }

        .favorite-btn.active {
            color: #FFD700;
        }

        .favorite-btn:hover {
            transform: scale(1.2);
        }

        .app-icon {
            width: var(--app-icon-size);
            height: var(--app-icon-size);
            border-radius: 22px;
            background: var(--card-color);
            box-shadow: var(--shadow);
            margin-bottom: 8px;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: var(--transition);
            padding: 0;
        }

        .app-icon img {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
        }

        .app-name {
            font-size: 0.9rem;
            color: var(--text-color);
            font-weight: 500;
            max-width: 100px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            transition: var(--transition);
        }

        .add-app {
            display: flex;
            flex-direction: column;
            align-items: center;
            position: relative;
        }

        .add-app .app-icon {
            background: var(--primary-gradient);
        }

        .add-app .app-icon i {
            color: white;
            font-size: 2rem;
        }

        .add-app .app-name {
            color: var(--text-color);
        }

        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: var(--card-color);
            box-shadow: 0 -2px 10px rgba(0, 0, 0, 0.1);
            display: flex;
            justify-content: space-around;
            padding: 12px 0;
            z-index: 100;
        }

        .nav-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            color: var(--secondary-color);
            text-decoration: none;
            font-size: 0.8rem;
        }

        .nav-item.active {
            color: var(--primary-color);
        }

        .nav-icon {
            font-size: 1.4rem;
            margin-bottom: 3px;
        }

        /* Modal styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            opacity: 0;
            transition: var(--transition);
            justify-content: center;
            align-items: center;
            backdrop-filter: blur(5px);
        }

        .modal.active {
            display: flex;
            opacity: 1;
        }

        .modal-content {
            background: var(--card-color);
            border-radius: var(--border-radius);
            padding: 25px;
            width: 90%;
            max-width: 500px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            transform: translateY(-20px);
            transition: var(--transition);
            position: relative;
            overflow: hidden;
        }

        .modal.active .modal-content {
            transform: translateY(0);
        }

        .modal-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 20px;
        }

        .modal-title {
            font-size: 1.5rem;
            font-weight: 600;
            background: var(--primary-gradient);
            -webkit-background-clip: text;
            background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .close {
            font-size: 1.5rem;
            color: var(--secondary-color);
            cursor: pointer;
            transition: var(--transition);
            width: 32px;
            height: 32px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
        }

        .close:hover {
            color: var(--text-color);
            background-color: rgba(0, 0, 0, 0.05);
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: var(--text-color);
        }

        input[type="text"],
        input[type="url"],
        select {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid rgba(0, 0, 0, 0.1);
            border-radius: var(--border-radius);
            background: var(--card-color);
            color: var(--text-color);
            font-size: 1rem;
            transition: var(--transition);
        }

        input[type="text"]:focus,
        input[type="url"]:focus,
        select:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(98, 70, 234, 0.2);
            outline: none;
        }

        .submit-btn {
            background: var(--primary-gradient);
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 25px;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: var(--transition);
            display: flex;
            align-items: center;
            gap: 8px;
            width: 100%;
            justify-content: center;
        }

        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(98, 70, 234, 0.3);
        }

        .icon-preview {
            width: 48px;
            height: 48px;
            border-radius: 12px;
            overflow: hidden;
            margin-top: 10px;
            display: none;
            background: #f1f1f1;
        }

        .icon-preview img {
            width: 100%;
            height: 100%;
            object-fit: contain;
        }

        /* Empty state */
        .empty-state {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            text-align: center;
            padding: 40px 20px;
            border-radius: var(--border-radius);
            background: var(--card-color);
            box-shadow: var(--shadow);
            min-height: 200px;
            margin-top: 20px;
        }

        .empty-icon {
            font-size: 3rem;
            color: var(--secondary-color);
            margin-bottom: 20px;
            opacity: 0.5;
        }

        .empty-text {
            font-size: 1.2rem;
            color: var(--secondary-color);
            margin-bottom: 20px;
        }

        /* Animations */
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Animation delay for grid items */
        .app-item:nth-child(1) { animation-delay: 0.05s; }
        .app-item:nth-child(2) { animation-delay: 0.1s; }
        .app-item:nth-child(3) { animation-delay: 0.15s; }
        .app-item:nth-child(4) { animation-delay: 0.2s; }
        .app-item:nth-child(5) { animation-delay: 0.25s; }
        .app-item:nth-child(6) { animation-delay: 0.3s; }
        .app-item:nth-child(n+7) { animation-delay: 0.35s; }
        
        /* For mobile optimization */
        @media (max-width: 768px) {
            .logo {
                font-size: 1.8rem;
                margin-bottom: 10px;
            }
            
            .app-grid {
                grid-template-columns: repeat(auto-fill, minmax(var(--app-icon-size-mobile), 1fr));
                gap: 15px;
            }
            
            .app-icon {
                width: var(--app-icon-size-mobile);
                height: var(--app-icon-size-mobile);
            }
            
            body, .container {
                padding-left: 5px;
                padding-right: 5px;
            }
            
            .search-bar input {
                padding: 12px 15px 12px 45px;
            }
            
            .nav-icon {
                font-size: 1.2rem;
            }
            
            .nav-text {
                font-size: 0.7rem;
            }
        }
    </style>
</head>
<body class="light-mode">
    <div class="container">
        <header>
            <h1 class="logo">DEZKI APP</h1>
            <div class="search-bar">
                <i class="fas fa-search search-icon"></i>
                <input type="text" id="searchInput" placeholder="Search apps...">
            </div>
        </header>

        <div class="content-area">
            <!-- All Apps Content -->
            <div class="tab-content" id="all-content">
                <div class="app-grid" id="appGrid">
                    <?php foreach ($apps as $index => $app): ?>
                        <div class="app-item" 
                            data-app-name="<?php echo strtolower($app['name']); ?>"
                            data-category="<?php echo strtolower($app['category']); ?>"
                            data-favorite="<?php echo $app['favorite'] ? 'true' : 'false'; ?>">
                            
                            <button class="favorite-btn <?php echo $app['favorite'] ? 'active' : ''; ?>" 
                                   data-app-name="<?php echo htmlspecialchars($app['name']); ?>">
                                <i class="fas fa-star"></i>
                            </button>
                            
                            <a href="<?php echo $app['url']; ?>" target="_blank" class="app-link">
                                <div class="app-icon">
                                    <img src="<?php echo $app['icon']; ?>" alt="<?php echo $app['name']; ?>">
                                </div>
                                <div class="app-name"><?php echo $app['name']; ?></div>
                            </a>
                        </div>
                    <?php endforeach; ?>

                    <!-- Add App Button -->
                    <div class="app-item add-app" id="addAppBtn">
                        <div class="app-icon">
                            <i class="fas fa-plus"></i>
                        </div>
                        <div class="app-name">Add App</div>
                    </div>
                </div>
            </div>
            
            <!-- Favorites Content (initially hidden) -->
            <div class="tab-content" id="favorites-content" style="display: none;">
                <?php if (count($favoriteApps) > 0): ?>
                    <div class="app-grid" id="favoritesGrid">
                        <?php foreach ($favoriteApps as $app): ?>
                            <div class="app-item" 
                                data-app-name="<?php echo strtolower($app['name']); ?>"
                                data-category="<?php echo strtolower($app['category']); ?>"
                                data-favorite="true">
                                
                                <button class="favorite-btn active" 
                                       data-app-name="<?php echo htmlspecialchars($app['name']); ?>">
                                    <i class="fas fa-star"></i>
                                </button>
                                
                                <a href="<?php echo $app['url']; ?>" target="_blank" class="app-link">
                                    <div class="app-icon">
                                        <img src="<?php echo $app['icon']; ?>" alt="<?php echo $app['name']; ?>">
                                    </div>
                                    <div class="app-name"><?php echo $app['name']; ?></div>
                                </a>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-star empty-icon"></i>
                        <div class="empty-text">No favorite apps yet</div>
                        <p>Mark apps as favorites to see them here</p>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Categories Content (initially hidden) -->
            <div class="tab-content" id="categories-content" style="display: none;">
                <?php foreach ($categories as $category): ?>
                    <div class="category-group" data-category="<?php echo strtolower($category); ?>">
                        <div class="app-grid">
                            <?php 
                            // Filter apps for this category
                            $categoryApps = array_filter($apps, function($app) use ($category) {
                                return $app['category'] === $category;
                            });
                            
                            foreach ($categoryApps as $app): 
                            ?>
                                <div class="app-item" 
                                    data-app-name="<?php echo strtolower($app['name']); ?>"
                                    data-category="<?php echo strtolower($app['category']); ?>"
                                    data-favorite="<?php echo $app['favorite'] ? 'true' : 'false'; ?>">
                                    
                                    <button class="favorite-btn <?php echo $app['favorite'] ? 'active' : ''; ?>" 
                                           data-app-name="<?php echo htmlspecialchars($app['name']); ?>">
                                        <i class="fas fa-star"></i>
                                    </button>
                                    
                                    <a href="<?php echo $app['url']; ?>" target="_blank" class="app-link">
                                        <div class="app-icon">
                                            <img src="<?php echo $app['icon']; ?>" alt="<?php echo $app['name']; ?>">
                                        </div>
                                        <div class="app-name"><?php echo $app['name']; ?></div>
                                    </a>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <div class="bottom-nav">
        <a href="#" class="nav-item active" data-tab="all">
            <i class="fas fa-th-large nav-icon"></i>
            <span class="nav-text">All Apps</span>
        </a>
        <a href="#" class="nav-item" data-tab="favorites">
            <i class="fas fa-star nav-icon"></i>
            <span class="nav-text">Favorites</span>
        </a>
        <a href="#" class="nav-item" data-tab="categories">
            <i class="fas fa-folder nav-icon"></i>
            <span class="nav-text">Categories</span>
        </a>
        <a href="#" class="nav-item" id="toggleTheme">
            <i class="fas fa-moon nav-icon"></i>
            <span class="nav-text">Theme</span>
        </a>
    </div>

    <!-- Add App Modal -->
    <div class="modal" id="addAppModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title">Add New App</h2>
                <span class="close" id="closeModal">&times;</span>
            </div>
            <form method="POST" action="" id="addAppForm">
                <div class="form-group">
                    <label for="app_name">App Name</label>
                    <input type="text" id="app_name" name="app_name" required placeholder="Enter app name">
                </div>
                <div class="form-group">
                    <label for="app_icon">Icon URL</label>
                    <input type="url" id="app_icon" name="app_icon" required placeholder="Enter icon URL">
                    <div id="iconPreview" class="icon-preview"></div>
                </div>
                <div class="form-group">
                    <label for="app_url">App URL</label>
                    <input type="url" id="app_url" name="app_url" required placeholder="Enter app URL">
                </div>
                <div class="form-group">
                    <label for="app_category">Category</label>
                    <select id="app_category" name="app_category" required>
                        <?php foreach ($categories as $category): ?>
                            <option value="<?php echo htmlspecialchars($category); ?>"><?php echo htmlspecialchars($category); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <button type="submit" name="submit" class="submit-btn">
                    <i class="fas fa-plus"></i> Add App
                </button>
            </form>
        </div>
    </div>

    <script>
        // Wait for the DOM to be fully loaded
        document.addEventListener('DOMContentLoaded', function() {
            // DOM Elements
            const body = document.body;
            const modal = document.getElementById('addAppModal');
            const addAppBtn = document.getElementById('addAppBtn');
            const closeModal = document.getElementById('closeModal');
            const searchInput = document.getElementById('searchInput');
            const navItems = document.querySelectorAll('.nav-item');
            const tabContents = document.querySelectorAll('.tab-content');
            const toggleThemeBtn = document.getElementById('toggleTheme');
            const iconInput = document.getElementById('app_icon');
            const iconPreview = document.getElementById('iconPreview');
            
            // Open modal when Add App button is clicked
            addAppBtn.addEventListener('click', function() {
                modal.classList.add('active');
                document.body.style.overflow = 'hidden'; // Prevent background scrolling
            });
            
            // Close modal when X is clicked
            closeModal.addEventListener('click', function() {
                modal.classList.remove('active');
                document.body.style.overflow = 'auto'; // Re-enable scrolling
            });
            
            // Close modal when clicking outside the modal content
            window.addEventListener('click', function(event) {
                if (event.target === modal) {
                    modal.classList.remove('active');
                    document.body.style.overflow = 'auto'; // Re-enable scrolling
                }
            });
            
            // Navigation tabs
            navItems.forEach(item => {
                if (item.id === 'toggleTheme') return; // Skip theme toggle
                
                item.addEventListener('click', function(e) {
                    e.preventDefault();
                    
                    // Remove active class from all tabs
                    navItems.forEach(navItem => {
                        if (navItem.id !== 'toggleTheme') {
                            navItem.classList.remove('active');
                        }
                    });
                    
                    // Add active class to clicked tab
                    this.classList.add('active');
                    
                    // Hide all tab contents
                    tabContents.forEach(content => {
                        content.style.display = 'none';
                    });
                    
                    // Show selected tab content
                    const tabId = this.getAttribute('data-tab');
                    document.getElementById(`${tabId}-content`).style.display = 'block';
                });
            });
            
            // Toggle theme mode
            toggleThemeBtn.addEventListener('click', function(e) {
                e.preventDefault();
                body.classList.toggle('dark-mode');
                body.classList.toggle('light-mode');
                
                const isDarkMode = body.classList.contains('dark-mode');
                const themeIcon = this.querySelector('.nav-icon');
                themeIcon.className = isDarkMode ? 'fas fa-sun nav-icon' : 'fas fa-moon nav-icon';
                
                localStorage.setItem('theme', isDarkMode ? 'dark' : 'light');
            });
            
            // Load saved theme preference
            function loadThemePreference() {
                const savedTheme = localStorage.getItem('theme');
                if (savedTheme === 'dark') {
                    body.classList.remove('light-mode');
                    body.classList.add('dark-mode');
                    const themeIcon = toggleThemeBtn.querySelector('.nav-icon');
                    themeIcon.className = 'fas fa-sun nav-icon';
                }
            }
            
            loadThemePreference();
            
            // Search functionality
            searchInput.addEventListener('input', function() {
                const searchTerm = this.value.toLowerCase().trim();
                
                document.querySelectorAll('.app-item:not(.add-app)').forEach(function(item) {
                    const appName = item.getAttribute('data-app-name');
                    
                    if (appName.includes(searchTerm)) {
                        item.style.display = 'flex';
                    } else {
                        item.style.display = 'none';
                    }
                });
            });
            
            // Toggle favorite status
            document.addEventListener('click', function(e) {
                if (e.target.closest('.favorite-btn')) {
                    e.preventDefault();
                    e.stopPropagation();
                    
                    const btn = e.target.closest('.favorite-btn');
                    const appName = btn.getAttribute('data-app-name');
                    
                    // Send AJAX request to toggle favorite status
                    const xhr = new XMLHttpRequest();
                    xhr.open('POST', '', true);
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.onreadystatechange = function() {
                        if (xhr.readyState === 4 && xhr.status === 200) {
                            const response = JSON.parse(xhr.responseText);
                            if (response.success) {
                                // Toggle active class
                                btn.classList.toggle('active');
                                
                                // Update data-favorite attribute
                                const isFavorite = btn.classList.contains('active');
                                const appItem = btn.closest('.app-item');
                                appItem.setAttribute('data-favorite', isFavorite ? 'true' : 'false');
                                
                                // Refresh if on favorites tab
                                if (document.querySelector('.nav-item[data-tab="favorites"]').classList.contains('active')) {
                                    location.reload();
                                }
                            }
                        }
                    };
                    xhr.send(`toggle_favorite=1&app_name=${encodeURIComponent(appName)}`);
                }
            });
            
            // Icon preview
            iconInput.addEventListener('input', function() {
                const iconUrl = this.value.trim();
                if (iconUrl && iconUrl.match(/\.(jpeg|jpg|gif|png|svg)$/i)) {
                    iconPreview.style.display = 'block';
                    iconPreview.innerHTML = `<img src="${iconUrl}" alt="Icon Preview">`;
                } else {
                    iconPreview.style.display = 'none';
                    iconPreview.innerHTML = '';
                }
            });
            
            // Form validation
            document.getElementById('addAppForm').addEventListener('submit', function(event) {
                let valid = true;
                
                // Validate icon URL
                const iconValue = iconInput.value;
                if (!iconValue.match(/\.(jpeg|jpg|gif|png|svg)$/i)) {
                    alert('Please enter a valid image URL (JPEG, JPG, GIF, PNG, SVG)');
                    valid = false;
                }
                
                // Validate app URL
                const urlValue = document.getElementById('app_url').value;
                if (!urlValue.match(/^(http|https):\/\/[^ "]+$/)) {
                    alert('Please enter a valid URL starting with http:// or https://');
                    valid = false;
                }
                
                if (!valid) {
                    event.preventDefault();
                }
            });
        });
    </script>
</body>
</html>