<?php
// Include primary files
require_once 'config.php';
require_once 'inventory.php';

// Check if user is authenticated
if (!isAuthenticated()) {
    header('Location: index.php?action=login');
    exit;
}

$action = isset($_GET['action']) ? $_GET['action'] : 'inventory';

// Process form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if ($action == 'add_product') {
        $name = isset($_POST['name']) ? $_POST['name'] : '';
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        $unit = isset($_POST['unit']) ? $_POST['unit'] : '';
        
        if (empty($name)) {
            $error = 'Product name is required';
        } elseif (empty($unit)) {
            $error = 'Unit is required';
        } else {
            $result = addProduct($name, $description, $unit);
            
            if (is_numeric($result)) {
                header('Location: inventory-ui.php?action=products');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'edit_product') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        $name = isset($_POST['name']) ? $_POST['name'] : '';
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        $unit = isset($_POST['unit']) ? $_POST['unit'] : '';
        
        if (empty($name)) {
            $error = 'Product name is required';
        } elseif (empty($unit)) {
            $error = 'Unit is required';
        } else {
            $result = updateProduct($id, $name, $description, $unit);
            
            if ($result === true) {
                header('Location: inventory-ui.php?action=products');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'delete_product') {
        $id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
        
        $result = deleteProduct($id);
        
        if ($result === true) {
            header('Location: inventory-ui.php?action=products');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'adjust_inventory') {
        $entityId = isset($_POST['entity_id']) ? (int)$_POST['entity_id'] : 0;
        $productId = isset($_POST['product_id']) ? (int)$_POST['product_id'] : 0;
        $quantity = isset($_POST['quantity']) ? (float)$_POST['quantity'] : 0;
        $purchasePrice = isset($_POST['purchase_price']) ? (float)$_POST['purchase_price'] : 0;
        $notes = isset($_POST['notes']) ? $_POST['notes'] : '';
        
        if ($entityId <= 0) {
            $error = 'Entity is required';
        } elseif ($productId <= 0) {
            $error = 'Product is required';
        } elseif ($quantity == 0) {
            $error = 'Quantity cannot be zero';
        } elseif ($purchasePrice <= 0 && $quantity > 0) {
            $error = 'Purchase price must be positive for stock additions';
        } else {
            $result = adjustInventory($entityId, $productId, $quantity, $purchasePrice, $notes);
            
            if ($result === true) {
                header('Location: inventory-ui.php?action=inventory');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'update_pricing') {
        $entityId = isset($_POST['entity_id']) ? (int)$_POST['entity_id'] : 0;
        $productId = isset($_POST['product_id']) ? (int)$_POST['product_id'] : 0;
        $purchasePrice = isset($_POST['purchase_price']) ? (float)$_POST['purchase_price'] : 0;
        $sellingPrice = isset($_POST['selling_price']) ? (float)$_POST['selling_price'] : 0;
        
        if ($entityId <= 0) {
            $error = 'Entity is required';
        } elseif ($productId <= 0) {
            $error = 'Product is required';
        } elseif ($purchasePrice <= 0) {
            $error = 'Purchase price must be positive';
        } elseif ($sellingPrice <= 0) {
            $error = 'Selling price must be positive';
        } else {
            $result = updateInventoryPricing($entityId, $productId, $purchasePrice, $sellingPrice);
            
            if ($result === true) {
                header('Location: inventory-ui.php?action=inventory');
                exit;
            } else {
                $error = $result;
            }
        }
    } elseif ($action == 'transfer_money') {
        $fromEntityId = isset($_POST['from_entity_id']) ? (int)$_POST['from_entity_id'] : 0;
        $toEntityId = isset($_POST['to_entity_id']) ? (int)$_POST['to_entity_id'] : 0;
        $amount = isset($_POST['amount']) ? (float)$_POST['amount'] : 0;
        $description = isset($_POST['description']) ? $_POST['description'] : '';
        
        $result = transferMoney($fromEntityId, $toEntityId, $amount, $description);
        
        if (is_numeric($result)) {
            header('Location: inventory-ui.php?action=transfers');
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'create_sale') {
        $sellerEntityId = isset($_POST['seller_entity_id']) ? (int)$_POST['seller_entity_id'] : 0;
        $buyerEntityId = isset($_POST['buyer_entity_id']) ? (int)$_POST['buyer_entity_id'] : 0;
        $notes = isset($_POST['notes']) ? $_POST['notes'] : '';
        
        $result = createSale($sellerEntityId, $buyerEntityId, $notes);
        
        if (is_numeric($result)) {
            header('Location: inventory-ui.php?action=edit_sale&id=' . $result);
            exit;
        } else {
            $error = $result;
        }
    } elseif ($action == 'add_sale_item') {
        $saleId = isset($_POST['sale_id']) ? (int)$_POST['sale_id'] : 0;
        $productId = isset($_POST['product_id']) ? (int)$_POST['product_id'] : 0;
        $quantity = isset($_POST['quantity']) ? (float)$_POST['quantity'] : 0;
        $unitPrice = isset($_POST['unit_price']) ? (float)$_POST['unit_price'] : 0;
        
        $result = addSaleItem($saleId, $productId, $quantity, $unitPrice);
        
        if ($result === true) {
            header('Location: inventory-ui.php?action=edit_sale&id=' . $saleId);
            exit;
        } else {
            $error = $result;
            $editSale = getSale($saleId);
        }
    } elseif ($action == 'remove_sale_item') {
        $saleItemId = isset($_POST['sale_item_id']) ? (int)$_POST['sale_item_id'] : 0;
        $saleId = isset($_POST['sale_id']) ? (int)$_POST['sale_id'] : 0;
        
        $result = removeSaleItem($saleItemId);
        
        if ($result === true) {
            header('Location: inventory-ui.php?action=edit_sale&id=' . $saleId);
            exit;
        } else {
            $error = $result;
            $editSale = getSale($saleId);
        }
    } elseif ($action == 'complete_sale') {
        $saleId = isset($_POST['sale_id']) ? (int)$_POST['sale_id'] : 0;
        
        $result = completeSale($saleId);
        
        if ($result === true) {
            header('Location: inventory-ui.php?action=view_sale&id=' . $saleId);
            exit;
        } else {
            $error = $result;
            $editSale = getSale($saleId);
        }
    } elseif ($action == 'cancel_sale') {
        $saleId = isset($_POST['sale_id']) ? (int)$_POST['sale_id'] : 0;
        
        $result = cancelSale($saleId);
        
        if ($result === true) {
            header('Location: inventory-ui.php?action=sales');
            exit;
        } else {
            $error = $result;
        }
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png"  href="https://app.dezki.biz.id/linkpic/assets/uploads/67dd7aa9c2f75.png">
    <title>Invenly</title>
    <!-- Add Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <!-- Material Icons -->
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <style>
        :root {
            --theme-color: #0e543d;
            --secondary-color: #ffffff;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            line-height: 1.6;
            background-color: #f5f5f5;
            color: #333;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 15px;
        }
        
        header {
            background-color: var(--theme-color);
            color: white;
            padding: 10px 0;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        header .container {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo h1 {
            font-size: 24px;
            font-weight: 500;
        }
        
        nav ul {
            list-style: none;
            display: flex;
        }
        
        nav ul li {
            margin-left: 20px;
        }
        
        nav ul li a {
            color: white;
            text-decoration: none;
            font-weight: 500;
            display: flex;
            align-items: center;
        }
        
        nav ul li a i {
            margin-right: 5px;
        }
        
        .content {
            padding: 20px 0;
        }
        
        .card {
            background: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            margin-bottom: 20px;
            overflow: hidden;
        }
        
        .card-header {
            background-color: var(--theme-color);
            color: white;
            padding: 15px 20px;
            font-weight: 500;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .card-body {
            padding: 20px;
        }
        
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
        }
        
        table th, table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        
        table th {
            background-color: #f2f2f2;
        }
        
        form .form-group {
            margin-bottom: 15px;
        }
        
        form label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        form input[type="text"], 
        form input[type="number"], 
        form input[type="password"], 
        form input[type="date"], 
        form textarea, 
        form select {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 16px;
        }
        
        form textarea {
            resize: vertical;
        }
        
        .btn {
            display: inline-block;
            background-color: var(--theme-color);
            color: white;
            padding: 10px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            text-decoration: none;
            font-size: 16px;
        }
        
        .btn-sm {
            padding: 5px 10px;
            font-size: 14px;
        }
        
        .btn-secondary {
            background-color: #6c757d;
        }
        
        .btn-success {
            background-color: #28a745;
        }
        
        .btn-danger {
            background-color: #dc3545;
        }
        
        .status-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 3px;
            font-size: 12px;
            font-weight: 500;
            text-transform: uppercase;
        }
        
        .status-draft {
            background-color: #fff3cd;
            color: #856404;
        }
        
        .status-completed {
            background-color: #d4edda;
            color: #155724;
        }
        
        .status-cancelled {
            background-color: #f8d7da;
            color: #721c24;
        }
        
        .filter-bar {
            display: flex;
            margin-bottom: 20px;
            align-items: center;
        }
        
        .filter-bar label {
            margin-right: 10px;
            font-weight: 500;
        }
        
        .filter-bar select {
            margin-right: 20px;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .material-icons {
            vertical-align: middle;
            line-height: 1;
        }
        
        /* Mobile navigation */
        .mobile-menu-toggle {
            display: none;
            background: none;
            border: none;
            color: white;
            font-size: 24px;
            cursor: pointer;
        }
        
        @media (max-width: 768px) {
            .mobile-menu-toggle {
                display: block;
            }
            
            nav ul {
                display: none;
                flex-direction: column;
                width: 100%;
            }
            
            nav ul.active {
                display: flex;
            }
            
            nav ul li {
                margin: 10px 0;
            }
        }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .inventory-card {
            background: white;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            padding: 15px;
        }
        
        .inventory-card h3 {
            margin-bottom: 10px;
            color: var(--theme-color);
        }
        
        .inventory-detail {
            margin-bottom: 5px;
        }
        
        .inventory-detail strong {
            display: inline-block;
            width: 120px;
        }
    </style>
</head>
<body>

<header>
    <div class="container">
        <div class="logo">
            <h1>INVENLY</h1>
        </div>
        
        <button class="mobile-menu-toggle" id="menuToggle">
            <span class="material-icons">menu</span>
        </button>
        
        <nav>
            <ul id="mainNav">
                <li><a href="index.php"><i class="material-icons">dashboard</i> Main App</a></li>
                <li><a href="inventory-ui.php?action=inventory"><i class="material-icons">inventory_2</i> Inventory</a></li>
                <li><a href="inventory-ui.php?action=products"><i class="material-icons">category</i> Products</a></li>
                <li><a href="inventory-ui.php?action=transfers"><i class="material-icons">swap_horiz</i> Transfers</a></li>
                <li><a href="inventory-ui.php?action=sales"><i class="material-icons">shopping_cart</i> Sales</a></li>
                <li><a href="inventory-ui.php?action=stock_movements"><i class="material-icons">analytics</i> Stock Movements</a></li>
            </ul>
        </nav>
    </div>
</header>

<div class="container">
    <div class="content">
        <?php if (isset($error)): ?>
            <div class="alert alert-danger"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?php echo $success; ?></div>
        <?php endif; ?>
        
        <?php if ($action == 'inventory' || $action == ''): ?>
            <!-- Inventory List -->
            <div class="card">
                <div class="card-header">
                    Inventory
                    <div>
                        <a href="inventory-ui.php?action=adjust_inventory" class="btn btn-sm">Adjust Stock</a>
                        <a href="inventory-ui.php?action=update_pricing" class="btn btn-sm">Update Pricing</a>
                    </div>
                </div>
                <div class="card-body">
                    <?php 
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    $entities = getEntities();
                    $inventory = getInventory($entityId);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="window.location.href='inventory-ui.php?action=inventory&entity_id='+this.value">
                            <option value="">All Entities</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (count($inventory) > 0): ?>
                        <div class="grid">
                            <?php foreach ($inventory as $item): ?>
                                <?php if ($item['quantity'] > 0): ?>
                                <div class="inventory-card">
                                    <h3><?php echo $item['product_name']; ?></h3>
                                    <div class="inventory-detail">
                                        <strong>Entity:</strong> <?php echo $item['entity_name']; ?>
                                    </div>
                                    <div class="inventory-detail">
                                        <strong>Quantity:</strong> <?php echo number_format($item['quantity'], 2) . ' ' . $item['unit']; ?>
                                    </div>
                                    <div class="inventory-detail">
                                        <strong>Purchase Price:</strong> <?php echo formatCurrency($item['purchase_price']); ?>
                                    </div>
                                    <div class="inventory-detail">
                                        <strong>Selling Price:</strong> <?php echo formatCurrency($item['selling_price']); ?>
                                    </div>
                                    <div class="inventory-detail">
                                        <strong>Total Value:</strong> <?php echo formatCurrency($item['quantity'] * $item['purchase_price']); ?>
                                    </div>
                                    <div style="margin-top: 10px;">
                                        <a href="inventory-ui.php?action=update_pricing&entity_id=<?php echo $item['entity_id']; ?>&product_id=<?php echo $item['product_id']; ?>" class="btn btn-sm">Update Pricing</a>
                                        <a href="inventory-ui.php?action=adjust_inventory&entity_id=<?php echo $item['entity_id']; ?>&product_id=<?php echo $item['product_id']; ?>" class="btn btn-sm">Adjust Stock</a>
                                    </div>
                                </div>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p>No inventory found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'products'): ?>
            <!-- Products List -->
            <div class="card">
                <div class="card-header">
                    Products
                    <a href="inventory-ui.php?action=add_product" class="btn btn-sm">Add New Product</a>
                </div>
                <div class="card-body">
                    <?php $products = getProducts(); ?>
                    
                    <?php if (count($products) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>Description</th>
                                    <th>Unit</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($products as $product): ?>
                                    <tr>
                                        <td><?php echo $product['name']; ?></td>
                                        <td><?php echo $product['description']; ?></td>
                                        <td><?php echo $product['unit']; ?></td>
                                        <td>
                                            <a href="inventory-ui.php?action=edit_product&id=<?php echo $product['id']; ?>" class="btn btn-sm">Edit</a>
                                            <a href="inventory-ui.php?action=delete_product&id=<?php echo $product['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this product?')">Delete</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No products found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'add_product'): ?>
            <!-- Add Product Form -->
            <div class="card">
                <div class="card-header">Add New Product</div>
                <div class="card-body">
                    <form method="post" action="inventory-ui.php?action=add_product">
                        <div class="form-group">
                            <label for="name">Name:</label>
                            <input type="text" id="name" name="name" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label for="unit">Unit:</label>
                            <input type="text" id="unit" name="unit" required placeholder="e.g., pcs, kg, box">
                        </div>
                        
                        <button type="submit" class="btn">Add Product</button>
                        <a href="inventory-ui.php?action=products" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'edit_product'): ?>
            <!-- Edit Product Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $product = getProduct($id);
            
            if ($product): 
            ?>
                <div class="card">
                    <div class="card-header">Edit Product</div>
                    <div class="card-body">
                        <form method="post" action="inventory-ui.php?action=edit_product">
                            <input type="hidden" name="id" value="<?php echo $product['id']; ?>">
                            
                            <div class="form-group">
                                <label for="name">Name:</label>
                                <input type="text" id="name" name="name" value="<?php echo $product['name']; ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="description">Description:</label>
                                <textarea id="description" name="description" rows="3"><?php echo $product['description']; ?></textarea>
                            </div>
                            
                            <div class="form-group">
                                <label for="unit">Unit:</label>
                                <input type="text" id="unit" name="unit" value="<?php echo $product['unit']; ?>" required>
                            </div>
                            
                            <button type="submit" class="btn">Update Product</button>
                            <a href="inventory-ui.php?action=products" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Product not found.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'delete_product'): ?>
            <!-- Delete Product Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $product = getProduct($id);
            
            if ($product): 
            ?>
                <div class="card">
                    <div class="card-header">Delete Product</div>
                    <div class="card-body">
                        <p>Are you sure you want to delete the product "<?php echo $product['name']; ?>"?</p>
                        
                        <form method="post" action="inventory-ui.php?action=delete_product">
                            <input type="hidden" name="id" value="<?php echo $product['id']; ?>">
                            
                            <button type="submit" class="btn btn-danger">Delete Product</button>
                            <a href="inventory-ui.php?action=products" class="btn btn-secondary">Cancel</a>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Product not found.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'adjust_inventory'): ?>
            <!-- Adjust Inventory Form -->
            <div class="card">
                <div class="card-header">Adjust Inventory</div>
                <div class="card-body">
                    <?php 
                    $entities = getEntities();
                    $products = getProducts();
                    
                    $selectedEntityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : 0;
                    $selectedProductId = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;
                    
                    $currentItem = null;
                    if ($selectedEntityId > 0 && $selectedProductId > 0) {
                        $currentItem = getInventoryItem($selectedEntityId, $selectedProductId);
                    }
                    ?>
                    
                    <form method="post" action="inventory-ui.php?action=adjust_inventory">
                        <div class="form-group">
                            <label for="entity_id">Entity:</label>
                            <select id="entity_id" name="entity_id" required <?php echo $selectedEntityId > 0 ? 'disabled' : ''; ?>>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>" <?php echo $selectedEntityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                            <?php if ($selectedEntityId > 0): ?>
                                <input type="hidden" name="entity_id" value="<?php echo $selectedEntityId; ?>">
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="product_id">Product:</label>
                            <select id="product_id" name="product_id" required <?php echo $selectedProductId > 0 ? 'disabled' : ''; ?>>
                                <option value="">Select Product</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo $product['id']; ?>" <?php echo $selectedProductId == $product['id'] ? 'selected' : ''; ?>><?php echo $product['name']; ?> (<?php echo $product['unit']; ?>)</option>
                                <?php endforeach; ?>
                            </select>
                            <?php if ($selectedProductId > 0): ?>
                                <input type="hidden" name="product_id" value="<?php echo $selectedProductId; ?>">
                            <?php endif; ?>
                        </div>
                        
                        <?php if ($currentItem): ?>
                            <div class="form-group">
                                <p><strong>Current Quantity:</strong> <?php echo number_format($currentItem['quantity'], 2) . ' ' . $currentItem['unit']; ?></p>
                                <p><strong>Current Purchase Price:</strong> <?php echo formatCurrency($currentItem['purchase_price']); ?></p>
                            </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label for="quantity">Adjustment Quantity:</label>
                            <input type="number" id="quantity" name="quantity" step="0.01" required>
                            <small>Use positive values to add stock, negative values to remove stock</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="purchase_price">Purchase Price (for additions):</label>
                            <input type="number" id="purchase_price" name="purchase_price" step="0.01" min="0" value="<?php echo $currentItem ? $currentItem['purchase_price'] : '0'; ?>">
                            <small>Only required for stock additions</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="notes">Notes:</label>
                            <textarea id="notes" name="notes" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn">Adjust Inventory</button>
                        <a href="inventory-ui.php?action=inventory" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'update_pricing'): ?>
            <!-- Update Inventory Pricing Form -->
            <div class="card">
                <div class="card-header">Update Inventory Pricing</div>
                <div class="card-body">
                    <?php 
                    $entities = getEntities();
                    $products = getProducts();
                    
                    $selectedEntityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : 0;
                    $selectedProductId = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;
                    
                    $currentItem = null;
                    if ($selectedEntityId > 0 && $selectedProductId > 0) {
                        $currentItem = getInventoryItem($selectedEntityId, $selectedProductId);
                    }
                    ?>
                    
                    <form method="post" action="inventory-ui.php?action=update_pricing">
                        <div class="form-group">
                            <label for="entity_id">Entity:</label>
                            <select id="entity_id" name="entity_id" required <?php echo $selectedEntityId > 0 ? 'disabled' : ''; ?>>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>" <?php echo $selectedEntityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                            <?php if ($selectedEntityId > 0): ?>
                                <input type="hidden" name="entity_id" value="<?php echo $selectedEntityId; ?>">
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="product_id">Product:</label>
                            <select id="product_id" name="product_id" required <?php echo $selectedProductId > 0 ? 'disabled' : ''; ?>>
                                <option value="">Select Product</option>
                                <?php foreach ($products as $product): ?>
                                    <option value="<?php echo $product['id']; ?>" <?php echo $selectedProductId == $product['id'] ? 'selected' : ''; ?>><?php echo $product['name']; ?> (<?php echo $product['unit']; ?>)</option>
                                <?php endforeach; ?>
                            </select>
                            <?php if ($selectedProductId > 0): ?>
                                <input type="hidden" name="product_id" value="<?php echo $selectedProductId; ?>">
                            <?php endif; ?>
                        </div>
                        
                        <div class="form-group">
                            <label for="purchase_price">Purchase Price:</label>
                            <input type="number" id="purchase_price" name="purchase_price" step="0.01" min="0.01" required value="<?php echo $currentItem ? $currentItem['purchase_price'] : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="selling_price">Selling Price:</label>
                            <input type="number" id="selling_price" name="selling_price" step="0.01" min="0.01" required value="<?php echo $currentItem ? $currentItem['selling_price'] : ''; ?>">
                        </div>
                        
                        <button type="submit" class="btn">Update Pricing</button>
                        <a href="inventory-ui.php?action=inventory" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'stock_movements'): ?>
            <!-- Stock Movements List -->
            <div class="card">
                <div class="card-header">Stock Movements</div>
                <div class="card-body">
                    <?php 
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    $productId = isset($_GET['product_id']) ? (int)$_GET['product_id'] : null;
                    
                    $entities = getEntities();
                    $products = getProducts();
                    $movements = getStockMovements($entityId, $productId);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="updateStockMovementsFilter()">
                            <option value="">All Entities</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                        
                        <label for="product">Product:</label>
                        <select id="product" onchange="updateStockMovementsFilter()">
                            <option value="">All Products</option>
                            <?php foreach ($products as $product): ?>
                                <option value="<?php echo $product['id']; ?>" <?php echo $productId == $product['id'] ? 'selected' : ''; ?>><?php echo $product['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (count($movements) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Entity</th>
                                    <th>Product</th>
                                    <th>Type</th>
                                    <th>Quantity</th>
                                    <th>Unit Price</th>
                                    <th>Notes</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($movements as $movement): ?>
                                    <tr>
                                        <td><?php echo date('d M Y H:i', strtotime($movement['created_at'])); ?></td>
                                        <td><?php echo $movement['entity_name']; ?></td>
                                        <td><?php echo $movement['product_name']; ?></td>
                                        <td>
                                            <?php 
                                            switch ($movement['movement_type']) {
                                                case 'purchase':
                                                    echo '<span style="color: green;">Purchase</span>';
                                                    break;
                                                case 'sale':
                                                    echo '<span style="color: red;">Sale</span>';
                                                    break;
                                                case 'adjustment':
                                                    if ($movement['quantity'] > 0) {
                                                        echo '<span style="color: green;">Stock In</span>';
                                                    } else {
                                                        echo '<span style="color: red;">Stock Out</span>';
                                                    }
                                                    break;
                                                case 'transfer_in':
                                                    echo '<span style="color: blue;">Transfer In</span>';
                                                    break;
                                                case 'transfer_out':
                                                    echo '<span style="color: orange;">Transfer Out</span>';
                                                    break;
                                            }
                                            ?>
                                        </td>
                                        <td><?php echo number_format($movement['quantity'], 2) . ' ' . $movement['unit']; ?></td>
                                        <td><?php echo formatCurrency($movement['unit_price']); ?></td>
                                        <td><?php echo $movement['notes']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No stock movements found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'transfers'): ?>
            <!-- Money Transfers List -->
            <div class="card">
                <div class="card-header">
                    Money Transfers
                    <a href="inventory-ui.php?action=add_transfer" class="btn btn-sm">Add New Transfer</a>
                </div>
                <div class="card-body">
                    <?php 
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    $entities = getEntities();
                    $transfers = getTransfers($entityId);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="window.location.href='inventory-ui.php?action=transfers&entity_id='+this.value">
                            <option value="">All Entities</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (count($transfers) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Date</th>
                                    <th>From</th>
                                    <th>To</th>
                                    <th>Amount</th>
                                    <th>Description</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transfers as $transfer): ?>
                                    <tr>
                                        <td><?php echo $transfer['id']; ?></td>
                                        <td><?php echo date('d M Y H:i', strtotime($transfer['transfer_date'])); ?></td>
                                        <td><?php echo $transfer['from_entity_name']; ?></td>
                                        <td><?php echo $transfer['to_entity_name']; ?></td>
                                        <td><?php echo formatCurrency($transfer['amount']); ?></td>
                                        <td><?php echo $transfer['description']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No transfers found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'add_transfer'): ?>
            <!-- Add Transfer Form -->
            <div class="card">
                <div class="card-header">Add Money Transfer</div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <form method="post" action="inventory-ui.php?action=transfer_money">
                        <div class="form-group">
                            <label for="from_entity_id">From Entity:</label>
                            <select id="from_entity_id" name="from_entity_id" required onchange="updateFromEntityBalance()">
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>" data-balance="<?php echo $entity['balance']; ?>"><?php echo $entity['name']; ?> (Balance: <?php echo formatCurrency($entity['balance']); ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="to_entity_id">To Entity:</label>
                            <select id="to_entity_id" name="to_entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="amount">Amount:</label>
                            <input type="number" id="amount" name="amount" step="0.01" min="0.01" required>
                            <div id="balance-info" style="color: #6c757d; margin-top: 5px;"></div>
                        </div>
                        
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn">Transfer Money</button>
                        <a href="inventory-ui.php?action=transfers" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'sales'): ?>
            <!-- Sales List -->
            <div class="card">
                <div class="card-header">
                    Sales
                    <a href="inventory-ui.php?action=new_sale" class="btn btn-sm">New Sale</a>
                </div>
                <div class="card-body">
                    <?php 
                    $entityId = isset($_GET['entity_id']) ? (int)$_GET['entity_id'] : null;
                    $status = isset($_GET['status']) ? $_GET['status'] : null;
                    
                    $entities = getEntities();
                    $sales = getSales($entityId, $status);
                    ?>
                    
                    <div class="filter-bar">
                        <label for="entity">Entity:</label>
                        <select id="entity" onchange="updateSalesFilter()">
                            <option value="">All Entities</option>
                            <?php foreach ($entities as $entity): ?>
                                <option value="<?php echo $entity['id']; ?>" <?php echo $entityId == $entity['id'] ? 'selected' : ''; ?>><?php echo $entity['name']; ?></option>
                            <?php endforeach; ?>
                        </select>
                        
                        <label for="status">Status:</label>
                        <select id="status" onchange="updateSalesFilter()">
                            <option value="">All Statuses</option>
                            <option value="draft" <?php echo $status == 'draft' ? 'selected' : ''; ?>>Draft</option>
                            <option value="completed" <?php echo $status == 'completed' ? 'selected' : ''; ?>>Completed</option>
                            <option value="cancelled" <?php echo $status == 'cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                        </select>
                    </div>
                    
                    <?php if (count($sales) > 0): ?>
                        <table>
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Date</th>
                                    <th>Seller</th>
                                    <th>Buyer</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($sales as $sale): ?>
                                    <tr>
                                        <td><?php echo $sale['id']; ?></td>
                                        <td><?php echo date('d M Y H:i', strtotime($sale['sale_date'])); ?></td>
                                        <td><?php echo $sale['seller_name']; ?></td>
                                        <td><?php echo $sale['buyer_name']; ?></td>
                                        <td><?php echo formatCurrency($sale['total_amount']); ?></td>
                                        <td>
                                            <span class="status-badge status-<?php echo $sale['status']; ?>">
                                                <?php echo $sale['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <a href="inventory-ui.php?action=view_sale&id=<?php echo $sale['id']; ?>" class="btn btn-sm">View</a>
                                            <?php if ($sale['status'] == 'draft'): ?>
                                                <a href="inventory-ui.php?action=edit_sale&id=<?php echo $sale['id']; ?>" class="btn btn-sm">Edit</a>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <p>No sales found.</p>
                    <?php endif; ?>
                </div>
            </div>
            
        <?php elseif ($action == 'new_sale'): ?>
            <!-- New Sale Form -->
            <div class="card">
                <div class="card-header">New Sale</div>
                <div class="card-body">
                    <?php $entities = getEntities(); ?>
                    
                    <form method="post" action="inventory-ui.php?action=create_sale">
                        <div class="form-group">
                            <label for="seller_entity_id">Seller:</label>
                            <select id="seller_entity_id" name="seller_entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="buyer_entity_id">Buyer:</label>
                            <select id="buyer_entity_id" name="buyer_entity_id" required>
                                <option value="">Select Entity</option>
                                <?php foreach ($entities as $entity): ?>
                                    <option value="<?php echo $entity['id']; ?>"><?php echo $entity['name']; ?> (Balance: <?php echo formatCurrency($entity['balance']); ?>)</option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label for="notes">Notes:</label>
                            <textarea id="notes" name="notes" rows="3"></textarea>
                        </div>
                        
                        <button type="submit" class="btn">Create Sale</button>
                        <a href="inventory-ui.php?action=sales" class="btn btn-secondary">Cancel</a>
                    </form>
                </div>
            </div>
            
        <?php elseif ($action == 'edit_sale'): ?>
            <!-- Edit Sale Form -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $sale = isset($editSale) ? $editSale : getSale($id);
            
            if ($sale && $sale['status'] == 'draft'): 
            ?>
                <div class="card">
                    <div class="card-header">
                        Edit Sale #<?php echo $sale['id']; ?>
                        <div>
                            <form method="post" action="inventory-ui.php?action=complete_sale" style="display: inline;">
                                <input type="hidden" name="sale_id" value="<?php echo $sale['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-success" onclick="return confirm('Are you sure you want to complete this sale?')">Complete Sale</button>
                            </form>
                            <form method="post" action="inventory-ui.php?action=cancel_sale" style="display: inline;">
                                <input type="hidden" name="sale_id" value="<?php echo $sale['id']; ?>">
                                <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to cancel this sale?')">Cancel Sale</button>
                            </form>
                        </div>
                    </div>
                    <div class="card-body">
                        <div style="margin-bottom: 20px;">
                            <strong>Seller:</strong> <?php echo $sale['seller_name']; ?><br>
                            <strong>Buyer:</strong> <?php echo $sale['buyer_name']; ?><br>
                            <strong>Date:</strong> <?php echo date('d M Y H:i', strtotime($sale['sale_date'])); ?><br>
                            <strong>Status:</strong> <span class="status-badge status-<?php echo $sale['status']; ?>"><?php echo $sale['status']; ?></span><br>
                            <strong>Total Amount:</strong> <?php echo formatCurrency($sale['total_amount']); ?><br>
                            <?php if ($sale['notes']): ?>
                                <strong>Notes:</strong> <?php echo $sale['notes']; ?>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Sale Items -->
                        <h3>Sale Items</h3>
                        
                        <?php if (count($sale['items']) > 0): ?>
                            <table>
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Quantity</th>
                                        <th>Unit Price</th>
                                        <th>Total Price</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($sale['items'] as $item): ?>
                                        <tr>
                                            <td><?php echo $item['product_name']; ?></td>
                                            <td><?php echo number_format($item['quantity'], 2) . ' ' . $item['unit']; ?></td>
                                            <td><?php echo formatCurrency($item['unit_price']); ?></td>
                                            <td><?php echo formatCurrency($item['total_price']); ?></td>
                                            <td>
                                                <form method="post" action="inventory-ui.php?action=remove_sale_item">
                                                    <input type="hidden" name="sale_item_id" value="<?php echo $item['id']; ?>">
                                                    <input type="hidden" name="sale_id" value="<?php echo $sale['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to remove this item?')">Remove</button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else: ?>
                            <p>No items in this sale.</p>
                        <?php endif; ?>
                        
                        <!-- Add Item Form -->
                        <h3 style="margin-top: 20px;">Add Item</h3>
                        
                        <?php 
                        $products = getProducts();
                        $inventory = getInventory($sale['seller_entity_id']);
                        
                        // Filter products that have inventory for this seller
                        $availableProducts = array();
                        foreach ($inventory as $item) {
                            if ($item['quantity'] > 0) {
                                $availableProducts[] = array(
                                    'id' => $item['product_id'],
                                    'name' => $item['product_name'],
                                    'unit' => $item['unit'],
                                    'quantity' => $item['quantity'],
                                    'selling_price' => $item['selling_price']
                                );
                            }
                        }
                        ?>
                        
                        <?php if (count($availableProducts) > 0): ?>
                            <form method="post" action="inventory-ui.php?action=add_sale_item">
                                <input type="hidden" name="sale_id" value="<?php echo $sale['id']; ?>">
                                
                                <div class="form-group">
                                    <label for="product_id">Product:</label>
                                    <select id="product_id" name="product_id" required onchange="updateProductInfo()">
                                        <option value="">Select Product</option>
                                        <?php foreach ($availableProducts as $product): ?>
                                            <option value="<?php echo $product['id']; ?>" 
                                                    data-price="<?php echo $product['selling_price']; ?>"
                                                    data-quantity="<?php echo $product['quantity']; ?>"
                                                    data-unit="<?php echo $product['unit']; ?>">
                                                <?php echo $product['name']; ?> (Available: <?php echo number_format($product['quantity'], 2) . ' ' . $product['unit']; ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label for="quantity">Quantity:</label>
                                    <input type="number" id="quantity" name="quantity" step="0.01" min="0.01" required>
                                    <div id="quantity-info" style="color: #6c757d; margin-top: 5px;"></div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="unit_price">Unit Price:</label>
                                    <input type="number" id="unit_price" name="unit_price" step="0.01" min="0.01" required>
                                    <div id="price-info" style="color: #6c757d; margin-top: 5px;"></div>
                                </div>
                                
                                <button type="submit" class="btn">Add Item</button>
                            </form>
                        <?php else: ?>
                            <p>No products with available inventory for this seller.</p>
                        <?php endif; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Sale not found or cannot be edited.</div>
            <?php endif; ?>
            
        <?php elseif ($action == 'view_sale'): ?>
            <!-- View Sale Details -->
            <?php 
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $sale = getSale($id);
            
            if ($sale): 
            ?>
                <div class="card">
                    <div class="card-header">
                        Sale #<?php echo $sale['id']; ?>
                        <a href="inventory-ui.php?action=sales" class="btn btn-sm">Back to Sales</a>
                    </div>
                    <div class="card-body">
                        <div style="margin-bottom: 20px;">
                            <strong>Seller:</strong> <?php echo $sale['seller_name']; ?><br>
                            <strong>Buyer:</strong> <?php echo $sale['buyer_name']; ?><br>
                            <strong>Date:</strong> <?php echo date('d M Y H:i', strtotime($sale['sale_date'])); ?><br>
                            <strong>Status:</strong> <span class="status-badge status-<?php echo $sale['status']; ?>"><?php echo $sale['status']; ?></span><br>
                            <strong>Total Amount:</strong> <?php echo formatCurrency($sale['total_amount']); ?><br>
                            <?php if ($sale['notes']): ?>
                                <strong>Notes:</strong> <?php echo $sale['notes']; ?>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Sale Items -->
                        <h3>Sale Items</h3>
                        
                        <?php if (count($sale['items']) > 0): ?>
                            <table>
                                <thead>
                                    <tr>
                                        <th>Product</th>
                                        <th>Quantity</th>
                                        <th>Unit Price</th>
                                        <th>Total Price</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($sale['items'] as $item): ?>
                                        <tr>
                                            <td><?php echo $item['product_name']; ?></td>
                                            <td><?php echo number_format($item['quantity'], 2) . ' ' . $item['unit']; ?></td>
                                            <td><?php echo formatCurrency($item['unit_price']); ?></td>
                                            <td><?php echo formatCurrency($item['total_price']); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <th colspan="3" style="text-align: right;">Total:</th>
                                        <th><?php echo formatCurrency($sale['total_amount']); ?></th>
                                    </tr>
                                </tfoot>
                            </table>
                        <?php else: ?>
                            <p>No items in this sale.</p>
                        <?php endif; ?>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">Sale not found.</div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<script>
    // Mobile menu toggle
    document.getElementById('menuToggle').addEventListener('click', function() {
        document.getElementById('mainNav').classList.toggle('active');
    });
    
    // Update filter for stock movements
    function updateStockMovementsFilter() {
        const entityId = document.getElementById('entity').value;
        const productId = document.getElementById('product').value;
        
        let url = 'inventory-ui.php?action=stock_movements';
        
        if (entityId) {
            url += '&entity_id=' + entityId;
        }
        
        if (productId) {
            url += '&product_id=' + productId;
        }
        
        window.location.href = url;
    }
    
    // Update filter for sales
    function updateSalesFilter() {
        const entityId = document.getElementById('entity').value;
        const status = document.getElementById('status').value;
        
        let url = 'inventory-ui.php?action=sales';
        
        if (entityId) {
            url += '&entity_id=' + entityId;
        }
        
        if (status) {
            url += '&status=' + status;
        }
        
        window.location.href = url;
    }
    
    // Update from entity balance info
    function updateFromEntityBalance() {
        const select = document.getElementById('from_entity_id');
        const balanceInfo = document.getElementById('balance-info');
        
        if (select.selectedIndex > 0) {
            const option = select.options[select.selectedIndex];
            const balance = option.getAttribute('data-balance');
            
            balanceInfo.textContent = 'Available Balance: ' + formatCurrency(balance);
        } else {
            balanceInfo.textContent = '';
        }
    }
    
    // Update product info for sale item
    function updateProductInfo() {
        const select = document.getElementById('product_id');
        const quantityInfo = document.getElementById('quantity-info');
        const priceInfo = document.getElementById('price-info');
        const quantityInput = document.getElementById('quantity');
        const priceInput = document.getElementById('unit_price');
        
        if (select.selectedIndex > 0) {
            const option = select.options[select.selectedIndex];
            const price = option.getAttribute('data-price');
            const availableQuantity = option.getAttribute('data-quantity');
            const unit = option.getAttribute('data-unit');
            
            quantityInfo.textContent = 'Available: ' + formatNumber(availableQuantity) + ' ' + unit;
            quantityInput.max = availableQuantity;
            
            priceInfo.textContent = 'Suggested Price: ' + formatCurrency(price);
            priceInput.value = price;
        } else {
            quantityInfo.textContent = '';
            priceInfo.textContent = '';
            priceInput.value = '';
        }
    }
    
    // Format currency
    function formatCurrency(amount) {
        return 'Rp ' + formatNumber(amount);
    }
    
    // Format number
    function formatNumber(number) {
        return parseFloat(number).toLocaleString('id-ID');
    }
</script>

</body>
</html>