<?php
// Set timezone to WIB (Western Indonesia Time)
date_default_timezone_set('Asia/Jakarta');

// config.php - Configuration and utility functions
class KerjainApp {
    private $dataFile = 'kerjain_data.json';
    
    public function __construct() {
        // Create data file if it doesn't exist
        if (!file_exists($this->dataFile)) {
            $this->initializeData();
        }
    }
    
    private function initializeData() {
        $defaultData = [
            'profile' => [
                'name' => '',
                'position' => '',
                'company' => '',
                'email' => '',
                'phone' => '',
                'address' => ''
            ],
            'settings' => [
                'workHours' => ['start' => 8, 'end' => 15],
                'workDays' => [
                    'monday' => true,
                    'tuesday' => true,
                    'wednesday' => true,
                    'thursday' => true,
                    'friday' => true,
                    'saturday' => false,
                    'sunday' => false
                ],
                'attendanceFlexibility' => [
                    'checkInEarly' => 60,    // minutes before work start
                    'checkInLate' => 60,     // minutes after work start
                    'checkOutEarly' => 60,   // minutes before work end
                    'checkOutLate' => 0      // unlimited minutes after work end (0 = unlimited)
                ],
                'notifications' => true,
                'autoSave' => true
            ],
            'taskTemplates' => [
                // Recurring tasks that can be auto-assigned
            ],
            'scheduledTasks' => [
                // Tasks scheduled for specific dates from Tasks menu
            ],
            'dailyData' => [],
            'statistics' => [
                'totalWorkDays' => 0,
                'totalDaily3Completed' => 0,
                'totalBonusCompleted' => 0,
                'averageDaily3Rate' => 0,
                'streakDays' => 0
            ]
        ];
        file_put_contents($this->dataFile, json_encode($defaultData, JSON_PRETTY_PRINT));
    }
    
    public function loadData() {
        return json_decode(file_get_contents($this->dataFile), true);
    }
    
    public function saveData($data) {
        return file_put_contents($this->dataFile, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    public function getDayData($date) {
        $data = $this->loadData();
        $dayData = isset($data['dailyData'][$date]) ? $data['dailyData'][$date] : null;
        
        // Auto-assign scheduled tasks for this date
        if ($dayData) {
            $dayData = $this->assignScheduledTasks($date, $dayData);
        }
        
        return $dayData;
    }
    
    public function saveDayData($date, $dayData) {
        $data = $this->loadData();
        $data['dailyData'][$date] = $dayData;
        $data['dailyData'][$date]['lastUpdated'] = date('Y-m-d H:i:s');
        return $this->saveData($data);
    }
    
    public function createScheduledTaskSet($taskSet, $startDate, $days) {
        $data = $this->loadData();
        
        $dates = [];
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', strtotime($startDate . ' +' . $i . ' days'));
            // Only add work days
            if ($this->isWorkDay($date)) {
                $dates[] = $date;
            }
        }
        
        $taskSetId = uniqid('taskset_');
        $data['scheduledTasks'][$taskSetId] = [
            'id' => $taskSetId,
            'title' => $taskSet['setTitle'],
            'type' => 'taskset',
            'dates' => $dates,
            'daily3' => $taskSet['daily3'],
            'daily3Durations' => $taskSet['daily3Durations'],
            'bonusTasks' => $taskSet['bonusTasks'],
            'description' => $taskSet['description'],
            'priority' => $taskSet['priority'],
            'created' => date('Y-m-d H:i:s'),
            'assigned' => []
        ];
        
        return $this->saveData($data);
    }
    
    public function assignScheduledTasks($date, $dayData) {
        $data = $this->loadData();
        $scheduledTasks = $data['scheduledTasks'] ?? [];
        
        foreach ($scheduledTasks as $taskId => $task) {
            if (in_array($date, $task['dates']) && !isset($task['assigned'][$date])) {
                if ($task['type'] === 'taskset') {
                    // Assign Daily 3 tasks
                    for ($i = 0; $i < 3; $i++) {
                        if (!empty($task['daily3'][$i]) && empty($dayData['daily3'][$i])) {
                            $dayData['daily3'][$i] = $task['daily3'][$i];
                            $dayData['expectedDuration'][$i] = $task['daily3Durations'][$i] ?? '';
                        }
                    }
                    
                    // Assign bonus tasks
                    if (!empty($task['bonusTasks'])) {
                        foreach ($task['bonusTasks'] as $bonusTask) {
                            if (!empty($bonusTask)) {
                                $dayData['bonus'][] = $bonusTask;
                                $dayData['completedBonus'][] = false;
                            }
                        }
                    }
                    
                    // Set priority if specified
                    if (!empty($task['priority'])) {
                        $dayData['priority'] = $task['priority'];
                    }
                } else {
                    // Legacy single task assignment
                    if ($task['type'] === 'daily3') {
                        for ($i = 0; $i < 3; $i++) {
                            if (empty($dayData['daily3'][$i])) {
                                $dayData['daily3'][$i] = $task['title'];
                                $dayData['expectedDuration'][$i] = $task['duration'] ?? '';
                                break;
                            }
                        }
                        if ($i === 3) {
                            $dayData['bonus'][] = $task['title'];
                            $dayData['completedBonus'][] = false;
                        }
                    } else {
                        $dayData['bonus'][] = $task['title'];
                        $dayData['completedBonus'][] = false;
                    }
                }
                
                // Mark as assigned
                $data['scheduledTasks'][$taskId]['assigned'][$date] = true;
                $this->saveData($data);
            }
        }
        
        return $dayData;
    }
    
    public function createScheduledTask($title, $type, $startDate, $days, $duration = '', $description = '') {
        $data = $this->loadData();
        
        $dates = [];
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', strtotime($startDate . ' +' . $i . ' days'));
            // Only add work days
            if ($this->isWorkDay($date)) {
                $dates[] = $date;
            }
        }
        
        $taskId = uniqid('task_');
        $data['scheduledTasks'][$taskId] = [
            'id' => $taskId,
            'title' => $title,
            'type' => $type,
            'dates' => $dates,
            'duration' => $duration,
            'description' => $description,
            'created' => date('Y-m-d H:i:s'),
            'assigned' => []
        ];
        
        return $this->saveData($data);
    }
    
    public function getScheduledTasks() {
        $data = $this->loadData();
        return $data['scheduledTasks'] ?? [];
    }
    
    public function deleteScheduledTask($taskId) {
        $data = $this->loadData();
        if (isset($data['scheduledTasks'][$taskId])) {
            unset($data['scheduledTasks'][$taskId]);
            return $this->saveData($data);
        }
        return false;
    }
    
    public function canCheckIn($currentTime, $workStartHour) {
        $data = $this->loadData();
        $flexibility = $data['settings']['attendanceFlexibility'];
        
        $workStart = strtotime(sprintf('%02d:00:00', $workStartHour));
        $earliest = $workStart - ($flexibility['checkInEarly'] * 60);
        $latest = $workStart + ($flexibility['checkInLate'] * 60);
        
        $current = strtotime($currentTime);
        
        return [
            'allowed' => $current >= $earliest && $current <= $latest,
            'earliest' => date('H:i', $earliest),
            'latest' => date('H:i', $latest),
            'current' => date('H:i', $current)
        ];
    }
    
    public function canCheckOut($currentTime, $workEndHour, $checkInTime = null) {
        $data = $this->loadData();
        $flexibility = $data['settings']['attendanceFlexibility'];
        
        $workEnd = strtotime(sprintf('%02d:00:00', $workEndHour));
        $earliest = $workEnd - ($flexibility['checkOutEarly'] * 60);
        
        $current = strtotime($currentTime);
        
        // Check if checked in (must be checked in to check out)
        if (!$checkInTime) {
            return [
                'allowed' => false,
                'reason' => 'Belum absen masuk',
                'earliest' => date('H:i', $earliest),
                'current' => date('H:i', $current)
            ];
        }
        
        // Must work at least minimum hours if checked in late
        $checkInTimestamp = strtotime($checkInTime);
        $minWorkDuration = 4 * 3600; // minimum 4 hours work
        $minCheckOut = $checkInTimestamp + $minWorkDuration;
        
        $allowedTime = max($earliest, $minCheckOut);
        
        return [
            'allowed' => $current >= $allowedTime,
            'earliest' => date('H:i', $allowedTime),
            'current' => date('H:i', $current),
            'minWorkHours' => $checkInTimestamp > strtotime(sprintf('%02d:00:00', $workStartHour))
        ];
    }
    
    public function getDayFocus($date) {
        $dayOfWeek = date('N', strtotime($date)); // 1=Monday, 7=Sunday
        switch($dayOfWeek) {
            case 1: case 3: return 'Cita Cerdas'; // Monday, Wednesday
            case 2: case 4: return 'Arena Kompetisi'; // Tuesday, Thursday
            case 5: return 'Cita Shop, Dezki Print'; // Friday
            default: return 'Hari Libur/Fleksibel';
        }
    }
    
    public function isWorkDay($date) {
        $data = $this->loadData();
        $workDays = $data['settings']['workDays'];
        $dayName = strtolower(date('l', strtotime($date)));
        return isset($workDays[$dayName]) ? $workDays[$dayName] : false;
    }
    
    public function calculateStats() {
        $data = $this->loadData();
        $dailyData = $data['dailyData'];
        
        $totalDays = count($dailyData);
        $totalDaily3 = 0;
        $completedDaily3Total = 0;
        $totalBonus = 0;
        
        foreach ($dailyData as $dayData) {
            if (isset($dayData['completedDaily3'])) {
                $completedDaily3Total += count(array_filter($dayData['completedDaily3']));
                $totalDaily3 += 3;
            }
            if (isset($dayData['completedBonus'])) {
                $totalBonus += count(array_filter($dayData['completedBonus']));
            }
        }
        
        return [
            'totalDays' => $totalDays,
            'daily3CompletionRate' => $totalDaily3 > 0 ? round(($completedDaily3Total / $totalDaily3) * 100, 1) : 0,
            'totalBonusCompleted' => $totalBonus,
            'averageDaily3PerDay' => $totalDays > 0 ? round($completedDaily3Total / $totalDays, 1) : 0,
            'averageBonusPerDay' => $totalDays > 0 ? round($totalBonus / $totalDays, 1) : 0
        ];
    }
}

// Initialize app
$app = new KerjainApp();

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'save_day_data':
            $date = $_POST['date'];
            $dayData = [
                'daily3' => $_POST['daily3'] ?? ['', '', ''],
                'bonus' => $_POST['bonus'] ?? [''],
                'completedDaily3' => array_map('boolval', $_POST['completedDaily3'] ?? [false, false, false]),
                'completedBonus' => array_map('boolval', $_POST['completedBonus'] ?? [false]),
                'reminders' => $_POST['reminders'] ?? '',
                'notes' => $_POST['notes'] ?? '',
                'focus' => $_POST['focus'] ?? '',
                'priority' => $_POST['priority'] ?? 'medium',
                'expectedDuration' => $_POST['expectedDuration'] ?? ['', '', ''],
                'attendance' => [
                    'checkedIn' => $_POST['checkedIn'] === 'true',
                    'checkInTime' => $_POST['checkInTime'] ?? null,
                    'checkOutTime' => $_POST['checkOutTime'] ?? null
                ]
            ];
            $app->saveDayData($date, $dayData);
            echo json_encode(['success' => true]);
            exit;
            
        case 'check_in':
            $currentTime = date('H:i:s');
            $data = $app->loadData();
            $workStartHour = $data['settings']['workHours']['start'];
            
            $checkInValidation = $app->canCheckIn($currentTime, $workStartHour);
            
            if ($checkInValidation['allowed']) {
                echo json_encode([
                    'success' => true,
                    'time' => $currentTime,
                    'message' => 'Absen masuk berhasil!'
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => "Tidak bisa absen masuk sekarang. Jam absen masuk: {$checkInValidation['earliest']} - {$checkInValidation['latest']}. Sekarang: {$checkInValidation['current']}"
                ]);
            }
            exit;
            
        case 'check_out':
            $currentTime = date('H:i:s');
            $checkInTime = $_POST['checkInTime'] ?? null;
            $data = $app->loadData();
            $workEndHour = $data['settings']['workHours']['end'];
            
            $checkOutValidation = $app->canCheckOut($currentTime, $workEndHour, $checkInTime);
            
            if ($checkOutValidation['allowed']) {
                echo json_encode([
                    'success' => true,
                    'time' => $currentTime,
                    'message' => 'Absen keluar berhasil!'
                ]);
            } else {
                $message = isset($checkOutValidation['reason']) 
                    ? $checkOutValidation['reason']
                    : "Tidak bisa absen keluar sekarang. Minimal keluar jam: {$checkOutValidation['earliest']}. Sekarang: {$checkOutValidation['current']}";
                    
                echo json_encode([
                    'success' => false,
                    'message' => $message
                ]);
            }
            exit;
            
        case 'create_scheduled_taskset':
            $taskSet = [
                'setTitle' => $_POST['setTitle'] ?? '',
                'daily3' => [
                    $_POST['daily3_1'] ?? '',
                    $_POST['daily3_2'] ?? '',
                    $_POST['daily3_3'] ?? ''
                ],
                'daily3Durations' => [
                    $_POST['duration_1'] ?? '',
                    $_POST['duration_2'] ?? '',
                    $_POST['duration_3'] ?? ''
                ],
                'bonusTasks' => array_filter($_POST['bonusTasks'] ?? []),
                'description' => $_POST['description'] ?? '',
                'priority' => $_POST['priority'] ?? 'medium'
            ];
            $startDate = $_POST['startDate'] ?? date('Y-m-d');
            $days = intval($_POST['days'] ?? 1);
            
            if (!empty($taskSet['setTitle'])) {
                $app->createScheduledTaskSet($taskSet, $startDate, $days);
                echo json_encode(['success' => true, 'message' => 'Set tugas berhasil dijadwalkan!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Judul set tugas wajib diisi!']);
            }
            exit;
            
        case 'create_scheduled_task':
            $title = $_POST['title'] ?? '';
            $type = $_POST['type'] ?? 'bonus';
            $startDate = $_POST['startDate'] ?? date('Y-m-d');
            $days = intval($_POST['days'] ?? 1);
            $duration = $_POST['duration'] ?? '';
            $description = $_POST['description'] ?? '';
            
            if (!empty($title)) {
                $app->createScheduledTask($title, $type, $startDate, $days, $duration, $description);
                echo json_encode(['success' => true, 'message' => 'Tugas berhasil dijadwalkan!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Judul tugas wajib diisi!']);
            }
            exit;
            
        case 'delete_scheduled_task':
            $taskId = $_POST['taskId'] ?? '';
            if ($app->deleteScheduledTask($taskId)) {
                echo json_encode(['success' => true, 'message' => 'Tugas berhasil dihapus!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Gagal menghapus tugas!']);
            }
            exit;
            
        case 'save_profile':
            $data = $app->loadData();
            $data['profile'] = [
                'name' => $_POST['name'] ?? '',
                'position' => $_POST['position'] ?? '',
                'company' => $_POST['company'] ?? '',
                'email' => $_POST['email'] ?? '',
                'phone' => $_POST['phone'] ?? '',
                'address' => $_POST['address'] ?? ''
            ];
            $app->saveData($data);
            echo json_encode(['success' => true]);
            exit;
            
        case 'save_settings':
            $data = $app->loadData();
            $data['settings']['workHours'] = [
                'start' => intval($_POST['workStart']),
                'end' => intval($_POST['workEnd'])
            ];
            $data['settings']['workDays'] = [
                'monday' => isset($_POST['monday']),
                'tuesday' => isset($_POST['tuesday']),
                'wednesday' => isset($_POST['wednesday']),
                'thursday' => isset($_POST['thursday']),
                'friday' => isset($_POST['friday']),
                'saturday' => isset($_POST['saturday']),
                'sunday' => isset($_POST['sunday'])
            ];
            $data['settings']['attendanceFlexibility'] = [
                'checkInEarly' => intval($_POST['checkInEarly'] ?? 60),
                'checkInLate' => intval($_POST['checkInLate'] ?? 60),
                'checkOutEarly' => intval($_POST['checkOutEarly'] ?? 60),
                'checkOutLate' => intval($_POST['checkOutLate'] ?? 0)
            ];
            $data['settings']['notifications'] = isset($_POST['notifications']);
            $data['settings']['autoSave'] = isset($_POST['autoSave']);
            $app->saveData($data);
            echo json_encode(['success' => true]);
            exit;
            
        case 'export_data':
            $data = $app->loadData();
            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename="kerjain-data-' . date('Y-m-d') . '.json"');
            echo json_encode($data, JSON_PRETTY_PRINT);
            exit;
            
        case 'import_data':
            if (isset($_FILES['importFile']) && $_FILES['importFile']['error'] === UPLOAD_ERR_OK) {
                $importData = json_decode(file_get_contents($_FILES['importFile']['tmp_name']), true);
                if ($importData) {
                    $app->saveData($importData);
                    echo json_encode(['success' => true, 'message' => 'Data berhasil diimport!']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'File JSON tidak valid!']);
                }
            } else {
                echo json_encode(['success' => false, 'message' => 'Error uploading file!']);
            }
            exit;
    }
}

// Get current data
$currentDate = $_GET['date'] ?? date('Y-m-d');
$activeMenu = $_GET['menu'] ?? 'dashboard';
$appData = $app->loadData();
$dayData = $app->getDayData($currentDate);
$stats = $app->calculateStats();
$scheduledTasks = $app->getScheduledTasks();

// Default day data if not exists
if (!$dayData) {
    $dayData = [
        'daily3' => ['', '', ''],
        'bonus' => [''],
        'completedDaily3' => [false, false, false],
        'completedBonus' => [false],
        'reminders' => '',
        'notes' => '',
        'focus' => '',
        'priority' => 'medium',
        'expectedDuration' => ['', '', ''],
        'attendance' => [
            'checkedIn' => false,
            'checkInTime' => null,
            'checkOutTime' => null
        ]
    ];
    
    // Auto-assign scheduled tasks
    $dayData = $app->assignScheduledTasks($currentDate, $dayData);
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kerjain Pro - PHP Edition</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .animate-pulse { animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite; }
        @keyframes pulse { 0%, 100% { opacity: 1; } 50% { opacity: .5; } }
        .sidebar-active { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); color: white; }
        .priority-high { background: #fef2f2; border-color: #fecaca; color: #dc2626; }
        .priority-medium { background: #fffbeb; border-color: #fed7aa; color: #d97706; }
        .priority-low { background: #f0fdf4; border-color: #bbf7d0; color: #16a34a; }
        
        /* Notification styles */
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 1000;
            max-width: 400px;
            padding: 1rem;
            border-radius: 0.5rem;
            color: white;
            font-weight: 500;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            transform: translateX(100%);
            transition: transform 0.3s ease-in-out;
        }
        
        .notification.show {
            transform: translateX(0);
        }
        
        .notification.success {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        }
        
        .notification.error {
            background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
        }
        
        .notification.warning {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        }
    </style>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <!-- Notification Container -->
    <div id="notificationContainer"></div>
    
    <!-- Header -->
    <header class="bg-white shadow-sm border-b border-gray-200">
        <div class="max-w-7xl mx-auto px-4 py-4">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-4">
                    <i class="fas fa-bullseye text-3xl text-indigo-600"></i>
                    <div>
                        <h1 class="text-2xl font-bold text-gray-900">Kerjain Pro</h1>
                        <p class="text-sm text-gray-600">Pengelola Tugas Harian - Advanced PHP Edition</p>
                    </div>
                </div>
                
                <div class="flex items-center space-x-6">
                    <div class="text-right">
                        <div class="flex items-center space-x-2 text-lg font-mono">
                            <i class="fas fa-clock text-gray-600"></i>
                            <span id="currentTime" class="text-gray-900"></span>
                        </div>
                        <p class="text-sm text-gray-600" id="currentDate"></p>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <div class="flex max-w-7xl mx-auto">
        <!-- Sidebar Navigation -->
        <nav class="w-64 bg-white shadow-sm border-r border-gray-200 min-h-screen">
            <div class="p-4">
                <div class="space-y-2">
                    <a href="?menu=dashboard&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'dashboard' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-home w-5 h-5"></i>
                        <span class="font-medium">Dashboard</span>
                    </a>
                    
                    <a href="?menu=tasks&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'tasks' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-tasks w-5 h-5"></i>
                        <span class="font-medium">Tasks</span>
                        <?php if (count($scheduledTasks) > 0): ?>
                            <span class="bg-blue-500 text-white text-xs px-2 py-1 rounded-full"><?php echo count($scheduledTasks); ?></span>
                        <?php endif; ?>
                    </a>
                    
                    <a href="?menu=history&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'history' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-history w-5 h-5"></i>
                        <span class="font-medium">Riwayat</span>
                    </a>
                    
                    <a href="?menu=statistics&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'statistics' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-chart-bar w-5 h-5"></i>
                        <span class="font-medium">Statistik</span>
                    </a>
                    
                    <a href="?menu=profile&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'profile' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-user w-5 h-5"></i>
                        <span class="font-medium">Profil</span>
                    </a>
                    
                    <a href="?menu=settings&date=<?php echo $currentDate; ?>" 
                       class="w-full flex items-center space-x-3 px-4 py-3 rounded-lg transition-colors <?php echo $activeMenu === 'settings' ? 'sidebar-active' : 'text-gray-700 hover:bg-gray-50'; ?>">
                        <i class="fas fa-cog w-5 h-5"></i>
                        <span class="font-medium">Pengaturan</span>
                    </a>
                </div>
            </div>
        </nav>

        <!-- Main Content -->
        <main class="flex-1 p-6">
            <?php if ($activeMenu === 'dashboard'): ?>
                <!-- Dashboard Content -->
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                    <!-- Left Column - Status & Actions -->
                    <div class="space-y-6">
                        <div class="bg-white rounded-xl shadow-sm p-6">
                            <h2 class="text-lg font-semibold text-gray-900 mb-4">Status Hari Ini</h2>
                            
                            <div class="space-y-4">
                                <div class="flex items-center justify-between p-3 bg-blue-50 border border-blue-200 rounded-lg">
                                    <div class="flex items-center space-x-2">
                                        <i class="fas fa-calendar text-blue-600"></i>
                                        <span class="font-medium text-blue-900">
                                            <?php echo date('l, d F Y', strtotime($currentDate)); ?>
                                        </span>
                                    </div>
                                    <input type="date" id="dateSelector" value="<?php echo $currentDate; ?>" 
                                           class="text-sm border border-blue-300 rounded px-2 py-1">
                                </div>

                                <div class="p-4 bg-amber-50 border border-amber-200 rounded-lg">
                                    <div class="flex items-center space-x-2 mb-2">
                                        <i class="fas fa-bullseye text-amber-600"></i>
                                        <span class="font-medium text-amber-900">Fokus</span>
                                    </div>
                                    <p class="text-amber-800 font-medium"><?php echo $app->getDayFocus($currentDate); ?></p>
                                </div>

                                <div class="p-3 border rounded-lg priority-<?php echo $dayData['priority']; ?>">
                                    <div class="flex items-center justify-between">
                                        <span class="font-medium">Prioritas Hari Ini</span>
                                        <select id="priority" class="text-sm border rounded px-2 py-1 bg-white">
                                            <option value="low" <?php echo $dayData['priority'] === 'low' ? 'selected' : ''; ?>>Rendah</option>
                                            <option value="medium" <?php echo $dayData['priority'] === 'medium' ? 'selected' : ''; ?>>Sedang</option>
                                            <option value="high" <?php echo $dayData['priority'] === 'high' ? 'selected' : ''; ?>>Tinggi</option>
                                        </select>
                                    </div>
                                </div>

                                <!-- Attendance with flexible time -->
                                <div class="space-y-3">
                                    <?php if (!$dayData['attendance']['checkedIn']): ?>
                                        <button id="checkInBtn" 
                                                class="w-full flex items-center justify-center space-x-2 bg-green-600 hover:bg-green-700 text-white px-4 py-3 rounded-lg font-medium transition-colors">
                                            <i class="fas fa-sign-in-alt"></i>
                                            <span>Absen Masuk</span>
                                        </button>
                                        <div class="text-xs text-gray-600 text-center">
                                            <?php 
                                            $flexibility = $appData['settings']['attendanceFlexibility'];
                                            $workStart = $appData['settings']['workHours']['start'];
                                            $earliest = date('H:i', strtotime(sprintf('%02d:00', $workStart)) - ($flexibility['checkInEarly'] * 60));
                                            $latest = date('H:i', strtotime(sprintf('%02d:00', $workStart)) + ($flexibility['checkInLate'] * 60));
                                            ?>
                                            Jam absen masuk: <?php echo $earliest; ?> - <?php echo $latest; ?>
                                        </div>
                                    <?php else: ?>
                                        <div class="space-y-3">
                                            <div class="p-3 bg-green-50 border border-green-200 rounded-lg">
                                                <p class="text-sm text-green-800">
                                                    <strong>Masuk:</strong> <?php echo $dayData['attendance']['checkInTime'] ? date('H:i:s', strtotime($dayData['attendance']['checkInTime'])) : '-'; ?>
                                                </p>
                                            </div>
                                            <button id="checkOutBtn" 
                                                    class="w-full flex items-center justify-center space-x-2 bg-red-600 hover:bg-red-700 text-white px-4 py-3 rounded-lg font-medium transition-colors">
                                                <i class="fas fa-sign-out-alt"></i>
                                                <span>Absen Keluar</span>
                                            </button>
                                            <div class="text-xs text-gray-600 text-center">
                                                <?php 
                                                $workEnd = $appData['settings']['workHours']['end'];
                                                $earliestOut = date('H:i', strtotime(sprintf('%02d:00', $workEnd)) - ($flexibility['checkOutEarly'] * 60));
                                                ?>
                                                Minimal keluar jam: <?php echo $earliestOut; ?>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <!-- Progress Card -->
                        <div class="bg-white rounded-xl shadow-sm p-6">
                            <h2 class="text-lg font-semibold text-gray-900 mb-4">Progress Hari Ini</h2>
                            
                            <div class="space-y-4">
                                <div>
                                    <div class="flex justify-between items-center mb-2">
                                        <span class="text-sm font-medium text-gray-700">Daily 3</span>
                                        <span class="text-sm text-gray-600" id="daily3Progress">
                                            <?php echo count(array_filter($dayData['completedDaily3'])); ?>/3
                                        </span>
                                    </div>
                                    <div class="w-full bg-gray-200 rounded-full h-2">
                                        <div id="daily3ProgressBar" class="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                             style="width: <?php echo (count(array_filter($dayData['completedDaily3'])) / 3) * 100; ?>%"></div>
                                    </div>
                                </div>

                                <div class="flex items-center justify-between p-3 bg-amber-50 border border-amber-200 rounded-lg">
                                    <div class="flex items-center space-x-2">
                                        <i class="fas fa-award text-amber-600"></i>
                                        <span class="font-medium text-amber-900">Bonus</span>
                                    </div>
                                    <span class="text-amber-800 font-bold" id="bonusCount">
                                        <?php echo count(array_filter($dayData['completedBonus'])); ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Scheduled Tasks for Today -->
                        <?php 
                        $todayScheduled = [];
                        foreach ($scheduledTasks as $task) {
                            if (in_array($currentDate, $task['dates']) && !isset($task['assigned'][$currentDate])) {
                                $todayScheduled[] = $task;
                            }
                        }
                        if (!empty($todayScheduled)): 
                        ?>
                        <div class="bg-blue-50 border border-blue-200 rounded-xl p-4">
                            <h3 class="text-sm font-semibold text-blue-900 mb-2">
                                <i class="fas fa-calendar-plus text-blue-600 mr-2"></i>
                                Tugas Terjadwal Hari Ini
                            </h3>
                            <div class="space-y-2">
                                <?php foreach ($todayScheduled as $task): ?>
                                    <?php if ($task['type'] === 'taskset'): ?>
                                        <div class="bg-white rounded-lg p-3 border border-blue-100">
                                            <div class="text-sm text-blue-800 font-medium flex items-center mb-2">
                                                <i class="fas fa-layer-group text-blue-600 mr-2"></i>
                                                <?php echo htmlspecialchars($task['title']); ?>
                                                <span class="text-xs text-blue-600 ml-2">(Set Tugas)</span>
                                            </div>
                                            <div class="text-xs text-blue-700 space-y-1">
                                                <?php 
                                                $taskCount = 0;
                                                for ($i = 0; $i < 3; $i++) {
                                                    if (!empty($task['daily3'][$i])) {
                                                        $taskCount++;
                                                        echo '<div class="flex items-center"><i class="fas fa-star text-blue-500 mr-1"></i>' . htmlspecialchars($task['daily3'][$i]) . '</div>';
                                                    }
                                                }
                                                if (!empty($task['bonusTasks'])) {
                                                    foreach ($task['bonusTasks'] as $bonusTask) {
                                                        $taskCount++;
                                                        echo '<div class="flex items-center"><i class="fas fa-award text-amber-500 mr-1"></i>' . htmlspecialchars($bonusTask) . '</div>';
                                                    }
                                                }
                                                ?>
                                                <div class="text-blue-600 font-medium mt-2">
                                                    Total: <?php echo $taskCount; ?> tugas akan ditambahkan
                                                </div>
                                            </div>
                                        </div>
                                    <?php else: ?>
                                        <div class="text-sm text-blue-800 flex items-center">
                                            <i class="fas fa-arrow-right text-blue-600 mr-2"></i>
                                            <?php echo htmlspecialchars($task['title']); ?>
                                            <span class="text-xs text-blue-600 ml-2">(<?php echo $task['type'] === 'daily3' ? 'Daily 3' : 'Bonus'; ?>)</span>
                                        </div>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </div>
                            <p class="text-xs text-blue-700 mt-3 bg-blue-100 rounded p-2">
                                <i class="fas fa-info-circle mr-1"></i>
                                Tugas akan otomatis muncul setelah refresh/save halaman
                            </p>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Middle Column - Daily 3 Tasks -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <div class="flex justify-between items-center mb-4">
                            <h2 class="text-lg font-semibold text-gray-900">Daily 3 - Tugas Utama</h2>
                            <button id="saveBtn" class="flex items-center space-x-1 text-blue-600 hover:text-blue-700 text-sm">
                                <i class="fas fa-save"></i>
                                <span>Simpan</span>
                            </button>
                        </div>
                        
                        <div class="space-y-4" id="daily3Container">
                            <?php for ($i = 0; $i < 3; $i++): ?>
                                <div class="space-y-3 p-4 border border-gray-200 rounded-lg">
                                    <div class="flex items-center space-x-2">
                                        <button class="daily3-checkbox" data-index="<?php echo $i; ?>">
                                            <i class="<?php echo $dayData['completedDaily3'][$i] ? 'fas fa-check-square text-green-600' : 'far fa-square text-gray-400'; ?>"></i>
                                        </button>
                                        <span class="text-sm font-medium text-gray-700">Tugas <?php echo $i + 1; ?></span>
                                    </div>
                                    
                                    <input type="text" class="daily3-task" data-index="<?php echo $i; ?>"
                                           value="<?php echo htmlspecialchars($dayData['daily3'][$i]); ?>"
                                           placeholder="Tugas utama <?php echo $i + 1; ?>..."
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent <?php echo $dayData['completedDaily3'][$i] ? 'bg-green-50 line-through text-green-700' : ''; ?>">
                                    
                                    <input type="text" class="expected-duration" data-index="<?php echo $i; ?>"
                                           value="<?php echo htmlspecialchars($dayData['expectedDuration'][$i]); ?>"
                                           placeholder="Estimasi waktu (misal: 2 jam)"
                                           class="w-full px-3 py-1 text-sm border border-gray-300 rounded focus:ring-1 focus:ring-blue-500">
                                </div>
                            <?php endfor; ?>
                        </div>

                        <!-- Fokus Khusus -->
                        <div class="mt-6 pt-6 border-t border-gray-200">
                            <h3 class="text-md font-medium text-gray-900 mb-3">Fokus Khusus Hari Ini</h3>
                            <input type="text" id="todayFocus" value="<?php echo htmlspecialchars($dayData['focus']); ?>"
                                   placeholder="Fokus atau goal khusus untuk hari ini..."
                                   class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                        </div>

                        <!-- Reminders -->
                        <div class="mt-4">
                            <h3 class="text-md font-medium text-gray-900 mb-3">Pengingat</h3>
                            <textarea id="reminders" placeholder="Catatan dan pengingat..."
                                      class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                      rows="3"><?php echo htmlspecialchars($dayData['reminders']); ?></textarea>
                        </div>
                    </div>

                    <!-- Right Column - Bonus Tasks & Notes -->
                    <div class="space-y-6">
                        <div class="bg-white rounded-xl shadow-sm p-6">
                            <div class="flex justify-between items-center mb-4">
                                <h2 class="text-lg font-semibold text-gray-900">Bonus Tasks</h2>
                                <button id="addBonusBtn" class="flex items-center space-x-1 text-amber-600 hover:text-amber-700 text-sm font-medium">
                                    <i class="fas fa-plus"></i>
                                    <span>Tambah</span>
                                </button>
                            </div>
                            
                            <div class="space-y-3" id="bonusContainer">
                                <?php for ($i = 0; $i < count($dayData['bonus']); $i++): ?>
                                    <div class="flex items-start space-x-2 bonus-task-item">
                                        <button class="bonus-checkbox flex-shrink-0 mt-1" data-index="<?php echo $i; ?>">
                                            <i class="<?php echo $dayData['completedBonus'][$i] ? 'fas fa-check-square text-amber-600' : 'far fa-square text-gray-400'; ?>"></i>
                                        </button>
                                        <input type="text" class="bonus-task" data-index="<?php echo $i; ?>"
                                               value="<?php echo htmlspecialchars($dayData['bonus'][$i]); ?>"
                                               placeholder="Tugas bonus..."
                                               class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent text-sm <?php echo $dayData['completedBonus'][$i] ? 'bg-amber-50 line-through text-amber-700' : ''; ?>">
                                        <?php if (count($dayData['bonus']) > 1): ?>
                                            <button class="remove-bonus flex-shrink-0 text-red-500 hover:text-red-700 mt-1" data-index="<?php echo $i; ?>">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                <?php endfor; ?>
                            </div>
                        </div>

                        <!-- Daily Notes -->
                        <div class="bg-white rounded-xl shadow-sm p-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-3">Catatan Harian</h3>
                            <textarea id="dailyNotes" placeholder="Catatan, refleksi, atau pembelajaran hari ini..."
                                      class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                      rows="5"><?php echo htmlspecialchars($dayData['notes']); ?></textarea>
                        </div>
                    </div>
                </div>

            <?php elseif ($activeMenu === 'tasks'): ?>
                <!-- Enhanced Tasks Content -->
                <div class="space-y-6">
                    <!-- Create Complete Task Set -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h2 class="text-lg font-semibold text-gray-900 mb-6">
                            <i class="fas fa-layer-group text-blue-600 mr-2"></i>
                            Buat Set Tugas Lengkap (Daily 3 + Bonus)
                        </h2>
                        
                        <form id="taskSetForm" class="space-y-6">
                            <!-- Set Title and Basic Info -->
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Nama Set Tugas *</label>
                                    <input type="text" name="setTitle" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                           placeholder="Contoh: Project Alpha - Week 1">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Prioritas</label>
                                    <select name="priority" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                        <option value="low">Rendah</option>
                                        <option value="medium" selected>Sedang</option>
                                        <option value="high">Tinggi</option>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Tanggal Mulai</label>
                                    <input type="date" name="startDate" value="<?php echo date('Y-m-d'); ?>"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Jumlah Hari Kerja</label>
                                    <input type="number" name="days" value="1" min="1" max="30"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                </div>
                            </div>
                            
                            <!-- Daily 3 Tasks -->
                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-6">
                                <h3 class="text-lg font-semibold text-blue-900 mb-4">
                                    <i class="fas fa-star text-blue-600 mr-2"></i>
                                    Daily 3 - Tugas Utama
                                </h3>
                                
                                <div class="space-y-4">
                                    <?php for ($i = 1; $i <= 3; $i++): ?>
                                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4 p-4 bg-white rounded-lg border border-blue-100">
                                            <div class="md:col-span-2">
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Tugas <?php echo $i; ?></label>
                                                <input type="text" name="daily3_<?php echo $i; ?>"
                                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                                       placeholder="Masukkan tugas utama <?php echo $i; ?>">
                                            </div>
                                            <div>
                                                <label class="block text-sm font-medium text-gray-700 mb-2">Estimasi Waktu</label>
                                                <input type="text" name="duration_<?php echo $i; ?>"
                                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                                       placeholder="2 jam, 30 menit">
                                            </div>
                                        </div>
                                    <?php endfor; ?>
                                </div>
                            </div>
                            
                            <!-- Bonus Tasks -->
                            <div class="bg-amber-50 border border-amber-200 rounded-lg p-6">
                                <div class="flex justify-between items-center mb-4">
                                    <h3 class="text-lg font-semibold text-amber-900">
                                        <i class="fas fa-award text-amber-600 mr-2"></i>
                                        Bonus Tasks
                                    </h3>
                                    <button type="button" id="addBonusTaskBtn" 
                                            class="bg-amber-600 hover:bg-amber-700 text-white px-3 py-1 rounded text-sm">
                                        <i class="fas fa-plus mr-1"></i>
                                        Tambah Bonus
                                    </button>
                                </div>
                                
                                <div id="bonusTasksContainer" class="space-y-3">
                                    <div class="flex items-center space-x-2">
                                        <input type="text" name="bonusTasks[]" 
                                               class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                                               placeholder="Bonus task 1 (opsional)">
                                        <button type="button" class="remove-bonus-task text-red-500 hover:text-red-700 px-2 py-2 hidden">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Description -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Deskripsi Set Tugas</label>
                                <textarea name="description" 
                                          class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                          rows="3" placeholder="Deskripsi umum untuk set tugas ini (opsional)"></textarea>
                            </div>
                            
                            <!-- Submit Button -->
                            <div class="flex justify-end space-x-3">
                                <button type="button" onclick="document.getElementById('taskSetForm').reset();" 
                                        class="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                                    Reset Form
                                </button>
                                <button type="submit" 
                                        class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg flex items-center space-x-2">
                                    <i class="fas fa-calendar-plus"></i>
                                    <span>Buat Set Tugas Terjadwal</span>
                                </button>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Single Task Form (Legacy) -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <details class="cursor-pointer">
                            <summary class="text-lg font-semibold text-gray-900 mb-4">
                                <i class="fas fa-plus-circle text-green-600 mr-2"></i>
                                Tambah Tugas Tunggal (Mode Legacy)
                            </summary>
                            
                            <form id="scheduledTaskForm" class="grid grid-cols-1 md:grid-cols-2 gap-6 mt-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Judul Tugas *</label>
                                    <input type="text" name="title" required
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                           placeholder="Masukkan judul tugas">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Tipe Tugas</label>
                                    <select name="type" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                        <option value="daily3">Daily 3 (Tugas Utama)</option>
                                        <option value="bonus">Bonus Task</option>
                                    </select>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Tanggal Mulai</label>
                                    <input type="date" name="startDate" value="<?php echo date('Y-m-d'); ?>"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Jumlah Hari Kerja</label>
                                    <input type="number" name="days" value="1" min="1" max="30"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Estimasi Waktu</label>
                                    <input type="text" name="duration" placeholder="contoh: 2 jam, 30 menit"
                                           class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Deskripsi</label>
                                    <textarea name="description" placeholder="Deskripsi detail tugas (opsional)"
                                              class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                              rows="3"></textarea>
                                </div>
                                
                                <div class="md:col-span-2 flex justify-end">
                                    <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg flex items-center space-x-2">
                                        <i class="fas fa-plus"></i>
                                        <span>Buat Tugas Tunggal</span>
                                    </button>
                                </div>
                            </form>
                        </details>
                    </div>
                    
                    <!-- Existing Scheduled Tasks -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h2 class="text-lg font-semibold text-gray-900 mb-6">Tugas Terjadwal</h2>
                        
                        <?php if (empty($scheduledTasks)): ?>
                            <div class="text-center py-8 text-gray-500">
                                <i class="fas fa-calendar-alt text-4xl mb-4"></i>
                                <p>Belum ada tugas terjadwal.</p>
                                <p class="text-sm">Gunakan form di atas untuk menambah tugas terjadwal.</p>
                            </div>
                        <?php else: ?>
                            <div class="space-y-4">
                                <?php foreach ($scheduledTasks as $taskId => $task): ?>
                                    <div class="border border-gray-200 rounded-lg p-4">
                                        <div class="flex justify-between items-start mb-3">
                                            <div class="flex-1">
                                                <h3 class="font-medium text-gray-900 flex items-center">
                                                    <?php if ($task['type'] === 'taskset'): ?>
                                                        <i class="fas fa-layer-group text-blue-600 mr-2"></i>
                                                        <?php echo htmlspecialchars($task['title']); ?>
                                                        <span class="ml-2 text-xs px-2 py-1 rounded-full bg-blue-100 text-blue-800">Set Tugas</span>
                                                    <?php else: ?>
                                                        <i class="fas fa-task text-gray-600 mr-2"></i>
                                                        <?php echo htmlspecialchars($task['title']); ?>
                                                        <span class="ml-2 text-xs px-2 py-1 rounded-full <?php echo $task['type'] === 'daily3' ? 'bg-blue-100 text-blue-800' : 'bg-amber-100 text-amber-800'; ?>">
                                                            <?php echo $task['type'] === 'daily3' ? 'Daily 3' : 'Bonus'; ?>
                                                        </span>
                                                    <?php endif; ?>
                                                </h3>
                                                
                                                <?php if (!empty($task['description'])): ?>
                                                    <p class="text-sm text-gray-600 mt-1"><?php echo htmlspecialchars($task['description']); ?></p>
                                                <?php endif; ?>
                                                
                                                <?php if ($task['type'] === 'taskset'): ?>
                                                    <!-- Show task set details -->
                                                    <div class="mt-3 grid grid-cols-1 md:grid-cols-2 gap-4">
                                                        <div>
                                                            <h4 class="text-sm font-medium text-blue-900 mb-2">Daily 3:</h4>
                                                            <ul class="text-sm text-gray-700 space-y-1">
                                                                <?php for ($i = 0; $i < 3; $i++): ?>
                                                                    <?php if (!empty($task['daily3'][$i])): ?>
                                                                        <li class="flex items-start">
                                                                            <i class="fas fa-star text-blue-500 mr-2 mt-0.5 text-xs"></i>
                                                                            <span><?php echo htmlspecialchars($task['daily3'][$i]); ?></span>
                                                                            <?php if (!empty($task['daily3Durations'][$i])): ?>
                                                                                <span class="text-xs text-gray-500 ml-2">(<?php echo htmlspecialchars($task['daily3Durations'][$i]); ?>)</span>
                                                                            <?php endif; ?>
                                                                        </li>
                                                                    <?php endif; ?>
                                                                <?php endfor; ?>
                                                            </ul>
                                                        </div>
                                                        
                                                        <?php if (!empty($task['bonusTasks'])): ?>
                                                            <div>
                                                                <h4 class="text-sm font-medium text-amber-900 mb-2">Bonus Tasks:</h4>
                                                                <ul class="text-sm text-gray-700 space-y-1">
                                                                    <?php foreach ($task['bonusTasks'] as $bonusTask): ?>
                                                                        <li class="flex items-start">
                                                                            <i class="fas fa-award text-amber-500 mr-2 mt-0.5 text-xs"></i>
                                                                            <span><?php echo htmlspecialchars($bonusTask); ?></span>
                                                                        </li>
                                                                    <?php endforeach; ?>
                                                                </ul>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php else: ?>
                                                    <!-- Legacy single task -->
                                                    <?php if (!empty($task['duration'])): ?>
                                                        <p class="text-xs text-gray-500 mt-1">
                                                            <i class="fas fa-clock mr-1"></i>
                                                            <?php echo htmlspecialchars($task['duration']); ?>
                                                        </p>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                            </div>
                                            
                                            <button class="delete-scheduled-task text-red-500 hover:text-red-700 ml-4" 
                                                    data-task-id="<?php echo $taskId; ?>">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                        
                                        <div class="flex items-center space-x-4 text-sm text-gray-600">
                                            <span>
                                                <i class="fas fa-calendar mr-1"></i>
                                                <?php echo count($task['dates']); ?> hari kerja
                                            </span>
                                            <span>
                                                <i class="fas fa-play mr-1"></i>
                                                Mulai: <?php echo date('d M Y', strtotime($task['dates'][0])); ?>
                                            </span>
                                            <span>
                                                <i class="fas fa-stop mr-1"></i>
                                                Selesai: <?php echo date('d M Y', strtotime(end($task['dates']))); ?>
                                            </span>
                                            <?php if ($task['type'] === 'taskset' && !empty($task['priority'])): ?>
                                                <span class="text-xs px-2 py-1 rounded-full priority-<?php echo $task['priority']; ?>">
                                                    Prioritas: <?php echo ucfirst($task['priority']); ?>
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <!-- Progress tracking -->
                                        <div class="mt-3">
                                            <?php 
                                            $assignedCount = count($task['assigned'] ?? []);
                                            $totalDates = count($task['dates']);
                                            $progressPercent = $totalDates > 0 ? ($assignedCount / $totalDates) * 100 : 0;
                                            ?>
                                            <div class="flex justify-between items-center mb-1">
                                                <span class="text-xs text-gray-600">Progress: <?php echo $assignedCount; ?>/<?php echo $totalDates; ?> hari</span>
                                                <span class="text-xs text-gray-600"><?php echo round($progressPercent, 1); ?>%</span>
                                            </div>
                                            <div class="w-full bg-gray-200 rounded-full h-2">
                                                <div class="bg-green-600 h-2 rounded-full" style="width: <?php echo $progressPercent; ?>%"></div>
                                            </div>
                                        </div>
                                        
                                        <!-- Show upcoming dates -->
                                        <div class="mt-3">
                                            <details class="text-sm">
                                                <summary class="cursor-pointer text-blue-600 hover:text-blue-700">
                                                    Lihat jadwal lengkap
                                                </summary>
                                                <div class="mt-2 grid grid-cols-4 gap-2">
                                                    <?php foreach ($task['dates'] as $date): ?>
                                                        <div class="text-xs p-2 rounded <?php echo isset($task['assigned'][$date]) ? 'bg-green-100 text-green-800' : ($date === $currentDate ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-600'); ?>">
                                                            <?php echo date('d/m', strtotime($date)); ?>
                                                            <?php if (isset($task['assigned'][$date])): ?>
                                                                <i class="fas fa-check ml-1"></i>
                                                            <?php elseif ($date === $currentDate): ?>
                                                                <i class="fas fa-arrow-left ml-1"></i>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </details>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

            <?php elseif ($activeMenu === 'history'): ?>
                <!-- History Content -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-6">Riwayat Kerja</h2>
                    
                    <div class="space-y-4">
                        <?php 
                        $dailyData = $appData['dailyData'];
                        krsort($dailyData); // Sort by date descending
                        $count = 0;
                        foreach ($dailyData as $date => $data): 
                            if ($count >= 10) break;
                            $count++;
                        ?>
                            <div class="border border-gray-200 rounded-lg p-4">
                                <div class="flex justify-between items-start mb-3">
                                    <div>
                                        <h3 class="font-medium text-gray-900">
                                            <?php echo date('l, d F Y', strtotime($date)); ?>
                                        </h3>
                                        <p class="text-sm text-gray-600">
                                            Daily 3: <?php echo count(array_filter($data['completedDaily3'] ?? [])); ?>/3 | 
                                            Bonus: <?php echo count(array_filter($data['completedBonus'] ?? [])); ?>
                                        </p>
                                    </div>
                                    <a href="?menu=dashboard&date=<?php echo $date; ?>" 
                                       class="text-blue-600 hover:text-blue-700 text-sm">Lihat Detail</a>
                                </div>
                                
                                <div class="space-y-2">
                                    <?php if (isset($data['daily3'])): ?>
                                        <?php for ($i = 0; $i < 3; $i++): ?>
                                            <?php if (!empty($data['daily3'][$i])): ?>
                                                <div class="flex items-center space-x-2 text-sm">
                                                    <i class="<?php echo ($data['completedDaily3'][$i] ?? false) ? 'fas fa-check-circle text-green-600' : 'fas fa-times-circle text-red-500'; ?>"></i>
                                                    <span class="<?php echo ($data['completedDaily3'][$i] ?? false) ? 'line-through text-gray-500' : ''; ?>">
                                                        <?php echo htmlspecialchars($data['daily3'][$i]); ?>
                                                    </span>
                                                </div>
                                            <?php endif; ?>
                                        <?php endfor; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

            <?php elseif ($activeMenu === 'statistics'): ?>
                <!-- Statistics Content -->
                <div class="space-y-6">
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h2 class="text-lg font-semibold text-gray-900 mb-6">Statistik Produktivitas</h2>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-blue-900">Total Hari Kerja</h3>
                                <p class="text-2xl font-bold text-blue-600"><?php echo $stats['totalDays']; ?></p>
                            </div>
                            
                            <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-green-900">Daily 3 Rate</h3>
                                <p class="text-2xl font-bold text-green-600"><?php echo $stats['daily3CompletionRate']; ?>%</p>
                            </div>
                            
                            <div class="bg-amber-50 border border-amber-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-amber-900">Total Bonus</h3>
                                <p class="text-2xl font-bold text-amber-600"><?php echo $stats['totalBonusCompleted']; ?></p>
                            </div>
                            
                            <div class="bg-purple-50 border border-purple-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-purple-900">Rata-rata Daily 3</h3>
                                <p class="text-2xl font-bold text-purple-600"><?php echo $stats['averageDaily3PerDay']; ?>/hari</p>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Scheduled Tasks Statistics -->
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h2 class="text-lg font-semibold text-gray-900 mb-6">Statistik Tugas Terjadwal</h2>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div class="bg-indigo-50 border border-indigo-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-indigo-900">Total Tugas Terjadwal</h3>
                                <p class="text-2xl font-bold text-indigo-600"><?php echo count($scheduledTasks); ?></p>
                            </div>
                            
                            <div class="bg-teal-50 border border-teal-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-teal-900">Tugas Aktif</h3>
                                <p class="text-2xl font-bold text-teal-600">
                                    <?php 
                                    $activeTasks = 0;
                                    foreach ($scheduledTasks as $task) {
                                        if (in_array(date('Y-m-d'), $task['dates']) || max($task['dates']) >= date('Y-m-d')) {
                                            $activeTasks++;
                                        }
                                    }
                                    echo $activeTasks;
                                    ?>
                                </p>
                            </div>
                            
                            <div class="bg-rose-50 border border-rose-200 rounded-lg p-4">
                                <h3 class="text-sm font-medium text-rose-900">Progress Rata-rata</h3>
                                <p class="text-2xl font-bold text-rose-600">
                                    <?php 
                                    $totalProgress = 0;
                                    $taskCount = count($scheduledTasks);
                                    if ($taskCount > 0) {
                                        foreach ($scheduledTasks as $task) {
                                            $assigned = count($task['assigned'] ?? []);
                                            $total = count($task['dates']);
                                            $totalProgress += $total > 0 ? ($assigned / $total) * 100 : 0;
                                        }
                                        echo round($totalProgress / $taskCount, 1);
                                    } else {
                                        echo "0";
                                    }
                                    ?>%
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

            <?php elseif ($activeMenu === 'profile'): ?>
                <!-- Profile Content -->
                <div class="bg-white rounded-xl shadow-sm p-6">
                    <h2 class="text-lg font-semibold text-gray-900 mb-6">Profil</h2>
                    
                    <form id="profileForm" class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Nama Lengkap</label>
                                <input type="text" name="name" value="<?php echo htmlspecialchars($appData['profile']['name']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                       placeholder="Masukkan nama lengkap">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Posisi/Jabatan</label>
                                <input type="text" name="position" value="<?php echo htmlspecialchars($appData['profile']['position']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                       placeholder="Masukkan posisi">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Perusahaan</label>
                                <input type="text" name="company" value="<?php echo htmlspecialchars($appData['profile']['company']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                       placeholder="Masukkan nama perusahaan">
                            </div>
                        </div>
                        
                        <div class="space-y-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                                <input type="email" name="email" value="<?php echo htmlspecialchars($appData['profile']['email']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                       placeholder="email@example.com">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Telepon</label>
                                <input type="tel" name="phone" value="<?php echo htmlspecialchars($appData['profile']['phone']); ?>"
                                       class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                       placeholder="08xx-xxxx-xxxx">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Alamat</label>
                                <textarea name="address" 
                                          class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                                          rows="3" placeholder="Alamat lengkap"><?php echo htmlspecialchars($appData['profile']['address']); ?></textarea>
                            </div>
                        </div>
                        
                        <div class="md:col-span-2">
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg">
                                Simpan Profil
                            </button>
                        </div>
                    </form>
                </div>

            <?php elseif ($activeMenu === 'settings'): ?>
                <!-- Enhanced Settings Content -->
                <div class="space-y-6">
                    <div class="bg-white rounded-xl shadow-sm p-6">
                        <h2 class="text-lg font-semibold text-gray-900 mb-6">Pengaturan</h2>
                        
                        <form id="settingsForm" class="space-y-6">
                            <!-- Work Hours -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-3">Jam Kerja</label>
                                <div class="flex space-x-4">
                                    <div class="flex-1">
                                        <label class="block text-xs text-gray-600 mb-1">Mulai</label>
                                        <select name="workStart" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                            <?php for ($i = 0; $i < 24; $i++): ?>
                                                <option value="<?php echo $i; ?>" <?php echo $appData['settings']['workHours']['start'] == $i ? 'selected' : ''; ?>>
                                                    <?php echo sprintf('%02d:00', $i); ?>
                                                </option>
                                            <?php endfor; ?>
                                        </select>
                                    </div>
                                    <div class="flex-1">
                                        <label class="block text-xs text-gray-600 mb-1">Selesai</label>
                                        <select name="workEnd" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                            <?php for ($i = 0; $i < 24; $i++): ?>
                                                <option value="<?php echo $i; ?>" <?php echo $appData['settings']['workHours']['end'] == $i ? 'selected' : ''; ?>>
                                                    <?php echo sprintf('%02d:00', $i); ?>
                                                </option>
                                            <?php endfor; ?>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Attendance Flexibility -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-3">Fleksibilitas Absensi</label>
                                <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                                    <div>
                                        <label class="block text-xs text-gray-600 mb-1">Masuk Lebih Awal (menit)</label>
                                        <input type="number" name="checkInEarly" min="0" max="180"
                                               value="<?php echo $appData['settings']['attendanceFlexibility']['checkInEarly']; ?>"
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    <div>
                                        <label class="block text-xs text-gray-600 mb-1">Masuk Terlambat (menit)</label>
                                        <input type="number" name="checkInLate" min="0" max="180"
                                               value="<?php echo $appData['settings']['attendanceFlexibility']['checkInLate']; ?>"
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    <div>
                                        <label class="block text-xs text-gray-600 mb-1">Keluar Lebih Awal (menit)</label>
                                        <input type="number" name="checkOutEarly" min="0" max="180"
                                               value="<?php echo $appData['settings']['attendanceFlexibility']['checkOutEarly']; ?>"
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                    </div>
                                    <div>
                                        <label class="block text-xs text-gray-600 mb-1">Keluar Terlambat (0=unlimited)</label>
                                        <input type="number" name="checkOutLate" min="0" max="999"
                                               value="<?php echo $appData['settings']['attendanceFlexibility']['checkOutLate']; ?>"
                                               class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                                    </div>
                                </div>
                                <p class="text-xs text-gray-600 mt-2">
                                    * Contoh: Jam kerja 08:00-15:00, toleransi masuk ±60 menit = bisa absen 07:00-09:00
                                </p>
                            </div>

                            <!-- Work Days -->
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-3">Hari Kerja</label>
                                <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
                                    <?php 
                                    $dayNames = [
                                        'monday' => 'Senin',
                                        'tuesday' => 'Selasa', 
                                        'wednesday' => 'Rabu',
                                        'thursday' => 'Kamis',
                                        'friday' => 'Jumat',
                                        'saturday' => 'Sabtu',
                                        'sunday' => 'Minggu'
                                    ];
                                    foreach ($dayNames as $day => $label): 
                                    ?>
                                        <label class="flex items-center space-x-2">
                                            <input type="checkbox" name="<?php echo $day; ?>" 
                                                   <?php echo $appData['settings']['workDays'][$day] ? 'checked' : ''; ?>
                                                   class="rounded border-gray-300 text-blue-600 focus:ring-blue-500">
                                            <span class="text-sm text-gray-700"><?php echo $label; ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>

                            <!-- Data Management -->
                            <div class="pt-6 border-t border-gray-200">
                                <h3 class="text-md font-medium text-gray-900 mb-4">Manajemen Data</h3>
                                <div class="flex flex-wrap gap-3">
                                    <button type="button" id="exportBtn" class="flex items-center space-x-2 bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg">
                                        <i class="fas fa-download"></i>
                                        <span>Export Data</span>
                                    </button>
                                    
                                    <label class="flex items-center space-x-2 bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg cursor-pointer">
                                        <i class="fas fa-upload"></i>
                                        <span>Import Data</span>
                                        <input type="file" id="importFile" accept=".json" class="hidden">
                                    </label>
                                </div>
                                
                                <p class="text-sm text-gray-600 mt-3">
                                    Data tersimpan dalam file kerjain_data.json. Backup secara berkala untuk keamanan data.
                                </p>
                            </div>

                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg">
                                Simpan Pengaturan
                            </button>
                        </form>
                    </div>
                </div>
            <?php endif; ?>
        </main>
    </div>

    <!-- Checkout Modal -->
    <div id="checkoutModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
        <div class="bg-white rounded-xl max-w-lg w-full p-6">
            <h3 class="text-lg font-semibold text-gray-900 mb-4">Konfirmasi Absen Keluar</h3>
            
            <div id="checkoutContent" class="space-y-4 mb-6">
                <!-- Content will be populated by JavaScript -->
            </div>

            <div class="flex space-x-3">
                <button id="cancelCheckout" class="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                    Kembali
                </button>
                <button id="confirmCheckout" class="flex-1 px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-lg">
                    Konfirmasi Keluar
                </button>
            </div>
        </div>
    </div>

    <script>
        // Global variables
        let currentData = <?php echo json_encode($dayData); ?>;
        let currentDate = '<?php echo $currentDate; ?>';
        
        // Notification system
        function showNotification(message, type = 'success') {
            const container = document.getElementById('notificationContainer');
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.innerHTML = `
                <div class="flex items-center justify-between">
                    <div class="flex items-center space-x-2">
                        <i class="fas ${type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-times-circle' : 'fa-exclamation-triangle'}"></i>
                        <span>${message}</span>
                    </div>
                    <button onclick="this.parentElement.parentElement.remove()" class="text-white hover:text-gray-200">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `;
            
            container.appendChild(notification);
            
            // Show notification
            setTimeout(() => {
                notification.classList.add('show');
            }, 100);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                notification.classList.remove('show');
                setTimeout(() => {
                    if (notification.parentElement) {
                        notification.remove();
                    }
                }, 300);
            }, 5000);
        }
        
        // Update time display with WIB timezone
        function updateTime() {
            const now = new Date();
            // Use device's local time zone
            document.getElementById('currentTime').textContent = now.toLocaleTimeString('id-ID', {
                timeZone: 'Asia/Jakarta',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            document.getElementById('currentDate').textContent = now.toLocaleDateString('id-ID', {
                timeZone: 'Asia/Jakarta',
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric'
            });
        }
        
        setInterval(updateTime, 1000);
        updateTime();

        // Date selector change
        document.getElementById('dateSelector')?.addEventListener('change', function() {
            window.location.href = `?menu=<?php echo $activeMenu; ?>&date=${this.value}`;
        });

        // Save day data function
        function saveDayData() {
            const formData = new FormData();
            formData.append('action', 'save_day_data');
            formData.append('date', currentDate);
            
            // Collect Daily 3 data
            const daily3 = [];
            const completedDaily3 = [];
            const expectedDuration = [];
            
            document.querySelectorAll('.daily3-task').forEach((input, index) => {
                daily3[index] = input.value;
            });
            
            document.querySelectorAll('.daily3-checkbox').forEach((btn, index) => {
                completedDaily3[index] = btn.querySelector('i').classList.contains('fa-check-square');
            });
            
            document.querySelectorAll('.expected-duration').forEach((input, index) => {
                expectedDuration[index] = input.value;
            });
            
            // Collect Bonus data
            const bonus = [];
            const completedBonus = [];
            
            document.querySelectorAll('.bonus-task').forEach((input, index) => {
                bonus[index] = input.value;
            });
            
            document.querySelectorAll('.bonus-checkbox').forEach((btn, index) => {
                completedBonus[index] = btn.querySelector('i').classList.contains('fa-check-square');
            });
            
            // Other data
            formData.append('daily3[]', ...daily3);
            formData.append('bonus[]', ...bonus);
            formData.append('completedDaily3[]', ...completedDaily3);
            formData.append('completedBonus[]', ...completedBonus);
            formData.append('expectedDuration[]', ...expectedDuration);
            formData.append('reminders', document.getElementById('reminders')?.value || '');
            formData.append('notes', document.getElementById('dailyNotes')?.value || '');
            formData.append('focus', document.getElementById('todayFocus')?.value || '');
            formData.append('priority', document.getElementById('priority')?.value || 'medium');
            formData.append('checkedIn', currentData.attendance.checkedIn);
            formData.append('checkInTime', currentData.attendance.checkInTime);
            formData.append('checkOutTime', currentData.attendance.checkOutTime);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update progress displays
                    updateProgressDisplays();
                    
                    // Show save confirmation
                    const saveBtn = document.getElementById('saveBtn');
                    if (saveBtn) {
                        const originalText = saveBtn.innerHTML;
                        saveBtn.innerHTML = '<i class="fas fa-check"></i> <span>Tersimpan!</span>';
                        saveBtn.classList.add('text-green-600');
                        setTimeout(() => {
                            saveBtn.innerHTML = originalText;
                            saveBtn.classList.remove('text-green-600');
                        }, 2000);
                    }
                }
            })
            .catch(error => console.error('Error saving data:', error));
        }
        
        // Update progress displays
        function updateProgressDisplays() {
            const completedDaily3 = document.querySelectorAll('.daily3-checkbox i.fa-check-square').length;
            const completedBonus = document.querySelectorAll('.bonus-checkbox i.fa-check-square').length;
            
            document.getElementById('daily3Progress').textContent = `${completedDaily3}/3`;
            document.getElementById('daily3ProgressBar').style.width = `${(completedDaily3 / 3) * 100}%`;
            document.getElementById('bonusCount').textContent = completedBonus;
        }

        // Daily 3 checkbox handlers
        document.querySelectorAll('.daily3-checkbox').forEach(btn => {
            btn.addEventListener('click', function() {
                const icon = this.querySelector('i');
                const index = this.dataset.index;
                const taskInput = document.querySelector(`.daily3-task[data-index="${index}"]`);
                
                if (icon.classList.contains('fa-check-square')) {
                    icon.className = 'far fa-square text-gray-400';
                    taskInput.classList.remove('bg-green-50', 'line-through', 'text-green-700');
                } else {
                    icon.className = 'fas fa-check-square text-green-600';
                    taskInput.classList.add('bg-green-50', 'line-through', 'text-green-700');
                }
                
                updateProgressDisplays();
                saveDayData();
            });
        });

        // Bonus checkbox handlers
        document.querySelectorAll('.bonus-checkbox').forEach(btn => {
            btn.addEventListener('click', function() {
                const icon = this.querySelector('i');
                const index = this.dataset.index;
                const taskInput = document.querySelector(`.bonus-task[data-index="${index}"]`);
                
                if (icon.classList.contains('fa-check-square')) {
                    icon.className = 'far fa-square text-gray-400';
                    taskInput.classList.remove('bg-amber-50', 'line-through', 'text-amber-700');
                } else {
                    icon.className = 'fas fa-check-square text-amber-600';
                    taskInput.classList.add('bg-amber-50', 'line-through', 'text-amber-700');
                }
                
                updateProgressDisplays();
                saveDayData();
            });
        });

        // Add bonus task
        document.getElementById('addBonusBtn')?.addEventListener('click', function() {
            const container = document.getElementById('bonusContainer');
            const index = container.children.length;
            
            const newTaskHTML = `
                <div class="flex items-start space-x-2 bonus-task-item">
                    <button class="bonus-checkbox flex-shrink-0 mt-1" data-index="${index}">
                        <i class="far fa-square text-gray-400"></i>
                    </button>
                    <input type="text" class="bonus-task" data-index="${index}"
                           placeholder="Tugas bonus..."
                           class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500 focus:border-transparent text-sm">
                    <button class="remove-bonus flex-shrink-0 text-red-500 hover:text-red-700 mt-1" data-index="${index}">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', newTaskHTML);
            
            // Add event listeners to new elements
            const newCheckbox = container.lastElementChild.querySelector('.bonus-checkbox');
            const newRemoveBtn = container.lastElementChild.querySelector('.remove-bonus');
            
            newCheckbox.addEventListener('click', function() {
                const icon = this.querySelector('i');
                const index = this.dataset.index;
                const taskInput = document.querySelector(`.bonus-task[data-index="${index}"]`);
                
                if (icon.classList.contains('fa-check-square')) {
                    icon.className = 'far fa-square text-gray-400';
                    taskInput.classList.remove('bg-amber-50', 'line-through', 'text-amber-700');
                } else {
                    icon.className = 'fas fa-check-square text-amber-600';
                    taskInput.classList.add('bg-amber-50', 'line-through', 'text-amber-700');
                }
                
                updateProgressDisplays();
                saveDayData();
            });
            
            newRemoveBtn.addEventListener('click', function() {
                this.closest('.bonus-task-item').remove();
                saveDayData();
            });
        });

        // Remove bonus task handlers
        document.querySelectorAll('.remove-bonus').forEach(btn => {
            btn.addEventListener('click', function() {
                this.closest('.bonus-task-item').remove();
                saveDayData();
            });
        });

        // Auto-save on input changes
        document.querySelectorAll('input, textarea, select').forEach(element => {
            element.addEventListener('change', saveDayData);
        });

        // Save button
        document.getElementById('saveBtn')?.addEventListener('click', saveDayData);

        // Enhanced Check in/out handlers with validation
        document.getElementById('checkInBtn')?.addEventListener('click', function() {
            if (currentDate !== new Date().toISOString().split('T')[0]) {
                showNotification('Tidak bisa absen untuk hari yang berbeda!', 'error');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'check_in');
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    currentData.attendance.checkedIn = true;
                    currentData.attendance.checkInTime = new Date().toISOString();
                    saveDayData();
                    showNotification(data.message, 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Terjadi kesalahan saat absen masuk', 'error');
            });
        });

        document.getElementById('checkOutBtn')?.addEventListener('click', function() {
            if (currentDate !== new Date().toISOString().split('T')[0]) {
                showNotification('Tidak bisa absen untuk hari yang berbeda!', 'error');
                return;
            }
            
            const formData = new FormData();
            formData.append('action', 'check_out');
            formData.append('checkInTime', currentData.attendance.checkInTime);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showCheckoutModal();
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Terjadi kesalahan saat validasi absen keluar', 'error');
            });
        });

        // Checkout modal functions
        function showCheckoutModal() {
            const modal = document.getElementById('checkoutModal');
            const content = document.getElementById('checkoutContent');
            
            let html = '<div><h4 class="font-medium text-gray-900 mb-2">Daily 3 Tasks:</h4><div class="space-y-1">';
            
            document.querySelectorAll('.daily3-task').forEach((input, index) => {
                const isCompleted = document.querySelector(`.daily3-checkbox[data-index="${index}"] i`).classList.contains('fa-check-square');
                const task = input.value || `Tugas ${index + 1} (kosong)`;
                
                html += `
                    <div class="flex items-center space-x-2">
                        <i class="fas ${isCompleted ? 'fa-check-square text-green-600' : 'fa-square text-red-500'}"></i>
                        <span class="text-sm ${isCompleted ? 'text-green-700' : 'text-red-600'}">${task}</span>
                    </div>
                `;
            });
            
            html += '</div></div>';
            
            // Check if there are bonus tasks
            const bonusTasks = [];
            document.querySelectorAll('.bonus-task').forEach(input => {
                if (input.value.trim()) bonusTasks.push(input);
            });
            
            if (bonusTasks.length > 0) {
                html += '<div><h4 class="font-medium text-gray-900 mb-2">Bonus Tasks:</h4><div class="space-y-1">';
                
                bonusTasks.forEach((input, index) => {
                    const isCompleted = document.querySelector(`.bonus-checkbox[data-index="${input.dataset.index}"] i`).classList.contains('fa-check-square');
                    
                    html += `
                        <div class="flex items-center space-x-2">
                            <i class="fas ${isCompleted ? 'fa-check-square text-amber-600' : 'fa-square text-gray-400'}"></i>
                            <span class="text-sm ${isCompleted ? 'text-amber-700' : 'text-gray-600'}">${input.value}</span>
                        </div>
                    `;
                });
                
                html += '</div></div>';
            }
            
            content.innerHTML = html;
            modal.classList.remove('hidden');
        }

        document.getElementById('cancelCheckout')?.addEventListener('click', function() {
            document.getElementById('checkoutModal').classList.add('hidden');
        });

        document.getElementById('confirmCheckout')?.addEventListener('click', function() {
            currentData.attendance.checkedIn = false;
            currentData.attendance.checkOutTime = new Date().toISOString();
            saveDayData();
            showNotification('Absen keluar berhasil!', 'success');
            setTimeout(() => location.reload(), 1500);
        });

        // Task Set form - Add/Remove bonus tasks
        document.getElementById('addBonusTaskBtn')?.addEventListener('click', function() {
            const container = document.getElementById('bonusTasksContainer');
            const bonusItems = container.children.length;
            
            const newBonusHTML = `
                <div class="flex items-center space-x-2">
                    <input type="text" name="bonusTasks[]" 
                           class="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-amber-500"
                           placeholder="Bonus task ${bonusItems + 1} (opsional)">
                    <button type="button" class="remove-bonus-task text-red-500 hover:text-red-700 px-2 py-2">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', newBonusHTML);
            
            // Add event listener to new remove button
            const newRemoveBtn = container.lastElementChild.querySelector('.remove-bonus-task');
            newRemoveBtn.addEventListener('click', function() {
                this.closest('.flex').remove();
                updateBonusTaskVisibility();
            });
            
            updateBonusTaskVisibility();
        });

        // Update bonus task remove button visibility
        function updateBonusTaskVisibility() {
            const container = document.getElementById('bonusTasksContainer');
            const removeButtons = container.querySelectorAll('.remove-bonus-task');
            
            removeButtons.forEach((btn, index) => {
                if (container.children.length <= 1) {
                    btn.classList.add('hidden');
                } else {
                    btn.classList.remove('hidden');
                }
            });
        }

        // Initial setup for existing remove buttons
        document.querySelectorAll('.remove-bonus-task').forEach(btn => {
            btn.addEventListener('click', function() {
                this.closest('.flex').remove();
                updateBonusTaskVisibility();
            });
        });

        // Task Set form submission
        document.getElementById('taskSetForm')?.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'create_scheduled_taskset');
            
            // Collect bonus tasks
            const bonusInputs = this.querySelectorAll('input[name="bonusTasks[]"]');
            formData.delete('bonusTasks[]'); // Clear existing
            bonusInputs.forEach(input => {
                if (input.value.trim()) {
                    formData.append('bonusTasks[]', input.value.trim());
                }
            });
            
            const submitBtn = this.querySelector('button[type="submit"]');
            setLoadingState(submitBtn, true);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                setLoadingState(submitBtn, false);
                if (data.success) {
                    showNotification(data.message, 'success');
                    this.reset();
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                setLoadingState(submitBtn, false);
                showNotification('Terjadi kesalahan saat membuat set tugas', 'error');
            });
        });

        // Scheduled task form (legacy single task)
        document.getElementById('scheduledTaskForm')?.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'create_scheduled_task');
            
            const submitBtn = this.querySelector('button[type="submit"]');
            setLoadingState(submitBtn, true);
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                setLoadingState(submitBtn, false);
                if (data.success) {
                    showNotification(data.message, 'success');
                    this.reset();
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                setLoadingState(submitBtn, false);
                showNotification('Terjadi kesalahan saat membuat tugas', 'error');
            });
        });

        // Delete scheduled task
        document.querySelectorAll('.delete-scheduled-task').forEach(btn => {
            btn.addEventListener('click', function() {
                if (confirm('Yakin ingin menghapus tugas terjadwal ini?')) {
                    const taskId = this.dataset.taskId;
                    const formData = new FormData();
                    formData.append('action', 'delete_scheduled_task');
                    formData.append('taskId', taskId);
                    
                    fetch('', {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            showNotification(data.message, 'success');
                            setTimeout(() => location.reload(), 1500);
                        } else {
                            showNotification(data.message, 'error');
                        }
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        showNotification('Terjadi kesalahan saat menghapus tugas', 'error');
                    });
                }
            });
        });

        // Profile form
        document.getElementById('profileForm')?.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'save_profile');
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Profil berhasil disimpan!', 'success');
                } else {
                    showNotification('Gagal menyimpan profil!', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Terjadi kesalahan saat menyimpan profil', 'error');
            });
        });

        // Settings form
        document.getElementById('settingsForm')?.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            formData.append('action', 'save_settings');
            
            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification('Pengaturan berhasil disimpan!', 'success');
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification('Gagal menyimpan pengaturan!', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Terjadi kesalahan saat menyimpan pengaturan', 'error');
            });
        });

        // Export data
        document.getElementById('exportBtn')?.addEventListener('click', function() {
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const input = document.createElement('input');
            input.name = 'action';
            input.value = 'export_data';
            form.appendChild(input);
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
            
            showNotification('Data sedang didownload...', 'success');
        });

        // Import data
        document.getElementById('importFile')?.addEventListener('change', function() {
            if (this.files[0]) {
                const formData = new FormData();
                formData.append('action', 'import_data');
                formData.append('importFile', this.files[0]);
                
                fetch('', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    showNotification(data.message, data.success ? 'success' : 'error');
                    if (data.success) {
                        setTimeout(() => location.reload(), 1500);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showNotification('Terjadi kesalahan saat import data', 'error');
                });
            }
        });

        // Priority change handler
        document.getElementById('priority')?.addEventListener('change', function() {
            const priorityDiv = this.closest('.p-3');
            priorityDiv.className = `p-3 border rounded-lg priority-${this.value}`;
            saveDayData();
        });

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl/Cmd + S to save
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                saveDayData();
                showNotification('Data tersimpan!', 'success');
            }
            
            // Ctrl/Cmd + 1-6 for navigation
            if ((e.ctrlKey || e.metaKey) && e.key >= '1' && e.key <= '6') {
                e.preventDefault();
                const menus = ['dashboard', 'tasks', 'history', 'statistics', 'profile', 'settings'];
                const menuIndex = parseInt(e.key) - 1;
                if (menus[menuIndex]) {
                    window.location.href = `?menu=${menus[menuIndex]}&date=${currentDate}`;
                }
            }
        });

        // Auto-refresh every 5 minutes to check for scheduled tasks
        setInterval(() => {
            if (window.location.href.includes('menu=dashboard')) {
                // Only refresh dashboard silently to update scheduled tasks
                const currentUrl = new URL(window.location);
                currentUrl.searchParams.set('_refresh', Date.now());
                // Uncomment line below if you want auto-refresh
                // window.location.href = currentUrl.toString();
            }
        }, 5 * 60 * 1000);

        // Initialize tooltips for better UX
        function initTooltips() {
            const tooltipElements = document.querySelectorAll('[title]');
            tooltipElements.forEach(element => {
                element.addEventListener('mouseenter', function() {
                    const tooltip = document.createElement('div');
                    tooltip.className = 'absolute bg-gray-900 text-white text-xs px-2 py-1 rounded shadow-lg z-50';
                    tooltip.textContent = this.getAttribute('title');
                    
                    const rect = this.getBoundingClientRect();
                    tooltip.style.left = rect.left + 'px';
                    tooltip.style.top = (rect.top - 30) + 'px';
                    
                    document.body.appendChild(tooltip);
                    this.tooltipElement = tooltip;
                    
                    // Clear title to prevent default browser tooltip
                    this.originalTitle = this.getAttribute('title');
                    this.removeAttribute('title');
                });
                
                element.addEventListener('mouseleave', function() {
                    if (this.tooltipElement) {
                        document.body.removeChild(this.tooltipElement);
                        this.tooltipElement = null;
                    }
                    // Restore title
                    if (this.originalTitle) {
                        this.setAttribute('title', this.originalTitle);
                    }
                });
            });
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            initTooltips();
            updateBonusTaskVisibility(); // Initialize bonus task buttons
            
            // Show welcome message for new users
            if (<?php echo empty($appData['profile']['name']) ? 'true' : 'false'; ?>) {
                setTimeout(() => {
                    showNotification('Selamat datang! Lengkapi profil Anda di menu Profil untuk pengalaman yang lebih personal.', 'warning');
                }, 2000);
            }
            
            // Check if there are scheduled tasks for today
            const scheduledTasksToday = <?php echo count($todayScheduled ?? []); ?>;
            if (scheduledTasksToday > 0 && window.location.href.includes('menu=dashboard')) {
                setTimeout(() => {
                    showNotification(`Ada ${scheduledTasksToday} tugas terjadwal untuk hari ini! Refresh halaman untuk melihat tugas yang sudah ditambahkan.`, 'warning');
                }, 3000);
            }
            
            // Show timezone info
            const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
            console.log('🕐 Aplikasi menggunakan timezone:', timezone);
            console.log('🇮🇩 Server timezone: Asia/Jakarta (WIB)');
        });

        // Mobile responsiveness improvements
        function handleMobileView() {
            const isMobile = window.innerWidth < 768;
            const sidebar = document.querySelector('nav');
            
            if (isMobile) {
                // Add mobile menu toggle functionality here if needed
                sidebar.classList.add('mobile-sidebar');
            } else {
                sidebar.classList.remove('mobile-sidebar');
            }
        }

        window.addEventListener('resize', handleMobileView);
        handleMobileView(); // Call on load

        // Performance optimization: Debounce auto-save
        let saveTimeout;
        function debouncedSave() {
            clearTimeout(saveTimeout);
            saveTimeout = setTimeout(() => {
                saveDayData();
            }, 1000); // Save 1 second after user stops typing
        }

        // Replace immediate save with debounced save for input fields
        document.querySelectorAll('input[type="text"], textarea').forEach(element => {
            element.removeEventListener('change', saveDayData);
            element.addEventListener('input', debouncedSave);
        });

        // Add loading states for better UX
        function setLoadingState(element, loading = true) {
            if (loading) {
                element.disabled = true;
                element.classList.add('opacity-50', 'cursor-not-allowed');
                const icon = element.querySelector('i');
                if (icon) {
                    icon.className = 'fas fa-spinner fa-spin';
                }
            } else {
                element.disabled = false;
                element.classList.remove('opacity-50', 'cursor-not-allowed');
            }
        }

        // Enhanced error handling
        window.addEventListener('error', function(e) {
            console.error('JavaScript Error:', e.error);
            showNotification('Terjadi kesalahan pada aplikasi. Silakan refresh halaman.', 'error');
        });

        // Service Worker registration for offline capability (optional)
        if ('serviceWorker' in navigator) {
            // Register service worker for offline functionality
            // This would require a separate service-worker.js file
            // navigator.serviceWorker.register('/service-worker.js');
        }

        console.log('🚀 Kerjain Pro Advanced Edition initialized successfully!');
        console.log('⌨️ Keyboard shortcuts:');
        console.log('- Ctrl/Cmd + S: Save data');
        console.log('- Ctrl/Cmd + 1-6: Navigate between menus');
        console.log('📅 New Features:');
        console.log('- Task Sets: Create 3 Daily 3 + Bonus tasks for multiple days');
        console.log('- Flexible Attendance: ±60 minutes tolerance (configurable)');
        console.log('- WIB Timezone: Asia/Jakarta timezone support');
        console.log('🕐 Current server time (WIB):', new Date().toLocaleString('id-ID', {timeZone: 'Asia/Jakarta'}));
    </script>
</body>
</html>